static char RCSid[] = "$Id: op_simcomb.c,v 1.8 1992/07/26 23:16:57 waite Exp $";
/* Copyright, 1989, The Regents of the University of Colorado */

/*    Often, rules exist in a grammar which have the same grammatic
 * structure, and which differ only on a single literal or identifier.
 * Such rules can be combined to simplify the grammar.
 *
 * Example:
 *
 *	The two rules
 *
 *		xExpr ::= xExpr '+' xExpr.
 *		xExpr ::= xExpr '-' xExpr.
 *
 *	can be combined into one rule by adding a new nonterminal
 *	named xOp as follows:
 *
 *		xExpr ::= xExpr xOp xExpr.
 *		xOp ::= '+' .
 *		xOp ::= '-' .
 *
 * Although the above manipulation results in the number of rules increasing,
 * it is an improvement because writing the attribution grammar for the
 * second set of rules is simpler.
 *
 *    Ideally, cagt would find all such similar rules, and allow the
 * user to combine them all at once. However, determining such groups
 * is a difficult problem that possibly may not be efficiently solvable,
 * and which is not attempted here. Instead, cagt allows the user to specify
 * that any two rules should be combined. Cagt checks the rules for
 * compatibility, prompts the user for an identifier name, and does the
 * operation on those two rules. In this manner, the user guides the overall
 * process of combining similar rules.
 *
 * Routines
 *	1) right_side_one_off - Compares two right sides to see if
 *				they are "one off" (i.e. differing
 *				only by a literal or identifier).
 *	2) one_off_rule - Compares two complete rules.
 *	3) combine_one_off - Accepts rule pairs from the user and
 *			     combines them if possible.
*/



#include <stdio.h>
#include <curses.h>
#include "cagt_config.h"
#include "support.h"
#include "queue.h"
#include "io_curses.h"
#include "gram.h"








public int right_side_one_off(rule1,rule2,diff1,diff2,diff_seen)
   RIGHT_NODE_PTR rule1,
		  rule2;
   RIGHT_NODE_PTR *diff1,
		  *diff2;
   char diff_seen;
/*
 * On Entry:
 *	rule1 contains a pointer to a node on the right side of a
 *	   rule in the EBNF internal structure.
 *	rule2 contains a pointer to a node on the right side of a
 *	   rule in the EBNF internal structure.
 *	diff1 is a pointer to a right_node passed by reference.
 *	diff2 is a pointer to a right_node passed by reference.
 *	diff_seen indicates that a difference has already been
 *	   seen in the right sides.
 *
 * On Exit:
 *	right_side_one_off returns TRUE if the right side of the two rules,
 *         starting from the nodes pointed at by rule1 and rule2 represent
 *	   EBNF subtrees of the same structure that differ only in the
 *	   textual representation of a single literal or nonterminal.
 *	if right_side_one_off returns TRUE then diff1 and diff2 point
 *	   at the differing nodes in the two right sides.
 *
 *
 * RESTRICTION: This routine does not currently treat NULT nodes
 *		specially. Hence, two rules that are one off
 *		except that one has an extra NULT node will
 *		not be returned as such.
*/
   {
   int result;

   if ((rule1 == (RIGHT_NODE_PTR) 0) || (rule2 == (RIGHT_NODE_PTR) 0))
	if ((rule1 == (RIGHT_NODE_PTR) 0) &&
	    (rule2 == (RIGHT_NODE_PTR) 0))
		return(diff_seen);				/* Dead End */
	   else
		 return(FALSE);		   /* Rule structures are different */

   if (rule1->code != rule2->code)
      if ( ((rule1->code != IDNT) && (rule1->code != LITT)) ||
           ((rule2->code != IDNT) && (rule2->code != LITT)) )
         return(FALSE);

   switch (rule1->code) {
	/* Instantly positive */
	case DOTT :
	case RPNT:
	case RBKT:
		{
		result = diff_seen;
		break;
		};

	/* Single continuation */
	case NULT :					    /* Deleted node */
	case  IST :
	case GTRT : 
	case LSST :
		{
		result = right_side_one_off(rule1->next,rule2->next,
					    diff1,diff2,diff_seen);
		break;
		}

	/* Literals and Identifiers */
	case LITT :
	case IDNT:
		{
		/*
		 * The following test works always only because the symbol
		 * table stores textually identical IDNT and LITT separately.
                */
		if (rule1->x.text->symbol_ptr != rule2->x.text->symbol_ptr)
		   if (diff_seen)
			 {
			 result = FALSE;	/* Only want one difference */
			 break;
			 }
		      else
			 {
			 diff_seen = TRUE;	  /* First difference found */
			 *diff1 = rule1;
			 *diff2 = rule2;
			 }
		result = right_side_one_off(rule1->next,rule2->next,
					    diff1,diff2,diff_seen);
		break;
		}

	/* Two continuations */
	case AMPT : 
	case  ATT :
	case DOLT :
	case ASTT : 
	case PLST : 
	case LPNT :
	case LBKT :
		{
		if (result = right_side_one_off(rule1->x.nest,rule2->x.nest,
						diff1,diff2,diff_seen))
			result = right_side_one_off(rule1->next,rule2->next,
						    diff1,diff2,diff_seen);
		break;
		}

	/* Three continuations */
	case BART : 
	case SEPT : 
		{
		if(result = right_side_one_off(rule1->x.infix->left,
				 rule2->x.infix->left,diff1,diff2,diff_seen) )
		   if (result = right_side_one_off(rule1->x.infix->right,
				rule2->x.infix->right,diff1,diff2,diff_seen) )
		      result = right_side_one_off(rule1->next,rule2->next,
						  diff1,diff2,diff_seen);
		break;
		}
	}

   return(result);
   }







public int one_off_rule(rule1,rule2,diff1,diff2)
   LEFT_NODE_PTR rule1,
		 rule2;
   RIGHT_NODE_PTR *diff1,
		  *diff2;
/*
 * On Entry:
 *	rule1 contains a pointer to the left side of a rule in the
 *	   EBNF internal structure.
 *	rule2 contains a pointer to the left side of a rule in the
 *	   EBNF internal structure.
 *	Neither left side has it's code field set to NULT.
 *	diff1 is a pointer to a right_node passed by reference.
 *	diff2 is a pointer to a right_node passed by reference.
 * On Exit:
 *	same_left_side returns TRUE if the two rules have the same
 *	   structure and differ by exactly on literal or identifier
 *	   on the right side.
*/
   {
   int result;

   if (result = same_left_side(rule1,rule2))
	result = right_side_one_off(rule1->right_side,rule2->right_side,
				    diff1,diff2,FALSE);
   return(result);
   }







public void combine_one_off(grammar)
   LEFT_NODE_PTR grammar;
/*
 * On Entry:
 *	grammar points at a list of left sides of an EBNF grammar structure.
 *	A valid symbol table for grammar must be active.
 *
 *
 * On Exit:
 *	User specified rules have been combined where possible.
*/
   {
   WINDOW *win;
   int cur_line,
       cur_col;
   LEFT_NODE_PTR rule1,
   		 rule2,
		 bottom_rule;
   RIGHT_NODE_PTR diff1,
		  diff2;
   char combine;
   char first_pass;
   QUEUE_PTR queue;					/* Rules to combine */
   QUEUE_PTR tmp_queue;
   QUEUE_PTR diff_queue;   
   QUEUE_PTR dup_diff_queue;   
   SYMBOL new_nt;
   char non_term = TRUE;



#ifdef lint                             /* Makes lint think cur_col is used */
   cur_col = 1;
   cur_line = cur_col++;
#endif

   /* Get a window to cover up the invocation window */
   win = newwin(0,0,0,0);
   output_centered(win,1,TRUE,"Similar Rule Combination");
   mvwaddstr(win,2,1,LINE);

   queue = init_queue();
   diff_queue = init_queue();


   /* Iterate until we can combine the selected rules */
   do {
      combine = TRUE;
      empty_queue(diff_queue);
      c_select_grammar_rules(grammar,queue,"Similar Rule Selection");
      if (count_queue(queue) < 2) break;
      tmp_queue = dup_queue(queue);
      rule1 = (LEFT_NODE_PTR) dequeue(tmp_queue);
      first_pass = TRUE;
      while (!empty(tmp_queue))
	    {
	    rule2 = (LEFT_NODE_PTR) dequeue(tmp_queue);
	    if (!one_off_rule(rule1,rule2,&diff1,&diff2))
	       {
	       combine = FALSE;
	       cur_line = 4;
	       cur_col = START_COL;
	       (void) wmove(win,4,START_COL);
	       (void) wclrtobot(win);
	       (void) wmove(win,4,START_COL);
	       c_output_rule_pair(win,rule1,rule2, ((RIGHT_NODE_PTR) 0),
						   ((RIGHT_NODE_PTR) 0));
	       box(win,'|','-');
	       getyx(win,cur_line,cur_col);
	       message_and_pause(win,(cur_line + 3),
					   "These rules cannot be combined.");
	       break;
	       }
	     if (first_pass)
		{
		enqueue(diff_queue,((void *) diff1));
		first_pass = FALSE;
		}
	     enqueue(diff_queue,((void *) diff2));
	     }
      delete_queue(tmp_queue);
      } while (!combine);


   /* If not enough rules selected, quietly quit */
   if (count_queue(queue) < 2) return;


   /* Find bottom rule - New rules have to go at bottom */
   bottom_rule = grammar;
   while (bottom_rule->next_rule) bottom_rule = bottom_rule->next_rule;


   /* Show all the difference nodes */
   dup_diff_queue = dup_queue(diff_queue);
   cur_col = START_COL;
   cur_line = 4;
   (void) wmove(win,cur_line,cur_col);
   (void) wclrtobot(win);
   cur_col = 3;
   (void) wmove(win,cur_line,cur_col);
   while (!empty(dup_diff_queue))
	{
	diff1 = (RIGHT_NODE_PTR) dequeue(dup_diff_queue);
	if ((cur_col + diff1->x.text->symbol_ptr->l) > COLS)
		{
		cur_line++;
		cur_col = 3;
		(void) wmove(win,cur_line,cur_col);
		}
	cur_col += c_prtsym(win,diff1->x.text->symbol_ptr) + 5;
	(void) wmove(win,cur_line,cur_col);
	}
   delete_queue(dup_diff_queue);

   /* Get an identifier from the user */
   mvwaddstr(win,(cur_line + 2),START_COL,
		"Enter a name to be used in place of the indicated items > ");
   box(win,'|','-');
   (void) wrefresh(win);
   new_nt = get_identifier(win,TRUE, FALSE, (SYMBOL) 0,&non_term);


   /* Combine the rules */
   rule1 = (LEFT_NODE_PTR) dequeue(queue);
   diff1 = (RIGHT_NODE_PTR) dequeue(diff_queue);
   while (!empty(queue))
	{
	rule2 = (LEFT_NODE_PTR) dequeue(queue);
	diff2 = (RIGHT_NODE_PTR) dequeue(diff_queue);
	/* Combine rules and add combination generated rules */
	if (new_nt != diff1->x.text->symbol_ptr)
		{
		add_comb_gen_rule(bottom_rule,new_nt,diff1);
		bottom_rule = bottom_rule->next_rule;
		}
	if (new_nt != diff2->x.text->symbol_ptr)
		{
		add_comb_gen_rule(bottom_rule,new_nt,diff2);
		bottom_rule = bottom_rule->next_rule;
		}
	change_idnt_or_litt(diff1,new_nt,IDNT);
	change_idnt_or_litt(diff2,new_nt,IDNT);
	(void) NULT_redundant_rules(grammar);
	}


   delete_queue(queue);
   delete_queue(diff_queue);

   (void) wclear(win);
   (void) wrefresh(win);
   delwin(win);

   }
