/*
 * $Id: assign_errors.c,v 1.1 1992/07/26 18:12:24 tony Exp $
 * Assign errors to their real locations.
 */

/*
 * Input is two files.  First is a list of errors in the standard format:
 *	filename line col message...
 * Second is a list of mapping lines:
 *      fromfile fromline fromcol tofile toline tocol
 * Output on standard output is a list of errors in standard format based on
 * the input errors where every reference to a line and col in a tofile has
 * been mapped to the corresponding line and col in a fromfile.
 *
 * We assume that the error file has been sorted on the first three
 * fields in that order.  The map file must be sorted on the to* fields.
 */

#include <stdio.h>

/* The current error */
char errfile[BUFSIZ], errmsg[BUFSIZ];
int errline, errcol;

/* The current map entry */
FILE *mapfil;
char fromfile[BUFSIZ], tofile[BUFSIZ];
int fromline, fromcol, toline, tocol;

/* The previous map entry position */
char p_fromfile[BUFSIZ], p_tofile[BUFSIZ];
int p_fromline, p_fromcol, p_toline, p_tocol;

/*
 * read_map_entry
 */

int read_map_entry ()
{
	int ret;

	/* Remember the beginning of the range */
	strcpy (p_fromfile, fromfile);
	p_fromline = fromline;
	p_fromcol = fromcol;
	strcpy (p_tofile, tofile);
	p_toline = toline;
	p_tocol = tocol;

	/* Get the next one */
	ret = fscanf (mapfil, "%s %d %d %s %d %d\n", fromfile,
		      &fromline, &fromcol, tofile, &toline, &tocol);
	if ((ret == EOF) && (p_toline != -1)) {
		toline = -1;
		return 0;
	} else
		return ret;
}

/*
 * read_error_entry
 */

int read_error_entry ()
{
	int ret;

	ret = fscanf (stdin, "%s %d %d", errfile, &errline, &errcol);
	if (ret != EOF) ret = (fgets (errmsg, BUFSIZ, stdin) != NULL);
	return ret;
}

/*
 * output_error
 */

void output_error (filename, line, col, msg)
char *filename, *msg;
int line, col;
{
	printf ("%s %d %d%s", filename, line, col, msg);
}

/*
 * in_range
 */

int in_range ()
{			
	return ((toline == -1)
		|| ((strcmp (p_tofile, errfile) == 0) &&
		    ((strcmp (tofile, errfile) != 0) ||
		     (errline < toline) ||
		     (errline == toline) && (errcol < tocol))));
}

/*
 * main
 */

int main (argc, argv)
int argc;
char *argv[];
{
	int ret = 0;

	if (argc != 2) {
		fprintf (stderr, "usage: assign_errors mapfile <errfile\n");
		exit (1);
	}

	/* Open the map file */
	if ((mapfil = fopen (argv[1], "r")) == (FILE *) NULL) {
		fprintf (stderr, "assign_errors: can't read '%s'\n",
			 argv[1]);
		exit (1);
	}
      
	/* Loop reading errors and mapping them */
	while (read_error_entry () != EOF) {
		while ((ret != EOF) && (!in_range ()))
			ret = read_map_entry ();
		if (ret == EOF) {
			fprintf (stderr, "assign_errors: no map for '%s'\n",
				 errfile);
			exit (1);
		} else
			output_error (p_fromfile, 
				      p_fromline + (errline - p_toline),
				      errcol + (errline == p_toline ?
						p_fromcol - p_tocol :
						0),
				      errmsg);
	}

	/* Close the map file */
	fclose (mapfil);
	
	exit (0);
}

 
