static char RcsId[] = "$Id: obstack.c,v 1.6 1993/02/25 21:20:14 kadhim Exp $";
/* obstack.c - subroutines used implicitly by object stack macros
   Copyright (C) 1988 Free Software Foundation, Inc.

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 1, or (at your option) any
later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

In other words, you are welcome to use, share and improve this program.
You are forbidden to forbid anyone else to use, share and improve
what you give them.   Help stamp out software-hoarding!  */


#include <stdlib.h>
#include "obstack.h"

#define POINTER void *

/* Size of a page in virtual memory */
#ifndef VIRTUAL_PAGE_SIZE
#define VIRTUAL_PAGE_SIZE 4096
#endif

/* Size of the overhead information required by the storage allocator */
#ifndef ALLOCATOR_OVERHEAD
#define ALLOCATOR_OVERHEAD 4
#endif

/* Determine default alignment.  */
struct fooalign {char x; double d;};
#define DEFAULT_ALIGNMENT ((char *)&((struct fooalign *) 0)->d - (char *)0)
/* If malloc were really smart, it would round addresses to DEFAULT_ALIGNMENT.
   But in fact it might be less smart and round addresses to as much as
   DEFAULT_ROUNDING.  So we prepare for it to do that.  */
union fooround {long x; double d;};
#define DEFAULT_ROUNDING (sizeof (union fooround))

/* The non-GNU-C macros copy the obstack address into this global variable
   to avoid multiple evaluation.  */

ObstackP _obstack;

/* Initialize an obstack H for use.  Specify chunk size SIZE (0 means default).
   Objects start on multiples of ALIGNMENT (0 means use default).
   CHUNKFUN is the function to use to allocate chunks,
   and FREEFUN the function to free them.  */

#if defined(__cplusplus) || defined(__STDC__)
void
_obstack_begin(
  ObstackP h, int size, int alignment,
    POINTER (*chunkfun) (), POINTER (*expandfun) (), void (*freefun) ())
#else
void
_obstack_begin(h, size, alignment, chunkfun, expandfun, freefun)
ObstackP h; int size; int alignment;
  POINTER (*chunkfun) (); POINTER (*expandfun) (); void (*freefun) ();
#endif
{
  register struct _obstack_chunk* chunk; /* points to new chunk */

  if (alignment == 0) alignment = DEFAULT_ALIGNMENT;
  if (size == 0) {
    int extra = ALLOCATOR_OVERHEAD;
    if (extra < DEFAULT_ROUNDING) extra = DEFAULT_ROUNDING;
    size = VIRTUAL_PAGE_SIZE - extra;
  }

  h->chunkfun = (struct _obstack_chunk * (*)()) chunkfun;
  h->expandfun = (struct _obstack_chunk * (*)()) expandfun;
  h->freefun = freefun;
  h->chunk_size = size;
  h->alignment_mask = alignment - 1;

  chunk	= h->chunk = (*h->chunkfun) (h->chunk_size);
  h->next_free = h->object_base = chunk->contents;
  h->chunk_limit = chunk->limit = (char *) chunk + h->chunk_size;
  chunk->prev = 0;

  /* Guarantee alignment */
  (void)obstack_alloc(h, 0);
}

/* Allocate a new current chunk for the obstack *H
   on the assumption that LENGTH bytes need to be added
   to the current object, or a new object of length LENGTH allocated.
   Copies any partial object from the end of the old chunk
   to the beginning of the new one.  */

#if defined(__cplusplus) || defined(__STDC__)
void *
_obstack_malloc (int length)
#else
void *
_obstack_malloc (length)
int length;
#endif
{
  void *temp = malloc((unsigned)length);

  if (temp) return temp;

  (void)perror("_obstack_malloc"); exit(1);
  return (void *)0;	/* To keep lint happy */
}

#if defined(__cplusplus) || defined(__STDC__)
void *
_obstack_realloc(struct _obstack_chunk *chunk, int length)
#else
void *
_obstack_realloc (chunk, length)
struct _obstack_chunk *chunk; int length;
#endif
{
  void *temp = realloc((void *)chunk, (unsigned)length);

  if (temp) return temp;

  (void)perror("_obstack_realloc"); exit(1);
  return (void *)0;	/* To keep lint happy */
}

#if defined(__cplusplus) || defined(__STDC__)
int
_obstack_newchunk(ObstackP h, int length)
#else
int
_obstack_newchunk (h, length)
     ObstackP h;
     int length;
#endif
{
  register struct _obstack_chunk*	old_chunk = h->chunk;
  register struct _obstack_chunk*	new_chunk;
  long	new_size;
  char *obj_loc = h->object_base;
  int obj_size = h->next_free - h->object_base;
  int Strategy;

  /* Reset the default chunk size if necessary */
  if (h->chunk_size == 0) {
    int extra = ALLOCATOR_OVERHEAD;
    if (extra < DEFAULT_ROUNDING) extra = DEFAULT_ROUNDING;
    h->chunk_size = VIRTUAL_PAGE_SIZE - extra;
  }

  /* Compute size for new chunk.  */
  new_size = (obj_size + length) << 1;
  if (new_size < h->chunk_size)
    new_size = h->chunk_size;

  /* Determine the allocation strategy for the new chunk:
   *   Strategy 0: Allocate a new chunk, move the immature object,
   *     set the previous pointer of the new chunk to the old chunk
   *   Strategy 1: Allocate a new chunk, move the immature object,
   *     set the previous pointer of the new chunk to the previous
   *     pointer of the old chunk, free the old chunk
   *   Strategy 2: Reallocate the old chunk with the new size
   */
  if (obj_loc != old_chunk->contents || !old_chunk->prev) Strategy = 0;
  else if (__PTR_TO_INT((char *)old_chunk) & h->alignment_mask) Strategy = 1;
  else Strategy = 2;

  if (Strategy == 2) {
    new_chunk = h->chunk = (*h->expandfun) (old_chunk, new_size);
    h->next_free = h->object_base = new_chunk->contents;
    h->chunk_limit = new_chunk->limit = (char *) new_chunk + new_size;
  } else {
    new_chunk = h->chunk = (*h->chunkfun) (new_size);
    h->next_free = h->object_base = new_chunk->contents;
    h->chunk_limit = new_chunk->limit = (char *) new_chunk + new_size;
  
    /* Guarantee alignment */
    { int save = h->temp; (void)obstack_alloc(h, 0); h->temp = save; }
  
    /* Move the existing object to the new chunk. */
    (void)memcpy(h->next_free, obj_loc, obj_size);
    h->next_free += obj_size;

    /* Deal with the old chunk appropriately */
    if (Strategy == 0) new_chunk->prev = old_chunk;
    else { new_chunk->prev = old_chunk->prev; (*h->freefun)(old_chunk); }
  }
  return 0;
}

/* Return nonzero if object OBJ has been allocated from obstack H.
   This is here for debugging.
   If you use it in a program, you are probably losing.  */

#if defined(__cplusplus) || defined(__STDC__)
int
_obstack_allocated_p(ObstackP h, POINTER obj)
#else
int
_obstack_allocated_p (h, obj)
     ObstackP h;
     POINTER obj;
#endif
{
  register struct _obstack_chunk*  lp;	/* below addr of any objects in this chunk */
  register struct _obstack_chunk*  plp;	/* point to previous chunk if any */

  lp = (h)->chunk;
  while (lp != 0 && (lp > (struct _obstack_chunk*)obj ||
                     lp->limit < (char *)obj))
    {
      plp = lp -> prev;
      lp = plp;
    }
  return lp != 0;
}

/* Free objects in obstack H, including OBJ and everything allocate
   more recently than OBJ.  If OBJ is zero, free everything in H.  */

#if defined(__cplusplus) || defined(__STDC__)
#undef obstack_free
char *
obstack_free (ObstackP h, POINTER obj)
#else
char *
_obstack_free (h, obj)
     ObstackP h;
     POINTER obj;
#endif
{
  register struct _obstack_chunk*  lp;	/* below addr of any objects in this chunk */
  register struct _obstack_chunk*  plp;	/* point to previous chunk if any */

  lp = (h)->chunk;
  while (lp != 0 && (lp > (struct _obstack_chunk*)obj ||
                     lp->limit < (char *)obj))
    {
      plp = lp -> prev;
      (*h->freefun) (lp);
      lp = plp;
    }
  if (lp)
    {
      (h)->object_base = (h)->next_free = (char *)(obj);
      (h)->chunk_limit = lp->limit;
      (h)->chunk = lp;
    }
  else if (obj != 0)
    /* obj is not in any of the chunks! */
    abort ();
  return (char *)(obj);
}

/* copy the contents of a string into an obstack */

#if defined(__cplusplus) || defined(__STDC__)
void
obstack_strgrow(ObstackP obstk, char *data)
#else
void
obstack_strgrow(obstk, data)
ObstackP obstk; char *data;
#endif
{
  register char c, *p = data;

  if (p) while (c = *p++) obstack_1grow(obstk, c);
}
