;
;; EuLisp (FEEL) Module                  Copyright (C) University Of Bath 1991 
;

;
;; tqueues
;

(defmodule tqueues

  (standard0 utils) ()

  ;
  ;;
  ;;; Object structure
  ;; 
  ;

  (defstruct tqueue ()
    ((semaphore
       initform (make-semaphore)
       reader tqueue-semaphore)
     (threads                         ; ((pred . thread) (pred . thread) ...)
       initform ()
       accessor tqueue-threads)) 
    constructor make-tqueue)

  (export tqueue make-tqueue)

  ;
  ;;
  ;;; Functionality
  ;;
  ;

  ;
  ;; Suspend the current thread onto the given queue closing any
  ;; given semaphores pending the given condition. 
  ;; Returns the non-nil result of the satisfied condition.
  ;

  (defgeneric suspend-to-queue (queue condition . sems))

  ;
  ;; Restart the threads who's conditions are met.
  ;; Returns the queue.
  ;

  (defgeneric dequeue-satisfied-threads (queue))

  ;
  ;; Forcibly dequeue the first thread with the result of applying
  ;; the condition function.
  ;

  (defgeneric dequeue-first-thread (queue))

  (export suspend-to-queue dequeue-satisfied-threads dequeue-first-thread)

  ;
  ;;
  ;;; Basic methods
  ;; 
  ;

  ;
  ;; Utility
  ;

  (defun thread-start-when-ready (th . args)
    ;;(await (eq (thread-state th) 'limbo))
    (apply thread-start th args))

  (defmethod suspend-to-queue ((q tqueue) pred . sems)
    (let ((sem (tqueue-semaphore q)))
      (open-semaphore sem)  ; OPEN
      ((setter tqueue-threads) 
        q (nconc (tqueue-threads q) (list (cons pred (current-thread)))))
      (mapc close-semaphore sems)
      (close-semaphore sem) ; CLOSE
      (car (thread-suspend))))

  (defmethod dequeue-satisfied-threads ((q tqueue))
    (let ((sem (tqueue-semaphore q))
	  (new ()))
      (open-semaphore sem)   ; OPEN
      (mapc
        (lambda (pair)
	  (let ((result ((car pair))))
	    (if (null result) (setq new (nconc new (list pair)))
	      (thread-start-when-ready (cdr pair) result))))
	(tqueue-threads q))
      ((setter tqueue-threads) q new)
      (close-semaphore sem)) ; CLOSE
    q)
      
  (defmethod dequeue-first-thread ((q tqueue))
    (let ((sem (tqueue-semaphore q))
	  (result ()))
      (open-semaphore sem)  ; OPEN
      (let ((ths (tqueue-threads q)))
	(if (null ths) (setq result ())
	  (progn
	    (setq result (cdr (car ths)))
	    (thread-start-when-ready result ((car (car ths)))))))
      (close-semaphore sem) ; CLOSE
      result))

  (defmethod generic-write ((q tqueue) s)
    (format s "#<tqueue: ~a>" (tqueue-threads q))
    s)

  (defmethod generic-prin ((q tqueue) s)
    (format s "#<tqueue: ~a>" (tqueue-threads q))
    s)

)

