;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;  EuLisp Module                     Copyright (C) University of Bath 1991  ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;   EuLisp Module  -   Copyright (C) Codemist and University of Bath 1989   ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;; Name: linda                                                               ;;
;;                                                                           ;;
;; Author: Keith Playford                                                    ;;
;;                                                                           ;;
;; Date: 31 May 1990                                                         ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;

;; Change Log:
;;   Version 1.0 (31/5/90)

;;

(defmodule linda

  (lists
   list-operators
   extras
   arith
   classes
   streams
   threads
   semaphores
   vectors
   calls
   others

   linda-base
   linda-tabs) ()

  ;;

  ;; Parameters...

  ;;

  (deflocal *default-tuple-space-size* 500)

  ;;

  ;; Linda objects...

  ;;

  ;; Tuple space object...

  (defstruct linda-pool linda-object
    ((lock initform (make-semaphore)
	   accessor linda-pool-lock)
     (tuple-table initform (make-linda-tuple-table)
		   accessor linda-pool-tuple-table)
     (max-tuples initform *default-tuple-space-size*
		 initargs (max-tuples)
		 accessor linda-pool-max-tuples)
     (tuple-count initform 0
		  accessor linda-pool-tuple-count)
     (out-blocked initform nil
		  accessor linda-pool-out-blocked))
    constructor make-linda-pool)

  (export make-linda-pool 
	  linda-pool-lock
	  linda-pool-tuple-table
	  linda-pool-max-tuples
	  linda-pool-tuple-count
	  linda-pool-out-blocked)

  ;;

  ;; Basic operations...

  ;;  (linda-out <space> <tuple>)
  ;;  (linda-in <space> <pattern>)
  ;;  (linda-read <space> <pattern>)

  ;;

  ;; 'in'...

  ;;

  (defun linda-in (pool pattern)
    (let ((lock (linda-pool-lock pool)))
      (open-semaphore lock)
      (let ((match (in-match (linda-pool-tuple-table pool) pattern lock)))
	((setter linda-pool-tuple-count) pool
	   (- (linda-pool-tuple-count pool) 1))
	(if (= (linda-pool-tuple-count pool) 
	       (- (linda-pool-max-tuples pool) 1))
	  (progn
	    (let ((blocked (linda-pool-out-blocked pool)))
	      (if (null blocked) nil
		(progn
		  (thread-start (car blocked))
		  ((setter linda-pool-out-blocked) pool (cdr blocked))))))
	  nil)
	(close-semaphore lock)
	(thread-reschedule)
	match)))

  (defun in-match (tab pattern lock)
    (let ((match (tuple-table-in tab pattern)))
      (if (null match)
        ;; Blocked on in...
	(tilnil
;;	  (print "IN-BLOCKED!!!")
	  (close-semaphore lock)
	  (thread-reschedule)
	  (open-semaphore lock)
	  (setq match (tuple-table-in tab pattern))
	  (null match))
	match)))

  ;;

  ;; 'read'

  ;;

  (defun linda-read (pool pattern)
    (let ((lock (linda-pool-lock pool)))
      (open-semaphore lock)
      (let ((match (read-match (linda-pool-tuple-table pool) pattern lock)))
	(close-semaphore lock)
	match)))

  (defun read-match (tab pattern)
    (let ((match (tuple-table-read tab pattern)))
      (if (null match)
        ;; Blocked on read...
        (progn
	  (close-semaphore lock)
	  (thread-reschedule)
	  (open-semaphore lock)
	  (read-match tab pattern))
	match)))

  ;;

  ;; 'out'...

  ;;

  (defun linda-out (pool tuple)
    (let ((lock (linda-pool-lock pool)))
      (open-semaphore lock)
      (cond ((= (linda-pool-tuple-count pool) (linda-pool-max-tuples pool))
	       ((setter linda-pool-out-blocked) pool
		 (nconc (linda-pool-out-blocked pool) 
			(list (current-thread))))
	       (close-semaphore lock)
	       (print "OUT-BLOCKED")
	       (thread-suspend)
	       ;; Restarted...
	       (out pool tuple))
	    (t (tuple-table-out (linda-pool-tuple-table pool) tuple)
	       ((setter linda-pool-tuple-count) pool
		 (+ (linda-pool-tuple-count pool) 1))
	       (close-semaphore lock)
	       (thread-reschedule)	
	       tuple))))

  (export linda-out linda-in linda-read)

  ;;

  ;; Scheduling malarky...

  ;;

  (deflocal scheduler-active-flag nil)

  (defun linda-scheduler-active-p () scheduler-active-flag)

  (export linda-scheduler-active-p)

  (deflocal process-queue nil)

  (defun linda-queue-process (pair)
    (setq process-queue (nconc process-queue (list pair)))
    (car pair))

  (export linda-queue-process)

  (defmacro linda-start (fun . args)
    `(let ((\@thread\@ (make-thread ,fun)))
       (if (linda-scheduler-active-p)
	 (thread-start \@thread\@ ,@args)
	 (linda-queue-process (cons \@thread\@ ,args)))
       \@thread\@))

  (export linda-start)

  (defun linda-scheduler () 
    (print "Linda scheduler started")
;;    (print process-queue)
    (setq scheduler-active-flag t)
    (linda-scheduler-aux process-queue))

  (defun linda-scheduler-aux (ll)
    (if (null ll) (thread-suspend)
      (progn
	(apply thread-start (car ll))
	(linda-scheduler-aux (cdr ll)))))

  (export linda-scheduler)

  ;;

  ;; Sundry exportations...

  ;;

  ;; (export make-linda-tuple tuple *vector-size* *linda-wild-card*)

)
