;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;   EuLisp Module  -   Copyright (C) Codemist and University of Bath 1990   ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;; Name: defs                                                                ;;
;;                                                                           ;;
;; Author: Keith Playford                                                    ;;
;;                                                                           ;;
;; Date: 21 August 1990                                                      ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;

;; Change Log:
;;   Version 1.0 (21/8/90)

;;

;; Lisp version of defclass... */

(defmodule odefs

  (standard) ()

  ;; 'defstruct'...

  ;; Utils... 

  (defconstant *key-list-fail* nil)

  (defconstant *nothing* (gensym))

  (defun search-key-list (l k)
    (cond ((null l) *key-list-fail*)
	  ((eqcar l k) (cadr l))
	  (t (search-key-list (cddr l) k))))

  (defcondition invalid-slot-options () options nil)

  (deflocal *name* nil)
  (deflocal *readers* nil)
  (deflocal *writers* nil)
  (deflocal *accessors* nil)

  (defun canonicalise (ops)
    (when (symbolp ops) (setq ops (list ops)))
    (unless (consp ops) (error "slot options not a list"
			       invalid-slot-options 'options ops))
    (let ((name *nothing*)
	  (slot-class *nothing*)
	  (slot-initargs *nothing*)
	  (initform *nothing*)
	  (initargs nil)
	  (readers nil)
	  (writers nil)
	  (accessors nil))
      (labels
        ((inner (l)
	  (unless (null l) 
	    (let ((key (car l)) 
		  (val (cadr l)))
	      (cond ((eq key 'initarg)
		      (setq initargs (nconc initargs (list val))))
		    ((eq key 'initform)
		      (if (eq initform *nothing*)
			  (setq initform `(lambda () ,val))
			(error "bad initform"
			       invalid-slot-options 'options ops)))
		    ((eq key 'slot-class) 
		      (if (eq slot-class *nothing*)
			  (setq class val)
			(error "slot-class multiply defined"
			       invalid-slot-options 'options ops)))
		    ((eq key 'slot-initargs)
		      (if (eq slot-initargs *nothing)
			  (setq class-initargs val)
			(error "slot initargs multiply defined"
			       invalid-slot-options 'options ops)))
		    ((eq key 'reader)
		      (setq readers (cons (cons val name) readers)))
		    ((eq key 'writer)
		      (setq writers (cons (cons val name) writers)))
		    ((eq key 'accessor)
		      (setq accessors (cons (cons val name) accessors)))
		    (t (error "unknown slot option"
			      invalid-slot-options 'options ops))))
	    (inner (cddr l)))))
	(setq name (car ops))
	(inner (cdr ops))
	(setq *readers* (nconc readers *readers*))
	(setq *writers* (nconc writers *writers*))
	(setq *accessors* (nconc accessors *accessors*))
	(when (eq slot-class *nothing*) 
	      (setq slot-class 'local-slot-description))
	(when (eq slot-initargs *nothing*)
	      (setq slot-initargs nil))
	(nconc `(list 'name          ',name 
		      'slot-class    ,slot-class 
		      'slot-initargs ,slot-initargs
		      'initargs      ',initargs)
	       (if (eq initform *nothing*) nil `('initform initform))))))

  (defun reader-defs (o) 
    (mapcar 
      (lambda (pair) 
	`(defconstant ,(car pair) (make-reader ,*name* ',(cdr pair))))
      *readers*))

  (defun writer-defs (o) 
    (mapcar 
      (lambda (pair) 
	`(defconstant ,(car pair) (make-writer ,*name* ',(cdr pair))))
      *writers*))

  (defun accessor-defs (o) 
    (mapcar 
      (lambda (pair) 
	`(progn
	   (defconstant ,(car pair) (make-reader ,*name* ',(cdr pair)))
	   ((setter setter) ,(car pair) (make-writer ,*name* ',(cdr pair)))))
      *accessors*))

  (defmacro ldefstruct (name super slot-ops . class-ops)
    (setq *name* name)
    `(progn
       (defconstant ,name
	 (make-instance structure-class
	   'name ',name
	   'direct-superclasses ,(if super `(list super) '(list structure)) 
	   'direct-slot-descriptions
	     (list ,@(mapcar canonicalise slot-ops))
	   'metaclass-hypotheses nil))
       ,@(reader-defs slot-ops)
       ,@(writer-defs slot-ops)
       ,@(accessor-defs slot-ops)
       ',name))

  (export ldefstruct)

  (defmacro ldefclass (name supers slot-ops . class-ops)
    (setq *name* name)
    (let ((metaclass
	    (or (search-key-list class-ops 'metaclass) 'class))
	  (initargs 
	    (or (search-key-list class-ops 'metaclass-initargs) nil))
	  (readers)
	  (writers)
	  (accessors))
      `(progn
	 (defconstant ,name
	   (make-instance ,metaclass
	     'name ',name
	     'direct-superclasses ,(if supers `(list ,@supers) '(list object))
	     'direct-slot-descriptions
	       (list ,@(mapcar canonicalise slot-ops))
	     'metaclass-hypotheses nil))
	 ,@(reader-defs slot-ops)
	 ,@(writer-defs slot-ops)
	 ,@(accessor-defs slot-ops)
	 ',name)))

  (export ldefclass)

  (defmacro defreader (name class slot)
    `(defconstant ,name (make-reader ,class ',slot)))

  (defmacro defwriter (name class slot)
    `(defconstant ,name (make-writer ,class ',slot)))

  (defmacro defaccessor (name class slot)
    `(progn
       (defconstant ,name (make-reader ,class ',slot))
       ((setter setter) ,name (make-writer ,class ',slot))))

  (export defreader defwriter defaccessor)

  (defun sll-signature (ll)
    (cond ((not (consp ll)) nil)
	  ((consp (car ll)) (cons (cadar ll) (sll-signature (cdr ll))))
	  (t (cons 'object (sll-signature (cdr ll))))))

  (defun sll-formals (ll)
    (cond ((not (consp ll)) nil)
	  ((consp (car ll)) (cons (caar ll) (sll-formals (cdr ll))))
	  (t (cons (car ll) (sll-formals (cdr ll))))))

  (defun gf-class (ops)
    (let ((val (search-key-list ops 'class)))
      (if (eq val *key-list-fail*) 'generic-function val)))

  (defun gf-method-class (ops)
    (let ((val (search-key-list ops 'method-class)))
      (if (eq val *key-list-fail*) 'method val)))
	
  (defun gf-methods (ops mc)
    (let ((val (search-key-list ops 'methods)))
      (if (eq val *key-list-fail*) nil
	`(list
	    ,@(mapcar
	        (lambda (form)
	          `(make-instance ,mc
		     'signature (list ,@(sll-signature (car form)))
		     'function
		     (lambda (***method-args-handle***
			      ***method-status-handle***
			      ,@(sll-formals (car form)))
		       ,@(cdr form)))) 
		val)))))

  (defmacro ldefgeneric (name ll . ops)
    `(defconstant ,name
       (make-instance ,(gf-class ops)
	  'name ',name
          'lambda-list ',ll
	  'method-class ,(gf-method-class ops)
	  'methods ,(gf-methods ops (gf-method-class ops)))))

  (defmacro ldefmethod (name sll . body)
    `(progn
       (add-method 
	 ,name
	 (make-instance (generic-function-method-class ,name)
			'signature (list ,@(sll-signature sll))
			'function
			  (lambda (***method-args-handle***
				   ***method-status-handle***
				   ,@(sll-formals sll)) 
			    ,@body)))))

  (defclass lockable-gf (generic-function)
    ((lock initarg lock
	   initform nil
	   accessor lockable-gf-lock))
    metaclass generic-class)
    
)

