;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;  EuLisp Module                     Copyright (C) University of Bath 1991  ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;   EuLisp Module  -   Copyright (C) Codemist and University of Bath 1989   ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;; Name: futures                                                             ;;
;;                                                                           ;;
;; Author: Keith Playford                                                    ;;
;;                                                                           ;;
;; Date: 20 May 1990                                                         ;;
;;                                                                           ;;
;; Description: Eager evaluating futures using the EuLisp thread mechanism   ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;

;; Change Log:
;;   Version 1.0 (20/5/90)

;;

(defmodule futures

  (standard0) ()

  ;;
  ;; Book-keeping...
  ;;

  (deflocal future-count-value 0)

  (defun future-count () future-count-value)
  (defun set-future-count (n) (setq future-count-value n))
  ((setter setter) future-count set-future-count)

  (defun increment-future-count () 
    (setq future-count-value (+ future-count-value 1)))

  (defun zero-future-count () (setq future-count-value 0))

  (export future-count set-future-count 
          increment-future-count zero-future-count)

  ;;
  ;; Future structure...
  ;;

  (defstruct future-object ()
    ((function 
        accessor future-object-function)
     (thread 
        accessor future-object-thread)
     (value 
        accessor future-object-value)
     (done  
        initform nil
	accessor future-object-done))
    constructor make-future-object)

  (export future-object future-object-value future-object-function
	  future-object-done make-future-object future-object-thread)

  ;;
  ;; Predicate...
  ;;

  (defgeneric futurep (obj))

  (defmethod futurep ((obj object)) nil)
  (defmethod futurep ((f future-object)) t)

  (export futurep)

  ;;
  ;; Future macro...
  ;;

  (defmacro future exp
    `(let 
       ((@@future@@ (make-future-object))
	(@@task@@ (make-thread 
	            (lambda (future fun)
		      ((setter future-object-value) future (fun))
		      ((setter future-object-done) future t)
		      t))))
         ((setter future-object-thread) @@future@@ @@task@@)
   	 ((setter future-object-function) @@future@@ (lambda () ,@exp))
	 (thread-start @@task@@ @@future@@ (lambda () ,@exp)) 
	 (increment-future-count)
	 @@future@@))
       
  (export future)

  ;;
  ;; Evaluator...
  ;;

  (defun futureeval (fut)
    (if (futurep fut)
	(if (future-object-done fut) (futureeval (future-object-value fut))
	  (progn
	    (thread-value (future-object-thread fut))
	    (futureeval fut)))
	fut))
	
  (export futureeval)				  

  ;;
  ;; Test...
  ;;
     
  (defun future-done-p (fut) (future-object-done fut))

  (export future-done-p)

)
