
/*

    This file is a part of the GLASS source distribution 
    and therefore subjected to the copy notice below. 
    
    Copyright (C) 1989,1990  S.J. Klaver, R Doesborg
              email: simon@sagan.nl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation version 1

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/****************************************************************/
/*                                                              */
/* File: handleparms.c,     Parameter handling module           */
/* Author: S.J. Klaver                                          */
/*                                                              */
/****************************************************************/

#include <stdio.h>
#include <tmc.h>
#include <curses.h>
/* curses.h defines WINDOW, stdscr   */
/* used here touchwin, ...           */
#include "menuds.h"
#include "dprocs.h"
#include "gconst.h"
#include "checkglue.h"
#include "tctypes.h"

/* Where to go when <esc> is hit,                                  */
/* possible values are 0, ACCVAL and RESVAL.                       */
enum { ACCVAL = -1, RESVAL = -2, ESCAPEDEST = -2 };

#define yesno char
enum { YES = 1, NO = 0 };

/* bottomline is is the bottomline of the parameter window. Value  */
/* is computed in handle_parm and used in display_parname          */
int bottomline;


int to_upper (c)
  int c;
{
  if ( ('a' <= c) && (c <= 'z'))
    return (c-'a'+'A');
  else
    return (c);
}

char *get_parval (gl, parmsetid, parshellname)
  gluefile gl;
  num parmsetid;
  string parshellname;
{
  Parset ps;
  int i;
  yesno found;
  char *str;
  
  /* ps := parameter set with id parmsetid */
  if (search_parmset (gl, parmsetid, &ps)) {
    fprintf (stderr, "Parameter set %d not found.\n", parmsetid);
    return ((char*) 0);       /* error return */
  }

  found = NO;
  i = 0;
  while ((i<ps->parsetitem->sz) && (found == NO)) {
    if (strcmp(parshellname,
               ps->parsetitem->arr[i]->parshellname) == 0)
      found = YES;
    else
      i++;
  }
  /*  found == YES  or  i == ps->parsetitem->sz and found == NO */

  if (found == YES) {
    str = (char *) malloc (strlen(ps->parsetitem->arr[i]->pardefault)+1);
    strcpy (str, ps->parsetitem->arr[i]->pardefault);
    return (str);
  }
  else
    return ((char *) 0);
}


/**************************************************/
int parms_to_env (gl, parmsetid)
  gluefile gl;    /* parameter and menu data structure */
  num parmsetid;  /* parameter set identifier */
{
  int i, len;
  char *str;
  Parset ps;

  /* ps := parameter set with id parmsetid */
  if (search_parmset (gl, parmsetid, &ps)) {
    fprintf (stderr, "Parameter set %d not found.\n", parmsetid);
    return (1);  /* error return */
  }

  for (i=0; i<ps->parsetitem->sz; i++) {
      len = 2 + strlen (ps->parsetitem->arr[i]->parshellname)
              + strlen (ps->parsetitem->arr[i]->pardefault);
      str = (char *) malloc(len);
      sprintf (str, "%s=%s",
               ps->parsetitem->arr[i]->parshellname,
               ps->parsetitem->arr[i]->pardefault);
      putenv (str);
  }
}
/**************************************************/

int parms_to_file (gl, parmsetid, fstr, csh)
  gluefile gl;    /* parameter and menu data structure */
  num parmsetid;  /* parameter set identifier */
  string fstr;    /* filename to write parameters to */
  yesno csh;      /* format for set command: TRUE = csh, FALSE = sh */
{
  FILE *parfile;
  Parset ps;
  int i;
  char str[250];

  /* ps := parameter set with id parmsetid */
  if (search_parmset (gl, parmsetid, &ps)) {
    fprintf (stderr, "Parameter set %d not found.\n", parmsetid);
    return (1);  /* error return */
  }

  /* open parfile */
  parfile = fopen (fstr, "a");   
  if (parfile == (FILE*) 0) {
    fprintf (stderr, "Cannot open file %s.\n", fstr);
    return (1);
  }

  /* write environment setup strings to file */
  for (i=0; i<ps->parsetitem->sz; i++) {
    if (csh == NO) {  /* format for sh */
      sprintf (str, "%s=\"%s\"; export %s\n",
               ps->parsetitem->arr[i]->parshellname,
               ps->parsetitem->arr[i]->pardefault,
               ps->parsetitem->arr[i]->parshellname);
    } 
    else {            /* format for csh */
      sprintf (str, "setenv %s \"%s\"\n",
               ps->parsetitem->arr[i]->parshellname,
               ps->parsetitem->arr[i]->pardefault);
    }
    fprintf (parfile, str);
  }

  /* close parfile */
  fclose (parfile);
}


/* Put all defaults in environment */
int parms_to_envir(gl)
  gluefile gl;
{ 
  menparspec_list mplist;    /* list of menus and parameter sets */
  int i;

  if (gl == gluefileNIL) return;  /* error return */
  mplist = ((Glue) gl)->specs;
  for (i = 0; i < mplist->sz; i++) 
    if (mplist->arr[i]->tag == TAGParset) 
       parms_to_env (gl, ((Parset) (mplist->arr[i]))->parsetid);
}


/* Which parameter set 'ps' has id 'parmsetid'    */
int search_parmset (gl, parmsetid, ps)
  gluefile gl;
  num parmsetid;
  Parset *ps;
{ 
  menparspec_list mplist;    /* list of menus and parameter sets */
  int i;
  yesno found;

  if (gl == gluefileNIL) return (1);  /* error return */

  mplist = ((Glue) gl)->specs;
  found = NO;

  for (i = 0; (i < (mplist->sz)) && (found == NO); i++) 
    if (mplist->arr[i]->tag == TAGParset) 
       if ( ( (Parset) (mplist->arr[i]) )->parsetid == parmsetid ) {
         found = YES;
         *ps = (Parset) (mplist->arr[i]);
       }

  if (found) return (0);   /* OK return */
  else       return (1);   /* OK return */
}


/* Displaying of a parameter name or the option Accept */

void display_parname (parmwin, str, parpos, attr, idchar, idpos)
  string str;
  int parpos, attr;
  WINDOW *parmwin; 
{
  switch (parpos) {
    case ACCVAL:
    /*  if (attr == NORMAL) attr = BOLD; */
      wdisplay_string (parmwin, ACCEPTSTR, bottomline, PACCEPTX, attr);
      break;
    case RESVAL:
    /*  if (attr == NORMAL) attr = BOLD; */
      wdisplay_string (parmwin, RESTORESTR, bottomline, PRESTOREX, attr);
      break;
    default:
      wdisplay_keyword (parmwin, str, PORGY+parpos, PORGX, attr, idchar, idpos);
      break;
  }
}


/* print parameter names and current (default) value */

void erase_parvalues (parmwin, paritems, maxlen)
  paritemspec_list paritems;
  int maxlen;
  WINDOW *parmwin; 
{
  int i;
  for (i=0; i < paritems->sz; i++) {
    werase_word (parmwin, 
                 PORGY+i, PORGX+maxlen+8,
                 strlen(paritems->arr[i]->pardefault)); 
  }
}

void display_parsetnames (parmwin, paritems, maxlen)
  paritemspec_list paritems;
  int maxlen;
  WINDOW *parmwin; 
{
  int i;
  for (i=0; i < (paritems->sz); i++) {
    display_parname (parmwin, 
                     paritems->arr[i]->parname, i, NORMAL,
                     paritems->arr[i]->idchar,
                     paritems->arr[i]->idpos);
    wmove (parmwin, PORGY+i, PORGX+maxlen+2);
    switch (paritems->arr[i]->partype->tag) {
      case TAGBoolean: waddstr (parmwin, "<b>:  ");  break;
      case TAGReal:    waddstr (parmwin, "<r>:  ");  break;
      case TAGInteger: waddstr (parmwin, "<i>:  ");  break;
      case TAGEnum:    waddstr (parmwin, "<e>:  ");  break;
      case TAGString:  waddstr (parmwin, "<s>:  ");  break;
      default:         waddstr (parmwin, "<?>:  ");  break;
    }
    waddstr (parmwin, paritems->arr[i]->pardefault); 
  }
}  

void nextenum (par, partyp)
  string *par;
  typespec partyp;
{
  string_list evals;
  int newlen, i;

  evals = ((Enum) partyp)->enumvals;
  i = 0;
  while ( (i < evals->sz) && (strcmp (*par, evals->arr[i]) != 0)) i++;
  /* i=evals->sz  or  *par == evals->arr[i] */

  i = i + 1;
  if (i >= evals->sz) i = 0;

  /* if  current value not found or if current value is last
     in enum list, the new values is the first. Otherwise the
     new value is the next in the enum list 
  */

  free (*par);
  newlen = strlen (evals->arr[i]);
  *par = (string) malloc (newlen+2);
  if (*par != (string) 0) strcpy (*par, evals->arr[i]);
}

void ask_parm (parmwin, par, partyp, newy, newx, promptstr)
  string *par, promptstr;  
  typespec partyp;
  int newy, newx;
  WINDOW *parmwin; 
{
  int promptlen, newlen;
  static char newstr[250];

  promptlen = strlen (promptstr);
  wdisplay_string (parmwin, promptstr, newy, newx, NORMAL);
  wdisplay_string (parmwin, ": ", newy, newx+promptlen, NORMAL);

  strcpy (newstr, *par);
  wgetstr_edit(parmwin, newstr, 53); 
  newlen = strlen (newstr);

  if (type_correct (newstr, partyp) == 1) {
    (*par) = (string) malloc (newlen+2);
    if (*par != (string) 0) 
      strcpy (*par, newstr);
  }
  else {
    beep();  flash();
  }
  werase_word (parmwin, newy, newx, newlen+promptlen+2);
  mvwaddch (parmwin, newy, PWINX1, '|'); wrefresh (parmwin); 
}

void update_parm (parmwin, par, partyp, newy, newx, parname)
  string *par;   /*  *par is the address of the value of the parameter */
  typespec partyp;
  int newy, newx;
  string parname;
  WINDOW *parmwin; 
{
  int newlen;

  switch (partyp->tag) {
    case TAGBoolean:
      if (strcmp ((*par),"TRUE") == 0) {
        free (*par);
        (*par) = (string) malloc (8);
        if (*par != (string) 0) 
          strcpy (*par, "FALSE");
      }
      else {
        strcpy ((*par), "TRUE");
      }
      break;
    case TAGInteger:
      ask_parm (parmwin, par, partyp, newy, newx, parname);
      break;
    case TAGReal:
      ask_parm (parmwin, par, partyp, newy, newx, parname);
      break;
    case TAGString:
      ask_parm (parmwin, par, partyp, newy, newx, parname);
      break;
    case TAGEnum:
      /*  *par wordt de volgende string in Enum */
      nextenum (par, partyp);
      break;
  } 
}
 

/************************************************************************
 
  handle_parms       Main function of this file

************************************************************************/

int handle_parms (gl, parmsetid, parmline)
  gluefile gl;         
  num parmsetid;   
  int parmline;
{
  Parset ps;          
  int parpos, len, maxlen, i;
  int oldlen, key;
  string oldvalues[25];
  yesno dontasknextchar; 
  WINDOW *parmwin;   

  /* check if there is a menu data structure                      */
  /* return error if not                                          */
  if (gl == gluefileNIL) return (1);              

  /* check if this menu data structure has correct parameter set  */
  if (search_parmset (gl, parmsetid, &ps)) {
    printf ("Parameter set not found.\n");
    return (1);  /* error return */
  }
  /* ps is now a pointer to the parameter set                     */

  /* Copying current values, for possible later restoring         */
  for (i=0; i < (ps->parsetitem->sz); i++) {
    len = strlen (ps->parsetitem->arr[i]->pardefault);
    oldvalues[i] = (string) malloc (len+1);
    strcpy (oldvalues[i], ps->parsetitem->arr[i]->pardefault);
  }

  parmwin = newwin (0, 0, parmline, 0); 

  /* Determine the maximum length of a parameter name             */
  maxlen = 0;
  for (i=0; i < (ps->parsetitem->sz); i++) {
    len = strlen (ps->parsetitem->arr[i]->parname);
    if (len > maxlen) maxlen = len;
  }

  /* Calculate position of bottom line of parameter window        */
  bottomline = PWINY0+ps->parsetitem->sz + 3;
  if (bottomline > PWINY1) {
    printf ("To many parameters.\n");
    return (1);
  }

  /* Display of box + parameter set name                          */
  wdraw_box (parmwin, 
             PWINY0, PWINX0, 
             bottomline, PWINX1);
  wdraw_box (parmwin,
             bottomline, PWINX0,
             bottomline+2, PWINX1);
  len = strlen (ps->parsetname);
  wdisplay_string (parmwin, ps->parsetname, 
                   PWINY0, (PWINX0+PWINX1-len)/2, NORMAL);  


  /* display names and values of parameters, and Accept option   */
  display_parsetnames (parmwin, ps->parsetitem, maxlen);
  display_parname (parmwin, ACCEPTSTR, ACCVAL, NORMAL, ' ', 1);
  display_parname (parmwin, RESTORESTR, RESVAL, NORMAL, ' ', 1);


  /* current parameter is first */
  parpos = 0;  
  display_parname (parmwin, ps->parsetitem->arr[0]->parname, 0, 
                   REVERSE, ' ', 1); 
  wrefresh(parmwin);
  key = wget_key(parmwin);
  dontasknextchar = NO;

  do {
    if (parpos == ACCVAL)
      display_parname (parmwin, ACCEPTSTR, parpos, NORMAL,
                       ' ',
                       1);
    else if (parpos == RESVAL)
      display_parname (parmwin, RESTORESTR, parpos, NORMAL,
                       ' ',
                       1);
    else
      display_parname (parmwin, ps->parsetitem->arr[parpos]->parname, 
                       parpos, NORMAL,
                       ps->parsetitem->arr[parpos]->idchar,
                       ps->parsetitem->arr[parpos]->idpos);
    switch (key) {
      case KEY_UP: 
        switch (parpos) {
          case ACCVAL: parpos = ps->parsetitem->sz-1;
                       break;
          case RESVAL: parpos = ACCVAL;
                       break;
          case 0: parpos = RESVAL;
                  break;
          default: parpos = parpos - 1;
                   break;
        }
        break;
      case ' ':         /* <space> is the same as <down> */
      case KEY_DOWN: 
        switch (parpos) {
          case ACCVAL: parpos = RESVAL;
                       break;
          case RESVAL: parpos = 0;
                       break;
          default: if (parpos == ps->parsetitem->sz-1)
                     parpos = ACCVAL;
                   else
                     parpos = parpos + 1;
                   break;
        }
        break;
      case KEY_RIGHT:   /* <right> is the same as <return> */
      case KEY_RETURN:
        if (parpos != ACCVAL) {
          if (parpos == RESVAL) {
            erase_parvalues (parmwin, ps->parsetitem, maxlen);

            /* Restore previous values (and display them) */
            for (i=0; i < (ps->parsetitem->sz); i++) 
              ps->parsetitem->arr[i]->pardefault = oldvalues[i];
            display_parsetnames (parmwin, ps->parsetitem, maxlen);
           
            parpos = 0;
            /* after restore to first parameter */
          }
          else {
            oldlen = strlen (ps->parsetitem->arr[parpos]->pardefault);
            update_parm (parmwin, 
                         &(ps->parsetitem->arr[parpos]->pardefault),
                         ps->parsetitem->arr[parpos]->partype,
                         bottomline+1, PWINX0+2,
                         ps->parsetitem->arr[parpos]->parname
                        );
  
            /* display new parameter value */
            werase_word (parmwin, PORGY+parpos, PORGX+maxlen+8, oldlen); 
            wdisplay_string (parmwin, 
                             ps->parsetitem->arr[parpos]->pardefault,
                             PORGY+parpos, PORGX+maxlen+8,
                             NORMAL);
#ifdef AUTOADVANCE
            switch (ps->parsetitem->arr[parpos]->partype->tag) {
              case TAGEnum:
              case TAGBoolean:
                break;
              default:
                if (parpos == ps->parsetitem->sz-1)
                  parpos = ACCVAL;
                else parpos = parpos + 1;
            } 
#endif
          }
        }
        break;
      case KEY_ESCAPE:
        parpos = ESCAPEDEST;
        break;
      default: 
        key = to_upper (key);
        if (key == 'A') {
          parpos = ACCVAL;
          dontasknextchar = YES;
        } 
        if (key == 'R') {
          parpos = RESVAL;
          dontasknextchar = YES;
        }
        for (i=0; i < (ps->parsetitem->sz); i++) 
          if (key == to_upper(ps->parsetitem->arr[i]->idchar)) {
            parpos = i;
            dontasknextchar = YES;
          }
        key = KEY_RETURN;
        break;
    }
    
    if (parpos == ACCVAL) 
      display_parname (parmwin, ACCEPTSTR, parpos, REVERSE, ' ', 1);
    else if (parpos == RESVAL) 
      display_parname (parmwin, RESTORESTR, parpos, REVERSE, ' ', 1);
    else
      display_parname (parmwin, ps->parsetitem->arr[parpos]->parname, 
                       parpos, REVERSE, ' ', 1);
    
    if (dontasknextchar == NO) key = wget_key(parmwin);
    dontasknextchar = NO;

  } while ((key != KEY_RETURN) ||  (parpos != ACCVAL));

  /* key = <return>  and  parpos = Accept */

  wclear_screen (parmwin);
  touchwin (stdscr);       /* curses */

  parms_to_env (gl, parmsetid);
  return (0);              /* OK return */
}
