/* 
   Copyright (C) 1990 C van Reewijk, email: dutentb.uucp!reeuwijk

This file is part of GLASS.

GLASS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GLASS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GLASS; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* file: tmstring.c
   String manipulation routines.
 */

#include "tmdefs.h"
#include <ctype.h>
#include <tmc.h>

extern char *malloc();

#include "debug.h"
#include "tmds.h"
#include "tmstring.h"
#include "tmmisc.h"
#include "tmerror.h"

/* Given a pointer to a string 's' and a pointer to a character pointer 'w',
   scan the string for a 'word'.
   
   First, all characters matching isspace() are skipped.
   After that a word is scanned, a new string is allocated for the word,
   and assigned to '*w'. If there is no other word in the string, '*w'
   is set to stringNIL.

   A word is one of the following regular expressions:
   [^ \t\n\r\f\0]+: An arbitrary string of nonblanks and non-specials.
   "[^"\0]*": Arbitrary characters surrounded by "".
 */
char *scanword( s, w )
 char *s;
 char **w;
{
    register int ix;
    register unsigned int room;
    string buf;

    while( isspace( *s ) ) s++;
    if( *s == '\0' ){
	*w = stringNIL;
	return( s );
    }
    buf = new_string( s );
    ix = 0;
    room = strlen( s )+1;
    if( *s == DQUOTE ){
	s++;
	while( *s != DQUOTE && *s != '\0' ){
	    if( ix>= room ){
		room += STRSTEP;
		buf = ckrealloc( buf, room+1 );
	    }
	    buf[ix++] = *s++;
	}
	if( *s != DQUOTE ){
	    line_error( UNEXPECTEOL );
	}
	else {
	    s++;
	}
	buf[ix] = '\0';
	*w = buf;
	return( s );
    }
    while( *s != '\0' && !isspace( *s ) ){
	if( ix>= room ){
	    room += STRSTEP;
	    buf = ckrealloc( buf, room+1 );
	}
	buf[ix++] = *s++;
    }
    buf[ix] = '\0';
    *w = buf;
    return( s );
}

/***************************************************************
 *   parameter parsing                                         *
 ***************************************************************/

/* Given a parameter string 'p' and a pointer to a string pointer
   'p1', ensure that 'p' contains exactly one parameter and create
   a copy of the string to put in '*p1'.
 */
void scan1par( pl, p1 )
 char *pl;
 char **p1;
{
    pl = scanword( pl, p1 );
    if( *p1 == CHARNIL ){
	line_error( MISSINGPAR );
	return;
    }
    cknopar( pl );
}

/***************************************************************
 *                                                             *
 *   error handling                                            *
 *                                                             *
 ***************************************************************/

/* Given a string 's', ensure that it does not contain an other
   parameter, or else complain.
 */
void cknopar( s )
 char *s;
{
    while( isspace( *s ) ) s++;
    if( *s != '\0' ){
	(void) sprintf( errarg, "'%s'", s );
	line_error( TOOMANYARG );
    }
}

/* Given a string 's', ensure that it is a correct number,
   or else complain.
 */
void cknumpar( n )
 char *n;
{
    char *s;

    s = n;
    while( isspace( *s ) ) s++;
    if( *s == '-' || *s == '+' ) s++;
    while( isdigit( *s ) ) s++;
    while( isspace( *s ) ) s++;
    if( *s != '\0' ){
	(void) sprintf( errarg, "'%s'", n );
	line_error( BADNUMBER );
    }
}

/* Return a new "1" or "0" string reflecting the value of boolean 'b'. */
char *newboolstr( b )
 bool b;
{
    return( new_string( ( b ? "1" : "0" ) ) );
}

/* Return a new string reflecting the value of int 'n'. */
char *newintstr( n )
 int n;
{
    char buf[NUMBUFSIZE];

    (void) sprintf( buf, "%d", n );
    return( new_string( buf ) );
}

/* Given a string 'p', chop it into words using 'scanword' and return
 * a stringlist containing the pieces.
 */
string_list chopstring( p )
 char *p;
{
    char *s;
    string_list sl;

    sl = new_string_list();
    for(;;) {
	p = scanword( p, &s );
	if( s == CHARNIL ) break;
	app_string_list( sl, s );
    }
    return( sl );
}

/* Given a string_list 'sl' and a separation string 'sep', construct one
 * single (newly allocated) string from all strings in 'sl' separated from
 * each other with a copy of 'sep'.
 *
 * To prevent problems with a fixed buffer, the final length of the
 * string is calculated, and sufficient room is allocated for that
 * string. To keep the string statistics in balance a fake string
 * with length 0 is allocated using new_string, is freed using
 * free(), and is replaced by the locally allocated string.
 *
 * the total length of the required string is
 *  sum(<all joined strings>) + (n-1)*strlen(sep)
 */
string sepstrings( sl, sep )
 string_list sl;
 string sep;
{
    register char *cp;		/* pointer in copied strings  */
    char *cs;			/* string under construction */
    register char *bufp;	/* pointer in string under construction */
    register unsigned int ix;	/* index in string array */
    unsigned int len;		/* calculated length of string */

    len = 0;
    if( sl->sz!=0 ){
	len += (sl->sz-1)*strlen(sep);
    }
    for( ix=0; ix<sl->sz; ix++ ){
	len += strlen( sl->arr[ix] );
    }
    cs = new_string( "" );
    cs = ckrealloc( cs, len+1 );
    bufp = cs;
    for( ix=0; ix<sl->sz; ix++ ){
	if( ix!=0 ){
	    cp = sep;
	    while( *cp!='\0' ) *bufp++ = *cp++;
	}
	cp = sl->arr[ix];
	while( *cp!='\0' ) *bufp++ = *cp++;
    }
    *bufp = '\0';
    return( cs );
}

/* Given a string_list 'sl', construct one single string from by separating
 * all strings in 'sl' from each other with a ' '.
 */
string flatstrings( sl )
 string_list sl;
{
    return( sepstrings( sl, " " ) );
}

/* Return TRUE if this string represents FALSE. */
bool isfalsestr( s )
 register string s;
{
    while( isspace( *s ) ) s++;
    if( s[0] != '0' ) return( FALSE );
    return( isspace( s[1] ) || s[1] == '\0' );
}

/* Return TRUE if this string does not represent FALSE. */
bool istruestr( s )
 register string s;
{
    while( isspace( *s ) ) s++;
    if( s[0] != '0' ) return( TRUE );
    return( !(isspace( s[1] ) || s[1] == '\0') );
}
