-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: getuses.p
-- Author: Daniel Yellin
-- SCCS Info: "@(#)getuses.p	1.7 5/16/90"
-- description: 
-- getuses is passed some capabilities and returns a getuses
-- function.  This function is given the name of a Hermes file and returns
-- the file names of the dependencies (as given in the "using" and 
-- "linking" lists). 
getuses: using (getuses,uptolow,unix,terminalInits,terminalio,findfile)
process(setupQ: setupGetusesQ)

declare
  setupM: setupGetusesIntf;
  findfileDO: findfile!findfile_func;   -- findfile relative to DEFLIBDD path
  findfilePO: findfile!findfile_func;   -- findfile relative to SYSLIBDD path
  initGetString: terminalInits!initGetStringFunc;  -- process that returns 
                                  -- a getstring func for a particular file
  fgetc: unix!fgetc_function;     -- the unix fgetc function
  fopen: unix!fopen_function;     --  "       fopen      "
  fclose: unix!fclose_function;   --  "       fclose     "
  getnoncommentline: getlineFunc;
  charposition: charPosFunc;
  getnextuse: getNextUseFunc;
  uptolow: uptolowFn;
  getusesQ: getusesQ;
  predefinedPath: charstring;  
begin
-----------------------------------------------------------------------
---------------------------  initialization ---------------------------
-----------------------------------------------------------------------
  receive setupM from setupQ;

  fgetc := setupM.unix.stdio.fgetc;  
  fopen := setupM.unix.stdio.fopen;  
  fclose := setupM.unix.stdio.fclose;  
  findfileDO := setupM.findfileDO;
  findfilePO := setupM.findfilePO;

  initGetString <- procedure of (setupM.pathload("getstring2"));
-- the above must be procedure of and not create of.  Otherwise, on calls
-- to initGetString subsequent to the first, the system will be left
-- hanging.  the following also must be procedure of.
  getnoncommentline <- procedure of (setupM.pathload("getnoncommentline"));
  charposition <- procedure of (setupM.pathload("charposition"));
  getnextuse  <- procedure of (setupM.pathload("getnextuse"));
  uptolow <- procedure of setupM.pathload("uptolow"); 

-- create getusesFn and return callmessage
  new getusesQ;
  connect setupM.getuses to getusesQ;
  return setupM;

  predefinedPath <- findfileDO("predefined.do",access#'read');
-- predefinedPath is now the fullname (including path) of predefined.do

-----------------------------------------------------------------------
-------------------------  process requests ---------------------------
-----------------------------------------------------------------------
  while 'true' repeat
    block declare
      getusesM: getusesIntf;
      getstring: terminalio!getStringFunc;
      file_handle: unix!handle;
      line: charstring;
      use: charstring;
      link: charstring;
      token: charstring;
      pos: integer;
      rightParenFound: boolean;
      advance: boolean;       
      filename: charstring;
    begin
      receive getusesM from getusesQ;
      new getusesM.uses;

-- initialize getstring to be a getstring function for this file.
-- note that we are assuming that getusesM.name give the complete path name 
-- for the file.  
      file_handle <- fopen(getusesM.name, "r");
      getstring <- initGetString(fgetc, file_handle);

-------------------------------------------------------------------------------
------------------ find begininng of using list ------------------------------
-------------------------------------------------------------------------------

-- the following sets the variable line to the first noncomment
-- line with a left paren in it, and sets pos to the position of the left 
-- paren.
     line <- getnoncommentline(getstring);
     -- this returns the next non-comment line in the Hermes program.
     -- the returned line is stripped of leading and trailing spaces
     -- and of any comments.  
     pos <- charposition('(', line);  -- this returns the position of
          -- the first left paren '(' in line and -1 if it doesn't exist
      while pos = -1 repeat   
         line <- getnoncommentline(getstring);
         pos <- charposition('(', line);  
      end while;  
      if pos = -1 then exit badFileFormat1; end if;

-- the following loops searches through the line (and any additional lines
-- if needed) until it comes to the first token in the using list
      pos := pos + 1;  -- advance to next char
      advance <- 'true';
      while advance repeat  -- advance to first nonnull space
         select(line[pos])
          where('NL')       -- get another line
            line <- getnoncommentline(getstring);
            pos := 0;   -- reset pos to start of line
          where(' ')        -- skip over spaces
            pos <- pos + 1;
          otherwise     -- found beginning of token!
            advance <- 'false';
          end select;
      end while;
      -- pos now points to first token (name) in using list

      if line[pos] = ')'
      then rightParenFound <- 'true';
      else rightParenFound <- 'false';
      end if;

--------------------------------------------------------------------------
------- insert each definition dependency into uses list -----------------
--------------------------------------------------------------------------
      while not rightParenFound repeat
         use <- getnextuse(line, pos); -- this returns the token that begins
            -- at line[pos] and is terminated by the next occurence of
            -- a comma, a space, a right paren ')', or a 'NL'.  It 
            -- advances pos to point to the terminating char (a comma, a 
            -- right paren,  etc.)
         call uptolow(use);  -- convert to lowercase letters
         use <- use | ".do";  -- add the suffix to the filename
         block begin
-- find the complete file name with path prefix and insert into getusesM.uses.
           insert findfileDO(use,access#'read') into getusesM.uses;  
         on (findfile_Intf.file_Not_Found)
-- The .do file doesn't exist, so it needs to be made.  
-- By default, put it into current dir
           insert use into getusesM.uses;
         end block;
         advance <- 'true';
         while advance repeat    -- advance to next token if one exists
           select(line[pos])
             where('NL')         -- check next line
                 line <- getnoncommentline(getstring);
                 pos <- 0;
             where(',')          -- skip commas
                 pos <- pos + 1;
             where(' ')          -- skip blanks
                 pos <- pos + 1;
             where(')')          -- no more
                 advance <- 'false';
                 rightParenFound <- 'true';
                 pos <- pos + 1;
             otherwise
                 advance <- 'false';
           end select;
        end while;  -- advance to next use in using list
      end while;    -- scan using list

      block begin
      -- every module implicitly depends upon predefined.do.
      insert copy of predefinedPath into getusesM.uses; 
      on (duplicatekey)
-- predefined was already inserted
      end block;

-----------------------------------------------------------------------
--------------------- check if linking list exists --------------------
-----------------------------------------------------------------------

-- advance to next noncomment token
      advance <- 'true';
      while advance repeat  -- advance to next token
         select(line[pos])
           where('NL')
              line <- getnoncommentline(getstring);
              pos <- 0;
           where(' ')
              pos <- pos + 1;
           otherwise
              advance <- 'false';
         end select;
      end while;

      token <- getnextuse(line,pos);
      call uptolow(token);
      if token <> "linking" then exit ret; end if;

------------------------------------------------------------------------
----- linking lists exists, so find beginning of link dependencies -----
------------------------------------------------------------------------

-- advance to next '('
      advance <- 'true';
      while advance repeat  -- advance to '('
         select(line[pos])
           where('NL')
              line <- getnoncommentline(getstring);
              pos <- 0;
           where(' ')
              pos <- pos + 1;
           otherwise
              advance <- 'false';
         end select;
      end while;
      if line[pos] <> '(' 
      then exit badFileFormat2;
      end if;

-- advance to first token in linking list
      pos := pos + 1;  -- advance to next char
      advance <- 'true';
      while advance repeat  -- advance to first nonnull space
         select(line[pos])
          where('NL')
            line <- getnoncommentline(getstring);
            pos := 0;   -- reset pos to start of line
          where(' ')
            pos <- pos + 1;
          otherwise
            advance <- 'false';
          end select;
      end while;
      -- pos now points to first name in link list

      if line[pos] = ')'
      then rightParenFound <- 'true';
      else rightParenFound <- 'false';
      end if;

--------------------------------------------------------------------
------- insert each link dependency into uses list -----------------
--------------------------------------------------------------------

      while not rightParenFound repeat
         link <- getnextuse(line, pos); 
         call uptolow(link);  -- convert to lowercase letters
         link <- link | ".po";  -- add the suffix to the filename
         block begin
-- find the complete file name with path prefix and insert into getusesM.uses.
           insert findfilePO(link, access#'read') into getusesM.uses;  
         on (findfile_Intf.file_Not_Found)
-- if the .po file doesn't exist, it needs to be made.  
-- By default, put it into current dir
           insert link into getusesM.uses;
         end block;
         advance <- 'true';
         while advance repeat  -- advance to next token
           select(line[pos])
             where('NL')
                 line <- getnoncommentline(getstring);
                 pos <- 0;
             where(',')
                 pos <- pos + 1;
             where(' ')
                 pos <- pos + 1;
             where(')') 
                 advance <- 'false';
                 rightParenFound <- 'true';
             otherwise
                 advance <- 'false';
           end select;
        end while;  -- advance to next use in linking list
      end while;    -- scan linking list
      exit ret;

--------------------------------------------------------------------------
------------------------------  normal exit ------------------------------
--------------------------------------------------------------------------

    on exit (ret)
      call fclose(file_handle);
      return getusesM;

----------------------------------------------------------------------------
-------------------------- exceptional exits -------------------------------
----------------------------------------------------------------------------
    on exit (badFileFormat1)
       print charstring#"could not find the using list for the file ";
       print getusesM.name;
       print charstring#" Exception getusesfailure being raised";
       call fclose(file_handle);
       return getusesM exception getusesfailure;

    on exit (badFileFormat2)
       print charstring#"expecting '(' after keyword 'linking' in file ";
       print getusesM.name;
       print charstring#" Exception getusesfailure being raised";
       call fclose(file_handle);
       return getusesM exception getusesfailure;

    on (fopen_call.semantic_error)
       print charstring#"fopen failed for file "; 
       print getusesM.name;
       print charstring#" Exception getusesfailure being raised";
       return getusesM exception getusesfailure;

    on (getlineIntf.endOfInput)
-- can this exception ever occur??? yes! when a file cannot be found
       print charstring#"end of input exception for file"; 
       print getusesM.name;
       print charstring#" Exception getusesfailure being raised";
       call fclose(file_handle);
       return getusesM exception getusesfailure;

    on (getNextUseIntf.noUse)
       print charstring#"bad file format in using or linking list of file ";
       print getusesM.name;
       print charstring#" Exception getusesfailure being raised";
       call fclose(file_handle);
       return getusesM exception getusesfailure;

    on (DuplicateKey)
       print charstring#"A duplicate file is listed in a using or linking list";
       print getusesM.name;
       print charstring#" Exception getusesfailure being raised";
       call fclose(file_handle);
       return getusesM exception getusesfailure;
    end block;

  end while;
end process

