-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: tsapre.p
-- Author: Rob Strom
-- SCCS Info: @(#)tsapre.p	1.1 7/27/89

 -- Procedure to check precondition and lower typestate if needed    
 --                                                                  
 -- Rather than design a fancy representation of typestates which    
 -- would reveal the lattice structure and the compatibility         
 -- rules governing typestates, it was decided to leave              
 -- typestates as a simple list of attributes, and apply             
 -- language-specific rules for GLB, apply precondition, and         
 -- apply postcondition.  As a consequence, this procedure           
 -- reflects the current type family structure of Hermes, and          
 -- will have to be changed if the language is changed radically     
 -- enough to alter the compatibility rules.                         
 -- The places where we rely on assumptions about Hermes are         
 -- documented at the points where those assumptions are used.       
 

TSAPRE : USING( Predefined, tscheck, Positions, Errors, Coercions, TsApre  ) PROCESS ( ApplyPreconditionInit : ApplyPreconditionInport )
  DECLARE
    FP: ApplyPreconditionCall ;
    Nothing: Empty; -- what an empty variant is
    AttemptToLower: ApplyPreconditionOutport; -- subprocedure to check deletions
  BEGIN
    
 -- Internal Procedure AttemptToLower
 -- Algorithm:                                                       
 -- 1. For each attribute in "Forbidden" set which is now present and not implied:
 --    1.1. obtain 2 pieces of information:                          
 --         the coercion needed to drop the attribute                
 --         the attributes which must be dropped as prereq to coercio
 --         the attributes which will be dropped together by coercion
 --    1.2. make sure the attributes which will be dropped are not RQ
 --    1.3. call myself to drop the prequisite attributes            
 --    1.4. insert the operation into the coercion list 
 --    1.5. remove the dropped attributes from the typestate         

    AttemptToLower <- ApplyPreconditionOutport # (PROCEDURE OF program # (PROCESS (AttemptToLowerInit: ApplyPreconditionInport)
      DECLARE
        FP: ApplyPreconditionCall;
        AttemptToLower: ApplyPreconditionOutport;  -- recursive call to myself
        Coercion: Statement; -- the statement which must be executed to drop some attribute
        Prerequisites: Typestate; -- attributes which must be dropped before coercion can be applied
        Corequisites: Typestate; -- attributes which are dropped when coercion is applied
        CoercionsForThisStatement: Coercion; -- set of coercions to be inserted before this statement
        Nothing: Empty; -- qualifier for coercions
	DroppedAttributes: Typestate;
      BEGIN
        RECEIVE FP FROM AttemptToLowerInit;
        -- 1.
        FOR ForbiddenAttribute IN FP.Forbidden WHERE(boolean # ('true'))
          INSPECT
            IF boolean # (EXISTS OF Attr IN FP.CurrentTS WHERE(boolean # (Attr = ForbiddenAttribute)))
              THEN
                IF FP.Services.ImpliedAttribute(FP.Services, FP.Declarations, FP.Definitions, FP.Context, ForbiddenAttribute, FP.CurrentTS)
                  THEN
                  ELSE
--		    /* debug */ CALL FP.Services.Outside.Terminal.PutLine("Forbidden:"|FP.Services.ebugFormatAttribute(FP.Services.Debug, FP.Definitions, FP.Declarations, FP.Context.InferredDcls, FP.Context.DefinitionsMap, FP.Context.ExecutableMap, ForbiddenAttribute));
		    -- 1.1
		    CALL FP.Services.AttemptToCoerce(FP.Services, FP.Declarations, FP.Context, FP.Definitions, ForbiddenAttribute, FP.CurrentTS, Coercion, Prerequisites, Corequisites);
		    -- 1.2
		    FOR DroppedAttribute IN Corequisites WHERE(boolean # ('true'))
		      INSPECT
			IF boolean # (EXISTS OF RequiredAttribute IN FP.Required WHERE(boolean # (RequiredAttribute = DroppedAttribute)))
			  THEN
			    INSERT (EVALUATE ErrorMessage: Error FROM
			      NEW ErrorMessage;
			      UNITE ErrorMessage.Position.APos FROM COPY OF FP.Position;
			      ErrorMessage.Code <-  errorcode # 'NotLowerable';
			      NEW ErrorMessage.Objects;
			      INSERT (EVALUATE ErrorObject: ErrorObject FROM
				UNITE ErrorObject.Attribute FROM Attribute # (COPY OF DroppedAttribute); 
				END) INTO ErrorMessage.Objects;
			      END) INTO FP.Context.ErrorMessages;
			  END IF;
		      END FOR;
		    -- 1.3
		    IF boolean # (integer # (SIZE OF Prerequisites)  > integer # (0) )
		      THEN
			AttemptToLower <- ApplyPreconditionOutport # (PROCEDURE OF program # CURRENTPROGRAM);
			-- fix mismatch between prerequisites and typestate
			CALL AttemptToLower(FP.Services, FP.Declarations, FP.Definitions, FP.Position, FP.Context, FP.Required, Prerequisites, FP.CurrentTS);
		      ELSE
		      END IF;
		    -- 1.4
		    /* CreateCoercionTableIfNeeded */ BLOCK
		      BEGIN
			REMOVE CoercionsForThisStatement FROM Coercion2 IN FP.Context.PreCoercions WHERE(boolean # (Coercion2.Position = FP.Position));
		      ON (NotFound)
			NEW CoercionsForThisStatement;
			CoercionsForThisStatement.Position := FP.Position;
			NEW CoercionsForThisStatement.Coercions;
		      END /* CreateCoercionTableIfNeeded */ BLOCK;
		    INSERT Coercion INTO CoercionsForThisStatement.Coercions;
		    INSERT CoercionsForThisStatement INTO FP.Context.Precoercions;
		    -- 1.5
		      EXTRACT DroppedAttributes FROM Attr1 IN FP.CurrentTS WHERE(EXISTS OF Attr2 IN Corequisites WHERE(Attr1 = Attr2));
--		    /* debug */ CALL FP.Services.Outside.Terminal.PutLine("Extracted:"|FP.Services.Debug.FormatTypestate(FP.Services.Debug, FP.Definitions, FP.Declarations, FP.Context.InferredDcls, FP.Context.DefinitionsMap, FP.Context.ExecutableMap, DroppedAttributes));
		  END IF;
              END IF;
	  END FOR;
	RETURN FP;
        
      END PROCESS));
 -- Algorithm:                                                       
 -- 1. For each required attribute, it can be present exactly, or else:
 --    It's implied by another attribute.  Currently, this happens if
 -- 2. If there is a coercion list for this point in the program     
 --    delete it. (This would happen if this is not the first        
 --    scan of this statement.)                                      
 -- 3. Call a recursive procedure which will try to drop all         
 --    the forbidden attributes while keeping the required ones.     
      
    RECEIVE FP FROM ApplyPreconditionInit ;
    FOR RequiredAttribute IN FP.Required WHERE(boolean # 'true')
      INSPECT
        IF (EVALUATE Satisfied: Boolean FROM
          Satisfied <- EXISTS OF Attribute IN FP.CurrentTS WHERE(Attribute = RequiredAttribute);
          IF Satisfied 
            THEN
            ELSE
              Satisfied <- FP.Services.ImpliedAttribute(FP.Services, FP.Declarations, FP.Definitions, FP.Context, RequiredAttribute, FP.CurrentTS);
            END IF;
          END)
          THEN
          ELSE
            /* typestate error: missing required attribute */
            INSERT (EVALUATE ErrorMessage: Error FROM
	      NEW ErrorMessage;
	      UNITE ErrorMessage.Position.APos FROM COPY OF FP.Position;
	      ErrorMessage.Code <- 'NotAsserted';
	      NEW ErrorMessage.Objects;
	      INSERT (EVALUATE ErrorObject: ErrorObject
		FROM
		  UNITE ErrorObject.Attribute FROM COPY OF RequiredAttribute;
		  END) INTO ErrorMessage.Objects;
	      END) INTO FP.Context.ErrorMessages;   
          END IF;
      END FOR;
--    SEND FP TO AttemptToLower; /* forward the original call */
    CALL AttemptToLower(FP.Services, FP.Declarations, FP.Definitions, FP.Position, FP.Context, FP.Required, FP.Forbidden, FP.CurrentTS);
    RETURN FP;
  END PROCESS
