-- (C) Copyright International Business Machines Corporation 16 September
-- 1991.  All Rights Reserved.
--
-- See the file USERAGREEMENT distributed with this software for full
-- terms and conditions of use.
-- SCCS Info: @(#)chprogram.p	1.3 2/17/92

-- This process is called for the main program to be transformed and for
-- each program literal.

-- Input is an li prog structure.
--

-- Output is a transformed prog, containing a residual li-program
-- (code table) and the C-coded transform in any or all of three
-- forms: C source text in the program object's text component, binary
-- object data in its object component, or the name of a file
-- containing the latter, in the program object's path component.

-- Other components of the input program are modified as follows:
--   size - typically this value is reduced, since the transformed
--     program usually needs fewer objects than the original.  
--   pool - qualifiers needed by the C-coded program are added to the
--     pool table of the prog (see chencode.p).  
--   type - copied unaltered from the original.
--   handle - initialized so as to cause the module to be dynamically
--     loaded when the program is first instantiated.
--   name - if the counter in ch.modifier is -1, the name is left
--     unchanged.  Otherwise, the counter is advanced and is used to
--     create new a unique name for this program.  In either case, the
--     resulting name is used in constructing the path component as
--     discussed above.  It is also prefixed with "CH_" to determine
--     the entry point name of the compiled object file.

-- The actual transformation is performed by the process chtransform.
-- That process may return with exception already, indicating that
-- this prog was already transformed.  In this case, the prog is left
-- unchanged by chtransform, although compilation and/or stashing of
-- source or object code in the prog may still take place and may
-- result in other changes to the prog.  Also, if compilation of the c
-- program fails, the program is left unaltered, and output from the
-- compilation is displayed.

chprogram: using(chinternal, chtransform, common, interpform, unix,getfile)
linking (chtransform,chcompile)

process(Q: programQ)
    
declare
  
  cm: programCm;
  progname: charstring;
  lipart: prog;
  chpart: charstring;
  object: charstring;
  failed: boolean;
begin
  receive cm from Q;
  failed <- 'false';
  -- make the reference name unique
  progname := cm.ch.main;
  if cm.ch.modifier >= 0 then
    -- it is either a program literal or a linked program
    insert '_' into progname;
    if progname = every of c in cm.liprog.name where
	  (position of c < size of progname) then
      -- prog lit or renamed in a previous chcode
      progname := cm.liprog.name;
    else
      -- linked or nested withina linked program... prefix the
      -- name with the main program name
      merge copy of cm.liprog.name into progname;
    end if;
  else
    cm.ch.modifier <- 0;	-- only -1 once, for main program
  end if;

  block begin
    insert copy of progname into cm.ch.names;
  on (duplicateKey)
    -- name already used... uniquify it
    insert '_' into progname;
    merge cm.ch.itoa(cm.ch.modifier) into progname;
    insert copy of progname into cm.ch.names;-- should be unique now
    cm.ch.modifier <- cm.ch.modifier + 1;
  end block;
  
  -- copy original prog to get things started
  lipart := cm.liprog;
  lipart.name := progname;
  lipart.id <- unique;		-- all liprogs that end up bound to
				-- programs must have unique id's

  if cm.ch.options.verbose and cm.ch.options.transform then
    call cm.ch.put("transform[" | progname | "] ");
    block begin
      call (transformFn#(create of process chtransform))
	  (cm.ch, lipart, chpart);
    on (transform.already)
      chpart <- "";		-- to satisfy typestate
      if cm.ch.options.verbose then
	call cm.ch.put("(already transformed; no change) ");
	if cm.ch.options.keepC and cm.ch.options.storeC 
	      and size of lipart.text = 0 then
	  -- try to locate the C source file and read it in
	  block begin
	    lipart.text <- cm.ch.getfile(cm.ch.options.storeCPath
		  | progname | ".c");
	  on (getfile.cantRead)
	    if not cm.ch.options.quiet then
	      call cm.ch.put("(C source not available for keeping) ");
	    end if;
	  end block;
	end if;
      end if;
    end block;
    if cm.ch.options.keepC then
      lipart.text := chpart;
    else
      lipart.text <- "";
    end if;

    if cm.ch.options.storeC or cm.ch.options.storeObj or cm.ch.options.keepObj
    then
      call cm.ch.putfile(cm.ch.options.storeCPath | progname | ".c", chpart);
    end if;
  else
    chpart <- "";		-- placeholder
  end if;

  -- compile C code for this program
  if cm.ch.options.storeObj or cm.ch.options.keepObj then
    block begin
      call (chcompileFn#(create of process chcompile))
	  (cm.ch, progname, object);
    on (others)
      -- diagnostic already written by chcompile... restore original prog
      lipart := cm.liprog;
      failed <- 'true';
      object <- "";
    end block;
  
    lipart.object <- "";
    if not failed then
      if cm.ch.options.keepObj then
	lipart.object <- object;
      end if;
    end if;
  end if;
    
  if not failed then
    if cm.ch.options.storeObj then
      if cm.ch.options.fullpaths then
	lipart.path <- cm.ch.options.storeObjPath | progname | ".o";
      else
	lipart.path <- progname | ".o";
      end if;
    else
      lipart.path <- "";
    end if;
  end if;

  -- remove C source file if it didn't have to be kept
  if not cm.ch.options.storeC and
	(cm.ch.options.keepObj or cm.ch.options.storeObj) then
    block declare
      pipe: stream;
    begin
      pipe <- cm.ch.runcmd("rm  -f " 
	    | cm.ch.options.storeCpath | progname | ".c", 'read');
      call pipe.pclose();
    on (others)
    end block;
  end if;
  
  if not failed then
    lipart.handle <- 0;		-- force dynamic object load on first
				-- instantiation
  end if;

  cm.newprog <- lipart;

  return cm;
    
end process
