/* (C) Copyright International Business Machines Corporation 23 January */
/* 1990.  All Rights Reserved. */
/*  */
/* See the file USERAGREEMENT distributed with this software for full */
/* terms and conditions of use. */
/* File: fegram.yacc */
/* Author: David F. Bacon */
/* SCCS Info: @(#)fegram.yacc	1.26 1/31/92 */

/*
 * yacc grammar for Hermes
 */

%{
#include "li.h"

#include "resolve.h"
#include "resfuncs.h"

#include "predefined.cd"
#include "errors.cd"

#undef TRUE
#undef FALSE
#undef CALLMESSAGE
#undef INITPORT

#define True -1
#define False 0

extern char yytext[];

static flag cons_type;		/* are we constructing a type? or attribute? */

/* we need a big stack because we're a big, important language! */
#define YYMAXDEPTH 500
%}


/*
 * token definitions are prepared automatically from the files "keywords" and
 * "othertokens".  we Include them here with m4.
 */

include(tokens.yacc)

%union {
	char *string;
	object *obj;
	dfd_boolean boolean;
	decl *declp;
	selrec *selector;
      }

%type <declp>		declaration
%type <obj>		simple_statement
%type <obj>		atposition
%type <string>		return_exception
%type <obj>		compound_statement
%type <obj>		else_clause
%type <obj>		select_clause
%type <obj>		event_guard
%type <obj>		boolean_guard
%type <obj>		otherwise_clause
%type <obj>		statement_clause
%type <selector>	selector
%type <selector>	the_element
%type <obj>		mapping
%type <obj>		expression_clause
%type <obj>		expression
%type <obj>		disjunction
%type <obj>		conjunction
%type <obj>		relation
%type <obj>		concat
%type <obj>		term
%type <obj>		factor
%type <obj>		secondary
%type <obj>		primary
%type <obj>		type_specifier
%type <obj>		typed_primary
%type <obj>		function_reference
%type <obj>		literal
%type <obj>		program_literal
%type <obj>		typename_literal
%type <obj>		attributename_literal
%type <obj>		key_definition
%type <obj>		minimum
%type <obj>		formal_typestate
%type <obj>		formal_attribute
%type <obj>		formal_attribute_arguments
%type <string>		module_name
%type <obj>		variable_name
%type <string>		link_name
%type <string>		base_variable
%type <string>		component_name
%type <obj>		formal_variable
%type <obj>		type_name
%type <obj>		attribute_name
%type <string>		definition_name
%type <obj>		exception_name
%type <string>		user_exception_name
%type <string>		builtin_exception_name
%type <string>		exit_name
%type <obj>		result_variable
%type <obj>		variant_variable
%type <obj>		variant_component
%type <obj>		table_variable
%type <obj>		element_variable
%type <obj>		outport_variable
%type <obj>		inport_variable
%type <obj> 		callmessage_variable
%type <obj>		message_variable
%type <obj>		polymorph_variable
%type <string>		enumeration_variable
%type <obj> 		definitions_module_variable
%type <obj>		definitions_library_variable
%type <obj>		source_expression
%type <obj>		element_expression
%type <obj>		table_expression
%type <obj>		position_expression
%type <obj>		outport_expression
%type <obj>		outport_primary
%type <obj>		polymorph_expression
%type <obj>		select_expression
%type <obj>		opt_select_expression
%type <obj>		selector_expression
%type <obj>		test_expression
%type <obj>		constraint_expression
%type <obj>		unwrapped_typestate
%type <obj>		inspect_typestate
%type <obj>		component_typestate
%type <obj>		element_typestate
%type <obj>		entry_typestate
%type <obj>		exit_typestate
%type <obj> 		minimum_typestate
%type <obj>		exception_typestate
%type <string>		pragma
%type <string>		real_literal
%type <string>		integer_literal
%type <string>		string_literal
%type <string>		named_literal
%type <obj>		declaration_list
%type <obj>		component_declaration_list
%type <obj>		named_literal_list
%type <obj>		exception_name_list
%type <obj>		exception_name_NElist
%type <obj>		exit_name_list
%type <obj>		exit_name_NElist
%type <obj>		formal_attributes_done
%type <obj>		mapping_list
%type <obj>		mapping_NElist
%type <obj>		select_clause_s
%type <obj>		opt_else_clause
%type <obj>		opt_key_definition
%type <obj>		opt_minimum
%type <boolean>		opt_ordered
%type <obj>		opt_atposition
%type <string>		opt_pragma
%type <string>		opt_return_exception
%type <string>		identifier
%type <declp>		enumeration_declaration
%type <declp>		component_declaration
%type <obj>		enumeration_type
%type <obj>		element_type
%type <obj>		inport_type
%type <obj>		callmessage_type


/*
 * start symbol -- a module
 */

%start module

%%

/*TEXBEGIN*/

/* TEXCOMMENT
A module is introduced by a header which names the module and
optionally supplies lists of definitions modules to be imported and
(for process modules) other process modules to be statically linked.
The module body follows.
*/

/* GOAL SYMBOL */
module : module_name ':'
	mod_init
	opt_imports
	opt_linking
	/*TEXBREAK*/
	process_module_body
	{ p_mod_end(); } ;
module : module_name ':'
	mod_init
	opt_imports
	/*TEXBREAK*/
	definitions_module_body
	{ p_mod_end(); } ;

/* Note: following ugly kluge is required in order to avoid a */
/* reduce/reduce conflict in the rules for module above.  The problem */
/* is that an action is required after the module name is scanned but */
/* before the optional USING and LINKING lists are scanned.  If the */
/* action is included explicitly in the 'module' rules, yacc ends up */
/* making new, distinct generated nonterminals for the two instances, */
/* and ends up not knowing which of those two nonterminals to reduce */
/* when USING appears.  Instead, we explicitly make our own */
/* nonterminal to represent the action, and therefore need to use */
/* yacc's ugly mechanism for accessing values from enclosing rules. */
/*TEXINLINE*/
mod_init: /* empty */
	{ p_mod_init($<string>-1); } ;

imports : USING '(' module_name_list ')' ;
linking : LINKING '(' link_name_list ')' ;


/* TEXCOMMENT
A process module body begins with a declaration for the init port, and
what then follows looks almost like a block body: an optional
declarations section, followed by the statements forming the process
body, optionally followed by one or more exception handler
specifications.
*/

process_module_body : PROCESS
	{ init_proc_module(); }
	'(' declaration ')'
	/*TEXBREAK*/
	opt_pragma 
	{ init_executables(); p_init_process($4, $6, False); }
	/*TEXBREAK*/
	opt_declaration_section
	/*TEXBREAK*/
	BEGIN
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_series
	/*TEXBREAK*/
	handler_s
	/*TEXBREAK*/
	END PROCESS
	{ p_end_proc_module(); } ;


/* TEXCOMMENT
A declaration contains an identifier (the variable being declared) and
the name of its type.  A pragma is optional prior to the type name.
*/

declaration_section : DECLARE declaration_series ;

declaration : base_variable ':' opt_pragma type_name 
	{ $$ = p_decl($1, $4, $3); } ;


/* TEXCOMMENT
Statements come in two basic varieties: simple and compound.  Either
may optionally be preceded by a pragma.
*/

statement : opt_pragma simple_statement 
	{ p_set_stmt_pragma($1, $2); p_add_stmt($2); };
statement : opt_pragma compound_statement 
	{ p_set_stmt_pragma($1, $2); p_add_stmt($2); };

/* seems to be ok lalr-wise....
statement : simple_statement EXCEPT statement_handler_s 

statement_handler : ON '(' exception_name_list ')' statement ;
*/

/* TEXCOMMENT
The simple statements:
*/

simple_statement : result_variable move source_expression
	{ $$ = p_movestmt($1, $3); } ;

simple_statement : result_variable assign source_expression
	{ $$ = p_copystmt($1, $3); } ;

simple_statement : ASSERT attribute 
	{ p_unsupported(); } ;

/* fix object_name to primary by getting rid of semantic init action */
simple_statement : CALL outport_variable
	{ p_init_call($2); }
	call_arguments
	{ $$ = p_end_call(); } ;
simple_statement : CALL '(' outport_expression ')'
	{ p_init_call($3); }
	call_arguments
	{ $$ = p_end_call(); } ;

/* this is bogus */
simple_statement : CHECKDEFINITIONS definitions_module_variable
	/*TEXBREAK*/
	/*TEXSPACE*/
	AGAINST definitions_library_variable
	{ p_unsupported(); } ;
/*	{ $$ = p_binaryop(operator__checkdefinitions, $2, $4); } ; */

/* in a connect, the outport is the "result */
simple_statement : CONNECT outport_variable TO inport_variable
	{ $$ = p_binaryop(operator__connect, $2, $4); } ;
	
simple_statement : DISCARD variable_name
	{ $$ = p_unaryop(operator__discard, $2); } ;

simple_statement : DISSOLVE variant_component INTO result_variable
	{ $$ = p_binaryop(operator__dissolve, $4, $2); } ;

simple_statement : DROP attribute
	{ p_unsupported(); } ;

simple_statement : EXIT exit_name
	{ $$ = p_exit_statement($2); } ;

simple_statement : EXTRACT table_variable FROM selector
	{ $$ = p_remove_extract(operator__extract, $2, $4); } ;

simple_statement : HIDE variant_variable 
	{ $$ = p_unaryop(operator__hide, $2); } ;

simple_statement : INSERT element_expression INTO table_variable 
	/*TEXBREAK*/
	/*TEXSPACE*/
	opt_atposition
	{ if ($5)
	    $$ = p_insertormergeat(operator__insert_at, $4, $2, $5);
	  else
	    $$ = p_binaryop(operator__insert, $4, $2);
	} ;

/*TEXINLINE*/
atposition : AT position_expression
	{ $$ = $2; } ;

simple_statement : MERGE table_expression INTO table_variable 
	/*TEXBREAK*/
	/*TEXSPACE*/
	opt_atposition
	{ if ($5)
	    $$ = p_insertormergeat(operator__merge_at, $4, $2, $5);
	  else
	    $$ = p_binaryop(operator__merge, $4, $2);
	} ;

simple_statement : NEW variable_name 
	{ $$ = p_unaryop(operator__new, $2); } ;

simple_statement : PRINT expression
	{ $$ = p_print($2); } ;

simple_statement : RECEIVE message_variable FROM inport_variable
	{ $$ = p_binaryop(operator__receive, $2, $4); } ;

simple_statement : REMOVE element_variable FROM selector
	{ $$ = p_remove_extract(operator__remove, $2, $4); } ;

simple_statement : RETURN callmessage_variable
	/*TEXBREAK*/
	/*TEXSPACE*/
	opt_return_exception
	{ $$ = p_return($2, $3); } ;

/*TEXINLINE*/
return_exception : EXCEPTION user_exception_name
	{ $$ = $2; } ;

simple_statement : REVEAL variant_component
	{ $$ = p_unaryop(operator__reveal, $2); } ;

simple_statement : SEND source_expression TO outport_expression 
	{ $$ = p_send($4, $2); } ;

simple_statement : UNITE variant_component FROM source_expression
	{ $$ = p_binaryop(operator__unite, $2, $4); } ;

simple_statement : UNWRAP result_variable
	/*TEXBREAK*/
	/*TEXSPACE*/
	FROM polymorph_expression
	{ p_set_resenv(lookup_object_typename($2)); }
	unwrapped_typestate
	{ $$ = p_unwrap($2, $4, $6); } ;

/* fix later: predefined says there is a wrap_qualifier on this statement, */
/* but unless the user specifies it, we have no way of knowing what the */
/* typestate is until after typestate checking, at which point the absprog */
/* is constant anyway.  */
simple_statement : WRAP source_expression AS polymorph_variable
	{ $$ = p_binaryop(operator__wrap, $4, $2); } ;



/* TEXCOMMENT
The compound statements:
*/
compound_statement : BLOCK 
	{ p_init_block(); }
	/*TEXBREAK*/
	opt_constant_section
	/*TEXBREAK*/
	opt_declaration_section
	/*TEXBREAK*/
	BEGIN
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_series
	/*TEXBREAK*/
	handler_s
	/*TEXBREAK*/
	END BLOCK
	{ $$ = p_end_compound_scope(); } ;

/* TEXCOMMENT
Various syntactic elements appearing in compound statements:
*/
constant_section : CONSTANT '(' base_variable_list ')'
	{ fe_error(Warning, errorcode__general_error, "Block constants list not yet supported"); } ;

handler : ON '(' exception_name_list ')'
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_clause
	{ p_handler($3, $5); } ;
handler : ON EXIT '(' exit_name_list ')'
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_clause
	{ p_handler($4, $6); } ;


/*-------------------------- FOR Statements --------------------------------*/

compound_statement : FOR selector
	{ p_init_inspect(operator__for_inspect, $2); }
	INSPECT
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_series
	/*TEXBREAK*/
	END FOR
	{ $$ = p_end_compound_scope(); } ;

compound_statement : FOR enumeration_declaration
	{ p_init_for_enumerate($2); }
	REPEAT
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_series
	/*TEXBREAK*/
	END FOR
	{ $$ = p_end_compound_scope(); } ;


/*---------------------------- IF Statement --------------------------------*/

compound_statement : IF test_expression
	/*TEXBREAK*/
	/*TEXSPACE*/
	THEN statement_clause
	/*TEXBREAK*/
	/*TEXSPACE*/
	opt_else_clause
	/*TEXBREAK*/
	END IF
	{ $$ = p_if($2, $4, $5); } ;

/*TEXINLINE*/
else_clause : ELSE statement_clause
	{ $$ = $2; } ;

/*-------------------- INSPECT Table or Polymorph --------------------------*/

compound_statement : INSPECT selector BEGIN
	{ p_init_inspect(operator__inspect_table, $2); }
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_series
	/*TEXBREAK*/
	END INSPECT
	{ $$ = p_end_compound_scope(); } ;

compound_statement : INSPECT declaration
	/*TEXBREAK*/
	/*TEXSPACE*/
	FROM polymorph_expression
	{ p_set_resenv($2->typename); }
	inspect_typestate
	{ p_inspect_polymorph($2, $4, $6) ; }
	/*TEXBREAK*/
	BEGIN
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_series
	/*TEXBREAK*/
	END INSPECT
	{ $$ = p_end_compound_scope(); } ;


/*-------------------------- SELECT Statement ------------------------------*/

compound_statement : SELECT opt_select_expression
	/*TEXBREAK*/
	/*TEXSPACE*/
	select_clause_s
	/*TEXBREAK*/
	/*TEXSPACE*/
	otherwise_clause
	/*TEXBREAK*/
	END SELECT
	{ $$ = p_select($2, $3, $4); } ;

select_clause : boolean_guard statement_clause
	{ $$ = p_select_guard_boolean($1, $2); } ;
select_clause : event_guard statement_clause
	{ $$ = p_select_guard_event($1, $2); } ;
select_clause : event_guard AND boolean_guard statement_clause
	{ $$ = p_select_guard_both($1, $3, $4); } ;

event_guard : EVENT inport_variable
	{ $$ = $2; } ;

boolean_guard : WHERE '(' test_expression ')'
	{ $$ = $3; } ;

otherwise_clause : OTHERWISE statement_clause
	{ $$ = $2; } ;

/*------------------------- WHILE Statement --------------------------------*/

compound_statement : WHILE test_expression REPEAT
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_clause
	/*TEXBREAK*/
	END WHILE
	{ $$ = p_while($2, $4); } ;

/*TEXINLINE*/
enumeration_declaration : enumeration_variable ':' enumeration_type
	{ $$ = p_decl($1, $3, (char *) nil); } ;

/*TEXINLINE*/
statement_clause :
	{ p_init_statement_clause(); }
	statement_series
	{ $$ = p_end_statement_clause(); } ;
	

/* TEXCOMMENT
Three syntaxes are available for selectors (used in table operations):
a long form (using \prkw{where}), and two shorthand forms (using square
brackets -- \pr{[$\ldots$]}).
*/

selector : base_variable IN table_variable
	/*TEXBREAK*/
	/*TEXSPACE*/
	WHERE
	{ p_init_selector($1, $3); }
	'(' selector_expression ')'
	{ $$ = p_selector($7); } ;
selector : base_variable IN table_variable
	/*TEXBREAK*/
	/*TEXSPACE*/
	'['
	{ p_init_selector($1, $3); }
	mapping_list ']'
	{ $$ = p_selector($6); } ; 
selector : table_variable '['
	{ p_init_selector(nil, $1); }
	mapping_list ']'
	{ $$ = p_selector($4); } ; 

/*TEXINLINE*/
mapping: expression
	{ $$ = make_mapping($1); };


/* TEXCOMMENT
The Hermes expression syntax follows.  All the Hermes binary operators
are left-associative, though the grammar shown here does not
explicitly represent this.
*/

/*TEXINLINE*/
expression_clause :
	{ p_init_expression_clause(); }
	expression
	{ $$ = p_end_expression_clause($2); } ;

expression : disjunction
	{ $$ = $1; } ;

disjunction : conjunction
	{ $$ = $1; } ;
disjunction : disjunction OR conjunction
	{ $$ = p_binary_expr(operator__or, $1, $3); } ;

conjunction : relation
	{ $$ = $1; } ; 
conjunction : conjunction AND relation
	{ $$ = p_binary_expr(operator__and, $1, $3); } ;

relation : concat
	{ $$ = $1; } ; 
relation : concat '=' concat
	{ $$ = p_binary_expr(operator__equal, $1, $3); } ; 
relation : concat '<' concat
	{ $$ = p_binary_expr(operator__less, $1, $3); } ; 
relation : concat '>' concat
	{ $$ = p_binary_expr(operator__greater, $1, $3); } ; 
relation : concat nequal concat
	{ $$ = p_binary_expr(operator__not_equal, $1, $3); } ;
relation : concat lequal concat
	{ $$ = p_binary_expr(operator__less_equal, $1, $3); } ;
relation : concat gequal concat
	{ $$ = p_binary_expr(operator__greater_equal, $1, $3); } ;

concat : term
	{ $$ = $1; } ; 
concat : concat '|' term
	{ $$ = p_binary_expr(operator__concatenate, $1, $3); } ;

term : factor
	{ $$ = $1; } ; 
term : '-' factor
	{ $$ = p_unary_expr(operator__unary_minus, $2); } ; 
term : term '+' factor
	{ $$ = p_binary_expr(operator__add, $1, $3); } ; 
term : term '-' factor
	{ $$ = p_binary_expr(operator__subtract, $1, $3); } ;

factor : secondary
	{ $$ = $1; } ; 
factor : factor '*' secondary
	{ $$ = p_binary_expr(operator__multiply, $1, $3); } ; 
factor : factor '/' secondary
	{ $$ = p_binary_expr(operator__divide, $1, $3); } ; 
factor : factor MOD secondary
	{ $$ = p_binary_expr(operator__mod, $1, $3); } ; 
factor : factor REM secondary
	{ $$ = p_binary_expr(operator__rem, $1, $3); } ;

/*--------------------------------------------------------------------------*/
/*------------------------------ Secondaries -------------------------------*/
/*--------------------------------------------------------------------------*/

secondary : primary
	{ $$ = $1; } ;

secondary : the_element
	{ $$ = p_unary_table_expr(operator__the_element, $1); } ;

secondary : CASE OF secondary
	{ $$ = p_unary_expr(operator__case, $3); } ;

secondary : CONVERT OF secondary
	{ $$ = p_unary_expr(operator__convert, $3); } ;

secondary : COPY OF secondary
	{ $$ = p_unary_expr(operator__copy, $3); } ;

secondary : CREATE OF secondary
	{ $$ = p_unary_expr(operator__create, $3); } ;

secondary : EMPTY OF secondary
	{ $$ = p_unary_expr(operator__empty, $3); } ;

secondary : EVALUATE declaration FROM
	{ p_init_exprblock($2); }
	statement_series END
	{ $$ = p_end_exprblock(); } ;

secondary : EVERY OF selector
	{ $$ = p_unary_table_expr(operator__every, $3); } ;

secondary : EXISTS OF selector
	{ $$ = p_unary_table_expr(operator__exists, $3); } ;

secondary : FORALL OF selector
	{ $$ = p_unary_table_expr(operator__forall, $3); } ;

secondary : NOT secondary
	{ $$ = p_unary_expr(operator__not, $2); } ;

secondary : POSITION OF element_variable
	{ $$ = p_unary_expr(operator__position_of_element, $3); } ;
secondary : POSITION OF selector
	{ $$ = p_unary_table_expr(operator__position_of_selector, $3); } ;

secondary : PROCEDURE OF secondary
	{ $$ = p_unary_expr(operator__procedure, $3); } ;

secondary : SIZE OF secondary
	{ $$ = p_unary_expr(operator__size, $3); } ;

secondary : TYPE OF secondary
	{ $$ = p_unary_expr(operator__type, $3); } ;

secondary : TYPESTATE OF secondary
	{ $$ = p_unary_expr(operator__typestate, $3); } ;

/*--------------------------------------------------------------------------*/
/*------------------------------ Primaries ---------------------------------*/
/*--------------------------------------------------------------------------*/

primary : type_specifier typed_primary
	{ $$ = p_typed_expr($1, $2); } ; 
primary : typed_primary
	{ $$ = $1; } ; 
primary : variable_name
	{ $$ = $1; } ; 
primary : function_reference
	{ $$ = $1; } ;

type_specifier : type_name '#'
	{ $$ = $1; } ;

typed_primary : literal
	{ $$ = $1; } ; 
typed_primary : '(' expression ')'
	{ $$ = $2; } ; 
typed_primary : CURRENTPROGRAM
	{ $$ = p_niladic_expr(operator__currentprogram); } ;
typed_primary : UNIQUE
	{ $$ = p_niladic_expr(operator__unique); } ; 


the_element : selector
	{ $$ = $1; } ;

function_reference : outport_primary
	{ p_init_call($1); }
	call_arguments
	{ $$ = p_function(); } ;

/* TEXCOMMENT
Literals of various types:
*/

literal : integer_literal
	{ $$ = p_literal(operator__integer_literal, $1); } ; 
literal : real_literal
	{ $$ = p_literal(operator__real_literal, $1); } ; 
literal : named_literal
	{ $$ = p_literal(operator__named_literal, $1); } ; 
literal : string_literal
	{ $$ = p_literal(operator__string_literal, $1); } ; 
literal : program_literal
	{ $$ = p_qliteral(operator__program_literal, $1,
			  qualifier_types__program_literal); } ; 
literal : attributename_literal
	{ $$ = $1; } ;
literal : typename_literal
	{ $$ = $1; } ;

program_literal : PROCESS '(' declaration ')'
	/*TEXBREAK*/
	opt_pragma
	{ p_init_process($3, $5, True); }
	/*TEXBREAK*/
	opt_declaration_section
	/*TEXBREAK*/
	BEGIN
	/*TEXBREAK*/
	/*TEXSPACE*/
	statement_series
	/*TEXBREAK*/
	handler_s
	/*TEXBREAK*/
	END PROCESS
	{ $$ = p_end_process(); } ; 
program_literal : PROCESS link_name
	{ $$ = p_link_process($2); } ;

/* double backslashes are used since they are quoting characters */
attributename_literal : '\\' ATTRIBUTENAME attribute_name '\\'
	{ $$ = p_qliteral(operator__attributename, $3,
			  qualifier_types__attributename); } ;
typename_literal : '\\' TYPENAME type_name '\\'
	{ $$ = p_qliteral(operator__typename, $3,
			  qualifier_types__typename); } ;


/* TEXCOMMENT
A definitions module comprises a collection of type and constraint
definitions.  A type definition associates a name with a type
construction, while a constraint defininition associates a name with
an attribute construction.  A pragma may optionally appear with either
type of definition.
*/

definitions_module_body : DEFINITIONS
	{ p_init_def_module(); }
	/*TEXBREAK*/
	/*TEXSPACE*/
	definition_series
	/*TEXBREAK*/
	END DEFINITIONS
	{ p_end_def_module(); } ;

definition : definition_name ':' opt_pragma
	{ p_def_name($1, $3); }
	construction ;

construction : type_construction ; 
construction : attribute_construction ;

/* TEXCOMMENT
Type constructions:
*/

type_construction : BOOLEAN '(' boolean_association ')' ;

/* TEXCOMMENT
Various syntactic elements used in type definitions:
*/

boolean_association : TRUE ':' named_literal ',' FALSE ':' named_literal
	{ p_boolean_def($3, $7); } ; 
boolean_association : FALSE ':' named_literal ',' TRUE ':' named_literal
	{ p_boolean_def($7, $3); } ;

/* resolution environment for the formal typestates is the callmessage def */
type_construction : CALLMESSAGE
	{ p_init_callmsg_def(); }
	'('
	/*TEXBREAK*/
	/*TEXSPACE*/
	component_declaration_list
	/*TEXBREAK*/
	')'
	{ p_set_callmsg($4); }
	/*TEXBREAK*/
	opt_constant_parameters
	/*TEXBREAK*/
	EXIT exit_typestate
	/*TEXBREAK*/
	opt_minimum
	/*TEXBREAK*/
	user_exception_s
	{ p_callmsg_def($9, $10); } ;

constant_parameters : CONSTANT '(' component_name_list ')' ;

minimum : MINIMUM minimum_typestate
	{ $$ = $2; } ;

user_exception : EXCEPTION user_exception_name exception_typestate
	{ p_callmsg_exception($2, $3); } ;

type_construction : opt_ordered ENUMERATION '('
	/*TEXBREAK*/
	/*TEXSPACE*/
	named_literal_list
	/*TEXBREAK*/
	')'
	{ p_enum_def($1, $4); } ;

/*TEXINLINE*/
ordered : ORDERED ;

/* the resolution environment for the formal typestate is the definition of */
/* the named type. */
type_construction : INPORT OF callmessage_type
	{ p_set_resenv($3); }
	entry_typestate
	{ p_inport_def($3,$5); } ;

type_construction : INTEGER
	{ p_integer_def(); } ;

type_construction : NOMINAL
	{ p_nominal_def(); } ;

type_construction : OUTPORT OF inport_type
	{ p_outport_def($3); } ;

type_construction : POLYMORPH
	{ p_polymorph_def(); } ;

type_construction : REAL OF ACCURACY integer_literal '/' integer_literal
	{ p_real_def($4,$6); } ;

type_construction : RECORD
	{ p_init_record_def(); }
	'('
	/*TEXBREAK*/
	/*TEXSPACE*/
	component_declaration_list
	/*TEXBREAK*/
	')'
	{ p_record_def($4); } ;

/* the resolution environment for the formal typestate and the formal */
/* variable names is the definition of the named type. */
type_construction : opt_ordered TABLE OF element_type
	{ p_set_resenv($4); }
	element_typestate
	/*TEXBREAK*/
	/*TEXSPACE*/
	opt_key_definition
	{ p_table_def($1, $4, $6, $7); } ;

/*TEXINLINE*/ 
key_definition : KEYS
	{ p_init_keyset(); }
	key_s
	{ $$ = p_get_keyset(); } ;

key : 	'(' 
	{ p_init_formal_object_name_list(); }
	formal_variable_list 
	')'
	{ p_add_key(p_get_formal_object_name_list()); } ;

type_construction : VARIANT OF enumeration_type
	{ p_init_variant_def($3); }
	'('
	/*TEXBREAK*/
	/*TEXSPACE*/
	case_declaration_list
	/*TEXBREAK*/
	')'
	{ p_variant_def(); } ;

/* the resolution environment for the formal typestate is the definition */
/* of the declared component. */

case_declaration : named_literal names_case component_declaration
	{ p_set_resenv($3->typename); }
	component_typestate
	{ p_add_variant_case($1, $3, $5); };

component_declaration : declaration
	{ $$ = $1; } ;

/* TEXCOMMENT
Attribute constructions:
*/

attribute_construction : CONSTRAINT '('
	/*TEXBREAK*/
	/*TEXSPACE*/
	declaration_list
	/*TEXBREAK*/
	')'
	/*TEXBREAK*/
	IS constraint_typestate constraint_expression
	{ p_unsupported(); } ;

/* TEXCOMMENT
Typestate specifications appear in attribute constructions.  A
typestate specification consists of a collection of attributes, each
of which includes an attribute name and a list of arguments.  Formal
typestates appear in the \prkw{unwrap} and \prkw{for-inspect}
statements, and in several type constructions (e.g.,  variants, tables,
inports).  In a formal typestate specification, each attribute
argument is taken to be a component of an assumed base variable which
depends on the context.
*/

typestate : '{' attribute_list '}' ;

attribute : attribute_name attribute_arguments ;

/* formal typestates */

formal_typestate : '{'
	{ p_init_formal_typestate(); }
	formal_attribute_list '}'
	{ $$ = p_get_formal_typestate(); } ;

formal_attribute : attribute_name formal_attribute_arguments
	{ $$ = p_formal_attribute($1, $2); } ;

/* TEXCOMMENT
Argument lists appear in various contexts, including \prkw{call}
statements and typestate attributes.  They match actual arguments to
parameters.  Positional matching is achieved with a comma-separated
list of arguments.  A keyword-based arguments list explicitly includes
the parameter names so that position is unimportant.

The various types of argument lists are distinguished only by what is
allowed as an argument.
*/

call_arguments : '(' expression_list ')' ;
/* this isn't a _list, since if it were it would cause an */
/* ambiguity -- both could generate the empty list "()" */
call_arguments : '(' labeled_expression_NElist ')'
	{ p_unsupported(); } ;

labeled_expression : identifier ':' expression ;

attribute_arguments : '(' variable_name_list ')' ;
attribute_arguments : '(' labeled_variable_name_NElist ')' ;

labeled_variable_name : identifier ':' variable_name
	{ p_unsupported(); } ;

formal_attribute_arguments : empty
	{ $$ = p_formal_whole_object(); } ; 
formal_attribute_arguments : '(' 
	formal_attributes_init formal_variable_list ')'
	formal_attributes_done
	{ $$ = $5; } ;

formal_attribute_arguments : '(' 
	formal_attributes_init labeled_formal_variable_NElist ')'
	formal_attributes_done
	{ p_unsupported(); } ; 
labeled_formal_variable : identifier ':' formal_variable ;

/* TEXINLINE */
formal_attributes_init :
	{ p_init_formal_object_name_list(); } ;
/* TEXINLINE */
formal_attributes_done :
	{ $$ = p_get_formal_object_name_list(); } ;

/* TEXCOMMENT
Names:
*/

attribute_name : definition_name
	{ $$ = p_attr_info(p_qual_name(nil, $1)); } ;
attribute_name : module_name '!' definition_name
	{ $$ = p_attr_info(p_qual_name($1, $3)); } ;

base_variable : identifier
	{ $$ = $1; } ;

builtin_exception_name : identifier
	{ $$ = $1; } ;

component_name : identifier
	{ $$ = $1; } ;

definition_name : identifier
	{ $$ = $1; } ;

exception_name : type_name '.' user_exception_name
	{ $$ = p_user_exception($1, $3); } ; 
exception_name : builtin_exception_name
	{ $$ = p_builtin_exception($1); } ;

exit_name : identifier
	{ $$ = $1; } ;

formal_variable : '*'
	{ $$ = p_whole_formal_object(); } ; 
formal_variable : component_name
	{ p_init_formal_object_name(); p_add_component_name($1); }
	dot_component_s
	{ $$ = p_get_formal_object_name(); p_reset_resenv(); } ;

link_name : identifier
	{ $$ = $1; } ;

module_name : identifier
	{ $$ = $1; } ;

type_name : definition_name
	{ $$ = p_type_or_attr_name(p_qual_name(nil, $1), True); } ;
type_name : module_name '!' definition_name
	{ $$ = p_type_or_attr_name(p_qual_name($1, $3), True); } ;

user_exception_name : identifier
	{ $$ = $1; } ;

variable_name : base_variable
	{ p_init_object_name($1); }
	dot_component_s
	{ $$ = p_get_object_name(); } ;

/*TEXINLINE*/
dot_component : '.' component_name
	{ p_add_component_name($2); } ;
	
/* TEXCOMMENT
Variable names appearing in particular contexts:
*/

callmessage_variable : variable_name
	{ $$ = $1; } ;

definitions_library_variable : variable_name
	{ $$ = $1; } ;

definitions_module_variable : variable_name
	{ $$ = $1; } ;

element_variable : variable_name
	{ $$ = $1; } ;

enumeration_variable : base_variable
	{ $$ = $1; } ;

inport_variable : variable_name
	{ $$ = $1; } ;

message_variable : variable_name
	{ $$ = $1; } ;
outport_variable : variable_name
	{ $$ = $1; } ;

polymorph_variable : variable_name
	{ $$ = $1; } ;

result_variable : variable_name
	{ $$ = $1; } ;

table_variable : variable_name
	{ $$ = $1; } ;

variant_variable : variable_name
	{ $$ = $1; } ;

variant_component : variable_name
	{ $$ = $1; } ;

/* TEXCOMMENT
Expressions appearing in particular contexts:
*/

constraint_expression : expression
	{ $$ = $1; } ;

element_expression : expression
	{ $$ = $1; } ;

outport_expression : expression
	{ $$ = $1; } ;

outport_primary : primary
	{ $$ = $1; } ;

polymorph_expression : expression
	{ $$ = $1; } ;

position_expression : expression
	{ $$ = $1; } ;

select_expression : expression
	{ $$ = $1; } ;

selector_expression : expression
	{ $$ = $1; } ;

source_expression : expression
	{ $$ = $1; } ;

table_expression : expression
	{ $$ = $1; } ;

test_expression : expression_clause
	{ $$ = $1; } ;

/* TEXCOMMENT
Types appearing in particular contexts:
*/

callmessage_type : type_name
	{ $$ = $1; } ;

element_type : type_name
	{ $$ = $1; } ;

enumeration_type : type_name
	{ $$ = $1; } ;

inport_type : type_name
	{ $$ = $1; } ;

/* TEXCOMMENT
Typestates appearing in particular contexts:
*/
component_typestate : formal_typestate
	{ $$ = $1; } ;

constraint_typestate : typestate ;

element_typestate : formal_typestate
	{ $$ = $1; } ;

entry_typestate : formal_typestate
	{ $$ = $1; } ;

exception_typestate : formal_typestate
	{ $$ = $1; } ;

exit_typestate : formal_typestate
	{ $$ = $1; } ;

inspect_typestate : formal_typestate
	{ $$ = $1; } ;

minimum_typestate : formal_typestate
	{ $$ = $1; } ;

unwrapped_typestate : formal_typestate
	{ $$ = $1; } ;


/* TEXCOMMENT
Various odds and ends.  Note that the pragma syntax is currently very
open-ended.  It may become more structured in the future.
*/

pragma : PRAGMA string_literal 
	{ $$ = $2; } ;

empty : ;

/*TEXEND*/


/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*---------------------------- Error Handling ------------------------------*/
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

simple_statement : error ';' 
	{ $$ = (objectp) nil; } ; /* fix later */
construction : error ';' ; 

declaration : error ';' 
	{ $$ = (decl *) nil; } ;  /* fix later */

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
/*----------------------------- UGLY ---------------------------------------*/
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* Here come the parts of the grammar not fit for human consumption */

names_case : T_NAMESCASE ;
move : T_MOVE ;
assign : T_ASSIGN ;
nequal : T_NEQUAL ;
lequal : T_LEQUAL ;
gequal : T_GEQUAL ;

real_literal : T_REAL 
	{ $$ = copystring(yytext); } ;

integer_literal : T_INTEGER 
	{ $$ = copystring(yytext); } ;

string_literal : T_STRING
	{ $$ = copystring(yytext); } ;

named_literal : T_NAME
	{ $$ = copystring(yytext); } ;


/* include automatically generated productions */

include(list.yacc)
include(kleene.yacc)
include(series.yacc)
include(opt.yacc)


/*
 * identifier -- an identifier is either a symbol or a non-reserved keyword.
 * the production is produces automatically from the files "keywords" and
 * "reswords", and Included with m4
 *
 * note that it does something that looks odd: instead of just returning a copy
 * of yytext in all cases, it only does so when the token is a symbol.  when
 * it is something like "pragma", which might be a keyword, it returns a copy
 * of that literal string.  this is because by the time we decide that "pragma"
 * is an identifier, we will already have looked ahead by one, and yytext will
 * contain the string value of the next token.
 */

include(ident.yacc)

%%
