-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: pcom.p
-- Author: David F. Bacon & Andy Lowry
-- SCCS Info: @(#)pcom.pp	1.37 2/17/92

-- This process drives the various phases in the compilation of one or
-- more Hermes process modules, namely: parsing, type checking,
-- typestate checking, and code generation.  Any phase can be
-- optionally suppressed by means of a command line switch.  The
-- switches and their consequences (roughly) are as follows:

--   -noParse
--	The source module is not parsed.  In order for any other stage
--	to succeed, an ".ao" file for the module must exist.
--   -noTCheck
--	The abstract program is not checked for type errors, and no
--	type inferencing is done.  Subsequent code generation may fail
--	if the source program was not completely typemarked.  Bizarre
--	run-time errors can result from incorrect programs, since
--	things like passing an object of the wrong type on an outport will 
--	be allowed.
--   -noTSCheck
--      The absprob is not checked for typestate errors, and no
--      coercions are generated.  Subsequent code generation should
--      produce an "apparently" correct compiled program, assuming the
--      program is correct.  Since coercions are not generated,
--      however, the resulting program will probably contain memory
--      leaks.  If the program is not typestate correct, skipping this
--      checking phase can result in bizarre runtime errors since
--      things like accessing fields of uninit records will be allowed.
--   -noObject
--	No compiled program object will be generated.  This is useful
--      for submitting a program to type or typestate checking only,
--      without overwriting an existing .po file, or in conjunction
--      with the -LIGen option
--   -LIGen
--      Disassemble the compiled module and write the result to a file
--      with extension '.li'
--   -strip
--      absprog.programs and absprog.definitions_modules will be 
--      set to empty tables before storing.
--   -stripdefs 
--      absprog.definitions_modules will be set to an empty table before
--      storing.
--   -keeppredef
--      The definitions module predefined will _not_ be removed from
--      absprog.definitions_modules before storing.  By default it is
--      removed.  If both -strip and -keeppredefs are specified, 
--      everything _except_ predefined will be stripped away.

-- This module is compiled four (4) times with different conditional 
-- compilation switches during the process of bootstrapping the compiler.
-- The phases are as follows:
--
-- CGBOOT 
-- Initial bootstrapping of the code generator.  Compile without the integrated
-- parser (only use loadprog), the type checker, or the typestate checker.
-- stdenv has getCwd, setCwd, libWriteObj, libStore, and terminal
-- uninit.
--
-- TCBOOT
-- Used to compile the type checker.  Gets a full stdenv, but does no
-- type or typestate checking.
--
-- TSBOOT
-- Used to compile the typestate checker.  Does no typestate checking.
--
-- No #ifdefs
-- Compiles the whole module.

-- Enforce strict hierarchy in the xxBOOT cpp variables
#ifdef CGBOOT
#  ifndef TCBOOT
#    define TCBOOT
#  endif
#endif

#ifdef TCBOOT
#  ifndef TSBOOT
#    define TSBOOT
#  endif
#endif

#ifdef TSBOOT
#  define ANYBOOT
#endif

#include "typemark.h"

-- following is required as a workaround for an AIX bug on the Rios
#define addspace(x,y) x y

#define AREF(elt,t,i) elt in t where (B(I(addspace(position of,elt)) = i))

pCom: using (main, common, terminalIO, stdenv, CLoad, root, unix, cwd,
  -- imports for all phases
  fileDef, posMap, errors, annotate, loadProg, parse, formatError, load
  -- imports for codegen
  , codegen, inferredType, listuff, interpform
#ifndef CGBOOT
  -- imports for parsing
  , getFile, initParse, findfile, initFindfile, pathload
#endif CGBOOT

#ifndef TCBOOT
  -- imports for type checking
  , typecheck, checking_table
#endif TCBOOT

#ifndef TSBOOT
  -- imports for typestate checking
  , typestate, coercions
#endif TSBOOT

#ifndef ANYBOOT
  -- imports for services beyond typestate checking
  , disassembler, sortErrors
#endif ANYBOOT
)
linking (liStuff, liUnstuff)

process (Q: mainQ)
  
declare
  args: main;
  argv: charstringList;
  options: annotations;
  procErrors: procErrors;
  errors: errors!errors;
  errorStrings: charStringList;
  codegenInit: codegenInitFn;
  fakeCodegenInit: codegenInitFn;
  codegen: codegenFn;
  loadProg: loadProgFunc;
  fakeLoadProg: loadProgFunc;
  putString: putStringFunc;
  putLine: putStringFunc;
  stdenv: stdenv;
  CLoad: CLoadFn;
  environment: root!environ;
  stdio: stdio;
  codeMap: codeMap;
  linkedCodeMap: codeMap;
  liStuff: liStuffFn;
  liUnstuff: liUnstuffFn;

#ifndef CGBOOT
  parse: parseProcFn;
  getFile: getFileFunc;
  getFileInit: getFileInitFunc;
  fakeGetFile: getFileFunc;
  formatError: formatErrorFn;
  fakeFormatError: formatErrorFn;
#endif CGBOOT

#ifndef TCBOOT
  checking_table: checking_table;
  typeCheck: typeCheckCapa;
  fakeTypeCheck: typeCheckCapa;
#endif TCBOOT

#ifndef TSBOOT
  tsCheck: typestateCheckOutport;
  fakeTsCheck: typestateCheckOutport;
#endif TSBOOT
  
#ifndef ANYBOOT
  disasm: disassemblerFn;
  fakeDisasm: disassemblerFn;
  sortErrs: sortErrorsFn;
  sortPErrs: sortProcErrorsFn;
#endif ANYBOOT
  
begin
  receive args from Q;
  
  -- Get some capabilities we need from the resource manager
  unwrap CLoad from polymorph#(args.rm.get(S("CLoader"), S(""))) {init};
  unwrap environment from polymorph#(args.rm.get(S("environ"), S(""))) {init};
#ifndef CGBOOT
  unwrap stdio from polymorph#(args.rm.get(S("stdio"), S(""))) 
      {init, init(fopen), init(access)};
#else
  new stdio;
  unwrap stdio.access from polymorph#(CLoad(S("Access"))) {init};
#endif

  -- build up a stdenv for compiler modules that were written with the
  -- old main interface in mind.  For CGBOOT the stdenv is not
  -- completely initialized, even though the codegen interface says it
  -- is.
  new stdenv;
  unwrap stdenv.load from polymorph#(args.rm.get(S("load"), S(""))) {init};
  unwrap stdenv.pathLoad from polymorph#(args.rm.get(S("pathLoad"), S("")))
      {init};
  unwrap stdenv.readObj from polymorph#(args.rm.get(S("readObj"), S(""))) 
      {init};
  unwrap stdenv.pathReadObj from polymorph#(
    args.rm.get(S("pathReadObj"), S(""))) {init};
  unwrap stdenv.writeObj from polymorph#(args.rm.get(S("writeObj"), S(""))) 
      {init};
  unwrap stdenv.store from polymorph#(args.rm.get(S("store"), S(""))) {init};

#ifndef CGBOOT
  unwrap stdenv.libWriteObj from polymorph#(
    args.rm.get(S("libWriteObj"), S(""))) {init};
  unwrap stdenv.libStore from polymorph#(args.rm.get(S("libStore"), S(""))) 
      {init};
  unwrap stdenv.getCwd from polymorph#(args.rm.get(S("getCwd"), S(""))) {init};
  unwrap stdenv.setCwd from polymorph#(args.rm.get(S("setCwd"), S(""))) {init};
  stdenv.terminal := args.terminal;
#endif CGBOOT
  
  -- Get capabilities to processes we may need, and get useless fake
  -- capabilities for processes that we might be able to avoid loading
  block declare
    loadIn: loadProgQ;
    cgIn: codegenInitQ;
  begin
    new loadIn;
    connect fakeLoadProg to loadIn;
    loadProg := fakeLoadProg;
    new cgIn;
    connect fakeCodegenInit to cgIn;
    codegenInit := fakeCodegenInit;
    liStuff <- liStuffFn#(procedure of program#(process liStuff));
    liUnstuff <- liUnstuffFn#(procedure of program#(process liUnstuff));
  end block;
  
#ifndef CGBOOT
  -- Set up the parser, which is a c-hermes function.  It is
  -- initialized with capas to some other standard processes.  Also
  -- get a fake capability to the error formatter and to getfile,
  -- which we may not need to load
  block declare
    initParse: initParseProcFn;
    fns: parseFns;
    fmtErrIn: formatErrorQ;
#ifndef ANYBOOT
    sortErrsIn: sortErrorsQ;
    sortPErrsIn: sortProcErrorsQ;
#endif ANYBOOT
    getFileIn: getFileQ;
    pathinit: initFindFileFn;
    bindir: charstring;		-- required by getpaths.pp, but unused
    defpath: charstring;
    syspath: charstring;
    linkpath: charstring;	-- this one we really need
    linkpathfinder: findfile_func;
    pathloadinit: pathload_init_func;
    ENVIRON: root!environ;
  begin
    unwrap initParse from polymorph#(CLoad(S("Parse Proc"))) {init};
    new fns;
    -- get a path loader that obeys HLINKPATH if set
    ENVIRON := environment;
#include "getpaths.pp"
    pathinit <- initFindFileFn#(procedure of program#(
	stdenv.pathLoad(S("findfile"))));
    linkpathfinder <- findfile_func#(
      pathinit(stdio.access, stdenv.getCwd, linkpath));
    pathloadinit <- pathload_init_func#(create of program#(
	stdenv.pathLoad(S("pathload"))));
    fns.load := load_func#(pathloadinit(stdenv.load, linkpathfinder));
    fns.store := stdenv.libStore;
    fns.readObj := stdenv.pathReadObj;
    fns.writeObj := stdenv.libWriteObj;
    unwrap fns.writeObj from polymorph#(args.rm.get(S("libWriteObj"), S(""))) 
	{init};
    fns.getCwd := stdenv.getCwd;
    parse <- parseProcFn#(initParse(fns));
    new fmtErrIn;
    connect fakeFormatError to fmtErrIn;
    formatError := fakeFormatError;
#ifndef ANYBOOT
    -- these need to beinitialized too, but will be connected to the
    -- real thing at the same time as formatError, so we don't need
    -- "fake" outports to check if we've loaded the real programs
    new sortErrsIn;
    connect sortErrs to sortErrsIn;
    new sortPErrsIn;
    connect sortPErrs to sortPErrsIn;
#endif ANYBOOT
    new getFileIn;
    connect fakeGetFile to getFileIn;
    getFile := fakeGetFile;
  end block;
#endif CGBOOT

#ifndef TCBOOT
  -- start with an empty checking table to satisfy typestate checking,
  -- but don't load it yet since we may not need it
  new checking_table;

  -- Get fake capabilities to the type checker, which we may not need
  -- to load
  block declare
    tcIn: typeCheckQueue;
  begin
    new tcIn;
    connect fakeTypeCheck to tcIn;
    typeCheck := fakeTypeCheck;
  end block;
#endif TCBOOT
  
#ifndef TSBOOT
  -- Get fake capability to typestate checker, in case we don't need
  -- to load them
  block declare
    tsIn: typestateCheckInport;
  begin
    new tsIn;
    connect fakeTsCheck to tsIn;
    tsCheck := fakeTsCheck;
  end block;
#endif TSBOOT

#ifndef ANYBOOT
  -- Get fake capabilities to disassembler in case we don't need it
  block declare
    disIn: disassemblerQ;
  begin
    new disIn;
    connect fakeDisasm to disIn;
    disasm := fakeDisasm;
  end block;
#endif ANYBOOT

  -- Shortcuts to simple i/o routines
  putLine := args.terminal.putLine;
  putString := args.terminal.putString;

  -- Pick out command line options
  new options;
  block declare
    option: annotation;
    optnames: charstringList;
    name: charstring;
    value: charstring;
    empty: empty;
    caseDiff: integer;
  begin
				-- for case conversions...
    caseDiff <- I(I(convert of C('a')) - I(convert of C('A')));
    -- get all arg strings that begin with a hyphen
    argv := args.argv;
    extract optnames from word in argv 
       where (B(C(AREF(tmp,word,ZERO)) = C('-')));
    -- incorporate all of the environmental variables, prefixed by "+"
    for e in environment[] inspect
	name <- S(S("+") | e.variable);
	if B(e.value = S("")) then
	    insert name into optnames;
	  else 
	    insert S(name | S(S("=") | e.value)) into optnames;
	  end if;
      end for;
    
    -- Turn each word into an option in the options table.  If the
    -- word begins "+pcom" it is a relevant environmental variable,
    -- and the effective word follows that prefix.  If the effective
    -- word contains an embedded equal sign ('='), the portion before
    -- the equal sign becomes the option name, and the following
    -- portion becomes the option value (a charstring)
    for optname in optnames[] inspect
      new option;
      new option.name;
      block begin
	inspect c in optname where (B(c = C('='))) begin
	  -- break up name and value
	  name <- charString#(every of c1 in optname 
		where (B(I(position of c1) < I(position of c))));
	  value <- charString#(every of c1 in optname 
		 where (B(I(position of c1) > I(position of c))));
	  wrap value as option.thing;	
	end inspect;
      on (NotFound)
	-- no equal sign... value is empty
	name := optname;
	wrap empty as option.thing;
      end block;
      -- Convert option name to lower case
      for c in name where (B(I(position of c) > ZERO)) inspect
	if B(B(c >= C('A')) and B(c <= C('Z'))) then
	  insert C(convert of I(I(convert of c) + caseDiff))
	      into option.name;
	else
	  insert C(copy of c) into option.name;
	end if;
      end for;
      -- Finished forming this option
    block begin
	if B(exists of c in name where(B(B(I(position of c) = I(0))
					and B(c = C('+'))))) then
	    -- its an environmental variable - check it for relevance!
	    if B(S(every of c in option.name
			   where(B(I(position of c) < I(4)))) = S("pcom")) then
		-- strip "pcom" from effective name
		option.name <- S(every of c in option.name
		       where(B((I(position of c) > I(3)))));
	      else
		exit notpcom;
	      end if;
	  end if;
        insert option into options;
      on (DuplicateKey)
        -- ignore
      on exit (notpcom)
	-- discard irrelevant environment variables
      end block;
    end for;
  end block;
  
  -- Now process each module named on the command line in turn,
  -- skipping the first two argument words which are the name of the
  -- shell and this module's name
  for fileName in argv where (B(I(position of fileName) >= I(2))) inspect
    block declare
      filenameCopy: charstring;
      modName: charString;
      absprog: program;
      links: linkedPrograms;
      defMaps: definitions_printmappings;
      procMaps: executable_printmappings;
      posMappings: position_mappings;
      infDefs: inferredDefinitions;
#ifndef TSBOOT
      coercions: coercions;
#endif TSBOOT
    begin

      filenameCopy := filename;
#ifndef ANYBOOT
      -- remove trailing ".p" if it's already there
      if (every of c in fileNameCopy 
	      where (position of c >= size of fileNameCopy - 2)) = ".p" then
	fileNameCopy <- every of c in fileNameCopy
	    where (position of c < size of fileNameCopy - 2);
      end if;

      -- remove cwd if it's a prefix of the filename prefix
      block declare
	cwd: charstring;
      begin
	cwd <- stdenv.getCwd();
	if (cwd | "/" = every of c in fileNameCopy 
		where (position of c <= size of cwd)) then
	  fileNameCopy <- every of c in fileNameCopy
	      where (position of c > size of cwd);
	end if;
      end block;
#endif
      -- extract module name by stripping leading path components
#ifdef ANYBOOT
      modName := fileNameCopy;
#else
      block declare
	slashPos: integer;
      begin
	slashPos <- size of fileNameCopy - 1;
	while slashPos >= 0 repeat
	  if fileNameCopy[slashPos] = '/' then
	    exit found;
	  end if;
	  slashpos <- slashpos - 1;
	end while;
	exit found;
      on exit(found)
	modName := every of c in fileNameCopy where (position of c > slashPos);
      end block;
#endif ANYBOOT      

      call putString(S(modname | S(":")));
      new infDefs;

      ---------------- parsing / loading ----------------

#ifndef CGBOOT
      if B(exists of options[S("noparse")]) then
#endif CGBOOT
	-- no parsing phase... read absprog from .ao file
	block declare
	  badMod: charstring;
	begin
	  call putString(S(S(" load[") | S(fileNameCopy | S(".ao]"))));
	  if B(loadProg = fakeLoadProg) then
	    loadProg <- loadProgFunc#(procedure of program#(
		stdenv.pathLoad(S("loadprog"))));
	  end if;
	  call loadProg(fileNameCopy, stdenv.pathReadObj,
	    absprog, links, defMaps, procMaps, posMappings, badMod);
	on (loadProgIntf.programNotFound)
	  call putLine(S(" ERRORS"));
	  call putLine
	      (S(S(S("Program module '") | modName) | S("' was not found.")));
	  exit failed;
	on (loadProgIntf.definitionNotFound)
	  call putLine(S(" ERRORS"));
	  call putLine
	      (S(S(S("Definitions module '") | badMod) 
		  | S("' was not found.")));
	  exit failed;
	on (loadProgIntf.definitionInconsistent)
	  call putLine(S(" ERRORS"));
	  call putLine
	      (S(S(S("Definitions module '") | badMod) 
		  | S("' is inconsistent.")));
	  exit failed;
	on (loadProgIntf.discarded)
	  call putLine(S("ERRORS"));
	  call putLine
	      (S(S(S("Unable to load program module '") | modName) | S("'")));
	  exit failed;
	end block;
      
#ifndef CGBOOT      
      else
	-- parse source file to yield absprog
	block declare
	  source: charstring;
	  imports: module_printmap;
	  pathName: charString;
	begin
	  call putString(S(S(" parse[") | S(fileNameCopy | S(".p]"))));
	  if B(getFile = fakeGetFile) then
	    getFileInit <- getFileInitFunc#(create of program#(
		stdenv.pathLoad(S("getfile"))));
	    getFile <- getFileFunc#(getFileInit(stdio.fopen));
	  end if;
#ifndef ANYBOOT
	  if fileNameCopy[0] = '/' then
#endif
	    pathName := fileNameCopy;
#ifndef ANYBOOT
	  else
	    pathName := S(S(S(stdenv.getCwd()) | S("/")) | fileNameCopy);
	  end if;
#endif
	  source <- S(getfile(S(pathName | S(".p"))));
	  call parse(source, S(fileNameCopy | S(".p")), absprog,
	    defMaps, procMaps, posMappings, imports, links, errors);
	  if B(I(size of errors) <> ZERO) then
	    exit parseErrors;
	  end if;
	  
	on (getFile.cantRead)
	  call putLine(S(" ERRORS"));
	  call putLine
	      (S(S(S("Unable to read source file '") | fileNameCopy) 
		  | S(".p'.")));
	  exit failed;
	end block;
      end if;
#endif CGBOOT

      ---------------- Static Linking -----------------
      -- merge definitions modules, programs, and compiled code from
      -- all linked modules into the absprog we're constructing
      codeMap <- codeMap#(liUnstuff(absprog));
      for linkentry in links[] inspect
	for defmod in linkentry.program.definitions_modules[] inspect
	  block begin
	    insert definitions_module#(copy of defmod) into
		absprog.definitions_modules;
	  on (DuplicateKey)
	    -- ignore duplicates
	  end block;
	end for;
	for proc in linkentry.program.programs[] inspect
	  block begin
	    insert predefined!proc#(copy of proc) into absprog.programs;
	  on (DuplicateKey)
	    inspect dupproc in absprog.programs[proc.id] begin
	      if (B(dupproc <> proc)) then
		-- maybe we should rename the proc if possible
		-- and then insert it.  For now just generate
		-- an error.
		call putLine(S(" ERRORS"));
		call putLine(S(S(S("Linked process module ") | linkentry.name)
			| S(" contains a process with same id as a different process also linked.")));
		exit failed;
	      end if;
	    end inspect;
	  end block;
	end for;
	
	linkedCodeMap <- codeMap#(liUnstuff(linkentry.program));
	for entry in linkedCodeMap[] inspect
	  block begin
	    insert interpform!codemapentry#(copy of entry) into codeMap;
	  on (duplicateKey)
	    inspect dupentry in codeMap[entry.processid] begin
	      block declare
		testentry: interpform!codemapEntry;
	      begin
		-- two compiled progs are interchangeable if they are
		-- equal other than id
		testentry := entry;
		testentry.liprog.id := dupentry.liprog.id;
		if B(dupentry <> testentry) then
		  -- conflicting code for the same program  - just drop
		  -- both of them, and hopefully it will get recompiled
		  remove testentry from codeMap[entry.processid];
		end if;
	      end block;
	    end inspect;
	  end block;
	end for;
      end for;
    
      -- initialize the procErrors table
      new procErrors;

#ifndef TCBOOT
      
      ---------------- type checking ----------------
      if B(not B(exists of options[S("notcheck")])) then
	-- Invoke the type checker on the absprog
	call putString(S(" typecheck"));
	if B(typeCheck = fakeTypeCheck) then
	  typeCheck <- typeCheckCapa#(procedure of program#(
	      stdenv.pathLoad(S("type"))));
	end if;
	if B(I(size of checking_table) = ZERO) then
	  unwrap checking_table from polymorph#(
	    stdenv.pathReadObj(S("checking_table.ho"))) {init};
	end if;
	for proc in absprog.programs where
	      (B(not B(exists of codemap[proc.id]))) inspect
	  block declare
	    newInfDefs: inferredDefinitions;
	  begin
	    call typeCheck(proc, absprog.definitions_modules,
	      checking_table, stdenv, errors, newInfDefs);
	    merge newInfDefs into infDefs;
	  on (typeCheckCall.typeErrors)
	    -- problems... stash away the errors
	    for error in errors[] inspect
	      block declare
		procError: procError;
	      begin
		new procError;
		procError.procID := proc.id;
		procError.error := error;
		insert procError into procErrors;
	      end block;
	    end for;
	  end block;
	end for;
	-- Stop now if any errors were reported
	if B(I(size of procErrors) <> ZERO) then
	  exit errors;
	end if;
      end if;
#endif TCBOOT

#ifndef TSBOOT
      ---------------- typestate checking ----------------
      if B(exists of options[S("notscheck")]) then
	-- suppress typestate checking, and pass on an empty coercions
	-- list
	new coercions;
      else
	call putString(S(" typestate"));
	if B(tsCheck = fakeTsCheck) then
	  tsCheck <- typestateCheckOutport#(procedure of program#(
	      stdenv.pathLoad(S("typestate"))));
	end if;
	if B(I(size of checking_table) = ZERO) then
	  unwrap checking_table from polymorph#(
	    stdenv.pathReadObj(S("checking_table.ho"))) {init};
	end if;
	new coercions;
	for proc in absprog.programs where
	      (not exists of codemap[proc.id]) inspect
	  block declare
	    newCoercions: coercions;
	  begin
	    call tscheck(stdenv, absprog, proc, checking_table,
	      defMaps, procMaps, infDefs, newCoercions, errors);
	    merge newCoercions into coercions;
	  on (typestateCheckCall.typestateErrors)
	    -- problems... stash away the errors
	    for error in errors[] inspect
	      block declare
		procError: procError;
	      begin
		new procError;
		procError.procID := proc.id;
		procError.error := error;
		insert procError into procErrors;
	      end block;
	    end for;
	  end block;
	end for;
	-- stop now if any errors were reported
	if B(I(size of procErrors) <> ZERO) then
	  exit errors;
	end if;
      end if;
#endif TSBOOT

      ---------------- Code Generation ----------------
      if B(B(not B(exists of options[S("noobject")]))
	      or B(exists of options[S("ligen")])) then
	call putString(S(" codegen"));
	if B(codegenInit = fakeCodegenInit) then
	  codegenInit <- codegenInitFn#(procedure of program#(
	      stdenv.pathLoad(S("codegen"))));
	end if;
	block declare
	  annotes: annotations;
	  annote: annotation;
	begin
	  -- Construct annotations for the codegen process
	  new annotes;

	  new annote;
	  annote.name <- S("Module Name");
	  wrap S(copy of modName) as annote.thing;
	  insert annote into annotes;

	  new annote;
	  annote.name <- S("Inferred Definitions");
	  wrap inferredDefinitions#(copy of infDefs) as annote.thing;
	  insert annote into annotes;
	  
#ifndef TSBOOT
	  new annote;
	  annote.name <- S("Coercions");
	  wrap coercions as annote.thing;
	  insert annote into annotes;
#endif TSBOOT  
	  
	  new annote;
	  annote.name <- S("Process Print Map");
	  wrap predefined!executable_printmappings#(copy of procMaps)
	      as annote.thing;
	  insert annote into annotes;
	  
	  new annote;
	  annote.name <- S("Definitions Print Map");
	  wrap predefined!definitions_printmappings#(copy of defMaps)
	      as annote.thing;
	  insert annote into annotes;
	  
	  new annote;
	  annote.name <- S("Position Map");
	  wrap posmap!position_mappings#(copy of posMappings)
	      as annote.thing;
	  insert annote into annotes;

	  codegen <- codegenFn#(codegenInit(absprog, annotes, options, 
	      stdenv.pathLoad, stdenv.pathReadObj));

	  for proc in absprog.programs where
		B(not B(exists of prog in codemap[proc.id])) inspect
	    block declare
	      code: interpform!prog;
	    begin
	      call codegen(proc, code, errors);
	      insert (evaluate newentry: codemapEntry from
		  new newentry;
		  newentry.processid := proc.id;
		  newentry.liprog := code;
		end) into codemap;
	    on (codegen.codegenErrors)
	      -- problems... stash away the errors
	      for error in errors[] inspect
		block declare
		  procError: procError;
		begin
		  new procError;
		  procError.procID := proc.id;
		  procError.error := error;
		  insert procError into procErrors;
		end block;
	      end for;
	    end block;
	  end for;
	  -- stop now if any errors were reported
	  if B(I(size of procErrors) <> ZERO) then
#ifdef CGBOOT
	    print charstring#"Errors (in raw form... have fun!)";
	    print procErrors;
	    exit failed;
#else
	    exit errors;
#endif
	  else
	    call listuff(absprog,codemap);
	  end if;

	  if B(not B(exists of options[S("noobject")])) then
	    call putString(S(S(" store[") | S(modName | S(".po]"))));
	    block begin
#ifdef CGBOOT
	      call stdenv.store(S(modName | S(".po")), 
#else
          call stdenv.libStore(modname,
#endif CGBOOT
              evaluate prog:program from
                  prog := absprog;
                  if B(exists of options[S("strip")]) then
                      new prog.programs;
                      if B(not B(exists of options[S("keeppredef")]))
                        then
                          new prog.definitions_modules;
                        else
                          -- strip everything _except_ predefined
                          block 
                            declare
                              predef: predefined!definitions_module;
                              typename: predefined!typename;
                            begin
                              -- remove and save predefined
                              typename := 
                                 typename#( \typename predefined!empty\ );
                              remove predef from
                              def in prog.definitions_modules 
                                 where (B(def.id = typename.moduleid));
                              -- blow away everything else
                              new prog.definitions_modules;
                              -- put back predefined
                              insert predef into prog.definitions_modules;
                            on (NotFound)
                              -- predefined not found
                            end block;
                        end if;
                    else if B(exists of options[S("stripdefs")]) then
                          new prog.definitions_modules;
                        else if B(not B(exists of options[S("keeppredef")]))
                            then
                              -- strip only predefined
                              block
                                declare
                                  predef: predefined!definitions_module;
                                  typename: predefined!typename;
                                begin
                                  -- remove predefined
                                  typename := 
                                     typename#( \typename predefined!empty\ );
                                  remove predef from
                                  def in prog.definitions_modules 
                                     where (B(def.id = typename.moduleid));
                                  discard predef;
                                on (NotFound)
                                  -- predefined not found
                                end block;
                            end if;
                        end if;
                    end if;
                end);

	    on (others)
	      call putLine(S("ERRORS"));
	      call putLine(
		S(S(S("Problems writing program object file '")
			| S(modName | S(".po"))) | S("'")));
	      exit failed;
	    end block;
	  end if;
	end block;
      end if;
      
#ifndef ANYBOOT
      if B(exists of options[S("ligen")]) then
	block declare
	  disasmPMap: disassembler!printmaps;
	  source: charString;
	  LIfile: stream;
	begin
	  call putString(S(" disassemble"));
	  if B(disasm = fakeDisasm) then
	    disasm <- disassemblerFn#(procedure of program#(
		stdenv.pathLoad(S("disassembler"))));
	  end if;
	  new disasmPMap;
	  disasmPMap.execs <- procMaps;
	  disasmPMap.defs <- defMaps;
	  unite disasmPMap.progid.pid
	      from processid#(copy of absprog.main_program);
	  block begin
	    source <- S(disasm(codemap#(LIUnstuff(absprog)),
		absprog.main_program,modName,stdenv, disasmPMap));
	  on (others)
	    call putLine(S(" ERRORS"));
	    call putLine(S("Disassembly failed"));
	    exit failed;
	  end block;

	  call putString(S(S(" write[") | S(modName | S(".li]"))));
	  block begin
	    LIfile <- stream#(stdio.fopen(S(modname | S(".li")), 
		openType#'write'));
	    call LIfile.fputs(source);
	    call LIfile.fclose();
	  on (others)
	    call putLine(S(" ERRORS"));
	    call putLine(S("Problems writing LI source file"));
	    exit failed;
	  end block;
	end block;
      end if;
      
#endif ANYBOOT

      call putLine(S(""));

#ifndef CGBOOT
    on exit (parseErrors)
      -- Here when there were parsing errors... we need to fake the
      -- normal arguments for formatError since we never got to the
      -- point of having a correct absprog (yuck!)
      call putLine(S(" ERRORS"));
      if B(formatError = fakeFormatError) then
	formatError <- formatErrorFn#(procedure of program#(
	    stdenv.pathLoad(S("formaterror"))));
#ifndef ANYBOOT
	sortErrs <- sortErrorsFn#(procedure of program#(
	    stdenv.pathLoad(S("sorterrors"))));
	sortPErrs <- sortProcErrorsFn#(procedure of program#(
	    stdenv.pathLoad(S("sortprocerrors"))));
#endif ANYBOOT
      end if;
      block declare
	scopes: scopes;
	defmods: definitions_modules;
	procMap: executable_printmap;
	posMaps: posMaps;
      begin
	new scopes;
	new defmods;
	new defMaps;
	new procMap;
	unite procMap.id.pid from processid#unique;
	new procMap.name;
	new procMap.roots;
	new procMap.exits;
	new posMaps;
#ifndef ANYBOOT
	call sortErrs(errors);
#endif
	for error in errors[] inspect
	  block declare
	    msg: charString;
	  begin
	    msg <- charstring#(
	      formatError(error, scopes, infDefs, defmods, defMaps,
		procMap, posMaps, stdenv.pathLoad));
	    call putLine(msg);
	  end block;
	end for;
      end block;
      call putLine
	  (S(S(S("Suppressing further processing for module '") | modName)
	      | S("'.")));
#endif CGBOOT

#ifndef TCBOOT
    on exit (errors)
      -- Here when there are type or typestate errors to report
      call putLine(S(" ERRORS"));
      if B(formatError = fakeFormatError) then
	formatError <- formatErrorFn#(procedure of program#(
	    stdenv.pathLoad(S("formaterror"))));
#ifndef ANYBOOT
	sortErrs <- sortErrorsFn#(procedure of program#(
	    stdenv.pathLoad(S("sorterrors"))));
	sortPErrs <- sortProcErrorsFn#(procedure of program#(
	    stdenv.pathLoad(S("sortprocerrors"))));
#endif ANYBOOT
      end if;
#ifndef ANYBOOT
      call sortPErrs(procErrors, posMappings);
#endif ANYBOOT

      for procError in procErrors[] inspect
	block declare
	  eid: executable_id;
	  msg: charString;
	begin
	  unite eid.pid from processid#(copy of procError.procID);
	  inspect proc in absprog.programs[procError.procID] begin
	    inspect procMap in procMaps[eid] begin
	      inspect posMap in posMappings[eid] begin
		msg <- charString#(
		  formatError(procError.error, proc.executable_part.scopes,
		    infDefs, absprog.definitions_modules, defMaps,
		    procMap, posMap.mapping, stdenv.pathLoad));
		call putLine(msg);
	      end inspect;
	    end inspect;
	  end inspect;
	end block;
      end for;
      call putLine
	  (S(S(S("Suppressing further processing for module '") | modName)
	      | S("'.")));
#endif TCBOOT

    on exit (failed)
      -- Here when a module failed one of the processing phases
      call putLine
	  (S(S(S("Suppressing further processing for module '") | modName)
	      | S("'.")));
      
    end block;
  end for;
  
  -- all finished
  return args;
end process
