-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: parseproc.p
-- Author: Andy Lowry
-- SCCS Info: @(#)parseproc.p	1.3 3/10/91

-- Process to parse a process specification from a pshell command line

parseProc: using (pshell, common, tokenize, plumbing)

process (Q: parseProcQ)
  
declare
  args: parseProc;
  label: charString;
  name: charString;
  argv: charStringList;
  state: procState;
  tok: token;
  punct: charString;
  delim: char;
begin
  receive args from Q;

  -- clear all result data and set initial state
  label <- "";
  name <- "";
  new argv;
  state <- 'start';
  
  -- insert name of shell in arg vector
  insert "pshell" into argv;

  -- following contains punctuation chars that are significant in the
  -- pipe syntax
  punct <- "|<>;&";

  -- get first token... there's guaranteed to be at least one
  remove tok from args.toks[0];
  
  while state <> 'done' repeat
    -- retrieve delimiter char now to simplify select guards below
    if case of tok = 'delimiter' then
      reveal tok.delimiter;
      delim := tok.delimiter;
    else
      delim <- ' ';
    end if;

    select state
    where ('start')
      -- Next token must be a word.  We assume it's the process name,
      -- so we'll need to fix things if we see a colon next
      if case of tok <> 'word' then
	exit syntaxError;
      end if;
      reveal tok.word;
      name := tok.word;
      -- advance to next token... if none, then we're finished
      block begin
	remove tok from args.toks[0];
	state <- 'getColon';
      on (notFound)
	insert copy of name into argv;
	state <- 'done';
      end block;
      
    where ('getColon')
      -- If the next token is a colon, then the word we thought was a
      -- process name was really a label... consume it and then go get
      -- the process name.   Otherwise we start accumulating args
      if case of tok = 'delimiter' and delim = ':' then
	label <- name;
	name <- "";
	-- advance to next token... if none, then this is a label reference
	block begin
	  remove tok from args.toks[0];
	  state <- 'gotLabel';
	on (notFound)
	  state <- 'done';
	end block;
      else
	-- not a colon... go accumulate args, and don't consume
	-- current token... add process name to arg vector
	insert copy of name into argv;
	state <- 'args';
      end if;
      
    where ('gotLabel')
      -- Here when we've picked up a label and need a following
      -- process name, which must be a word.  If next token isn't a
      -- word, we got a label reference.
      if case of tok = 'word' then
	reveal tok.word;
	name := tok.word;
	-- add process name to arg vector
	insert copy of name into argv;
	-- advance to next token and start gathering args... we're
	-- done if there's no next token
	block begin
	  remove tok from args.toks[0];
	  state <- 'args';
	on (notFound)
	  state <- 'done';
	end block;
      else
	-- no process name... add the current token back onto the
	-- token list so it will be rescanned
	insert copy of tok into args.toks at 0;
	state <- 'done';
      end if;
      
    where ('args')
      -- Here when we're accumulating args in a process specification.
      -- We stop when there are no more tokens or when the next token
      -- is one of the punctuation characters that are significant in
      -- the shell syntax.  In the latter case the token needs to be
      -- pushed back to be rescanned.
      if case of tok = 'delimiter' and exists of c in punct where (c = delim)
      then
	-- punctuation... push the token back for rescanning, and exit
	insert copy of tok into args.toks at 0;
	state <- 'done';
      else
	-- This is a real arg... gather the token text (sans quotes if
	-- it was a quoted string) and advance to the next token
	select case of tok
	where ('word')
	  reveal tok.word;
	  insert copy of tok.word into argv;
	where ('delimiter')
	  reveal tok.delimiter;
	  block declare
	    arg: charString;
	  begin
	    new arg;
	    insert copy of tok.delimiter into arg;
	    insert arg into argv;
	  end block;
	where ('quotedString')
	  reveal tok.quoted;
	  insert copy of tok.quoted.string into argv;
	where ('bracketedString')
	  -- this really shouldn't happen, since our tokenizer was
	  -- initialized with no bracket characters
	  reveal tok.bracketed;
	  insert copy of tok.bracketed.string into argv;
	otherwise
	  exit cantHappen;
	end select;
	block begin
	  remove tok from args.toks[0];
	on (notFound)
	  -- no more tokens... we're through
	  state <- 'done';
	end block;
      end if;

    otherwise
      exit cantHappen;
    end select;
  end while;
    
  -- Here when we finally hit the 'done' state... if a label was
  -- given, try to use it to look up the process ID.  Failing that,
  -- create a new process ID and make a label entry if a label was
  -- present
  block declare
    procLabel: procLabel;
    info: procInfo;
  begin
    if label = "" then
      -- no label present... generate a new proc ID
      args.proc <- unique;
    else
      -- label present... pull proc ID from existing entry if any
      block begin
	inspect plab in args.labels[label] begin
	  args.proc := plab.proc;
	end inspect;
      on (notFound)
	-- no prior entry... create a new entry with a new ID
	args.proc <- unique;
	new procLabel;
	procLabel.label := label;
	procLabel.proc := args.proc;
	procLabel.defined <- 'false';
	insert procLabel into args.labels;
      end block;
    end if;
    
    -- Now we have a proc ID.  If a definition is present, generate a
    -- procInfo record and make sure we're not defining a label twice
    if name <> "" then
      if label <> "" then
	remove procLabel from args.labels[label];
	if procLabel.defined then
	  exit syntaxError;
	else
	  procLabel.defined <- 'true';
	  insert procLabel into args.labels;
	end if;
      end if;

      new info;
      info.id := args.proc;
      info.name := name;
      info.args := argv;
      insert info into args.procs;
    end if;
  end block;
    
  return args;
  
on exit(syntaxError)
  return args exception syntaxError;
  
end process
