(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                            CAML                                       *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            Inria                                      *)
(*                      Domaine de Voluceau                              *)
(*                      78150  Rocquencourt                              *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* dir_gram The grammar of directives and pragmas                        *)
(*          Michel Mauny                                                 *)
(*          Pierre Weis                                                  *)

#standard arith true;;
#fast arith true;;

(********************************************)
(* Preliminaries: functions used in actions *)
(********************************************)

let parse_caml_impl () =
    match eval_prag_syntax(parse_caml_import_list())
    with dynamic(ll:MLspec list & string list) -> ll
       | _ -> system_error "parse_caml_import_list"
and parse_caml_expl () =
    match eval_prag_syntax(parse_caml_export_list())
    with dynamic(l:MLspec list) -> l
       | _ -> system_error "parse_caml_export_list"
;;

(*
   Parse conditionnally: only next phrase if plus_flag=true
   else the one after
*)
let parse_cond plus_flag =
    (if not plus_flag then parse_raw_caml_syntax());
    parse_raw_caml_syntax()
;;

(* Parse conditionnally according to the value of the next macro *)
let parse_conditionnally plus_flag =
    let e = parse_caml_expr0() in
    match eval_prag_syntax(e)
    with dynamic (true:bool) -> parse_cond plus_flag
       | dynamic (false:bool) -> parse_cond (not plus_flag)
       | d -> ill_typed_macro e d <:gtype<bool>>
;;

let eval_caml_pragma () =
    match eval_prag_syntax (parse_caml_pragma ())
    with dynamic (s: MLpragma) -> s
       | _ -> raise parse "syntax of pragmas is out of scope"
;;

let eval_caml_MLtype s =
    let gty =
    (try MLtype_to_gtype (parse_caml_typ ())
     with failure _ -> raise parse "Unknown type in default printer")
    in ML (MLapply (MLvar s, MLquote (dynamic gty),[]))
;;

let parse_caml_string_const () =
    match parse_caml_expr0 ()
    with MLconst(mlstring s) as S -> S
       | MLvar s -> MLconst(mlstring s)
       | _ -> raise parse ""
;;

let parse_caml_string () =
    match parse_caml_expr0 ()
    with MLconst(mlstring s) -> s
       | MLvar s -> s
       | _ -> raise parse ""
;;

let parse_caml_syntax_extension () =
    match eval_prag_syntax (parse_caml_expr ())
    with dynamic (e:ML) -> ML e
       | dynamic (e:MLdecl) -> MLdecl e
       | dynamic (e:MLgrammar_decl) -> MLgrammar e
       | dynamic (e:MLsyntax) -> e
       | _ -> raise parse "illegal syntactic extension"
;;

(*******************************)
(*                             *)
(*      The grammar itself     *)
(*                             *)
(*******************************)

grammar for values directives =

rule entry Commands =
    parse Literal "directive" (* To be executed in all modes *)
            -> MLdirective (eval_caml_pragma ())
        | Literal "pragma" -> MLpragma (eval_caml_pragma ())

        | Literal "use"; {parse_caml_expr ()} e ; Literal ";;"
            -> MLdirective
                (Pragmaexp(MLapply (MLvar "use",e,[])))
        | Literal "load"; {parse_caml_expr ()} e; Literal ";;"
            -> MLdirective
                (Pragmaexp(MLapply (MLvar "load",e,[])))
        | Literal "compile"; {parse_caml_expr ()} e; Literal ";;"
            -> MLdirective
                (Pragmaexp
                  (MLapply (MLvar "compile", e,[])))

        | Literal "infix"; Ident id; Literal ";;"
            -> MLdirective
                (Pragmaexp
                  (MLapply (MLvar "infix", MLconst (mlstring id),[])))
        | Literal "uninfix"; Ident id; Literal ";;"
            -> MLdirective
                (Pragmaexp
                  (MLapply (MLvar "uninfix", MLconst (mlstring id),[])))

        | Literal "fast"; Literal "arith"; {parse_caml_expr()} b;
          Literal ";;"
            -> MLpragma
                (Pragmaexp
                  (MLapply (MLvar "fast_arith", b,[])))

        | Literal "standard"; Literal "arith"; {parse_caml_expr()} b;
          Literal ";;"
            -> MLpragma
                (Pragmaexp
                  (MLapply (MLvar "standard_arith_switch", b,[])))

        | Literal "arith"; Literal "cautious";
          Literal ";;"
            -> MLdirective
                (Pragmaexp
                  (MLapply (MLvar "cautious_arith_switch",
                            (MLconst mlnull,[]))))

        | Literal "arith"; Arith b; Literal ";;" -> arith_switch b

        | Literal "open"; Literal "compilation"; {parse_caml_expr()} b;
          Literal ";;"
            -> MLpragma
                (Pragmaexp (MLapply (MLvar "open_compilation", b,[])))
        | Literal "open"; Literal "optimization"; {parse_caml_expr()} n;
          Literal ";;"
            -> MLpragma
                (Pragmaexp (MLapply (MLvar "open_optimization", n,[])))
        | Literal "open"; Literal "printing"; {parse_caml_expr()} b;
          Literal ";;"
            -> MLdirective
                (Pragmaexp (MLapply (MLvar "open_printing", b,[])))
        | Literal "open"; Literal "overloading"; {parse_caml_expr()} b;
          Literal ";;"
            -> MLpragma
                (Pragmaexp (MLapply (MLvar "open_overloading", b,[])))
        | Literal "open"; Literal "module"; {parse_caml_expr()} b;
          Literal ";;"
            -> MLdirective
                (Pragmaexp (MLapply (MLvar "module_open_mode", b,[])))
        | Literal "open"; Literal "typing"; {parse_caml_expr()} b;
          Literal ";;"
            -> MLpragma
                (Pragmaexp (MLapply (MLvar "open_typing", b,[])))
        | Literal "open"; Literal "arith";
          Literal ";;"
            -> MLpragma
                (Pragmaexp (MLapply (MLvar "open_arith", MLconst mlnull,[])))

        | Literal "set"; Literal "any"; {parse_caml_expr ()} e;
          Literal ";;"
            -> MLpragma
                (Pragmaexp (MLapply (MLvar "make_any", e,[])))
        | Literal "set"; Literal "exhaustive"; Literal "matches";
          {parse_caml_expr ()} e;
          Literal ";;"
            -> MLpragma
                (Pragmaexp (MLapply (MLvar "set_exhaustive_matches", e,[])))

        | Literal "set"; Literal "default"; Literal "grammar"; Syntax s;
          Literal ";;"
            -> MLpragma
                (Pragmaexp (MLapply (MLvar "set_default_grammar", s,[])))
        | Literal "default"; Literal "grammar"; Literal ";;"
            -> MLpragma
                (Pragmaexp
                  (MLapply (MLvar "default_grammar", MLconst mlnull,[])))

        | Literal "printer"; Printer_args args; Literal ";;"
            -> ML(MLapply (MLvar "new_printer", args))
        | Literal "default"; Literal "printer"; Literal "for"; Literal "type";
           {eval_caml_MLtype "default_printer"} e ; Literal ";;" -> e
        | Literal "sharing"; Literal "printer"; Literal "for"; Literal "type";
           {eval_caml_MLtype "add_type_to_print_with_sharing"} e ;
           Literal ";;" -> e

        | Literal "eval"; Literal "when"; Literal "print";
          {parse_caml_expr()} b; Literal ";;"
            -> MLdirective
                (Pragmaexp (MLapply
                             (MLvar "eval_when_print", b,[])))
        | Literal "quit"; Literal ";;"
            -> (quit();
                MLpragma
                 (Pragmaexp (MLapply (MLvar "quit", MLconst mlnull,[]))))
        | Literal "relet"; {lex_reread "let"; parse_caml_decl ()} D;
          Literal ";;" ->
                MLdecl (decl_from_relet_decl D)
        | Literal "extensible"; Literal "function"; Ident_dir name;
          Literal "with"; Literal "default"; Ident_dir default;
          Literal ";;" ->
           ML (new_extensible_function name default)
        | Literal "extend"; Ident_dir name;
          Literal "with"; Ident_dir extension;
          Literal ";;" ->
           ML (extend_function name extension)
        | {parse_caml_syntax_extension ()} e ; Literal ";;" -> e

(*** May be more user's friendly ???
and Arith_list =
    parse Arith a; ( * (parse Literal ","; Arith a -> a)) al -> a,al
***)

and Arith = parse Literal "num" -> num_arith
                | Literal "int" -> int_arith
                | Literal "big_int" -> big_int_arith
                | Literal "ratio" -> ratio_arith
                | Literal "float" -> float_arith
                | Literal "overloading" -> overloading_arith

and entry Auto =
    parse Ident_dir i;
          {lex_reread i;parse_caml_straint_list ()} str_l;
          Literal "from"; {parse_caml_expr ()} e; Literal ";;"
            -> MLautoload (str_l,e)
        | Literal "grammar"; {parse_caml_string()} gname;
          Literal "with"; Entries_parsers epl;
          Literal "from"; {parse_caml_expr ()} e; Literal ";;"
            -> MLautoload_grammar (gname,epl,e)

and Ident_dir = parse IDENT id -> id
                    | Literal "use" -> "use"
                    | Literal "relet" -> "relet"
                    | Literal "print" -> "print"

and Sig_imp =
    parse Literal ";;" ->
            ([],[])
        | Literal "using"; {parse_caml_impl ()} imp;
          Literal ";;"
            -> imp

and entry Module_import =
    parse {parse_caml_string()} name; Sig_imp sig
            -> MLbegin_mod(name,sig)

and entry Sys_module_import =
    parse Literal "module"; {parse_caml_string()} name; Sig_imp sig
            -> MLsys_begin_mod(name,sig)

and entry Module_export =
    parse "module"; Sig_exp sig
            -> MLend_mod sig

and Sig_exp =
    parse Literal ";;" -> []
        | Literal "with";
          {parse_caml_expl ()} imp; Literal ";;"
            -> imp
        | Literal "#with";
          {let e = (parse_caml_expr ())
            in match eval_prag_syntax e
               with dynamic (sl: MLspec list) -> sl
                  | d -> ill_typed_macro e d <:gtype<string list list>>
          } imp;
          Literal ";;"
            -> imp

and Syntax =
    parse {parse_caml_string_const()} syntax_name; Entry_name ext
            -> MLpair(syntax_name, ext)

and Entry_name =
    parse Literal ":" -> parse_caml_string_const()
        | -> MLconst(mlstring "")

and Entries_parsers =
    parse Entry_parser ep -> [ep]
        | Entries_parsers epl; Literal "and"; Entry_parser ep ->ep::epl

and Entry_parser =
    parse {parse_caml_string()} str1; Literal "at"; Literal "entry";
          {parse_caml_string()} str2  -> (str2,str1)

and Printer_args =
    parse Delim d1; {parse_caml_string_const()} pr; Delim d2
            -> (pr, [MLpair(MLconst(mlstring d1), MLconst(mlstring d2))])

and Delim = parse -> "" | STRING s -> s

and entry Plus_dir =
    parse -> parse_conditionnally true

and entry Minus_dir =
    parse -> parse_conditionnally false
;;

let parse_dir = (directives "Commands").Parse_raw
and parse_imp = (directives "Module_import").Parse_raw
and parse_exp = (directives "Module_export").Parse_raw
and parse_sys_imp = (directives "Sys_module_import").Parse_raw
and parse_pldir = (directives "Plus_dir").Parse_raw
and parse_mindir = (directives "Minus_dir").Parse_raw
and parse_auto = (directives "Auto").Parse_raw
;;

let parse_directive_gen parser () =
    match eval_syntax (parser())
    with dynamic (S: MLsyntax) -> S
       | _ -> system_error "parse_directive"
;;

let parse_directive = parse_directive_gen parse_dir
and parse_import = parse_directive_gen parse_imp
and parse_system_import = parse_directive_gen parse_sys_imp
and parse_export = parse_directive_gen parse_exp
and parse_plus_dir = parse_directive_gen parse_pldir
and parse_minus_dir = parse_directive_gen parse_mindir
and parse_autoload = parse_directive_gen parse_auto
;;
