-- Text I/O functions, a realization of [1] in LML.
--
-- "Get" and "Form" allow the programmer to easily create interactive
-- textual programs which run on vt100 terminals. This is only a start
-- since I really want to concentrate on interactive graphics. But you may
-- find them useful.
--
-- The use of the "Get" and "Form" functions are demonstrated in the files
-- sum2.m, formsum.m perm.m day.m . 
--
-- (C) E.C.R.C. GmBH. 1988
--
-- A.Dwelly 29.3.88
--
-- [1] Synchronizing the I/O behavior of Functional programs with feedback.
-- To be published May 88 Information processing letters, North-Holland.
--
-- Modifications
----------------
-- 

module

-- All the VT100 dependant codes are here

#include "vt100termcap.t"

export RecogType,FormType,Get,Form,Number,String;

rec type RecogType = Error +
		     Last Char +
		     Lastc Char Char + 
		     Cont Char Char (Char -> RecogType)
    
and type FormType =  Formelem Int Int (List (Char -> RecogType)) (List Char)


-- Get 
--
-- This seems to work pretty well, but it's not too elegant. I find that
-- the "let d1..dn in e" syntax pretty well backwards to the way I think. On
-- the other hand the more natural "e where d1..dn" doesn't seem to have
-- an adequate end marker. So I used "let" throughout this work.
--

and Get r c l = 

    let rec (Str,Feedback,Rest) = Gt r [] "" l

    and Gt r rl [] (h.t) & (h = DelC) = 
        let (Str,Feedback,Rest) = Gt r rl "" t in (Str,Bell @ Feedback,Rest)
    ||  Gt r (hr.tr) (hs.s) (h.t) & (h = DelC) =
        let (Str,Feedback,Rest) = Gt hr tr s t in (Str,EraseLeft@ Feedback,Rest)
    ||  Gt r rl s (h.t) = Gtt (r h) r rl s t

    and Gtt (Error) r rl s t = 
        let (Str,Feedback,Rest) = Gt r rl s t in (Str,Bell @ Feedback,Rest)
    ||  Gtt (Last f) r rl s t = (s,[f],t)
    ||  Gtt (Lastc a f) r rl s t = (a.s,[f],t)
    ||  Gtt (Cont a f fun) r rl s t = 
        let (Str,Feedback,Rest) = Gt fun (r.rl) (a.s) t in (Str,f.Feedback,Rest)

    in (c (reverse Str),Feedback,Rest)

-- Form
--
-- If anything I find this definition uglier than "Get", and I am not happy
-- with the way it reacts to completed recgonitions at the moment. Still I
-- have no more time for corrections and it's not too bad.
--
-- Currently <Tab> moves the user to the next input field and "*" enters
-- the entire form. Future modifications might include a <backtab>
--
-- This may be the worlds shortest forms handler.


and Form (rh.rt) conv l =
    let rec SubForm (fr.hr.tr) bl ('\t'.t) =
	    let (Out,OldForm,Feedback,Rest) = SubForm (hr.tr) (fr.bl) t
	    in  (Out,OldForm,GotoPos hr @ Feedback,Rest)

    ||      SubForm [r] bl ('\t'.t) =
            let (Out,OldForm,Feedback,Rest) = SubForm (reverse (r.bl)) [] t
	    in  (Out,OldForm,GotoPos (hd (reverse (r.bl))) @ Feedback,Rest)

    ||      SubForm rl bl ('*'.t) = 
	    (conv(map Answer (reverse bl @ rl)),reverse bl @ rl,"",t)

    ||      SubForm (form as ((Formelem x y f "").tr)) bl (h.t) & (h=DelC) =
	    let (Out,OldForm,Feedback,Rest) = 
		 SubForm form bl t
	    in (Out,OldForm,Bell @ Feedback,Rest)
    
    ||      SubForm ((Formelem x y (fh.ft) (sh.st)).tr) bl (h.t) & (h=DelC) =
	    let (Out,OldForm,Feedback,Rest) = 
		 SubForm ((Formelem x y ft st).tr) bl t
	    in (Out,OldForm,EraseLeft @ Feedback,Rest)

    ||      SubForm (form as ((Formelem x y (fh.th) str).tr)) bl (h.t) =
		RecForm (fh h) form bl t

    and     RecForm (Error) rl bl l =
	    let (Out,OldForm,Feedback,Rest) = SubForm rl bl l
	    in (Out,OldForm,Bell @ Feedback,Rest)
    
    ||      RecForm (Cont a f fun) ((Formelem x y fl s).tr) bl l =
	    let (Out,OldForm,Feedback,Rest) =
		SubForm ((Formelem x y (fun.fl) (a.s)).tr) bl l
	    in (Out,OldForm,f.Feedback,Rest)

    ||      RecForm (Last f) rl bl l =
	    let (Out,OldForm,Feedback,Rest) = SubForm rl bl ('\t'.l)
	    in (Out,OldForm,f.Feedback,Rest)

    ||      RecForm (Lastc a f) ((Formelem x y fl s).tr) bl l =
	    let (Out,OldForm,Feedback,Rest) =
		SubForm ((Formelem x y fl (a.s)).tr) bl ('\t'.l)
	    in (Out,OldForm,f.Feedback,Rest)

    and     GotoPos (Formelem x y fl str) = MoveTo x y @ reverse str

    and     Answer (Formelem x y fl str) = reverse str

    and (Out,OldForm,Feedback,Rest) = SubForm (rh.rt) [] l

    in  (Out,OldForm,concmap GotoPos rt @ GotoPos rh @ Feedback,Rest)

-- Here is a collection of useful recognisers, of type Char -> Recogtype --
---------------------------------------------------------------------------

and Number c & (isdigit c) = (Cont c c Number)
||  Number '\n'            = (Last  '\n')
||  Number c               = Error

and String '\n'            = (Last '\n')
||  String c               = (Cont c c String)

end
