/* SPIM S20 MIPS simulator.
   Description of a SPIM S20 instruction.
   (Layout does not correspond to MIPS machine.)
   Copyright (C) 1990 by James Larus (larus@cs.wisc.edu).

   SPIM is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 1, or (at your option) any
   later version.

   SPIM is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GNU CC; see the file COPYING.  If not, write to James R.
   Larus, Computer Sciences Department, University of Wisconsin--Madison,
   1210 West Dayton Street, Madison, WI 53706, USA or to the Free
   Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. */
/* $Header: /n/fs/vb/egs/src/spim/RCS/inst.h,v 1.5 92/07/08 14:53:46 egs Exp Locker: egs $
*/

/* Describes an expression that produce a value for an instruction's
   immediate field.  Immediates have the form: label +/- offset. */

typedef struct immexpr
{
  long offset;			/* Offset from symbol */
  struct lab *symbol;		/* Symbolic label */
  short bits;			/* > 0 => 31..16, < 0 => 15..0 */
  short pc_relative;		/* Non-zero => offset from label in code */
} imm_expr;

/* Describes an expression that produce an address for an instruction.
   Address have the form: label +/- offset (register). */
typedef struct addrexpr
{
  unsigned char reg_no;		/* Register number */
  imm_expr *imm;		/* The immediate part */
} addr_expr;

/* Choose bland names for fields and #define the real names below.
   This is better than a union since it doesn't require an additional
   level of syntactic indirection in references. */
typedef struct inst_s {
    short opcode;
    unsigned char f1;
    unsigned char f2;
    unsigned char f3;
    unsigned char f4;
    long f5;
    short f5x;
    imm_expr *f6;
    long word;      /* the word corresponding to this instruction; undecoded */
} instruction;


#define OPCODE(INST)	(INST)->opcode

#define RS(INST)	(INST)->f1
#define FS(INST)	(INST)->f1
#define BASE(INST)	(INST)->f1

#define RT(INST)	(INST)->f2
#define FT(INST)	(INST)->f2

#define RD(INST)	(INST)->f3
#define FD(INST)	(INST)->f3

#define SHAMT(INST)	(INST)->f4
#define FORMAT(INST)	(INST)->f4

#define IMM(INST)	(INST)->f5x
#define IOFFSET(INST)	(INST)->f5x
#define COND(INST)	(INST)->f5x

#define TARGET(INST)	(INST)->f5

#define EXPR(INST)	(INST)->f6

#define COND_UN		0x1
#define COND_EQ		0x2
#define COND_LT		0x4
#define COND_IN		0x8

/* Round V to next greatest B boundary */
#define ROUND(V, B) (((int) V + (B-1)) & ~(B-1))

/* Sign-extend a short to a long */
#define SIGN_EX(X)	(((X) & 0x8000) ? ((X) | 0xffff0000) : (X))

/* Raise an exception! */
#define RAISE_EXCEPTION(CAUSE, MISC)					\
	{								\
	  if (((CAUSE)<= LAST_REAL_EXCEPT) || (Status_Reg & 0x1))	\
	    {								\
	      Cause = (CAUSE) << 2;					\
	      exception_occurred = 1;					\
	      EPC = PC;							\
	      Status_Reg = (Status_Reg & 0xffffffc0) | ((Status_Reg & 0xf) << 2); \
	      MISC;							\
	    }								\
	}								\

/* Recognized exceptions (see Ch. 5): */
#define INT_EXCPT 0
#define ADDRL_EXCPT 4
#define ADDRS_EXCPT 5
#define IBUS_EXCPT 6
#define DBUS_EXCPT 7
#define SYSCALL_EXCPT 8
#define BKPT_EXCPT 9
#define RI_EXCPT 10
#define OVF_EXCPT 12
/* Floating point exceptions (Ch. 8): */
#define INEXACT_EXCEPT 13
#define INVALID_EXCEPT 14
#define DIV0_EXCEPT 15
#define FOVF_EXCEPT 16
#define FUNF_EXCEPT 17

#define LAST_REAL_EXCEPT FUNF_EXCEPT
/* External interrupts: */
#define CLOCK_EXCPT 0x100       /* IP[0] */
#define IO_EXCPT 0x200          /* IP[1] */

#ifndef OLDC /* the parser is still old C */
extern void text_begins_at_point(mem_addr);
extern void k_text_begins_at_point(mem_addr addr);
extern unsigned long int current_text_pc(void);
extern void increment_text_pc(int);
extern void user_kernel_text_segment(int);
extern void store_instruction(instruction *);
extern void i_type_inst(int, int, int, imm_expr *);
extern void j_type_inst(int, imm_expr *);
extern instruction *r_type_inst(int, int, int, int);
extern void r_sh_type_inst(int, int, int, int);
extern void r_cond_type_inst(int, int, int);
extern void print_inst(mem_addr);
extern int print_inst_internal(char *, instruction *, unsigned long int);
extern int opcode_is_branch(int);
extern int opcode_is_jump(int);
extern int opcode_is_load_store(int);
extern int inst_is_breakpoint(mem_addr);
extern instruction *set_breakpoint(mem_addr);
extern imm_expr *make_imm_expr(int, char *, int);
extern imm_expr *copy_imm_expr(imm_expr *);
extern imm_expr *upper_bits_of_expr(imm_expr *);
extern imm_expr *lower_bits_of_expr(imm_expr *);
extern imm_expr *const_imm_expr(long int);
extern imm_expr *incr_expr_offset(imm_expr *, long int);
extern imm_expr *addr_expr_imm(addr_expr *);
extern long int eval_imm_expr(imm_expr *);
extern int zero_imm(imm_expr *);
extern addr_expr *make_addr_expr(long int, char *, int);
extern int addr_expr_reg(addr_expr *);
extern long int inst_encode(instruction *);
extern instruction *inst_decode(long value);

/* parser.y y.tab.c */
extern void fix_current_label_address(mem_addr);
extern int op_to_imm_op(int);
extern int imm_op_to_op(int);
extern imm_expr *branch_offset(int);
#endif
