#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>
#include <setjmp.h>
#include "spim.h"
#include "inst.h"
#include "mem.h"
#include "reg.h"
#include "y.tab.h"
#include "sym_tbl.h"

/* Internal functions: */
static mem_addr copy_str_to_stack(char *);
static mem_addr copy_int_to_stack(int);
static void delete_all_breakpoints(void);

extern jmp_buf spim_top_level_env;           /* defined and set in spim.c */
/* Global Variables: */
char *input_file_name;		/* Name of file being parsed */
extern int yylval;		/* Value of token from YYLEX */
int source_file;                /* File binary or source */
void *message_out = 0, *console_out = 0;
mem_addr program_starting_address = 0;
int prog_fds[MAXFDS];

long initial_data_limit = DATA_LIMIT;
long initial_stack_size = STACK_SIZE;
long initial_stack_limit = STACK_LIMIT;
long initial_k_text_size = K_TEXT_SIZE;
long initial_k_data_size = K_DATA_SIZE;
long initial_k_data_limit = K_DATA_LIMIT;

/* Record of where a breakpoint was placed 
   and the instruction previously in memory. */
typedef struct bkptrec {
    mem_addr addr;
    instruction *inst;
    struct bkptrec *next;
} bkpt;
static bkpt *bkpts = NULL;

void init_prog_fds(void) {
    int i;

/*
    if(((prog_fds[0] = dup(0)) < 0) || 
       ((prog_fds[1] = dup(1)) < 0) ||
       ((prog_fds[2] = dup(2)) < 0))
	error("init_prog_fds");
    for(i=3; i<MAXFDS; ++i)
	prog_fds[i] = -1;
*/

    for(i=0; i<MAXFDS; ++i)
	prog_fds[i] = i;
}


/* Run a program starting at PC for N steps and display each
   instruction before executing if FLAG is non-zero.  If CONTINUE is
   non-zero, then step through a breakpoint.  Return non-zero if
   breakpoint is encountered. */
extern int init_and_run(mem_addr addr, int steps, int display, int cont, 
			int argcnt, char **argvc, char **env) {
    static int stopped = 0;

    if(stopped == 0) {
	init_prog_fds();
	initcache();
	initialize_run_stack(argcnt, argvc, env);
	addr = starting_address();
    }
    if (cont && inst_is_breakpoint(addr)) {
	delete_breakpoint(addr);
	exception_occurred = 0;
	run_spim(addr, 1, display);
	add_breakpoint(addr);
	steps -= 1;
    }
    exception_occurred = 0;
    stopped = run_spim(addr, steps, display);
    if(stopped == 0) {  /* program finished */
	stopped = 0;
	deallocate_mem();  /* only need to deallocate stack & heap */
	PC = 0;
	print_cstats();
    }
    return stopped;
}

extern int init_memory(long text_size, long data_size) {
    return make_memory(text_size, data_size, initial_stack_size, 
		       initial_k_text_size, initial_k_data_size);
}

/* Initialize or reinitialize the state of the machine. */
extern void initialize_world(int load_trap_handler)
{
    /* Allocate the floating point registers */
    if (FGR == NULL)
	FPR = (double *) malloc (16 * sizeof (double));
    initialize_registers ();
    initialize_symbol_table ();
    k_text_begins_at_point (K_TEXT_BOT);
    k_data_begins_at_point (K_DATA_BOT);
    data_begins_at_point (DATA_BOT);
    text_begins_at_point (TEXT_BOT);
    if (load_trap_handler)    {
	error("trap handler feature is disabled\n");
	/*      int old_bare = bare_machine; bare_machine = 0;
		if (read_assembly_file (DEFAULT_TRAP_HANDLER))
		fatal_error ("Cannot read trap handler");
		bare_machine = old_bare; user_kernel_data_segment(0);
		*/
    }
    initialize_scanner (stdin);
    delete_all_breakpoints ();
    write_output(message_out, "SPIM %s\n", SPIM_VERSION);
    write_output(message_out,
		 "Copyright 1990-92 by James R. Larus (larus@cs.wisc.edu)\n");
    write_output(message_out, "See the file COPYING for license information\n");
}

void initialize_registers(void)
{
    bzero(FPR, 16 * sizeof (double));
    FGR = (float *) FPR;
    FWR = (int *) FPR;
    bzero (R, 32 * sizeof (reg_word));
    R[29] = STACK_TOP - BYTES_PER_WORD;
    PC = 0;
    Cause = 0;
    EPC = 0;
    Status_Reg = 0;
    BadVAddr = 0;
    Context = 0;
    PRId = 0;
}

/* Read file NAME, which should contain assembly code. Return zero if
   successful and non-zero otherwise. */
int read_assembly_file (char *name)
{
    FILE *file = fopen(name, "r");
    
    source_file = 1;
    if (file == NULL) {
	error ("Cannot open file: `%s'\n", name);
	return (1);
    } else {
#ifdef mips
#include <sys/exec.h>
	unsigned short magic;
	
	fread(&magic, sizeof (short), 1, file);
	fclose (file);
	if(magic == MIPSEBMAGIC || magic == MIPSELMAGIC)
	    return read_aout_file(name);
#endif
	input_file_name = name;
	initialize_scanner (file);
	while(yyparse ()) ;
	fclose(file);
	flush_local_labels ();
	end_of_assembly_file ();
	return (0);
    }
}

mem_addr starting_address (void) {
    if (PC == 0)    {
	if (program_starting_address != 0)
	    return (program_starting_address);
	else
	    return (program_starting_address =
		    find_symbol_address (DEFAULT_RUN_LOCATION));
    }
    else
	return (PC);
}

/* Initialize the SPIM stack with ARGC, ARGV, and ENVP data. */
void initialize_run_stack(int argc, char **argv, char **env ) {
    char **p;
    int i, j = 0, env_j;
    mem_addr addrs[10000];
    
    /* Put strings on stack: */
    R[REG_A0]  = argc;
    R[REG_A2] = R[29];
    for (p = env; *p != '\0'; p++)
	addrs[j++] = copy_str_to_stack (*p);
    
    R[REG_A1]  = R[29];
    env_j = j;
    for (i = 0; i < argc; i++)
	addrs[j++] = copy_str_to_stack(argv[i]);
    
    R[29] -= 4;			/* Leave rest of word empty */
    /* Build vectors on stack: */
    for (i = env_j - 1; i >= 0; i--)
	copy_int_to_stack(addrs[i]);
    for (i = j - 1; i >= env_j; i--)
	copy_int_to_stack(addrs[i]);
    
    R[29] = copy_int_to_stack (argc); /* Leave pointing to argc */
}

static mem_addr copy_str_to_stack(char *s) {
    mem_addr str_start;
    int i = strlen (s);
    
    while (i >= 0)
    {
	SET_MEM_BYTE (R[29], s[i]);
	R[29] -= 1;
	i -= 1;
    }
    str_start = (mem_addr) R[29] + 1;
    R[29] = R[29] & 0xfffffffc;	/* Round down to word boundary */
    return (str_start);
}


static mem_addr copy_int_to_stack (int n) {
    SET_MEM_WORD (R[29], n);
    R[29] -= BYTES_PER_WORD;
    return ((mem_addr) R[29] + BYTES_PER_WORD);
}

/* Set a breakpoint at memory location ADDR. */
void add_breakpoint(mem_addr addr) {
    bkpt *rec = (bkpt *) malloc (sizeof (bkpt));

    rec->next = bkpts;
    rec->addr = addr;

    if ((rec->inst = set_breakpoint (addr)) != NULL)
	bkpts = rec;
    else {
	if (exception_occurred)
	    error ("Cannot put a breakpoint at address 0x%08x\n", addr);
	else
	    error ("Already have a breakpoint at address 0x%08x\n", addr);
	free (rec);
    }
}

/* Delete all breakpoints at memory location ADDR. */
void delete_breakpoint(mem_addr addr) {
    bkpt *p, *b;
    int deleted_one = 0;

    for (p = NULL, b = bkpts; b != NULL; )
	if (b->addr == addr) {
	    bkpt *n;
	    
	    SET_MEM_INST (addr, b->inst);
	    if (p == NULL)
		bkpts = b->next;
	    else
		p->next = b->next;
	    n = b->next;
	    free(b);
	    b = n;
	    deleted_one = 1;
	}
	else
	    p = b, b = b->next;
    if (!deleted_one)
	error ("No breakpoint to delete at 0x%08x\n", addr);
}

static void delete_all_breakpoints(void) {
    bkpt *b, *n;
    
    for (b = bkpts, n = NULL; b != NULL; b = n) {
	n = b->next;
	free (b);
    }
    bkpts = NULL;
}

/* List all breakpoints. */
void list_breakpoints(void) {
    bkpt *b;
    
    if (bkpts)
	for (b = bkpts;  b != NULL; b = b->next)
	    write_output (message_out, "Breakpoint at 0x%08x\n", b->addr);
    else
	write_output (message_out, "No breakpoints set\n");
}

/* Utility routines */
/* Print an error message. */
extern void error(char *fmt, ...) { /* Display only */
    va_list args;

    va_start(args, fmt);
    vfprintf(stderr, fmt, args);
    va_end(args);
}


/* Print an error message and return to top level. */
extern void run_error(char *fmt, ...) {
    va_list args;

    va_start(args, fmt);
    vfprintf(stderr, fmt, args);
    va_end(args);
    longjmp(spim_top_level_env, 1);
}

/* Print the error message then exit. */
extern void fatal_error (char *fmt, ...) {
    va_list ap;

    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    va_end(ap);
    exit (-1);
}

extern void write_output(void *f, char *fmt, ...) {
    va_list ap;
    
    va_start(ap, fmt);
    f = (f != NULL) ? f: stdout;
    vfprintf(f, fmt, ap);
    va_end(ap);
    fflush(f);
}

extern void read_input(char *str, int n) {
    fgets (str, n, stdin);
}

/* Return the entry in the hash TABLE of length LENGTH with key STRING.
   Return NULL if no such entry exists. */
inst_info *map_string_to_inst_info(inst_info tbl[], int tbl_len, char *id) {
    register int low = 0;
    register int hi = tbl_len - 1;
    
    while (low <= hi)
    {
	int mid = (low + hi) / 2;
	char *idp = id, *np = tbl[mid].name;

	while (*idp == *np && *idp != '\0') {idp ++; np ++;}

	if (*np == '\0' && *idp == '\0') /* End of both strings */
	    return (& tbl[mid]);
	else if (*idp > *np)
	    low = mid + 1;
	else
	    hi = mid - 1;
    }
    return NULL;
}


/* Return the entry in the hash TABLE of length LENGTH with VALUE1 field NUM.
   Return NULL if no such entry exists. */

inst_info *
map_int_to_inst_info (inst_info tbl[], int tbl_len, int num) {
    int low = 0;
    int hi = tbl_len - 1;
    
    while (low <= hi)
    {
	int mid = (low + hi) / 2;
	
	if (tbl[mid].value1 == num)
	    return (&tbl[mid]);
	else if (num > tbl[mid].value1)
	    low = mid + 1;
	else
	    hi = mid - 1;
    }
    return NULL;
}


#ifdef NEED_VSPRINTF
char *vsprintf (char *str, char *fmt, va_list *args) {
    FILE _strbuf;

    _strbuf._flag = _IOWRT+_IOSTRG;
    _strbuf._ptr = str;
    _strbuf._cnt = 32767;
    _doprnt(fmt, args, &_strbuf);
    putc('\0', &_strbuf);
    return(str);
}
#endif


#ifdef NEED_STRTOL
long strtol(char *str, char *eptr, int base) {
    long result;
    
    if (base != 16)
	fatal_error ("SPIM's strtol only works for base 16");
    if (*str == '0' && (*(str + 1) == 'x' || *(str + 1) == 'X'))
	str += 2;
    sscanf (str, "%lx", &result);
    return (result);
}
#endif
