(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Last modified on Tue Feb 11 16:18:06 PST 1992 by muller                   *)

INTERFACE Dir;

IMPORT Text, List;

(* The procedures in this interface raise an exception if the requested
   operation cannot be performed.  In general, the exception Error is raised
   and is an indication that an error was reported by the operating system. In
   some cases, a more specific exception can be raised, to give a better
   idea of the problem; however, an implementation can chose raise Error
   rather than the more specific one, and clients of this interface must be
   prepared for that. *) 
  
EXCEPTION
  Error;
  CannotAccess;

PROCEDURE Delete (name: Text.T) RAISES {CannotAccess, Error};
  (* delete the file 'name' *)

PROCEDURE Rename (old, new: Text.T) RAISES {CannotAccess, Error};
  (* rename the file 'old' to 'new' *)

PROCEDURE Contents (name: Text.T): List.T RAISES {CannotAccess, Error};
  (* return a list with Text.T elements, one for each entry in the 
     directory 'name' *)

PROCEDURE CreateAlias (name, alias: Text.T) RAISES {CannotAccess, Error};
  (* create an alias for the file 'name', under the name 'alias'.  The
     implementation may actually copy the file 'name'.  If the file 
     'name' is subsequently modified, there is no guarantee that these changes
     will be visible if reading the file using 'alias' *)

PROCEDURE FindFile (READONLY path: ARRAY OF Text.T; name: Text.T): Text.T
     RAISES {CannotAccess, Error};
  (* find the file 'name' by looking successively in the directories 
     in 'path' (starting at index 0).  If the file is not found, raise
     CannotAccess; otherwise, return the full name of the file. *)

END Dir.
