(* Copyright (C) 1991, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Thu Feb 20 22:18:31 PST 1992 by muller         *)

GENERIC INTERFACE Float(Real); 

TYPE T = Real.T;

(* The purpose of the interface is to provide access to the
   floating-point operations required or recommended by the IEEE
   floating-point standard.  Consult the standard for the precise
   specifications of the procedures, including when their arguments
   are NaNs, infinities, and signed zeros, and including what exceptions
   they can raise.  Our comments specify their effect when the arguments
   are ordinary numbers and no exception is raised.  Implementations on 
   non-IEEE machines that have values similar to NaNs and infinities 
   should explain how those values behave for IsNaN, Finite, etc. in
   an implementation guide *)

PROCEDURE Scalb(x: T; n: INTEGER): T;
(* Return x * (2 ** n). *)

PROCEDURE Logb(x: T): T;
(* Return the exponent of x.  More precisely, return the unique
   n such that ABS(x) / Base ** n is in the range [1..Base-1],
   unless x is denormalized, in which case return MinExp,
   where MinExp is the minimum exponent value for T. *)

PROCEDURE ILogb(x: T): INTEGER;
(* Like Logb, but returns an integer, never raises an exception, 
   and always returns the n such that ABS(x)/Base**N is in [1..Base-1]
   even for denormalized numbers. *)

PROCEDURE NextAfter(x, y: T): T;
(* Return the next representable neighbor of x in the direction
   towards y.  If x = y, return x *)

PROCEDURE CopySign(x, y: T): T;
(* Return x with the sign of y. *)

PROCEDURE Finite(x: T): BOOLEAN;
(* Return TRUE if x is strictly between -infinity and +infinity.
   This always returns TRUE on non-IEEE machines. *)

PROCEDURE IsNaN(x: T): BOOLEAN;
(* Return FALSE if x represents a numerical (possibly infinite) value,
   and TRUE if x does not represent a numerical value.  For example,
   on IEEE implementations, returns TRUE if x is a NaN, FALSE otherwise;
   on Vaxes, returns TRUE of x is a reserved operand, FALSE otherwise.
   *)

PROCEDURE Sign(x: T): [0..1];
(* Return the sign bit x.  For non-IEEE implementations,
   this is the same as ORD(x >= 0); for IEEE implementations,
   Sign(-0) = 1, Sign(+0) = 0. *)

PROCEDURE Differs(x, y: T): BOOLEAN;
(* RETURN (x < y OR y < x).  Thus, for IEEE implementations,
   Differs(NaN, x) is always FALSE; for non-IEEE implementations,
   Differs(x,y) is the same as x # y.  *)    
PROCEDURE Unordered(x, y: T): BOOLEAN;
(* Return NOT (x <= y OR y <= x).  For non-IEEE
   implementations, this always returns FALSE. *)

PROCEDURE Sqrt(x: T): T;
(* Return the square root of T.  Must be correctly rounded if 
   FloatMode.IEEE is TRUE.  *)

TYPE IEEEClass = 
  {SignalingNaN, QuietNaN, Infinity, Normal, Denormal, Zero};
  
PROCEDURE Class(x: T): IEEEClass;
(* Return the IEEE number class containing x.  On non-IEEE systems,
   the result will be Normal or Zero. *)
       
END Float.
