MODULE SimpleIO;	(*29.07.93. LB*)
(* Implements the procedures for simple file I/O.
   Error handling is somewhat strange: the idea was to
   catch all exceptions and convert them into simple error messages,
   in order to protect beginners from being confronted with exceptions. *)

  IMPORT
    Rd, Wr, Stdio, FileStream, Text;


  PROCEDURE GetFileName(userInfo: TEXT := ""): TEXT =
  BEGIN
    WITH so = Stdio.stdout DO
      Wr.PutText(so, userInfo & PromptText);
      Wr.Flush(so);
    END; (*WITH so*)
    RETURN Rd.GetLine(Stdio.stdin);
  END GetFileName;

  PROCEDURE Error(text: TEXT) =
  BEGIN
    WITH so = Stdio.stdout DO
      Wr.PutText(so, "Error at SimpleIO." & text & "\n");
      Wr.Flush(so);
    END; (*WITH so*)
  END Error;

  PROCEDURE OpenRead(name: TEXT := NIL): Reader =
  VAR rd: Reader := NIL; finish: BOOLEAN;
  BEGIN
    IF name = NIL THEN name:= GetFileName("input ") END;
    REPEAT
      finish:= TRUE;
      TRY
        IF Text.Empty(name) THEN rd:= Stdio.stdin
        ELSE                     rd:= FileStream.OpenRead(name) 
        END;
      EXCEPT
        Rd.Failure => 
          Error("OpenRead, " & name & " doesn't exists, try again");
          name:= GetFileName("input "); finish:= FALSE;
      ELSE
        Error("OpenRead");
      END; (*TRY*)
    UNTIL finish;
    RETURN rd;
  END OpenRead;

  PROCEDURE OpenWrite(name: TEXT := NIL): Writer =
  VAR wr: Writer := NIL; finish: BOOLEAN;
  BEGIN
    IF name = NIL THEN name:= GetFileName("output") END;
    TRY
      REPEAT
        finish:= TRUE;
        IF Text.Empty(name) THEN wr:= Stdio.stdout 
        ELSIF FileExists(name) THEN
            Error("OpenWrite, " & name & " already exists, try again");                   
            name:= GetFileName("output"); finish:= FALSE;
        ELSE
          wr:= FileStream.OpenWrite(name)
        END; (*IF*)
      UNTIL finish;
    EXCEPT ELSE
      Error("OpenWrite"); 
    END; (*TRY*)
    RETURN wr;
  END OpenWrite;

  PROCEDURE CloseRead(VAR rd: Reader) =
  BEGIN
    IF (rd # NIL) AND (rd # Stdio.stdin) THEN 
      TRY
        Rd.Close(rd)
      EXCEPT ELSE
        Error("CloseRead");
      END; (*TRY*)    
      rd:= NIL;
    END; (*IF rd # NIL*)
  END CloseRead;

  PROCEDURE CloseWrite(VAR wr: Writer) =
  BEGIN
    IF (wr = NIL) OR (wr = Stdio.stdout) THEN Wr.Flush(Stdio.stdout)
    ELSE
      TRY
        Wr.Close(wr)
      EXCEPT ELSE
        Error("CloseWrite");
      END; (*TRY*)
      wr:= NIL;
    END; (*IF wr = NIL*)
  END CloseWrite;

  PROCEDURE Flush(wr: Writer := NIL) =
  BEGIN
    IF (wr = NIL) THEN Wr.Flush(Stdio.stdout)
    ELSE Wr.Flush(wr)
    END;
  END Flush;

  PROCEDURE FileExists(name: TEXT): BOOLEAN =
  VAR e: BOOLEAN := TRUE;
  BEGIN
    TRY
      EVAL FileStream.OpenRead(name)
    EXCEPT
      Rd.Failure => e:= FALSE
    END; (*TRY*)
    RETURN e
  END FileExists;

  PROCEDURE GetChar(rd: Reader := NIL): CHAR =
  VAR ch: CHAR := '\000';
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    TRY
      ch:= Rd.GetChar(rd)
    EXCEPT ELSE
      Error("GetChar");
    END; (*TRY*)
    RETURN ch;
  END GetChar;

  PROCEDURE GetLine(rd: Reader := NIL): TEXT =
  VAR t: TEXT := "";
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    TRY
      t:= Rd.GetLine(rd)
    EXCEPT ELSE
      Error("GetLine"); 
    END; (*TRY*)
    RETURN t;
  END GetLine;

  PROCEDURE EOF(rd: Reader := NIL): BOOLEAN =
  VAR eof: BOOLEAN := TRUE;
  BEGIN
    IF (rd # NIL) AND (rd # Stdio.stdin) THEN (*on stdin return always true*)
      TRY
        eof:= Rd.EOF(rd)
      EXCEPT ELSE
        Error("EOF");
      END; (*TRY*)
    END; (*IF rd # NIL*)
    RETURN eof;
  END EOF; 

  PROCEDURE Availabe(rd: Reader := NIL): BOOLEAN =
  VAR av: BOOLEAN := FALSE;
  BEGIN
    IF rd = NIL THEN rd:= Stdio.stdin END;
    TRY
      av:= Rd.CharsReady(rd) > 0
    EXCEPT ELSE
      Error("Availabe");
    END; (*TRY*)
    RETURN av;
  END Availabe;

  PROCEDURE Reset(rd: Reader := NIL) =
  BEGIN
    IF rd # NIL THEN
      IF Rd.Seekable(rd) THEN 
        TRY
          Rd.Seek(rd, 0)
        EXCEPT ELSE
        Error("Reset");
        END; (*TRY*)
      ELSE
        Error("Reset, file cannot be reset")
      END; (*IF Rd.Seekable(rd)*)
    END; (*IF rd # NIL*)
  END Reset;

  PROCEDURE PutChar(ch: CHAR; wr: Writer := NIL) =
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    TRY
      Wr.PutChar(wr, ch);
      IF (wr = Stdio.stdout) AND (ch = '\n') THEN
        Wr.Flush(wr)
      END; (*IF wr*)
    EXCEPT ELSE
      Error("PutChar");
    END; (*TRY*)
  END PutChar;  

  PROCEDURE PutText(t: TEXT; wr: Writer := NIL) =
  VAR last: CHAR;
  BEGIN
    IF wr = NIL THEN wr:= Stdio.stdout END;
    TRY
      Wr.PutText(wr, t);
      IF (wr = Stdio.stdout) THEN
        last:= Text.GetChar(t, Text.Length(t)-1);
        IF (last = '\n') THEN Wr.Flush(wr) END;
      END; (*IF wr*)
    EXCEPT ELSE
      Error("PutText");
    END; (*TRY*)
  END PutText;

BEGIN
END SimpleIO.
