MODULE ProgFileIO	EXPORTS ProgFileIO;



IMPORT	Rd, Thread, Fmt, FileStream, Text, Char;

<* FATAL Thread.Alerted *>


REVEAL	T = Public BRANDED OBJECT
	    Reader:	Rd.T;	(* Reader, von dem alles kommt. *)
	    filename:	TEXT;

	    currentRow:=0;	(* Spalte, TABS expandiert. Setzt getChar()*)
	    row:=	0;	(* Spalte, TABS expandiert, des Beginn des *)
				(* letzten Wortes. Setzt getWord()!        *)
	
	    indent:=	-1;	(* Einrckung der aktuellen Zeile, i.e.    *)
				(* Zeichenabstand des ersten Nicht-Blanks  *)
				(* vom Zeilenanfang. Wenn negativ ungltig,*)
				(* nach Newline oder vor dem ersten Lesen. *)
	    line:=	0;	(* aktuelle Zeile. *)

	    backupRow:= 0;	(* fr ungetChar() *)
	    backupIndent:= 0;
	    backupLine:= 0;

	    lastToken:	TokenTypes;	(* setzt getWord() *)

	METHODS
	    (* Tokentyptester: *)
	    isKeyword(Word: TEXT): BOOLEAN:=		IsKeyword;
	    isOperator(Word: TEXT): BOOLEAN:=		IsOperator;
	    isPragmaStart(Word: TEXT): BOOLEAN:=	IsPragmaStart;
	    isCommentStart(Word: TEXT): BOOLEAN:=	IsCommentStart;

	OVERRIDES
	    getChar:=	GetChar;
	    ungetChar:=	UngetChar;
	    getWord:=	GetWord;
	    getNumber:=	GetNumber;
	    getComment:=GetComment;
	    getPragma:=	GetPragma;
	    getLiteral:=GetLiteral;
	    tokenType:=	TokenType;
	    lineIndent:=LineIndent;
	    currRow:=	CurrRow;
	    currLine:=	CurrLine;
	    eof:=	Eof;
	    close:=	Close;
	    open:=	Open;
	END; (*OBJECT*)



(****************************************************************************)
(*                     primitve Dateibehandlung                             *)
(****************************************************************************)


(* Bemerkung zur EOF und Dateisystemfehlerbehandlung:
 *
 * Die Prozeduren hier nehmen an, da T.eof geprft wurde, bevor alle Lese-
 * operationen versucht wurden, d.h. EOF ist eine Exeption (die irgendwo im 
 * Rd.GetChar aufgebracht wird).
 *
 * Ich bin zwar nicht glcklich darber, da das ffnen einer nicht existieren
 * Datei zum Lesen eine Exeption ist, doch die anderen Modula-3 Libraries be-
 * handeln das so, soda auch open eine Rd.Failure aufbringt.
 *)


PROCEDURE Open(self: T;  Name: TEXT) RAISES {Rd.Failure} =
BEGIN
	self.Reader:= FileStream.OpenRead(Name);
	self.filename:= Name;
END Open;

PROCEDURE Close(self: T) RAISES {Rd.Failure} =
BEGIN
	Rd.Close(self.Reader);
END Close;

PROCEDURE Eof(self: T): BOOLEAN  RAISES {Rd.Failure} =
BEGIN
	RETURN Rd.EOF(self.Reader);
END Eof;



PROCEDURE GetChar(self: T): CHAR   RAISES {Rd.Failure, Rd.EndOfFile} =
	(* Einzige Stelle,die die row und line-Felder ndert. *)
VAR	c:	CHAR;

BEGIN
	self.backupRow:= self.row;
	self.backupIndent:= self.indent;
	self.backupLine:= self.line;

	c:= Rd.GetChar(self.Reader);
	IF c = Char.HT THEN
	    self.row:= self.row + (self.tabStops - self.row MOD self.tabStops);
	ELSIF c = self.newLine THEN
	    self.row:= 0;
	    self.indent:= -1;
	    INC(self.line);
	ELSE
	    INC(self.row);
	END; (*IF*)
	RETURN c;
END GetChar;


PROCEDURE UngetChar(self: T) =
BEGIN
	self.row:= self.backupRow;
	self.indent:= self.backupIndent;
	self.line:= self.backupLine;

	Rd.UnGetChar(self.Reader);	
END UngetChar;

(****************************************************************************)
(*                     Tokentyp feststellen                                 *)
(****************************************************************************)


PROCEDURE IsPragmaStart(self: T;  Word: TEXT): BOOLEAN =
BEGIN
	(* Kommentarbeginn? *)
	FOR i:= FIRST(self.pragmaStart) TO LAST(self.pragmaStart) DO
	    IF Text.Equal(Word, self.pragmaStart[i]) THEN
		RETURN TRUE;
	    END; (*IF*)
	END; (*FOR*)

	(* nicht gefunden *)
	RETURN FALSE;
END IsPragmaStart;


PROCEDURE IsCommentStart(self: T;  Word: TEXT): BOOLEAN =
BEGIN
	(* Kommentarbeginn? *)
	FOR i:= FIRST(self.commentStart) TO LAST(self.commentStart) DO
	    IF Text.Equal(Word, self.commentStart[i]) THEN
		RETURN TRUE;
	    END; (*IF*)
	END; (*FOR*)

	(* nicht gefunden *)
	RETURN FALSE;
END IsCommentStart;


PROCEDURE IsKeyword(self: T;  Word: TEXT): BOOLEAN =
BEGIN
	(* Kommentarbeginn? *)
	FOR i:= FIRST(self.keywords) TO LAST(self.keywords) DO
	    IF Text.Equal(Word, self.keywords[i]) THEN
		RETURN TRUE;
	    END; (*IF*)
	END; (*FOR*)

	(* nicht gefunden *)
	RETURN FALSE;
END IsKeyword;


PROCEDURE IsOperator(self: T;  Word: TEXT): BOOLEAN =
BEGIN
	(* Kommentarbeginn? *)
	FOR i:= FIRST(self.operators) TO LAST(self.operators) DO
	    IF Text.Equal(Word, self.operators[i]) THEN
		RETURN TRUE;
	    END; (*IF*)
	END; (*FOR*)

	(* nicht gefunden *)
	RETURN FALSE;
END IsOperator;




(****************************************************************************)
(*                     G e t  W o r d  ( )                                  *)
(****************************************************************************)


PROCEDURE GetWord(self: T): TEXT
	RAISES {SyntaxError, Rd.EndOfFile, Rd.Failure} =

VAR	c:	CHAR;
	result:="";

BEGIN
	(* leading whitespaces berlesen: *)
	REPEAT
	    c:= self.getChar();
	UNTIL NOT c IN self.whiteSpaces;

	(* Spalten-Informationen aktualisieren: *)
	self.row:= self.currentRow;
	IF self.indent < 0 THEN
	    self.indent:= self.row;
	END; (*IF*)

	IF c IN Char.Digits THEN
	    self.ungetChar(); (*zurck mit der Ziffer, getNumber() liest sie!*)
	    self.lastToken:= TokenTypes.Number;
	    RETURN self.getNumber();

	ELSIF c = self.charQuote OR c = self.stringQuote THEN
	    IF c = self.charQuote THEN
		self.lastToken:= TokenTypes.CharacterLiteral;
	    ELSE
		self.lastToken:= TokenTypes.TextLiteral;
	    END; (*IF*)
	    RETURN self.getLiteral(c);

	ELSE
	    LOOP
		IF c IN Char.Letters THEN	(* Identifier, Keywords *)
		    WHILE c IN self.identChars DO
			result:= result & Text.FromChar(c);
			c:= self.getChar();
		    END; (*WHILE*)
		    EXIT;

		ELSE
		    result:= result & Text.FromChar(c);
    
		    IF self.isCommentStart(result) THEN
			self.lastToken:= TokenTypes.Comment;
			RETURN self.getComment(result);
		    END; (*IF*)
		    
		    IF self.isPragmaStart(result) THEN
			self.lastToken:= TokenTypes.Pragma;
			RETURN self.getPragma(result);
		    END; (*IF*)
    
		    (* nchstes Zeichen *)
		    c:= self.getChar();
    
		    (* wenn 'result & c' noch Teil eines Operators oder  *)
		    (* Kommentarklammernanfanges ist, dann weitermachen, *)
		    (* sonst gehrt c nicht mehr zum Wort!	         *)
		    IF c IN self.whiteSpaces THEN
			EXIT
		    ELSIF NOT (self.isCommentStart(result&Text.FromChar(c)) OR
			       self.isPragmaStart(result&Text.FromChar(c)) OR
			       self.isOperator(result&Text.FromChar(c)) )
		    THEN
			EXIT
		    END; (*IF*)
		END; (*IF*)
	    END; (*LOOP*)

	    self.ungetChar();	(* gehrt nicht mehr zum Wort: geht zurck! *)

	    IF self.isKeyword(result) THEN
		self.lastToken:= TokenTypes.Keyword;
	    ELSIF self.isOperator(result) THEN
		self.lastToken:= TokenTypes.Operator;
	    ELSE
		self.lastToken:= TokenTypes.Identifier;
	    END; (*IF*)

	    RETURN result;
	END; (*IF*)
END GetWord;



PROCEDURE GetNumber(self: T): TEXT
	RAISES {SyntaxError, Rd.EndOfFile, Rd.Failure} =

CONST	HexDigits = Char.Digits + SET OF CHAR {'a'..'f', 'A'..'F'};
	Exponents = SET OF CHAR {'e','E','d','D','x','X'};

VAR	c:	CHAR;
	result:="";

	PROCEDURE ReadDigits() RAISES {Rd.EndOfFile, Rd.Failure} =
	BEGIN
	    c:= self.getChar();
	    WHILE c IN Char.Digits DO
		result:= result & Text.FromChar(c);
		c:= self.getChar();
	    END; (*WHILE*)
	END ReadDigits;

	PROCEDURE ReadHexDigits() RAISES {Rd.EndOfFile, Rd.Failure} =
	BEGIN
	    c:= self.getChar();
	    WHILE c IN HexDigits DO
		result:= result & Text.FromChar(c);
		c:= self.getChar();
	    END; (*WHILE*)
	END ReadHexDigits;

BEGIN
	ReadDigits();
	CASE c OF
	    '_'=>	result:= result & Text.FromChar(c);
			ReadHexDigits();
	  | '.'=>	result:= result & Text.FromChar(c);
			ReadDigits();
			IF c IN Exponents THEN
			    result:= result & Text.FromChar(c);
			    c:= self.getChar();
			    IF c IN Char.Digits + SET OF CHAR{'+','-'} THEN
				result:= result & Text.FromChar(c);
				ReadDigits();
			    ELSE
				RAISE SyntaxError(self.filename & 
						": poor REAL-Literal in line "
						& Fmt.Int(self.line));
			    END; (*IF*)
			END; (*IF*)
	  ELSE		(* ok: Nur Cardinal gewesen. *)
	END; (*CASE*)

	(* Zahlen mssen mit Whitespace oder Operator aufhren *)
	IF c IN Char.Letters THEN
	    RAISE SyntaxError(self.filename & ": poor Number in line "
			      & Fmt.Int(self.line));
	END; (*IF*)

	self.ungetChar();
	RETURN result;
END GetNumber;


PROCEDURE GetLiteral (self: T;  Quote: CHAR): TEXT
	RAISES {SyntaxError, Rd.EndOfFile, Rd.Failure} =
	(* bernimmt Quote in den Resultatstring und liest solange, *)
	(* bis Quote im Inputstream wieder auftaucht. Das Auftauchen*)
	(* von self.newLine im Inputstream fhrt sum SyntaxError.   *)
	(* Quote kann mit Backspaces "escaped" werden: \'           *)
VAR	c:	CHAR;
	result:=Text.FromChar(Quote);

BEGIN
	REPEAT
	    c:= self.getChar();
	    IF c = self.newLine THEN
	        RAISE SyntaxError(self.filename&": string with no end in line "
			          & Fmt.Int(self.line));
	    ELSIF c = '\\' THEN	(* Backslash berlesen *)
		result:= result & Text.FromChar(c);
		c:= self.getChar();
	    END; (*IF*)
	    result:= result & Text.FromChar(c);
	UNTIL c = Quote;
	RETURN result;
END GetLiteral;



PROCEDURE GetComment (self: T; startSequenz: TEXT): TEXT
	RAISES {SyntaxError, Rd.Failure} =

VAR	c:	CHAR;
	result:=startSequenz;

BEGIN
    TRY
	c:= self.getChar();
	result:= result & Text.FromChar(c);
	LOOP
	    IF c = '*' THEN   (* erstes Zeichen, das einen Kommentar beendet *)
		c:= self.getChar();
		result:= result & Text.FromChar(c);
		IF c = ')' THEN (* zweites Zeichen gelesen -> Ende *)
		    RETURN result;
		END; (*IF*)
	    ELSIF c = '(' THEN (* mglicherweise Start einer Schachtelung *)
		c:= self.getChar();
		result:= result & Text.FromChar(c);
		IF c = '*' THEN (* zweites Zeichen gelesen -> Schachtelung *)
		    result:= result & self.getComment("");
		END; (*IF*)
	    ELSE
		c:= self.getChar();
		result:= result & Text.FromChar(c);
	    END; (*IF*)
	END; (*LOOP*)

    (* Bei Erreichen des Dateiendes fehlt offenbar eine schlieende Klammer! *)
    EXCEPT
	Rd.EndOfFile=>RAISE SyntaxError(self.filename&": comment with no end");
    END; (*TRY*)
END GetComment;



PROCEDURE GetPragma(self: T; startSequenz: TEXT): TEXT
	RAISES {SyntaxError, Rd.Failure} =

VAR	c:	CHAR;
	result:=startSequenz;

BEGIN
    TRY
	c:= self.getChar();
	result:= result & Text.FromChar(c);
	LOOP
	    IF c = '*' THEN   (* erstes Zeichen, das ein Pragma beendet *)
		c:= self.getChar();
		result:= result & Text.FromChar(c);
		IF c = '>' THEN (* zweites Zeichen gelesen -> Ende *)
		    RETURN result;
		END; (*IF*)
	    ELSE
		c:= self.getChar();
		result:= result & Text.FromChar(c);
	    END; (*IF*)
	END; (*LOOP*)

    (* Bei Erreichen des Dateiendes fehlt offenbar eine schlieende Klammer! *)
    EXCEPT
	Rd.EndOfFile=>RAISE SyntaxError(self.filename&": pragma with no end");
    END; (*TRY*)
END GetPragma;



PROCEDURE TokenType(self: T): TokenTypes =
BEGIN
	RETURN self.lastToken;
END TokenType;


PROCEDURE LineIndent(self: T): INTEGER =
BEGIN
	RETURN self.indent;
END LineIndent;


PROCEDURE CurrRow(self: T): INTEGER =
BEGIN
	RETURN self.row;
END CurrRow;


PROCEDURE CurrLine(self: T): INTEGER =
BEGIN
	RETURN self.line;
END CurrLine;


BEGIN
END ProgFileIO.