(*
   RPCSunUDP.m3
   Sun RPC on UDP datagrams.
   David Nichols, Xerox PARC
   July, 1991

   $Id: RPCSunUDP.m3,v 1.6 1992/03/31 01:21:26 nichols Exp $
*)

(* Copyright (c) 1991, 1992 Xerox Corporation.  All rights reserved.

   Use and copying of this software and preparation of derivative works
   based upon this software are permitted.  Any distribution of this
   software or derivative works must comply with all applicable United
   States export control laws.  This software is made available AS IS, and
   Xerox Corporation makes no warranty about the software, its performance
   or its conformity to any specification. *)

UNSAFE MODULE RPCSunUDP EXPORTS RPCSun, RPCSunPriv;

IMPORT RPC, RTScheduler, Thread, Time, Uin, Unix, Usocket, Utime, XDR,
       XDRMem;

TYPE
  UDPClient = Client OBJECT
                socket                    : INTEGER;
                bindingInfo               : BindingInfo;
                remoteAddr                : Uin.struct_sockaddr_in;
                xid                       : INTEGER                  := 0;
                sendBuffer, recvBuffer    : REF ARRAY OF CHAR;
                source                    : XDRMem.Source;
                sink                      : XDRMem.Sink;
                cred, verf                : Credentials;
                totalTimeout, retryTimeout: Time.T;
              OVERRIDES
                StartCall := StartCall;
                SendCall  := SendCall;
                EndCall   := EndCall;
                GetRemote := GetRemote;
                Destroy   := Destroy;
              END;

  UDPServer = Server OBJECT
                sp                    : ServerProc;
                socket                : INTEGER;
                prog, vers            : INTEGER;
                xid                   : INTEGER;
                sendBuffer, recvBuffer: REF ARRAY OF CHAR;
                source                : XDRMem.Source;
                sink                  : XDRMem.Sink;
                cred, verf            : Credentials;
              OVERRIDES
                StartReply := StartReply;
              END;


CONST
  BufSize   = 8800;             (* Magic number from Sun code. *)
  StackSize = 10000;            (* words *)

(*
 * Client
 *)

(* Import for UDP.  Assumes port has been looked up. *)
PROCEDURE UDPImportService (bi: BindingInfo): Client RAISES {RPC.Failed} =
  VAR c := NEW(UDPClient);
  BEGIN
    c.socket := GetUDPSocket();
    c.bindingInfo := bi;
    SetAddr(c.remoteAddr, bi.hostAddr, bi.port);
    c.sendBuffer := NEW(REF ARRAY OF CHAR, BufSize);
    c.recvBuffer := NEW(REF ARRAY OF CHAR, BufSize);
    c.sink := XDRMem.NewSink(c.sendBuffer);
    c.source := XDRMem.NewSource(c.recvBuffer);
    c.cred := NIL;
    c.verf := NIL;
    c.totalTimeout := Time.T{20, 0};
    c.retryTimeout := Time.T{2, 0};
    RETURN c;
  END UDPImportService;

PROCEDURE StartCall (c: UDPClient; proc: INTEGER): XDR.Sink
  RAISES {Erred, RPC.Failed, Thread.Alerted} =
  BEGIN
    TRY
      INC(c.xid);
      XDRMem.SetSinkPos(c.sink, 0);
      PutCallHeader(c.sink, c.xid, c.bindingInfo.progNum,
                    c.bindingInfo.progVersion, proc, c.cred, c.verf);
      (* Return the sink so the user can toss in his args. *)
      RETURN c.sink;
    EXCEPT
      XDR.Failed (e) =>
        RAISE RPC.Failed(
                NEW(RPC.ZeroTimesFailure,
                    info := "Couldn't marshal call header", subArg := e));
    END;
  END StartCall;

PROCEDURE SendCall (c: UDPClient): XDR.Source
  RAISES {Erred, RPC.Failed, Thread.Alerted} =
  VAR
    endTime, now, timeLeft               : Time.T;
    timeout                              : Utime.struct_timeval;
    sockSet                              : Unix.FDSet;
    xid, accept, code, authWhy, low, high: INTEGER;
    verf                                 : Credentials;
    dummyAddr                            : Uin.struct_sockaddr_in;
    nBytes                               : INTEGER;
  BEGIN
    (* TODO: Make sure we've consumed all the args. *)
    endTime := Time.Add(Time.Now(), c.totalTimeout);
    LOOP
      SendPacket(
        c.socket, SUBARRAY(c.sendBuffer^, 0, XDRMem.GetSinkPos(c.sink)),
        c.remoteAddr);
      (* Compute time to wait *)
      now := Time.Now();
      IF Time.Compare(now, endTime) > 0 THEN
        RAISE RPC.Failed(NEW(TimeoutFailure, info := "Call timed out."));
      END;
      timeLeft := Time.Subtract(endTime, now);
      IF Time.Compare(c.retryTimeout, timeLeft) < 0 THEN
        timeout := Utime.struct_timeval{
                     c.retryTimeout.seconds, c.retryTimeout.microseconds};
      ELSE
        timeout :=
          Utime.struct_timeval{timeLeft.seconds, timeLeft.microseconds};
      END;
      (* wait for reply: nBytes has number of bytes *)
      sockSet := Unix.FDSet{c.socket};
      IF RTScheduler.IOAlertSelect(
           Unix.MAX_FDSET, ADR(sockSet), NIL, ADR(sockSet), ADR(timeout))
           # 0 THEN
        TRY
          IF RecvPacket(c.socket, c.recvBuffer^, nBytes, dummyAddr) THEN
            XDRMem.SetSourcePos(c.source, 0);
            XDRMem.SetSourceLen(c.source, nBytes);
            GetReplyHeader(
              c.source, xid, accept, code, authWhy, low, high, verf);
            (* If xids match, we finally have a reply. *)
            IF xid = c.xid THEN EXIT END;
          END;
        EXCEPT
          XDR.Failed => EXIT;   (* just toss it if we can't decode it *)
        END;
      END;
    END;

    (* Should be smarter in processing error conditions. *)
    CASE accept OF
      MSG_ACCEPTED =>
        CASE code OF
          ACCEPT_SUCCESS => RETURN c.source;
        | ACCEPT_PROG_UNAVAIL =>
            RAISE RPC.Failed(NEW(RejectFailure,
                                 info := "Rejected: program unavailable"));
        | ACCEPT_PROC_UNAVAIL =>
            RAISE
              RPC.Failed(NEW(RejectFailure,
                             info := "Rejected: procedure unavailable"));
        | ACCEPT_GARBAGE_ARGS =>
            RAISE
              RPC.Failed(NEW(RejectFailure, info := "Rejected: bad args"));
        | ACCEPT_PROG_MISMATCH =>
            RAISE RPC.Failed(
                    NEW(RejectFailure,
                        info := "Rejected: program version unavailable"));
        ELSE
          RAISE RPC.Failed(
                  NEW(RPC.Failure, info := "Received bad accept code"));
        END;
    | MSG_DENIED =>
        CASE code OF
          REJECT_RPC_MISMATCH =>
            RAISE
              RPC.Failed(NEW(RejectFailure,
                             info := "Rejected: RPC version unavailable"));
        | REJECT_AUTH_ERROR =>
            RAISE
              RPC.Failed(NEW(RejectFailure,
                             info := "Rejected: Authentication failure"));
        ELSE
          RAISE RPC.Failed(
                  NEW(RPC.Failure, info := "Received bad reject code"));
        END;
    ELSE
      RAISE
        RPC.Failed(NEW(RPC.Failure, info := "Received bad reply packet"));
    END;
  END SendCall;

PROCEDURE EndCall (<*UNUSED*> c: UDPClient) RAISES {} =
  <* FATAL XDR.Failed, Thread.Alerted *>
  BEGIN
  END EndCall;

PROCEDURE GetRemote (c: UDPClient): BindingInfo =
  BEGIN
    RETURN c.bindingInfo;
  END GetRemote;

(* Shut down the connection.  We just close the socket to free up the file
   descrtiptor.  The buffers will float away with the client object. *)
PROCEDURE Destroy (c: UDPClient) RAISES {} =
  BEGIN
    EVAL Unix.close(c.socket);
  END Destroy;

(*
 * Server
 *)

PROCEDURE ExportUDP (sp: ServerProc; prog, vers: INTEGER; socket: INTEGER):
  BindingInfo RAISES {Erred, RPC.Failed, Thread.Alerted} =
  VAR
    host, port: INTEGER;
    s         : UDPServer;
  BEGIN
    GetHostPortFromSocket(socket, host, port);
    PortMapperRegister(prog, vers, port, Protocol.UDP);
    s := NEW(UDPServer);
    s.sp := sp;
    s.socket := socket;
    s.prog := prog;
    s.vers := vers;
    s.sendBuffer := NEW(REF ARRAY OF CHAR, BufSize);
    s.recvBuffer := NEW(REF ARRAY OF CHAR, BufSize);
    s.sink := XDRMem.NewSink(s.sendBuffer);
    s.source := XDRMem.NewSource(s.recvBuffer);
    s.cred := NIL;
    s.verf := NIL;
    EVAL
      Thread.Fork(NEW(ServerClosure, stackSize := StackSize, server := s));
    RETURN CreateBindingInfo(host, prog, vers, port, Protocol.UDP);
  END ExportUDP;

TYPE
  ServerClosure = Thread.SizedClosure OBJECT
                    server: UDPServer;
                  OVERRIDES
                    apply := ServerLoop;
                  END;

PROCEDURE ServerLoop (cl: ServerClosure): REFANY RAISES {} =
  VAR
    prog, vers, proc: INTEGER;
    server                                   := cl.server;
    ok              : BOOLEAN;
    fromAddr        : Uin.struct_sockaddr_in;
    sockSet         : Unix.FDSet;
    nBytes          : INTEGER;
  BEGIN
    TRY                         (* global error handler *)
      LOOP
        TRY                     (* one call *)
          ok := TRUE;
          sockSet := Unix.FDSet{cl.server.socket};
          EVAL RTScheduler.IOSelect(
                 Unix.MAX_FDSET, ADR(sockSet), NIL, ADR(sockSet));
          IF NOT RecvPacket(
                   server.socket, server.recvBuffer^, nBytes, fromAddr) THEN
            RETURN NIL;
          END;
          XDRMem.SetSourcePos(server.source, 0);
          XDRMem.SetSourceLen(server.source, nBytes);
          TRY
            GetCallHeader(server.source, server.xid, prog, vers, proc,
                          server.cred, server.verf);
          EXCEPT
            HeaderError =>
              SendRejection(server, RejectReason.RPCVersion);
              ok := FALSE;
          END;
          IF ok THEN
            IF prog # cl.server.prog THEN
              SendRejection(server, RejectReason.ProgUnavail);
            ELSIF vers # cl.server.vers THEN
              SendRejection(server, RejectReason.ProgMismatch);
            ELSE
              (* Do auth check here someday. *)
              server.sp.HandleCall(server, proc, server.source);
            END;
          END;
        EXCEPT
          XDR.Failed =>
            SendRejection(server, RejectReason.BadArgs);
            ok := FALSE;
        END;
        (* We've either sent the reply or a rejection, so send the
           packet. *)
        SendPacket(server.socket, SUBARRAY(server.sendBuffer^, 0,
                                           XDRMem.GetSinkPos(server.sink)),
                   fromAddr);
      END;
    EXCEPT
      (* If anything goes wrong, shut down the socket and give up. *)
      XDR.Failed, RPC.Failed, Erred, Thread.Alerted =>
        EVAL Unix.close(server.socket);
        RETURN NIL;
    END;
  END ServerLoop;

PROCEDURE StartReply (s: UDPServer): XDR.Sink RAISES {RPC.Failed} =
  <* FATAL XDR.Failed, Thread.Alerted *>
  BEGIN
    XDRMem.SetSinkPos(s.sink, 0);
    PutReplyHeader(s := s.sink, xid := s.xid, accept := MSG_ACCEPTED,
                   code := ACCEPT_SUCCESS, verf := s.verf);
    RETURN s.sink;
  END StartReply;

TYPE
  RejectReason = {RPCVersion, ProgUnavail, ProgMismatch, BadProc, BadArgs};
PROCEDURE SendRejection (s: UDPServer; why: RejectReason)
  RAISES {RPC.Failed} =
  <* FATAL XDR.Failed, Thread.Alerted *>
  BEGIN
    CASE why OF
      RejectReason.RPCVersion =>
        PutReplyHeader(
          s := s.sink, xid := s.xid, accept := MSG_DENIED,
          code := REJECT_RPC_MISMATCH, low := RPCVERS, high := RPCVERS);
    | RejectReason.ProgUnavail =>
        PutReplyHeader(s := s.sink, xid := s.xid, accept := MSG_ACCEPTED,
                       code := ACCEPT_PROG_UNAVAIL, verf := s.verf);
    | RejectReason.ProgMismatch =>
        PutReplyHeader(s := s.sink, xid := s.xid, accept := MSG_ACCEPTED,
                       code := ACCEPT_PROG_MISMATCH, low := s.vers,
                       high := s.vers, verf := s.verf);
    | RejectReason.BadProc =>
        PutReplyHeader(s := s.sink, xid := s.xid, accept := MSG_ACCEPTED,
                       code := ACCEPT_PROC_UNAVAIL, verf := s.verf);
    | RejectReason.BadArgs =>
        PutReplyHeader(s := s.sink, xid := s.xid, accept := MSG_ACCEPTED,
                       code := ACCEPT_GARBAGE_ARGS, verf := s.verf);
    END;
  END SendRejection;

(*
 * Sockets.
 *)

(* Get a generic UDP socket. *)
PROCEDURE GetUDPSocket (port := 0): INTEGER RAISES {RPC.Failed} =
  VAR
    s   : INTEGER;
    addr: Uin.struct_sockaddr_in;
  BEGIN
    s := Usocket.socket(Usocket.AF_INET, Usocket.SOCK_DGRAM, 0);
    IF s = -1 THEN
      RAISE
        RPC.Failed(NEW(RPC.Failure, info := "Couldn't get UDP socket."));
    END;
    SetAddr(addr, 0, port);
    IF Usocket.bind(s, ADR(addr), BYTESIZE(addr)) = -1 THEN
      EVAL Unix.close(s);
      RAISE RPC.Failed(
              NEW(RPC.Failure, info := "Bind to remote address failed."));
    END;
    RETURN s;
  END GetUDPSocket;

PROCEDURE SendPacket (         socket  : INTEGER;
                      READONLY data    : ARRAY OF CHAR;
                      READONLY destAddr: Uin.struct_sockaddr_in)
  RAISES {RPC.Failed} =
  BEGIN
    IF Usocket.sendto(
         s := socket, msg := ADR(data[0]), len := NUMBER(data), flags := 0,
         to := ADR(destAddr), tolen := BYTESIZE(destAddr)) = -1 THEN
      RAISE RPC.Failed(NEW(RPC.Failure, info := "sento failed"));
    END;
  END SendPacket;

PROCEDURE RecvPacket (         socket    : INTEGER;
                      READONLY data      : ARRAY OF CHAR;
                      VAR      nBytes    : INTEGER;
                      VAR      sourceAddr: Uin.struct_sockaddr_in):
  BOOLEAN =
  VAR
    result : INTEGER;
    addrLen: INTEGER := BYTESIZE(sourceAddr);
  BEGIN
    result := Usocket.recvfrom(
                s := socket, buf := ADR(data[0]), len := NUMBER(data),
                flags := 0, from := ADR(sourceAddr), fromlen := addrLen);
    IF result = -1 THEN
      RETURN FALSE;
    ELSE
      nBytes := result;
      RETURN TRUE;
    END;
  END RecvPacket;

BEGIN
END RPCSunUDP.
