/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 1997-2001
 *	Sleepycat Software.  All rights reserved.
 */

#include "db_config.h"

#ifndef lint
static const char revid[] = "$Id: cxx_db.cpp,v 11.52 2001/12/20 21:56:40 mjc Exp $";
#endif /* not lint */

#include <errno.h>
#include <string.h>

#include "db_cxx.h"
#include "cxx_int.h"

#include "db_int.h"
#include "db_page.h"
#include "db_auto.h"
#include "crdel_auto.h"
#include "db_ext.h"
#include "common_ext.h"

// Helper macros for simple methods that pass through to the
// underlying C method. It may return an error or raise an exception.
// Note this macro expects that input _argspec is an argument
// list element (e.g., "char *arg") and that _arglist is the arguments
// that should be passed through to the C method (e.g., "(db, arg)")
//
#define DB_METHOD(_name, _argspec, _arglist)			  \
int Db::_name _argspec						  \
{								  \
	int ret;						  \
	DB *db = unwrap(this);					  \
								  \
	if ((ret = db->_name _arglist) != 0)			  \
		DB_ERROR("Db::" # _name, ret, error_policy());	  \
	return (ret);						  \
}

#define DB_METHOD_CHECKED(_name, _cleanup, _argspec, _arglist)	  \
int Db::_name _argspec						  \
{								  \
	int ret;						  \
	DB *db = unwrap(this);					  \
								  \
	if (!db) {						  \
		DB_ERROR("Db::" # _name, EINVAL, error_policy()); \
		return (EINVAL);				  \
	}							  \
	if (_cleanup)						  \
		cleanup();					  \
	if ((ret = db->_name _arglist) != 0)			  \
		DB_ERROR("Db::" # _name, ret, error_policy());	  \
	return (ret);						  \
}

#define DB_METHOD_QUIET(_name, _argspec, _arglist)		  \
int Db::_name _argspec						  \
{								  \
	DB *db = unwrap(this);					  \
								  \
	return (db->_name _arglist);				  \
}

#define DB_METHOD_VOID(_name, _argspec, _arglist)		  \
void Db::_name _argspec 					  \
{								  \
	DB *db = unwrap(this);					  \
								  \
	db->_name _arglist;					  \
}

// A truism for the Db object is that there is a valid
// DB handle from the constructor until close().
// After the close, the DB handle is invalid and
// no operations are permitted on the Db (other than
// destructor).  Leaving the Db handle open and not
// doing a close is generally considered an error.
//
// We used to allow Db objects to be closed and reopened.
// This implied always keeping a valid DB object, and
// coordinating the open objects between Db/DbEnv turned
// out to be overly complicated.  Now we do not allow this.

Db::Db(DbEnv *env, u_int32_t flags)
:	imp_(0)
,	env_(env)
,	construct_error_(0)
,	flags_(0)
,	construct_flags_(flags)
{
	if (env_ == 0)
		flags_ |= DB_CXX_PRIVATE_ENV;
	initialize();
}

// Note: if the user has not closed, we call _destroy_check
// to warn against this non-safe programming practice.
// We can't close, because the environment may already
// be closed/destroyed.
//
Db::~Db()
{
	DB *db;

	db = unwrap(this);
	if (db != NULL) {
		DbEnv::_destroy_check("Db", 0);
		cleanup();
	}
}

// private method to initialize during constructor.
// initialize must create a backing DB object,
// and if that creates a new DB_ENV, it must be tied to a new DbEnv.
// If there is an error, construct_error_ is set; this is examined
// during open.
//
int Db::initialize()
{
	u_int32_t cxx_flags;
	DB *db;
	int err;
	DB_ENV *cenv = unwrap(env_);

	cxx_flags = construct_flags_ & DB_CXX_NO_EXCEPTIONS;

	// Create a new underlying DB object.
	// We rely on the fact that if a NULL DB_ENV* is given,
	// one is allocated by DB.
	//
	if ((err = db_create(&db, cenv,
			     construct_flags_ & ~cxx_flags)) != 0) {
		construct_error_ = err;
		return (err);
	}

	// Associate the DB with this object
	imp_ = wrap(db);
	db->cj_internal = this;

	// Create a new DbEnv from a DB_ENV* if it was created locally.
	// It is deleted in Db::close().
	//
	if ((flags_ & DB_CXX_PRIVATE_ENV) != 0)
		env_ = new DbEnv(db->dbenv, cxx_flags);

	return (0);
}

// private method to cleanup after destructor or during close.
// If the environment was created by this Db object, we optionally
// delete it, or return it so the caller can delete it after
// last use.
//
void Db::cleanup()
{
	DB *db = unwrap(this);

	if (db != NULL) {
		// extra safety
		db->cj_internal = 0;
		imp_ = 0;

		// we must dispose of the DbEnv object if
		// we created it.  This will be the case
		// if a NULL DbEnv was passed into the constructor.
		// The underlying DB_ENV object will be inaccessible
		// after the close, so we must clean it up now.
		//
		if ((flags_ & DB_CXX_PRIVATE_ENV) != 0) {
			env_->cleanup();
			delete env_;
			env_ = 0;
		}
	}
	construct_error_ = 0;
}

// Return a tristate value corresponding to whether we should
// throw exceptions on errors:
//   ON_ERROR_RETURN
//   ON_ERROR_THROW
//   ON_ERROR_UNKNOWN
//
int Db::error_policy()
{
	if (env_ != NULL)
		return (env_->error_policy());
	else {
		// If the env_ is null, that means that the user
		// did not attach an environment, so the correct error
		// policy can be deduced from constructor flags
		// for this Db.
		//
		if ((construct_flags_ & DB_CXX_NO_EXCEPTIONS) != 0) {
			return (ON_ERROR_RETURN);
		}
		else {
			return (ON_ERROR_THROW);
		}
	}
}

int Db::close(u_int32_t flags)
{
	DB *db = unwrap(this);
	int err;

	// after a DB->close (no matter if success or failure),
	// the underlying DB object must not be accessed,
	// so we clean up in advance.
	//
	cleanup();

	// It's safe to throw an error after the close,
	// since our error mechanism does not peer into
	// the DB* structures.
	//
	if ((err = db->close(db, flags)) != 0 && err != DB_INCOMPLETE)
		DB_ERROR("Db::close", err, error_policy());

	return (err);
}

// The following cast implies that Dbc can be no larger than DBC
DB_METHOD(cursor, (DbTxn *txnid, Dbc **cursorp, u_int32_t flags),
    (db, unwrap(txnid), (DBC **)cursorp, flags))

int Db::del(DbTxn *txnid, Dbt *key, u_int32_t flags)
{
	DB *db = unwrap(this);
	int err;

	if ((err = db->del(db, unwrap(txnid), key, flags)) != 0) {
		// DB_NOTFOUND is a "normal" return, so should not be
		// thrown as an error
		//
		if (err != DB_NOTFOUND) {
			DB_ERROR("Db::del", err, error_policy());
			return (err);
		}
	}
	return (err);
}

void Db::err(int error, const char *format, ...)
{
	va_list args;
	DB *db = unwrap(this);

	va_start(args, format);
	__db_real_err(db->dbenv, error, 1, 1, format, args);
	va_end(args);
}

void Db::errx(const char *format, ...)
{
	va_list args;
	DB *db = unwrap(this);

	va_start(args, format);
	__db_real_err(db->dbenv, 0, 0, 1, format, args);
	va_end(args);
}

int Db::fd(int *fdp)
{
	DB *db = unwrap(this);
	int err;

	if ((err = db->fd(db, fdp)) != 0) {
		DB_ERROR("Db::fd", err, error_policy());
		return (err);
	}
	return (0);
}

int Db::get(DbTxn *txnid, Dbt *key, Dbt *value, u_int32_t flags)
{
	DB *db = unwrap(this);
	int err;

	if ((err = db->get(db, unwrap(txnid), key, value, flags)) != 0) {
		// DB_NOTFOUND and DB_KEYEMPTY are "normal" returns,
		// so should not be thrown as an error
		//
		if (err != DB_NOTFOUND && err != DB_KEYEMPTY) {
			const char *name = "Db::get";
			if (err == ENOMEM && DB_OVERFLOWED_DBT(value))
				DB_ERROR_DBT(name, value, error_policy());
			else
				DB_ERROR(name, err, error_policy());
			return (err);
		}
	}
	return (err);
}

int Db::get_byteswapped(int *isswapped)
{
	DB *db = (DB *)unwrapConst(this);
	return (db->get_byteswapped(db, isswapped));
}

int Db::get_type(DBTYPE *dbtype)
{
	DB *db = (DB *)unwrapConst(this);
	return (db->get_type(db, dbtype));
}

// Dbc is a "compatible" subclass of DBC - that is, no virtual functions
// or even extra data members, so these casts, although technically
// non-portable, "should" always be okay.
DB_METHOD(join, (Dbc **curslist, Dbc **cursorp, u_int32_t flags),
    (db, (DBC **)curslist, (DBC **)cursorp, flags))

DB_METHOD(key_range,
    (DbTxn *txnid, Dbt *key, DB_KEY_RANGE *results, u_int32_t flags),
    (db, unwrap(txnid), key, results, flags))

// If an error occurred during the constructor, report it now.
// Otherwise, call the underlying DB->open method.
//
int Db::open(const char *file, const char *database,
	     DBTYPE type, u_int32_t flags, int mode)
{
	int err;
	DB *db = unwrap(this);

	if ((err = construct_error_) != 0)
		DB_ERROR("Db::open", construct_error_, error_policy());
	else if ((err = db->open(db, file, database, type, flags, mode)) != 0)
		DB_ERROR("Db::open", err, error_policy());

	return (err);
}

int Db::pget(DbTxn *txnid, Dbt *key, Dbt *pkey, Dbt *value, u_int32_t flags)
{
	DB *db = unwrap(this);
	int err;

	if ((err = db->pget(db, unwrap(txnid), key, pkey,
			    value, flags)) != 0) {
		// DB_NOTFOUND and DB_KEYEMPTY are "normal" returns,
		// so should not be thrown as an error
		//
		if (err != DB_NOTFOUND && err != DB_KEYEMPTY) {
			const char *name = "Db::pget";
			if (err == ENOMEM && DB_OVERFLOWED_DBT(value))
				DB_ERROR_DBT(name, value, error_policy());
			else
				DB_ERROR(name, err, error_policy());
			return (err);
		}
	}
	return (err);
}

int Db::put(DbTxn *txnid, Dbt *key, Dbt *value, u_int32_t flags)
{
	int err;
	DB *db = unwrap(this);

	if ((err = db->put(db, unwrap(txnid), key, value, flags)) != 0) {

		// DB_KEYEXIST is a "normal" return, so should not be
		// thrown as an error
		//
		if (err != DB_KEYEXIST) {
			DB_ERROR("Db::put", err, error_policy());
			return (err);
		}
	}
	return (err);
}
    
DB_METHOD_CHECKED(rename, true,
    (const char *file, const char *database, const char *newname, u_int32_t flags),
    (db, file, database, newname, flags))
DB_METHOD_CHECKED(remove, true,
    (const char *file, const char *database, u_int32_t flags),
    (db, file, database, flags))
DB_METHOD_CHECKED(truncate, false,
    (DbTxn *txnid, u_int32_t *countp, u_int32_t flags),
    (db, unwrap(txnid), countp, flags))
DB_METHOD_CHECKED(stat, false,
    (void *sp, u_int32_t flags), (db, sp, flags))

int Db::sync(u_int32_t flags)
{
	int err;
	DB *db = unwrap(this);

	if (!db) {
		DB_ERROR("Db::sync", EINVAL, error_policy());
		return (EINVAL);
	}
	if ((err = db->sync(db, flags)) != 0 && err != DB_INCOMPLETE) {
		DB_ERROR("Db::sync", err, error_policy());
		return (err);
	}
	return (err);
}

DB_METHOD_CHECKED(upgrade, false,
    (const char *name, u_int32_t flags), (db, name, flags))

////////////////////////////////////////////////////////////////////////
//
// callbacks
//
// *_intercept_c are 'glue' functions that must be declared
// as extern "C" so to be typesafe.  Using a C++ method, even
// a static class method with 'correct' arguments, will not pass
// the test; some picky compilers do not allow mixing of function
// pointers to 'C' functions with function pointers to C++ functions.
//
// One wart with this scheme is that the *_callback_ method pointer
// must be declared public to be accessible by the C intercept.
// It's possible to accomplish the goal without this, and with
// another public transfer method, but it's just too much overhead.
// These callbacks are supposed to be *fast*.
//
// The DBTs we receive in these callbacks from the C layer may be
// manufactured there, but we want to treat them as a Dbts.
// Technically speaking, these DBTs were not constructed as a Dbts,
// but it should be safe to cast them as such given that Dbt is a
// *very* thin extension of the DBT.  That is, Dbt has no additional
// data elements, does not use virtual functions, virtual inheritance,
// multiple inheritance, RTI, or any other language feature that
// causes the structure to grow or be displaced.  Although this may
// sound risky, a design goal of C++ is complete structure
// compatibility with C, and has the philosophy 'if you don't use it,
// you shouldn't incur the overhead'.  If the C/C++ compilers you're
// using on a given machine do not have matching struct layouts, then
// a lot more things will be broken than just this.
//
// The alternative, creating a Dbt here in the callback, and populating
// it from the DBT, is just too slow and cumbersome to be very useful.

#define DB_CALLBACK_C_INTERCEPT(_name, _rettype, _cargspec, _cthis,	\
    _return, _cxxargs)							\
extern "C" _rettype _db_##_name##_intercept_c _cargspec			\
{									\
	Db *cxxthis;							\
 									\
	DB_ASSERT(_cthis != NULL);					\
	cxxthis = (Db *)_cthis->cj_internal;				\
	DB_ASSERT(cxxthis != NULL);					\
	DB_ASSERT(cxxthis->associate_callback_ != 0);			\
									\
	_return (*cxxthis->_name##_callback_) _cxxargs;			\
}

#define DB_SET_CALLBACK(_cxxname, _name, _cxxargspec, _db, _cb, _cargs)	\
int Db::_cxxname _cxxargspec						\
{									\
	DB *cthis = unwrap(this);					\
									\
	_db->_name##_callback_ = _cb;					\
	return ((*(cthis->_cxxname)) _cargs);				\
}

DB_CALLBACK_C_INTERCEPT(associate,
	int, (DB *secondary, const DBT *key, const DBT *data, DBT *retval),
	secondary, return,
	(cxxthis, Dbt::get_const_Dbt(key), Dbt::get_const_Dbt(data), Dbt::get_Dbt(retval)))

DB_SET_CALLBACK(associate, associate,
    (Db *secondary, int (*callback)(Db *, const Dbt *, const Dbt *, Dbt *), u_int32_t flags),
    secondary, callback,
	(cthis, unwrap(secondary), _db_associate_intercept_c, flags)
)

DB_CALLBACK_C_INTERCEPT(feedback,
    void, (DB *cthis, int opcode, int pct),
    cthis, /* no return */ (void), (cxxthis, opcode, pct))

DB_SET_CALLBACK(set_feedback, feedback,
    (void (*arg)(Db *cxxthis, int opcode, int pct)),
    this, arg, (cthis, _db_feedback_intercept_c))

DB_CALLBACK_C_INTERCEPT(append_recno,
    int, (DB *cthis, DBT *data, db_recno_t recno),
    cthis, return, (cxxthis, Dbt::get_Dbt(data), recno))

DB_SET_CALLBACK(set_append_recno, append_recno,
    (int (*arg)(Db *cxxthis, Dbt *data, db_recno_t recno)),
    this, arg, (cthis, _db_append_recno_intercept_c))

DB_CALLBACK_C_INTERCEPT(bt_compare,
    int, (DB *cthis, const DBT *data1, const DBT *data2),
    cthis, return,
    (cxxthis, Dbt::get_const_Dbt(data1), Dbt::get_const_Dbt(data2)))

DB_SET_CALLBACK(set_bt_compare, bt_compare,
    (int (*arg)(Db *cxxthis, const Dbt *data1, const Dbt *data2)),
    this, arg, (cthis, _db_bt_compare_intercept_c))

DB_CALLBACK_C_INTERCEPT(bt_prefix,
    size_t, (DB *cthis, const DBT *data1, const DBT *data2),
    cthis, return,
    (cxxthis, Dbt::get_const_Dbt(data1), Dbt::get_const_Dbt(data2)))

DB_SET_CALLBACK(set_bt_prefix, bt_prefix,
    (size_t (*arg)(Db *cxxthis, const Dbt *data1, const Dbt *data2)),
    this, arg, (cthis, _db_bt_prefix_intercept_c))

DB_CALLBACK_C_INTERCEPT(dup_compare,
    int, (DB *cthis, const DBT *data1, const DBT *data2),
    cthis, return,
    (cxxthis, Dbt::get_const_Dbt(data1), Dbt::get_const_Dbt(data2)))

DB_SET_CALLBACK(set_dup_compare, dup_compare,
    (int (*arg)(Db *cxxthis, const Dbt *data1, const Dbt *data2)),
    this, arg, (cthis, _db_dup_compare_intercept_c))

DB_CALLBACK_C_INTERCEPT(h_hash,
    u_int32_t, (DB *cthis, const void *data, u_int32_t len),
    cthis, return, (cxxthis, data, len))

DB_SET_CALLBACK(set_h_hash, h_hash,
    (u_int32_t (*arg)(Db *cxxthis, const void *data, u_int32_t len)),
    this, arg, (cthis, _db_h_hash_intercept_c))


// This is a 'glue' function declared as extern "C" so it will
// be compatible with picky compilers that do not allow mixing
// of function pointers to 'C' functions with function pointers
// to C++ functions.
//
extern "C"
int _verify_callback_c(void *handle, const void *str_arg)
{
	char *str;
	OSTREAMCLASS *out;

	str = (char *)str_arg;
	out = (OSTREAMCLASS *)handle;

	(*out) << str;
	if (out->fail())
		return (EIO);

	return (0);
}

int Db::verify(const char *name, const char *subdb,
	       OSTREAMCLASS *ostr, u_int32_t flags)
{
	int err;
	DB *db = unwrap(this);

	if (!db) {
		DB_ERROR("Db::verify", EINVAL, error_policy());
		return (EINVAL);
	}
	if ((err = __db_verify_internal(db, name, subdb, ostr,
					_verify_callback_c, flags)) != 0) {
		DB_ERROR("Db::verify", err, error_policy());
		return (err);
	}
	return (0);
}

DB_METHOD(set_bt_compare, (bt_compare_fcn_type func), (db, func))
DB_METHOD(set_bt_maxkey, (u_int32_t bt_maxkey), (db, bt_maxkey))
DB_METHOD(set_bt_minkey, (u_int32_t bt_minkey), (db, bt_minkey))
DB_METHOD(set_bt_prefix, (bt_prefix_fcn_type func), (db, func))
DB_METHOD(set_dup_compare, (dup_compare_fcn_type func), (db, func))
DB_METHOD_VOID(set_errfile, (FILE *errfile), (db, errfile))
DB_METHOD_VOID(set_errpfx, (const char *errpfx), (db, errpfx))
DB_METHOD(set_flags, (u_int32_t flags), (db, flags))
DB_METHOD(set_h_ffactor, (u_int32_t h_ffactor), (db, h_ffactor))
DB_METHOD(set_h_hash, (h_hash_fcn_type func), (db, func))
DB_METHOD(set_h_nelem, (u_int32_t h_nelem), (db, h_nelem))
DB_METHOD(set_lorder, (int db_lorder), (db, db_lorder))
DB_METHOD(set_pagesize, (u_int32_t db_pagesize), (db, db_pagesize))
DB_METHOD(set_re_delim, (int re_delim), (db, re_delim))
DB_METHOD(set_re_len, (u_int32_t re_len), (db, re_len))
DB_METHOD(set_re_pad, (int re_pad), (db, re_pad))
DB_METHOD(set_re_source, (char *re_source), (db, re_source))
DB_METHOD(set_q_extentsize, (u_int32_t extentsize), (db, extentsize))

DB_METHOD_QUIET(set_alloc,
    (db_malloc_fcn_type malloc_fcn, db_realloc_fcn_type realloc_fcn,
        db_free_fcn_type free_fcn),
    (db, malloc_fcn, realloc_fcn, free_fcn))

void Db::set_errcall(void (*arg)(const char *, char *))
{
	env_->set_errcall(arg);
}

void *Db::get_app_private() const
{
	return unwrapConst(this)->app_private;
}

void Db::set_app_private(void *value)
{
	unwrap(this)->app_private = value;
}

DB_METHOD(set_cachesize, (u_int32_t gbytes, u_int32_t bytes, int ncache),
    (db, gbytes, bytes, ncache))

int Db::set_paniccall(void (*callback)(DbEnv *, int))
{
	return (env_->set_paniccall(callback));
}

void Db::set_error_stream(OSTREAMCLASS *error_stream)
{
	env_->set_error_stream(error_stream);
}
