;;; -*- Mode: Emacs-Lisp; Syntax: Common-lisp; Base: 10; -*-
;;; File: scr-title.el
;;; Author: Heinz Schmidt (hws@icsi.berkeley.edu)
;;; Created: Thu Nov 29 19:40:34 1990
;;; Copyright (C) 1990, International Computer Science Institute
;;;
;;; THIS CODE IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT.  WE
;;; DISTRIBUTE IT IN THE HOPE THAT IT WILL BE USEFUL. BUT EXCEPT WHEN OTHERWISE
;;; STATED IN WRITING, THE INTERNATIONAL COMPUTER SCIENCE INSTITUTE AND/OR THE
;;; AUTHOR(S) PROVIDE THIS CODE "AS IS" WITHOUT WARRANTY OF ANY KIND.
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: Explicit and startup setting of screen and icon titles,
;;;*           Native Epoch creates screens with current buffer title, but
;;;*           titles are not changed when other buffer is selected. 
;;;*
;;;* RELATED PACKAGES: builds on epoch.el hook *create-screen-hook*
;;;*
;;;* HISTORY: 
;;;* Last edited: Jan 16 16:00 1992 (hws)
;;;*  May 27 10:38 1991 (hws): rename buffer-title-screen to title-screen.
;;;*  Dec  5 21:08 1990 (hws): fixed buffer-title-screen to work like doc says.
;;;*  Nov 29 19:40 1990 (hws): buffer-title-screen split off my .epoch 
;;;*   and added the hook functionality.
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(defun title-screen (&optional arg)
  "Redefines the title of the screen to be the name of the current buffer.
The icon name is changed appropriately. 
With \\[universal-argument] 0 prompts for the title,
     \\[universal-argument] positive chooses a default title independent
of buffer names."
  (interactive "P")
  (let ((title 
	 (cond ((null arg)
		(concat 
		 (format "%s" (buffer-name (current-buffer)))
		 (if include-system-name (format " @ %s" (system-name)) "")))
	       ((zerop arg) (read-input "Screen title: "))
	       (t (default-screen-title)))))
    (title title)
    (icon-name (default-icon-title title))))

(defvar screen-count 0)

(defun default-screen-title ()
  "A new unique title each time it is called."
  (concat 
   (format "Epoch %d" (setq screen-count (+ 1 screen-count)))
   (if include-system-name		; respect user profile settings
       (format " @ %s" (system-name))
     "")))

(defun default-icon-title (title)
  (if include-system-name		; respect user profile settings
      ;; assume that icon titles are limited and user works epoch on several machines
      (let ((title-end (string-match "\\(@\\|$\\)" title))
	    (len (length title)))
	(format "%s%s" (substring title 
				    (max 0 (- title-end 3)) 
				    title-end)
		(system-name)))
    title))

(defvar epoch-screen-title-alist nil
  "* An alist mapping screen titles (symbols) to X default property lists.")

(defun title-create-screen-hook (alist)
  "A hook to push into *create-screen-alist-hook*. Defines the title for screens
and their icons. Set include-system-name to NIL if the title gets too long for
your window manager."
  (let* ((tit (assq 'title alist))
	 (props (assq (intern (cdr tit)) epoch-screen-title-alist))
	 (icon (assq 'icon-name alist))
	 (newtitle (default-screen-title)))
    (if  props
	(append props alist)
      (progn
	(rplacd tit newtitle)
	(rplacd icon (default-icon-title newtitle))
	alist))))

(defvar screen-title-is-buffer-name nil
  "*When T, the screen title is made the buffer name whenever the screen is 
selected.")		      ; can we do better? no hook when mode-line changes.

(defun auto-buffer-title-screen-hook ()
  (when screen-title-is-buffer-name (title-screen nil)))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Activate screen title commands

;close to title refresh \C-za, on a key free in native Epoch.
(global-set-key "\C-zb" 'title-screen) 

(push 'auto-buffer-title-screen-hook *select-screen-hook*)
(push 'title-create-screen-hook *create-screen-alist-hook*)

;;; The first screen is already created when this file is processed. Set
;;; title explicitly.

(title-screen 2)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Customize screen sizes on a mode basis. This shows how it works.
;;;
;
;(push '(lisp-mode (geometry . "80x48+20+100") (cursor-glyph . 2)
;		  (title . "Lisp") (icon-name . "Lisp")) epoch-mode-alist)
;(push '(emacs-lisp-mode (geometry . "81x52+20+100") (cursor-glyph . 2)
;			(title . "Lisp") (icon-name . "Lisp")) epoch-mode-alist)
;(push '(c-mode (geometry . "80x40+100+0") (cursor-glyph . 2)
;			(title . "C") (icon-name . "C")) epoch-mode-alist)
;

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Customize screen sizes on a title basis. This is useful for unique screens
;;; like *compilation* or a server screen serving emacsclients.
;;;
(push '(*compilation* (geometry . "80x48+10+100") (cursor-glyph . 2)
		      (title . "Compilation") (icon-name . "GnuComp"))
      epoch-screen-title-alist)
(push '(Server\ Screen (geometry . "80x45+50+50") (cursor-glyph . 2)
		      (title . "Server Screen") (icon-name . "GnuServe"))
      epoch-screen-title-alist)

		      
		      
