;;; -*- Mode: Emacs-Lisp; Syntax: Common-lisp; Base: 10; Package: SKY -*-
;;; File: epoch-mouse-base.el
;;; Author: Heinz Schmidt (hws@icsi.berkeley.edu)
;;; Created: Wed Nov 28 12:20:30 1990
;;; Copyright (C) 1990, International Computer Science Institute
;;;
;;; THIS CODE IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT
;;; UNDER THE TERMS OF THE GNU EMACS GENERAL PUBLIC LICENSE.
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: Epoch dependent realization of mouse primitives for sky-mouse.el
;;;* 
;;;* RELATED PACKAGES: sky-mouse.el, 
;;;*                   modifies Epoch select-screen,
;;;*                   extends Epoch mouse.el,
;;;*                   rebinds mouse-keys and set C-h z to mouse help.
;;;* HISTORY:
;;;* Last edited: Mar  8 14:38 1992 (hws)
;;;*  Nov 12 16:40 1991 (hws): adapt multi clicks to use "real" event time(stamp)
;;;*                           under Epoch 4
;;;*  Nov 11 19:15 1991 (hws): integrated Tom Emerson's patch for Epoch4
;;;*  May 26 22:35 1991 (hws): back to auto-raise-screen standard (in select-screen)
;;;*  Jan 25 17:57 1991 (hws): check mouse handling also when thrown out of
;;;*     motion handling.
;;;*  Jan 25 16:45 1991 (hws): mouse-set-point fixed to work on not
;;;*     selected window too. ditto for mouse-extend-drag-point.
;;;*  Jan 25 14:55 1991 (hws): By and by added multi-clicks.
;;;*     Made help mouse avoid an error when mouse table holds a non-function
;;;*     symbol (such as hyper-man mouse does).
;;;*     Fixed a bug in mouse-extend-drag-point. When it is the first operation
;;;*     after revert-buffer, a region marking wouldn't be forgot.
;;;*  Jan 20 17:13 1991 (hws): Add mouse feedback and make install command
;;;*     available. Now that there is a mouse-handler macro form, you
;;;*     better always run this file compiled.
;;;*     Fixed a case of wrong underlining with mouse-extend-drag-point.
;;;*  Dec 17 09:26 1990 (hws): add mouse-reset to help when mouse screws up.
;;;*  Dec 13 23:20 1990 (hws): force redisplay after mouse paste, fixes a bug
;;;*     that occurs when multi-line marking was beyond end of line, somehow
;;;*     redisplay thinks, there is input, or what?
;;;*     Comment helpful-mouse-handler installation.
;;;*     Mouse-set-point sets focus now consistently.
;;;*  Dec  6 02:00 1990 (hws): present mouse buttons acc. to Emacs conventions
;;;*  Dec  3 20:20 1990 (hws): cleaned Epoch redefinitions, only select-screen
;;;*     is redefined. Mouse help no longer modifies mouse::handler but
;;;*     is pushed/popped to event-handler stack.
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Note: Mouse key binding at the end of this file. We use an environment
;;; variable WINMGR, when olwm, key binding for region extensions follows
;;; the Openlook standard and is on Middle.  (Right pastes). Otherwise 
;;; extend is on right and middle pastes (convention for twm).
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Just to recall the interface sky-mouse requires on top of host system ...
;;;
;;; Abstract Mouse, used in all mouse commands
;;;
;;; mouse-set-point          - actually set point and mark, reset s/k/y state
;;; mouse-drag-point         - mouse-set-point and let point follow mouse
;;;                            if this is supported by host system.
;;; mouse-end-drag-point     - set point at current location and stop dragging
;;;                            point
;;; mark-region              - provide visual feedback for current region.
;;; forget-region            - remove visual feedback
;;;
;;; store-cut-buffer(from to),cut-buffer
;;;
;;; set-kbd-focus(x), release-kbd-focus
;;;
;;; mouse-warp-to-point
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 
(provide 'epoch-mouse-base)

(require 'motion)
(require 'mouse)

;;; Record the version of Epoch that we are running
;;; Make sure the code continues to run under Epoch version < 4 (hws)
(defvar sky-epoch-3
  (and (> (length epoch::version) 6) ;;#+Epoch-3
	    (string-equal (substring epoch::version 0 7) 
					    "Epoch 3")))

;;;****************************************************************************
;;; EPOCH EXTENSIONS
;;;****************************************************************************
;;; Note: For series of mouse commands the mouse must be able to interrupt a
;;; going minibuffer interaction, in particular when the mouse can be used
;;; across screens.  Epoch mouse::set-point, used in series of mouse commands,
;;; would leave a minibuffer interaction going (kbd stays with minibuffer) and
;;; set-mark or relatives would suddenly release another screen's point and put
;;; that screen into a state some time ago, unrelated to the current
;;; interaction.  Mouse commands require a simple way to define the point at
;;; which they apply or the region which they affect. Epoch end-mouse-drag
;;; leaves mark and point if we click on a single pixel, but moves them when
;;; button-up misses the position of the corresponding button-down transition.
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

;;; The following selection gestures exhibit common behavior for high level
;;; mouse command (cf. mouse-define).
;;; They save current point and ... for sort of mouse-save-excursion
;;; in command proper on button down.
;;; If they confine drag, they rely on the mouse-handler to reenable drag
;;; after button release and before dispatching to the release command.


(defun mouse-local-map () mouse::local-map)
(defun mouse-use-local-map (map)
  "Switch to local mouse map to be map, or if nil unset the current local map."
  (if map (use-local-mouse-map map)
    (setq mouse::local-map nil)))

(defmacro save-excursion-info ()
  '(setq mouse-save-excursion-info
	(list (mark) (point) (current-buffer) (selected-window) (current-screen))))

(defun mouse-set-point (arg)
  "Select window mouse is on, move point and mark to mouse position, i.e.
make the current region empty.
Programmers use this at the begin of Mouse Point Commands, mouse commands that
let the user select a point to operate from there. (Free-hand) Mouse Region
Commands, that require visual feedback to make a reliable selection are started
with the companion commands mouse-drag-point or mouse-extend-drag-point.

Novice mouse command programmers often try to use dragging for the wrong
purposes. Mouse-set-point is also used if the region between mouse and point is
affected and needs not be selected upon command completion, or simply if the
point and mouse cursors provide sufficient feedback. For instance
warping the mouse to point or killing the text between point and mouse
do not require that region to be marked."
  (interactive) 
  (let ((scr (nth 3 arg)))
    (save-excursion-info)
    (select-screen scr) 
    (select-window (nth 2 arg))
    (forget-region)
    (goto-char (car arg))
    (push-mark (point))
    (if current-focus-screen
	(set-kbd-focus scr))
    (sky-state-reset)
    (epoch::redisplay-screen)		;workaroud 'hanging redisplay bug'
    ;; TODO: allow to quit minibuffer cmds; 
    ;; something in the line of (force-kbd-input "\C-g\C-g"), say.
    ;; (setq inhibit-quit nil)            ; doesn't help really
    ;; (top-level)                        ; aborts callers
    ))

(defun mouse-drag-point (arg)
  "Select window mouse is on, move point and mark to mouse position, i.e.
make the current region empty. Then start dragging point.
Programmers use this at the begin of Mouse Drag Commands, mouse commands that
give the user feedback for a free-hand selection of a region.
Commands, that require visual feedback to make a reliable selection are started
with the companion commands mouse-drag-point or mouse-extend-drag-point.
Mouse drag commands consist of a pair of commands, an initiator and a
terminator and provide feedback on the currently selected region while the
mouse moves.  The initiator calls this function and is typically installed on a
button-down transition. The terminator calls mouse-end-drag and is installed on
the corresponding button-up transition.
In a click-oriented interaction style, initiator and terminator commands are
both installed on the same transition, two clicks are required to define the
region and move events trigger the region marking feedback.

Novice mouse command programmers often try to use dragging for the wrong
purposes. Use mouse-set-point instead, if the region between mouse and point is
affected and needs not be selected upon command completion.
Cf. the documentation of mouse-set-point."
  (interactive)
  (mouse-set-point arg)
  (start-mouse-drag arg)
  (push-event 'motion 'point-follow-mouse-handler))

(defun mouse-extend-drag-point (arg)
  "Extends the current region by moving point. Then starts dragging point.
On a double click, swap point and mark before starting to drag. That is 
when you want to extend a region at the 'mark' side, start the extension
with a double click.
For programming hints cf. mouse-drag-point."
  (interactive)
  ;;extend may go into a non-selected window
  (when (not (eq (selected-window) (nth 2 arg)))
    ;;(error "Select window first.")
    (select-screen (nth 3 arg))
    (select-window (nth 2 arg)))
   ;; at the startup of a buffer (mark) is nil
   ;; after revert buffer it is 1 and (point) maybe anywhere.
   ;; Then an extend request may go into a non-selected window.
   (cond ((mark)
	  (if multi-click-hint 
	    ;; undo the previous interpretation, multi-clicks go to same point
	    (let ((p (cadr mouse-save-excursion-info)))
	      (goto-char p)	    
	      (exchange-point-and-mark)))
	  (save-excursion-info)
	  (goto-char (car arg))
	  (if (null drag-button) (set-mouse-marker))
	  (mark-region (mark) (point))
	  (extend-mouse-drag arg)
	  (push-event 'motion 'point-follow-mouse-handler))
	 (t (mouse-drag-point arg))))

(defun mouse-end-drag-point (arg)
  "Assumes that dragging was initiated by mouse-drag-point or mouse-extend-drag-point.
Sets point to mouse and ends dragging. 
For programming hints cf. mouse-set-point."
  (setq mouse::last-point -1)		;always do this cleanup
  (when mouse::downp
	(let (motion-handler done) 
	  (unwind-protect		; leave handler stack intact
	      (progn
		(setq mouse::downp nil)
		(mouse::copy-button drag-button)
		(goto-char (car arg)))
	    (while (not done)
	      (setq motion-handler (pop-event 'motion))
	      (when (not (eq motion-handler 'point-follow-mouse-handler))
		    (setq done t)
		    (push-event 'motion motion-handler)))))
	;; make sure feedback is right we moved point
	(mark-region (mark) (point))))

(defun mouse-warp-to-point (&optional ignore)
  "Warps mouse to the current point."
  (interactive)
  (let ((coords (epoch::query-cursor)))
    (epoch::warp-mouse (car coords) (cdr coords) ;char line screen
		       (current-screen))))
  
;;; This was hacked by Tree, 21.10.91 to work with Epoch4.0a1.
;;; Evidently we need to explicitly set the style to be used for
;;; underlining in epoch::mark-region.  This seems to work.
(defvar sky-region-style (epoch::make-style))
(epoch::set-style-foreground sky-region-style (foreground))
(epoch::set-style-background sky-region-style (background))
(epoch::set-style-underline sky-region-style (foreground))

;;; Make sure the code continues to run under Epoch version < 4 (hws)
(if sky-epoch-3 (setq sky-region-style 1))

(defun mark-region (from to)
  "Mark the current region and put the region contents into
the cut buffer."
  ;; mark only if necessary
  (when (/= from to)
    (cond (drag-button
	   (epoch::move-button drag-button from to))
	  (t
	   (setq drag-button 
		 (epoch::add-button from to 						
				    sky-region-style 
				    nil))))
    (epoch::redisplay-screen)))

(defun forget-region ()
  (epoch::delete-button drag-button)
  (setq drag-button nil))
  
(fset 'get-cut-buffer (function epoch::get-cut-buffer))

(defun store-cut-buffer (from to)
  (epoch::store-cut-buffer (buffer-substring from to)))


(defun mouse-paste-cut-buffer (arg)
  "Like mouse::paste-cut-buffer, paste the cut buffer contents to where mouse
points without moving the region. Additionally, force redisplay."
  (mouse::paste-cut-buffer arg)
  (epoch::redisplay-screen))

;;;
;;; Focus
;;;

;;; Set-focus, i.e. kbp input stays with current screen (or window in
;;; plain emacs). Relies on changes to select-screen (epoch.el) below.

(defvar current-focus-screen nil "The currently focused screen.")

(defun set-kbd-focus (screen)
  (setq current-focus-screen screen))

(defun release-kbd-focus ()
  (setq current-focus-screen nil))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Mouse feedback
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;; Note: There is fundamental difficulty in providing mouse feedback.
;;; cursors are associated on a per/screen basis while mouse commands
;;; are allocated on a per buffer basis. We only get a consistent correspondence
;;; if buffers correspond to screens. Let's make it work at least under this
;;; condition.
;;;
;;; The screen cursor association reminds the user of the mouse mode for
;;; the current screen, in other words of what the user can expect to happen
;;; when using the mouse on the current buffer.
;;; Commands may change the cursor while executing, this time telling the
;;; user what the mouse is currently doing (everyone knows the watch for `busy'
;;; or various glyphs in graphic editing). However, commands have two
;;; facets, the interactive version must provide feedback while in command
;;; execution under the program control tries to avoid unnecessary interaction
;;; and feedback.
;;;
;;; Approach: The two views can be combined by separating the the command proper
;;; from an interactive version, installing the command proper on the button up
;;; transition and the undoable interaction/selection mechanism like dragging on
;;; the down transition and/or drag events.  Provided, down events have no
;;; (updating) side-effects (e.g. just move the point) and the interactive
;;; command can be interrupted before the command proper (up event) is executed,
;;; feedback can be provided outside the command proper.  To exit the
;;; interactive command, the user can pass by the command proper by releasing
;;; the button outside the buffer receiving the button press when s/he realizes
;;; given the mouse feedback, that s/he selected the `wrong' mouse tool.
;;; Program controled command execution calls commands proper, while only one
;;; interactive mouse command provides mouse mode feedback.
;;;
;;; Conclusion: To allow for a high-level mouse definition, the command proper
;;; is to be installed in a special form that specifies how the interactive
;;; version of the command works.

;;;
;;; Auxiliaries
;;;

(defun mouse-select-cursor ()
  "Steps through the set of existing mouse cursors and tells their code.
Press space to continue. Hold down space to run through."
  (interactive)
  (let ((i 
	 (do ((i 0) done j)
	     ((or done (> i 152)) j)
	     (cursor-glyph i)
	     (if (not (y-or-n-p (format "Cursor %s. Next? " i))) 
		 (setq done t))
	     (setq j i i (+ i 2)))))
    (cursor-glyph (mouse-regular-glyph))
    i))

;;;
;;; Glyphs
;;;

;;; avoid need of the cursors.el and provide a few symbolic glyphs instead
(defvar mouse-points-glyph 68 ;'cursor-left-ptr  :point
  "* Cursor glyph indicating mouse point function.")
(defvar mouse-drags-glyph  152 ; i-beam like xterm  :drag and :extend
  "* Cursor glyph indicating drag selection.")
(defvar mouse-syntactic-cue-glyph 22	; north :thing   (44 north east)
  "* Cursor glyph indicating syntax-directed thing selection.")

(defvar mouse-temporary-tool-glyph  92 ; question mark
  "* Cursor glyph indicating temporary and changing mouse function.")
(defvar mouse-follow-hot-spot-glyph 60  ;cursor-hand2
  "* Cursor glyph hot spot / button selection.")
(defvar mouse-window-glyph 64
  "* Cursor glyph indicating window related command such as fixing focus 
or popping up other windows or menus.")

;;; save/kill/yank  cut-and-paste
(defvar mouse-sky-glyph 50 ; exchange icon (kill-ring)
  "* Cursor glyph indicating save/kill/yank (kill-ring) related operation.")
(defvar mouse-yanks-here-glyph 50 "* Cursor glyph indicating pasting.")
(defvar mouse-yanks-there-glyph 50 "* Cursor glyph indicating yanking from here.")
(defvar mouse-kills-glyph 50 "* Cursor glyph indicating killing.")

(defvar mouse-formats-glyph 96 "* Cursor glyph indicating formating.")

(defvar mouse-dead-glyph 88 "*Cursor glyph indicating mouse death.")

;;;
;;; Default glyphs and abstraction
;;;

(defmacro with-busy-feedback (&rest body)
  "Body is executed while visual `busy' feedback is provided."
  (` (progn
       (dolist (scr (screen-list))
	       (cursor-glyph 150 scr)) ;watch
       (unwind-protect
	   (progn (,@ body))
	 (let ((sscr (current-screen)))
	   (dolist (scr (screen-list))
		   (epoch::select-screen scr)
		   (cursor-glyph (mouse-regular-glyph) scr))
	   (epoch::select-screen sscr))))))


;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Basis for high-level command installation
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(defun mouse-get-button-spec-and-install-command ()
  (beep)
  (if (y-or-n-p "Install commands in buffer map (else global editor map)? ")
      (rplaca (last *mouse-installation-spec*) ':local)
    (rplaca (last *mouse-installation-spec*) ':global))
  (message "Press a mouse button to install the command.")
  (push-event 'button 'mouse-select-button-spec-handler))

(defvar mouse-command-feedback-alist nil)  ;; elements ((down . up) . glyph)

(defun mouse-set-cursor-glyph (down-command up-command &optional screen)
  (cursor-glyph 
   (or (let ((l mouse-command-feedback-alist) glyph as p d u)
	 (while (and (not glyph) l)
	   (if (equal (caar l) (cons down-command up-command))
	       (setq glyph (cdar l)))
	   (setq l (cdr l)))
	 glyph)
       (cdr (assq down-command
		  (list 
		   (cons 'mouse-set-point mouse-points-glyph)
		   (cons 'mouse-drag-point mouse-drags-glyph)
		   (cons 'mouse-extend-drag-point mouse-drags-glyph)
		   (cons 'mouse-mark-thing mouse-syntactic-cue-glyph)
		   (cons 'mouse-mark-thing-extend mouse-drags-glyph))))
       (mouse-regular-glyph)
       mouse-points-glyph)
   screen))
   
(defun mouse-regular-glyph () ; glyph of current buffer, cache it to speed up
  (when (or (not (boundp 'mouse-buffer-cursor-glyph))
	    (null mouse-buffer-cursor-glyph))
	(make-local-variable 'mouse-buffer-cursor-glyph)
	(setq mouse-buffer-cursor-glyph 
	      (or (cdr (assq 'cursor-glyph 
			     (cdr (assq major-mode epoch-mode-alist))))
		  mouse-points-glyph)))
  mouse-buffer-cursor-glyph)

(defun mouse-define-internal (down-command up-command code feedback map)
  (let ((button (button-code-button code))
	(modstate (button-code-modifier code)))
    (cond ((eq map ':global) (setq map mouse::global-map))
	  ((eq map ':local) (setq map mouse::local-map)))
    (when (not (or (functionp down-command) (symbolp down-command)))
      (error "%s is not a mouse command." down-command))
    (when (not (or (functionp down-command) (symbolp down-command)))
      (error "%s is not a mouse command." up-command))
    (define-mouse map button modstate down-command)
    (define-mouse map button (+ mouse-up modstate) up-command)
    (if (numberp feedback)
	(push (cons (cons down-command up-command) feedback) mouse-command-feedback-alist))))

(defun button-code-button (code)
  (if (stringp code)
      (cond ((string-match "Left" code) mouse-left)
	    ((string-match "Middle" code) mouse-middle)
	    ((string-match "Right" code) mouse-right)
	    (t (button-code-error code)))
    (button-code-error code)))

(defun button-code-modifier (code)
  (if (stringp code)
      (+ (if (string-match "C-" code) mouse-control 0)
	 (if (string-match "M-" code) mouse-meta 0)
	 (if (string-match "S-" code) mouse-shift 0))
    (button-code-error code)))

(defun button-code-error (code)
  (error "Hm... This seems to be a new kind of button code %s. What now? \n\
See mouse-define for details of button code string representation."
		      code))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Mouse handlers
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

;;; Similar to mouse::handler from mouse.el

(defun macro-var (i var-list) 
  (let ((var (nth i var-list)))
    (if (eq var 'ignore) (gensym) var)))

(defvar mouse-handler-down-hook nil 
  "* A hook to run after processing button down events.")
(defvar mouse-handler-up-hook nil 
  "* A hook to run after processing button up events.")
(defmacro with-button-handler-info (n-down-c-m-s-i-arg-map-hdlr 
				    handler-args &rest body)
  ;; where is destructuring-bind?
  (when (not (and (consp n-down-c-m-s-i-arg-map-hdlr)
		  (= 9 (length n-down-c-m-s-i-arg-map-hdlr))))
	(error 
	 "with-button-info macro expects a list of 9 variables as arg 1."))
  (when (not (and (consp handler-args)
		  (= 2 (length handler-args))))
	(error
	 "with-button-info macro expects a list of 2 arguments as arg 2."))
  (let ((time (gensym))
	(coords (gensym))
	(value (car handler-args))
	(scr (cadr handler-args))
	(n (macro-var 0 n-down-c-m-s-i-arg-map-hdlr))
	(edge (macro-var 1 n-down-c-m-s-i-arg-map-hdlr))
	(control-down (macro-var 2 n-down-c-m-s-i-arg-map-hdlr))
	(meta-down (macro-var 3 n-down-c-m-s-i-arg-map-hdlr))
	(shift-down (macro-var 4 n-down-c-m-s-i-arg-map-hdlr))
	(index (macro-var 5 n-down-c-m-s-i-arg-map-hdlr))
	(arg (macro-var 6 n-down-c-m-s-i-arg-map-hdlr))
	(map (macro-var 7 n-down-c-m-s-i-arg-map-hdlr))
	(handler (macro-var 8 n-down-c-m-s-i-arg-map-hdlr))
	(valuev (gensym)) (scrv (gensym))
	(modstate (gensym)))
    (` (unwind-protect
	   ;; make timestamps visible for multi-click hints (hws)
	   (let* (((, time) (if sky-epoch-3 (the-time) (aref epoch::event 3)))
		  ((, coords) (query-pointer))
		  ((, valuev) (, value))
		  ((, scrv) (, scr))
		  ((, n) (1- (nth 3 (, valuev))))
		  ((, edge) (nth 0 (, valuev)))
		  ((, modstate) (nth 4 (, valuev)))
		  (epoch::event-handler-abort nil) ;prevent lossage
		  ((, shift-down) (/= 0 (logand (, modstate) shift-mod-mask)))
		  ((, control-down) (/= 0 (logand (, modstate) 
						  control-mod-mask)))
		  ((, meta-down) (/= 0 (logand (, modstate) meta-mod-mask)))
		  ((, index) (+ (if (, edge) mouse-down mouse-up)
				(if (, shift-down) mouse-shift 0)
				(if (, control-down) mouse-control 0)
				(if (, meta-down) mouse-meta 0)
				(* mouse::button-size (, n))))
		  ;; find the handler list and try to dispatch
		  ((, arg)
		   (coords-to-point1
		    (nth 1 (, valuev)) (nth 2 (, valuev)) (, scrv)))
		  ((, map) 
		   (if (and mouse::down-buffer (not (, edge)))
		       ;; force release into press buffer, for simulated grab
		       (symbol-buffer-value 'mouse::local-map 
					    mouse::down-buffer)
		     ;; ELSE if there's an arg, use the arg buffer
		     (and (, arg) 
			  (symbol-buffer-value 'mouse::local-map 
					       (nth 1 (, arg))))))
		  ((, map) (if (vectorp (, map)) (, map)
			     mouse::global-map))
		  ((, handler) (aref (, map) (, index))))
	     (setq mouse::down-buffer (and (, edge) (, arg) (nth 1 (, arg))))
	     ;; figure multi-click-hint, only on down and if timeout /= 0
	     (if (and (> multi-click-timeout 0) (, edge))
		 (provide-multi-click-hint (, index) (, time) (, coords)))
	     (,@ body)
	     (cond ((and (, edge) mouse-handler-down-hook)
		    (funcall mouse-handler-down-hook (, arg)))
		   (mouse-handler-up-hook
		    (funcall mouse-handler-up-hook (, arg)))))
	 (check-button-handler-works)))))

;;;?follow Epoch's drag paradigm. It seems to exclude drag-and-drop which
;;;must be simulated by two clicks for now.
(defvar mouse-feedback-screen nil)

;;; try to treat mode-line and rightmost column in a more robust way.
;;; Some commands move into different modes. The standard way to indicate this
;;; is to change the cursor-glyph property of the current buffer!
;;; Note: Some commands like mouse-warp operate on multiple screens.
;;; We have to reset where we change.
(defun visual-mouse-handler (type value scr)
  (with-button-handler-info 
   (number downp control meta shift index arg map handler) (value scr)
   (cond (downp				;down commands provide gesture feedback
	  (setq mouse-feedback-screen scr)
	  (mouse-set-cursor-glyph handler (aref map (+ index mouse-up)) scr))
	 (t				;up commands must take it back.
	  (cursor-glyph (mouse-regular-glyph) mouse-feedback-screen)))
   (cond ((null arg) (beep))		;click outside of buffer window
	 (t (if (and handler (functionp handler))
		(funcall handler arg)
	      ;; if down - undefined command
	      (if downp (beep)))))))

(defun help-with-epoch-mouse-handler (type value scr)
  (with-button-handler-info 
   (btn downp controlp metap shiftp index arg map handler) (value scr)
   ;; make sure minibuffer is visible
   (epoch::raise-screen 0)
   (describe-mouse-event handler downp btn controlp metap shiftp)))

;;; Install command handler
(defun mouse-select-button-spec-handler (type value scr)
  (with-button-handler-info 
   (btn downp controlp metap shiftp index arg map handler) (value scr)
   (let ((mod  (+ (if shiftp mouse-shift 0)
		  (if controlp mouse-control 0)
		  (if metap mouse-meta 0)))
	 installer)
     (cond ((null arg) (beep))		; click not exactly on window
	   (downp 
	    (setq installer
		  (if (eq (nth 3 *mouse-installation-spec*) ':local)
		      (function local-set-mouse)
		    (function global-set-mouse)))
	    (funcall installer btn mod (nth 0 *mouse-installation-spec*))
	    (funcall installer btn (+ mouse-up mod) (nth 1
							 *mouse-installation-spec*))
	    (message "...installed %s/%s on %s/-Up"
		     (nth 0 *mouse-installation-spec*)
		     (nth 1 *mouse-installation-spec*)
		     (mouse-button-string downp btn controlp metap shiftp nil))		     
	    (setq *mouse-installation-spec* nil))
	   ((null *mouse-installation-spec*) ; done
	    (pop-event 'button))))))

;;; No surprises with GC on macro expansion, please
(defun compiled-function-p (symbol)
  (and (functionp symbol)
       (eq (car (nth 2 (symbol-function symbol))) 'byte-code)))

(when (not (compiled-function-p 'visual-mouse-handler))
      (require 'byte-compile "bytecomp")
      (fset 'visual-mouse-handler (byte-compile 'visual-mouse-handler))
      (fset 'mouse-select-button-spec-handler 
	    (byte-compile 'mouse-select-button-spec-handler))
      (fset 'help-with-epoch-mouse-handler 
	    (byte-compile 'help-with-epoch-mouse-handler)))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Motion handlers
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(defun point-follow-mouse-handler (type value scr)
  (unwind-protect
      (progn
	(and value 
	     (let ((modstate (nth 4 value)))))
	(if (null mouse-down-marker) (set-mouse-marker))
	(cond ((and (boundp 'mouse::downp) mouse::downp)
	       (mouse-sweep-update)
	       (goto-char mouse::last-point))))
    (check-button-handler-works)))

;;; like epoch mouse.el mouse::handler except we visualize mouse mode and 
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Mouse help
;;;
;;; A simple help function to explain the mouse. It combines brief-key,
;;; describe and where-is by allowing to 'play' the mouse. The mouse is small
;;; enough to permit this.

(defun help-with-mouse (&rest ignore)
  "Describes what the mouse feels like under the current key bindings. Successive
mouse actions are described as they occur. Type RET to end the command."
  (interactive)
  (push-event 'button 'help-with-epoch-mouse-handler)
  (cursor-glyph mouse-glyph)
  (unwind-protect
      (read-input "Play Mouse (type RET to exit).")
    (pop-event 'button)
    (cursor-glyph (mouse-regular-glyph))
    (check-button-handler-works)))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Mouse death and revival
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; The mouse can be 'lost' by aborting inside one of the handlers
;;; if epoch::event-handler-abort is non-nil. Allow a simple way to fix
;;; the mouse, but do it automatically only if users chooses so.
(defun mouse-reset (arg)
  "Resets the mouse handling to defaults if the mouse was 'lost' by user abort 
or error occuring inside an unprotected handler. The following actions are 
taken:
epoch::event-handler is reset to event::handler.
push mouse::handler on button handler stack.
push motion::handler on motion handler stack.
With a prefix argument the last two actions are skipped."
  (interactive "P")
  ;; just overwrite any special handler the user may have running and 
  ;; by using the system defaults. But use push-event so the user may
  ;; pop some away again if necessary.
  ;; event.el
  (setq epoch::event-handler 'event::handler)
  (when (not arg)
	(cursor-glyph (mouse-regular-glyph))
	;; Mouse.el
	(push-event 'button 'visual-mouse-handler)
	(setq epoch::mouse-events t)
	;; inhibit motion handling, except when initiated by button down.
	(push-event 'motion t)		
	(dolist (s (screen-list t)) (epoch::set-motion-hints t s))
	)
  (check-button-handler-works)
  )

(defvar mouse-fix-on-failure ':query
  "*One of ':query, nil, or t meaning if a mouse button handler error
is detected, ask whether to invoke the command mouse-reset, invoke that command
without interaction, ignore the failure, respectively.")

(defun check-button-handler-works ()
  (let ((bh  (cadr (assq 'button event::functions))))
    (when (or (null epoch::event-handler) (eq t bh) (not (functionp bh)))
	  (cursor-glyph mouse-dead-glyph)
	  (beep)
	  (if (or 
	       (and (eq mouse-fix-on-failure ':query)
		    (yes-or-no-p 
		     "The mouse button handler seems to have suffered. Invoke mouse-reset? "))
	       (eq mouse-fix-on-failure t))
	      (mouse-reset nil)))))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; WORKAROUND coords-to-point error
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(defun coords-to-point1 (x y scr)
  "Convert X Y SCREEN in character coordinates to a list of
 (POINT BUFFER WINDOW SCREEN). Works like epoch::coords-to-point 
but also for 1-line minibuffer. Depending on the Epoch version, this
will not deal properly with the offset introduced by the minibuffer prompt."
  (or (epoch::coords-to-point x y scr)
      (if (and (eq scr (minibuf-screen)) ;minibuffer
	       (= y (1- (window-height (minibuffer-window))))) ; last line
	  (let* ((mwin (minibuffer-window))
		 (mbuf (window-buffer mwin)))
	      (save-excursion		;else find point
		(set-buffer mbuf)
		(beginning-of-line)
		(setq x (+ (point) x))
		(list x mbuf mwin scr))))))

;;;****************************************************************************
;;; EPOCH: REDEFINITIONS 
;;;
;;; select-screen: respect input (keyboard/mouse) focus.
;;;****************************************************************************
;;; Support focus and fix a bug producing a trail of raise events when
;;; auto-raise-screen is t, and minbuf and scr overlap under mouse.  Add a new
;;; value to auto-raise-screen 'minibuf that always and only raises minibuf.

;;; Modifies epoch.el in the hope that other commands written in terms of
;;; select-screen respect focus, calls to epoch::select-screen wouldn't anyway.
(defvar auto-raise-screen nil		; improve documentation
  "* Automatically raise screen when mouse focuses in. Values
 t        -- raise screen and minibuffer
 nil      -- no auto raise
'minibuf  -- raise minibuffer only when focusing into any screen
otherwise -- raise the screen on which we focused in.")

;;; Can't handle auto-raise in hook, needs scr as argument
(defun select-screen (&optional scr)	; this is invoked by focus events and programs
  (interactive)
  (if (or (eq auto-raise-screen t)	;both
	  (eq auto-raise-screen 'minibuf)) ;minibuffer only if other screen selected
      (epoch::raise-screen 0))
  (if (and auto-raise-screen 
	   (not (eq (current-screen) (minibuf-screen)))
	   (not (eq auto-raise-screen 'minibuf)))
      (raise-screen scr))
  (cond ((or (eq scr (current-screen)) 
	     (not current-focus-screen)) ; maybe raise but don 't select
	 ;; either no focus, or scr has focus
	 ;; also focus-in events enter here when no focus set.
	 (epoch::select-screen scr)
	 (run-hooks '*select-screen-hook*)))
  )

;;;****************************************************************************
;;; GLOBAL MOUSE COMMAND INSTALLATION, like it or lump it
;;;****************************************************************************

;;; These are installed by Epoch by default.

(defmouse :global  "Left" 'mouse-drag-point 'mouse-end-drag-point)
(defmouse :global  "Middle" 'mouse-paste-cut-buffer 'mouse-ignore)
(defmouse :global  "Right" 'mouse-extend-drag-point 'mouse-end-drag-point)

;;; Exchange Left and Right only if we run under Openlook (olwm).
;;; Middle is the ADJUST/EXTEND key of choice there.

(cond
 ((equal (getenv "WINMGR") "olwm")
  (defmouse :global  "Middle" 'mouse-extend-drag-point 'mouse-end-drag-point)
  (defmouse :global  "Right" 'mouse-paste-cut-buffer 'mouse-ignore
    mouse-yanks-here-glyph)
  ))

;;; CONSTRAINED marks/kills : Shift
;;; Make it easy to change your mind about what is marked.
;;;  toggle between freehand drag-extend and mark-thing by shift up-down.
;;;  use same modifier chord (here shift only) for save/kill/yank.

(defmouse :global  "S-Left" 'mouse-mark-thing 'mouse-ignore)
(defmouse :global  "S-Middle" 'mouse-save-kill-yank 'mouse-ignore mouse-sky-glyph)

  ;;; right reserved for visual kill-ring in separate screen
  ;;;               "more-above" scrolls up, 
  ;;;              ( one-line-per-kill yanks or yank-pops )*
  ;;;               "more-below" scrolls down

;;; FROM-MOUSE-TO-FOCUS: Meta
;;; Make the std sequence easy: meta-down,Left,Middle*,Right,Left,meta-up


(defmouse :global  "M-Left" 'mouse-toggle-focus-screen 'mouse-ignore 
		 mouse-window-glyph)
(defmouse :global  "M-Middle" 'mouse-yank-thing-to-point 'mouse-ignore
		 mouse-yanks-there-glyph)
(defmouse :global  "M-Right" 'mouse-warp-to-point 'mouse-ignore mouse-window-glyph)

;;; MORE MICE POWER: control

(defmouse :global  "C-Left" 'mouse-set-point-force-kbd 'mouse-ignore)
(defmouse :global  "C-Middle" 'mouse-extend-drag-point 'mouse-kill-region
  mouse-kills-glyph)
;;; don't do it the other way round else up transition is interpreted in other map
(defmouse :global  "C-Right" 'mouse-set-point 'mouse-scroll-mode)

;;;
;;; EXTENDED MOUSE: control-shift (sort of control-x of the mouse)
;;;
;;; requires habit since with most keys irrelevant,
;;; use for rare things.

;;; reserved for menu stuff like buffer-menu, help-menu etc.

;;;
;;; TEMPORARY MOUSE: meta-shift
;;;

(defmouse :global  "M-S-Left" 'mouse-set-point 'mouse-execute-kbd-macro
  mouse-temporary-tool-glyph)
(defmouse :global  "M-S-Middle" 'mouse-set-point 'mouse-execute-at-point
  mouse-temporary-tool-glyph)
(defmouse :global  "M-S-Right" 'mouse-drag-point 'mouse-execute-on-region
  mouse-temporary-tool-glyph)

;;; Mode MOUSE: meta-control
;;; Why comes meta-control Mouse to mind for mark-thing and yank-thing?!
;;; Make sure it works or does not do any damage if hit accidentally.

(defmouse :global  "C-M-Left" 'mouse-mark-thing 'mouse-ignore)
(defmouse :global  "C-M-Middle" 'mouse-yank-thing-to-point 'mouse-ignore
  mouse-yanks-there-glyph)

(defmouse :global  "C-M-Right" 'mouse-drag-point 'mouse-fill-or-indent
  mouse-formats-glyph)

;;; INSANE MOUSE: meta-control-shift

(defmouse :global "C-M-S-Left" 'mouse-set-point 'help-with-mouse mouse-glyph)
(defmouse :global "C-M-S-Middle" 'mouse-set-point 'help-with-mouse-tutorial 54)
(defmouse :global "C-M-S-Right" 'mouse-message 'mouse-message)

;;;----------------------------------------------------------------------------
;;; Scrolling is in a separate module now

(autoload 'mouse-scroll-mode "scrollm" "Starts mouse scrolling."  t)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Reset mouse to get visual feedback

(mouse-reset nil)

