;;; -*- Mode: Emacs-Lisp;  -*-
;;; File: sather.el
;;; Author: Heinz Schmidt (hws@ICSI.Berkeley.EDU) 
;;; Copyright (C) International Computer Science Institute, 1990, 1991
;;; 
;;; Stephen M. Omohundro (om@ICSI.Berkeley.EDU) wrote the original version of
;;; this Emacs mode which was based on an earlier mode for Eiffel including
;;; modifications made by Bob Weiner. Few is left of this and significant
;;; expansions and improvements were added.
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: Major mode for editing Sather programs. 
;;;* 
;;;*  The following two forms, placed in a .emacs file or site-init.el,
;;;*  will cause this file to be autoloaded, and sather-mode invoked, when
;;;*  visiting .sa files:
;;;*
;;;*	(autoload 'sather-mode "sather" "Sather mode" t nil)
;;;*    (setq auto-mode-alist
;;;*            (append
;;;*              (list (cons "\\.sa$" 'sather-mode)
;;;*                    (cons "\\.sather$" 'sather-mode))
;;;*              auto-mode-alist)))
;;;*  
;;;*  
;;;*  This mode works (at least) with Emacs 18.55-18.57, and Epoch 3.2--3.4.2 -- 4.
;;;*
;;;* RELATED PACKAGES: 
;;;*     The mode builds on language-tools, a set of language-independent 
;;;*     syntax-directed Emacs commands. It integrates with sky-mouse
;;;*     sky-mouse, a set of language-independent syntax-directed mouse
;;;*     commands that are independent and can optionally be loaded.
;;;* 
;;;* KNOWN BUGS:
;;;*     M-; on lines with comment-start in string.
;;;*     Emacs find-tag seems to visit some symbols twice even if they are
;;;*       in the tag table once only. This makes M-, peculiar sometimes.
;;;*
;;;* HISTORY:
;;;* Last edited: Mar 29 19:50 1992 (hws)
;;;*  Mar 26 18:17 1992 (hws): fixed to work with blocks
;;;*  Mar 20 12:17 1992 (hws): sather-find-tag t no longer changes focus on fail.
;;;*  Feb 16 23:49 1992 (hws): add Epooch 4.0 Beta compatibility patch
;;;*  Feb 16 16:30 1992 (hws): fix file name expansion for leading ../
;;;*  Dec 30 10:57 1991 (hws): merge fix respecting user defined styles
;;;*  Dec 30 10:15 1991 (hws): avoid language tools function in TAGS searches, may
;;;*                           require syntax. Fix sather-apropos asking for comment-start.
;;;*  Nov 21 15:44 1991 (hws): don't refer to epoch::version unconditionally
;;;*  Nov 11 21:18 1991 (hws): adapt to Epoch 4
;;;*  Aug 30 21:57 1991 (hws): fix skip-definition-head to not stumble "5.;" decimal.
;;;*    Improve sather-parents to present more useful about inheritance in one view:
;;;*       - Full distinguishes the kinds, not just all features on equal footing.
;;;*       - Add user-defined distinctions (filters applied to decl/def).
;;;*       - Attr includes type and init expr.
;;;*  Aug 29 22:21 1991 (hws): add completion for qualified id's FOO::abc,
;;;*                           allow alias and multi-line lists in general.
;;;*  Aug 28 23:31 1991 (hws): speedup beautify-buffer a little and correct doc.
;;;*  Aug 27 19:22 1991 (hws): avoid reading old TAGS.compl in sather-tags
;;;*  Aug 22 20:22 1991 (hws): removed (uncommented) all short abbrevs, seem to be
;;;*                           hardly used, but I may be wrong.
;;;*  Aug 21 21:24 1991 (hws): add initialize, alias syntax
;;;*  Aug 20 22:00 1991 (hws): add sather-parents option to include only attributes;
;;;*                           make sather-apropos less verbose.
;;;*  Aug 18 17:32 1991 (hws): 0 which-class shows feature too.
;;;*  Aug 18 14:26 1991 (hws): describe-sather-tag: don't look at file name in TAGS
;;;*  Aug 18 13:34 1991 (hws): sather-parents shows direct descendents to ease browsing
;;;*                           in inheritance hierarchies.
;;;*  Aug 16 22:19 1991 (hws): fix operator regexp in sather-indent-line to no include open,
;;:*                           s-unbalanced-exp-p don't count parens in string.
;;;*  Aug 16 21:56 1991 (hws): multi-line indent and/or improved
;;;*  Aug 15 21:27 1991 (hws): fix matching syntax to recognize loop-end without until
;;;*                    affects various commands like begin/end feature, marking etc.
;;;*  Aug 15 21:13 1991 (hws): fix sather-apropos to not show shared/constant/private
;;;*  Aug 13 21:49 1991 (hws): fix tags completion regexp to recognize 
;;;*                           "private shared" etc properly
;;;*  Jul 12 10:42 1991 (hws): fix extended attributes, also allow initializers
;;;*                           for public attributes.
;;;*  Jun  3 20:50 1991 (hws): use default-directory consistently with Emacs.
;;;*  Jun  3 20:23 1991 (hws): make doc-table-classes-exclude work if ""
;;;*  Jun  3 19:09 1991 (hws): let sather-tags prompt for .sather.
;;;*  Jun  2 18:35 1991 (hws): fix feature marking when preceded by ":= 50.;"
;;;*  Jun  1 22:59 1991 (hws): Make beautify treat 'end' right.
;;;*  Jun  1 18:56 1991 (hws): Add document-tag-table-classes.
;;;*  Jun  1 18:55 1991 (hws): Fix describe-public-features to not read beyond TAGS end.
;;;*  May 31 13:45 1991 (hws): Make sather-tags understand (sather_home),
;;;*                           know installation setup if missing.
;;;*  May 30 06:35 1991 (hws): plain emacs complained about multi-style missing.
;;;*  May 28 22:51 1991 (hws): don't let it uglify `and(this) or(that)'
;;;*  May 28 22:21 1991 (hws): include arithmetic operations as identifier separators.
;;;*  May 28 10:36 1991 (hws): polish multi-style to adapt to user selected fonts.
;;;*  May 27 00:17 1991 (hws): (mark) is nil on Emacs startup, fix various places
;;;*                           that won't work after startup
;;;*  May 26 16:45 1991 (hws): add 'and','or' to multi-line continuation logic.
;;;*  May 26 12:51 1991 (hws): fix beautify bug with char literals, monadic -, and
;;;*                           fixpoint literals; let beautify hightlight, too.
;;;*  May 22 14:31 1991 (hws): fix dot-sather option parsing to not break when missing 
;;;*  May 22 13:05 1991 (hws): fix doc and prompt of document-tag-table-files
;;;*  May 22 11:15 1991 (hws): fix sather-tags-multiple-replace-from-buffer to be really
;;;*                           silent like doc says. Only optionally prompt now.
;;;*  May 21 21:17 1991 (hws): add Sather copyright note on C-c n.
;;;*  May 17 18:41 1991 (hws): sather-class no longer worked on empty files after changes.
;;;*  May 13 00:00 1991 (hws): 'und' abbrev removed, bad for German
;;;*  May 12 23:50 1991 (hws): fix in-class-head to not forward-char at end of buffer
;;;*  May 10 21:41 1991 (hws): allow to find symbols of 'distributed' class 'C'
;;;*  May 10 14:45 1991 (hws): lineno-mode didn't have the right autoload name
;;;*  May  8 20:56 1991 (hws): fix a bug in beautify, no space in between '<=' et al
;;;*  May  6 12:34 1991 (hws): fix M-0 edit-definition to not ask for class on class
;;;*  May  6 10:29 1991 (hws): feature searching completely rewritten, faster and
;;;*        simpler; mult-line indentation improved; top-level indentation
;;;*        separated from in feature indentation for speed;
;;;*        attribute distinction added to sather-parents.; 
;;;*        sather-tags now also saves completion info to file;
;;;*        some generic language functions moved to language-tools.
;;;*  May  1 10:14 1991 (hws): multi line indentation polished and
;;;*        'edit currently visible definition' added.
;;;*  Apr 29 09:09 1991 (hws): mode doc overworked
;;;*  Apr 28 14:53 1991 (hws): fix endless loop in sather-tags files last .sather option
;;;*  Apr 24 18:14 1991 (hws): fix sather-apropos to work with new TAGS format.
;;;*  Apr 24 15:53 1991 (hws): improve parent tree for recognizing repeatition
;;;*  Apr 24 11:58 1991 (hws): doc current buffer and tags-table settled.
;;;*                           various fixes to the format of describes.
;;;*  Apr 23 21:59 1991 (hws): made doc-file-summary local via mode hook;
;;;*            mode hook is renamed sather-mode-hooks!
;;;*  Apr 23 14:36 1991 (hws): sather-documentation added; on visit-tags-table
;;;*            reset completion info; treat attributes correctly.
;;;*  Apr 22 10:10 1991 (hws): replace DEL in patterns to allow file transfer
;;;*  Apr 21 18:04 1991 (hws): added sather-parent
;;;*  Apr 19 05:30 1991 (hws): fixed several bugs with completion and 
;;;*     touched up the tags version. x,y: T; are ok now.
;;;*  Apr 17 10:58 1991 (hws): fix sather-comment to not auto insert blank,
;;;*            would break explicit indentation.
;;;*  Apr 17 06:22 1991 (hws): uglify/beautify added.
;;;*  Apr 17 01:45 1991 (hws): Fix various minor bugs:
;;;*            special indentation of expressions starting with op;
;;;*            explicit indentation when comment-start in quoted string;
;;;*            'is' with explicit indentation finds right offset now;
;;;*            Add completion filter to avoid Lisp symbol inclusion;
;;;*            Switch to sather-mode when making Sather TAGS from .sather.
;;;*  Apr 16 04:27 1991 (hws): Fix bugs in class positioning (sather-which-class
;;;*                           and with tags).
;;;*  Apr 15 10:53 1991 (hws): Added sather tags: edit-definition
;;;*  Apr 14 14:45 1991 (hws): fix s-comment-indent to not indent when 
;;;*      preceding comment line ends with 'is'.
;;;*  Apr  9 13:32 1991 (hws): added explicit indentation escape for
;;;*   for multi-line calls etc. Fixed indent if 'is' appears in string.
;;;*   Various short cuts to speed up indentation if locally obvious.
;;;*  Apr  8 13:25 1991 (hws): improved doc of convert-eiffel
;;;*  Apr  7 10:48 1991 (hws): added sdb mode hook
;;;*  Jan 15 11:44 1991 (hws): completed language-tool extraction for
;;;*      doc commands and syntax-oriented commands and 
;;;*      various minor fixes.
;;;*  Jan 12 18:41 1991 (hws): doc std changes, whoami is respected across 
;;;*      multiple doc logs and can be changed by prefix arg with any of the 
;;;*      doc log commands.
;;;*  10/14/90, Nov 18 90, Jan 12 91 (hws): adapted some of my Eiffel commands.
;;;* Created: Fri Sep  7 16:34:58 1990
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  
(require 'language-tools "lang-tools")		;basic mode independent things
(require 'tags)
(provide 'sather)			;don't load twice

;;; do we have mouse support?
(defvar sather-mouse-p
  (and (memq 'sky-mouse features) 
       (or (memq 'x-mouse features) (memq 'epoch-mouse-base features))))

;;; running under Epoch?

(defvar running-epoch (and (boundp 'epoch::version) epoch::version))


;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Standard Command table and command installation
;;; This is taken only if there is no active table.

(defvar sather-mode-map nil "Keymap for Sather mode.")
(if (not sather-mode-map)			
  (let ((map (make-sparse-keymap)))
    ;; language construct (C-c letter)
    (define-key map "\C-ca" 'sather-assert)
    (define-key map "\C-cc" 'sather-class)
    (define-key map "\C-cx" 'sather-except)
    (define-key map "\C-cd" 'sather-debug)
    (define-key map "\C-ch" 'doc-header)
    (define-key map "\C-ci" 'sather-if)
    (define-key map "\C-cl" 'sather-loop)
    (define-key map "\C-cm" 'doc-modification)
    (define-key map "\C-cn" 'doc-copyright-note)
    (define-key map "\C-co" 'sather-when)
    (define-key map "\C-cr" 'sather-routine)
    (define-key map "\C-cs" 'sather-switch)
    (define-key map "\C-cu" 'sather-until)
    (define-key map "\C-cw" 'sather-while)
    (define-key map "\C-cA" 'sather-extended-attribute) ; avoid accidental hit
    (define-key map "\C-c;" 'skip-layout)
    
    (define-key map "\C-\M-a" 'beginning-of-feature)
    (define-key map "\C-\M-e" 'end-of-feature)
    (define-key map "\C-\M-f" 'forward-matching-exp)

    (define-key map "\C-\M-b" 'backward-sexp-ignore-comments)
    
    (define-key map "\C-\M-u" 'uglify-region)
    (define-key map "\C-\M-g" 'beautify-region)

    ;; language command (C-c C-letter)
    (define-key map "\C-c\C-a" 'sather-apropos)
    (define-key map "\C-c\C-w" 'sather-which-class)
    (define-key map "\C-c\C-d" 'sather-documentation)
    (define-key map "\C-c\C-p" 'sather-parents)

    (define-key map "\C-c\C-b" 'beautify-region)
    (define-key map "\C-c\C-u" 'uglify-region)

    (define-key map "\C-c\t" 'sather-info) ; C-i == TAB
    (define-key map "\C-c\C-s" 'compile)
    (define-key map "\C-c," 'next-error) ;  , == C-,

    ;; symbols  (M letter)
    (define-key map "\M-." 'edit-definitions)
    (define-key map "\M-?" 'edit-callers)    
    (define-key map "\M-\t" 'sather-complete-symbol)
    ;; indentation    
    (define-key map "\t" 'sather-indent-line)
    (define-key map "\r" 'newline-and-indent)
    (define-key map "\n" 'newline)
    (define-key map "\177" 'backward-delete-char-untabify)
    (define-key map "\M-;" 'sather-comment)
    (setq sather-mode-map map)))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Syntax (fine tricks for mouse too?)
;;; This is taken only if there is no active table.

(defvar sather-mode-syntax-table nil
  "Syntax table in use in Sather-mode buffers.")

(if (not sather-mode-syntax-table)	;users may have a table of their own
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?\\ "\\ " table) ;escape
    (modify-syntax-entry ?\' "\"" table)  ; 'c' char
    ;; whitespace
    (modify-syntax-entry ?\t "    " table)
    (modify-syntax-entry ?\n ">   " table)
    (modify-syntax-entry ?\n ">   " table)
    (modify-syntax-entry ?\f ">   " table)

    ;; A...Za...z0...9 are word by default
    ;; _: are symbol by default
    (modify-syntax-entry ?: "." table)
    (modify-syntax-entry ?$ "." table)
    ;;(modify-syntax-entry ?= "_ " table)
    ;; let := be one thing even if res:=self is a symbol now, but res := self
    ;; works with yanking. Not worth it. Tag extraction would reuire special
    ;; logic to get rid of it again.
    (modify-syntax-entry ?= "." table)

    ;; arith: separate symbols
    (modify-syntax-entry ?/ "." table)    
    (modify-syntax-entry ?* "." table)
    (modify-syntax-entry ?+ "." table)     
    (modify-syntax-entry ?- ". 12" table) ; also comment start
    (modify-syntax-entry ?% "." table)
    (modify-syntax-entry ?& "." table)
    (modify-syntax-entry ?\| "." table)
    ;; bool opns: separate symbols
    (modify-syntax-entry ?< "." table)
    (modify-syntax-entry ?> "." table)

    (setq sather-mode-syntax-table table)))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Abbreviations
;;; This is taken only if there is no active table.
;;; Abbreviations are easier to customize for the user than commands;
;;; most often this simplicity is preferable even if it may require another 
;;; pointing action to get the cursor someplace inside.
;;; M-x write-abbrev-file saves including a count how often you use which ones.
;;; Cf. Emacs abbrev documentation for more info.

(if (not (boundp 'sather-mode-abbrev-table))
    (define-abbrev-table 'sather-mode-abbrev-table 
      '(
	;; classes / unfortunately all need $ version too to expand

	;; Short abbrevs seem to be used rarely
	;; ("itg" "INT" nil 0) ("$itg" "$INT" nil 0) 
	;; ("stg" "STR" nil 0) ("$stg" "$STR" nil 0) 
	;; ("boo" "BOOL" nil 0) ("$boo" "$BOOL" nil 0)
	;; ("cha" "CHAR" nil 0) ("$cha" "$CHAR" nil 0)
	;; ("rea" "REAL" nil 0) ("$rea" "$REAL" nil 0)
	;; ("hsh" "HASH{}" nil 0) ("$hsh" "$HASH{}" nil 0)
	;; ("err" "ERR" nil 0) ("$err" "$ERR" nil 0)
	;; ("fil" "FILE" nil 0) ("$fil" "$FILE" nil 0) 
	;; ("out" "OUT" nil 0) ;this is a bad one
	;; ("oba" "OB" nil 0) ("$oba" "$OB" nil 0)
	;; ("fob" "F_OB" nil 0) ("$fob" "$F_OB" nil 0)
	;; ("res" "res" nil 0)
	;; ("slf" "self" nil 0)

	;; Bad ones, just to not forget

	;;("lst" "LIST{}" nil 0) ("$lst" "$LIST{}" nil 0) -- used a lot in old code
	;;("und" "UNDEFINE" nil 0) -- real bad for German

	("udf" "UNDEFINE" nil 0)
	("dbl" "DOUBLE" nil 0) ("$dbl" "$DOUBLE" nil 0)

	;; code may still have it.

	("arr" "ARRAY{}" nil 0) ("$arr" "$ARRAY{}" nil 0)
	("ar2" "ARRAY2{}" nil 0) ("$ar2" "$ARRAY2{}" nil 0)
	("ar3" "ARRAY3{}" nil 0) ("$ar3" "$ARRAY3{}" nil 0)
	("ar4" "ARRAY4{}" nil 0) ("$ar4" "$ARRAY4{}" nil 0)

	("sty" "SELF_TYPE" nil 0) ("$sty" "$SELF_TYPE" nil 0)	
	
	;; keywords
	("cns" "constant" nil 0)
	("pri" "private" nil 0)
	("sha" "shared" nil 0)

	("elf" "elseif" nil 0)
	
	;; all the templates while we are typing

	("whe" "" sather-when 0)
	("exc" "" sather-except 0)
	("ifa" "" sather-if 0)
	("unt" "" sather-until 0)
	("whi" "" sather-while 0)
	("ast" "" sather-assert 0)
	("deb" "" sather-debug 0)
	("loo" "" sather-loop 0)
	("rou" "" sather-routine 0)
	("cla" "" sather-class 0)
	("swi" "" sather-switch 0)	
	
	;; Class sections
	("inh" "
--**
--**** INHERITANCE 
--**

--~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
-- Representation:
-- Abstraction:
-- For family:
-- Replaceable:
--~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
" nil 0)
	("rpr" "
--**
--**** REPRESENTATION 
--**
-- - - - - - - - - - - -  Class attributes - - - - - - - - - - - -
-- - - - - - - - - - -  Instance attributes  - - - - - - - - - - -
-- - - - - - - - - - - - - Initialization  - - - - - - - - - - - -
" nil 0)
	("pbl" "
--**
--**** FOR PUBLIC  - abstract data type
--**
" nil 0)
	("fml" "
--**
--**** FOR FAMILY - called-in by subclasses
--**
" nil 0)
	("rpl" "
--**
--**** REPLACEABLE - redefined by subclasses, we call-out
--**
" nil 0)    
	;; own secrets is NOT private, although it may be a subset of private;
	;; sometimes we want to share secrets, too
	("scr" "
--**
--**** OWN SECRETS - may change or go away
--**
" nil 0)
	("imp" "-- Implementation note:
--~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
" nil 0)
    ;;; comment separator lines
	;; line of tilde's
	("lnt" 
	 "--~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
" nil 0)
	;; line of minus'
	("lnm" 
	 "---------------------------------------------------------------------------
" nil 0)
	;; line dashed 
	("lnd" 
	 "--- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
" nil 0))))



;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; The mode, top-level
;;;
(defvar sather-mode-hooks nil
  "* A list of functions to run after sather-mode.")

(defun sather-mode ()
  
"A major mode for programming in the language Sather. 
The mode includes various programming environment tools and interfaces
to the Sather compiler and debugger. For a detailed description
see the file SATHER-MODE.doc.
The Sather syntax is controlled by the syntax table and the language tools'
variables thing-boundary-alist and matching-identifiers-alist. Various Emacs
commands respect these variables and become mode dependent only through
them.  Particularly syntactic objects, called 'things' for short, are
recognized by a number of commands and by the optional mouse package (Cf.
short recap below).

Paragraphs are separated by blank lines.
Delete converts tabs to spaces as it moves back.
Tab anywhere on a line indents it according to Sather conventions.
Return indents to the expected indentation for the new line. 
Newline inserts a newline without indentation.

Comments are begun with --.  M-; inserts and indents a comment on the line,
or indents an existing comment if there is one.

Turning on Sather mode calls the value of the variable sather-mode-hooks
with no args, if that value is non-nil.

This is the current command table including your personal bindings:
\\{sather-mode-map}
Initially the following groups of commands are available. 



                           INDENTATION

  <Tab>                    indents current line
  M-;                      indents to comment column and starts comment
  <RETURN>                 indents to expected column

  C-M-SPACE indent-region  runs indent on all lines of region
  C-M-u uglify-region      removes 'unecessary' horizontal white space
  C-M-g beautify-region    fill in 'necessary' horizontal white space

  M-x comment-region       comments all lines in region
  M-x uncomment-region     uncomments all lines in region

  M-x indent-buffer        indent whole buffer
  M-x uglify-buffer        uglify whole buffer
  M-x beautify-buffer      beautify whole buffer

Use C-h v sather TAB to find out more about variables controling
indentation.  Indentation is usually chosen to reflect the nesting level.
There are two exceptions, 'explicit indentation' and 'multi-line
expressions'.
A commented line whose comment-start (--) is followed by a non-blank
character is viewed as explicitly indented by the user. The indenter will
not change it.
Subsequent lines of a multi-line expression or call can be indented,
undented or left-aligned with respect to the preceding line. A multi-line
breakpoint is an operator like one of '.,*/+-<>=' that begins or ends a
line.  Indent takes effect for the second line of such an expression or if
the line starts with an operator and the previous one did not or it started
with a different operator. A line containing an assignment operator `:=' or
starting with `begin' is considered a 'first line'.  Undent takes effect
only if the current line is final, i.e. a continuation line
starting with `end_'.

Examples:

     foo(x,y,
         z,w: INT) is --- explicit indent is not subject to indentation

     panel: PANEL :=              -- multi-line continuation
         panel.create.            -- a 'second line' is indented
	 begin_menu(\"Foo\").       -- aligned
	    item(\"a\").            -- a 'second line' is indented
	    item(\"b\").            -- aligned
	    item(\"c\").            -- aligned
	 end_menu.                -- a 'final line' is undented
         button(\"Quit\");          -- aligned

     x: INT := foo + bar
	    + 7
	    + 4 *
	    ( a + b );


At various places Emacs language tools try to understand whether or not
they are in a string. Although Emacs language tools understand
multi-line strings, we recommend the use of inline strings for speed,
simplicity and better independence of the underlying platform.

Moreover we recommend not to change tab-width and related variables
in language buffers, so you can exchange well-indented code with other
programmers without the need for reindentation. Instead of setting
these variables globally, you can always restrict their redefinition
to a single buffer or non-language mode if you really need to change
the values.


                               TEMPLATES

Templates of the major Sather constructs can be inserted with commands, or,
when the abbrev-mode is on, you just type the first two or three letters of
the keyword followed by a blank and they expand into the templates while you
are typing them.  The mode includes various abbreviations for inline
documentation and most common keywords and types. Abbreviations are only
activated if there are no user abbreviations. Use write-abbrev-file to see
and customize the default abbreviations. We recommend the use of the Emacs
abbrev facility for inline class documentation. Use abbrev-mode to toggle
the minor abbreviation mode.  Initially abbrev-mode is off.

Template     Abbrev:   Command:
 
assert       ast       C-c a
class        cla       C-c c
debug        deb       C-c d
elseif       elf       C-c e
except       exc       C-c x    -- x cept
if           ifa       C-c i
loop         loo       C-c l
routine      rou       C-c r
switch       swi       C-c s
until        unt       C-c u
when         whe       C-c o    -- or else
while        whi       C-c w


Templates can be inserted in the middle of other constructs. They take care
for the proper spacing in the context. Try successive template expansions
with/without C-e (end-of-line) in between to see how templates nest.  The
boolean user variable` terse-template-spacing' controls vertical spacing.
We recommend the default `terse' spacing in which there will be no blank
lines in routine bodies unless explicitly entered. With the non-terse
spacing every transition between nesting levels results in a blank line.


                             SOURCE BROWSING

Movement commands are 

 C-c ; == C-c C-;               skip-layout (whitespace, comment)
 C-M-a                          beginning-of-feature
 C-M-e                          end-of-feature
 C-M-f                          forward-matching-exp
 C-M-b                          backward-sexp-ignore-comments


Syntactic Things:

Source level objects, called 'things', can be selected and operated upon
when the cursor is on them or with the mouse.  The default thing syntax
associates syntactic or semantic objects to the current point of Emacs. A
set of related mouse functions can be used to click to such hot spots of
things in order to get them marked (underlined), indented, or to yank them
across Emacs windows.  The mouse becomes a syntactic queue (cf.
mouse-tutorial for more info (\\[help-with-mouse-tutorial] when the package
is loaded.)  The following correspondence is used by the language tools
package on which the Sather-mode builds:

  You point to                       The command selects the

  end of line                        line
  comment-start                      comment up to the end of line
  word char                          word
  symbol char                        symbol
  punctuation char                   to end of next symbol
  open paren char                    group
  close paren char                   group
  whitespace                         whitespace
  keyword first char                 corresponding language construct


Oline documentation and browsing:

Online documentation is based on the Emacs Tags facility. Sather tags tables
contain only the pointers to the definitions in Sather source files and are
produced using the .sather file. Beside the Emacs Tags commands, special
Sather mode Tags commands take care of the peculiarities of Sather.  One can
see the class names of symbols that one is looking at and and skips over
irrelevant entries if necessary. The most important commands in this context
are

 C-c C-a  sather-apropos          finds all symbols containing substring
 C-c C-d  sather-documentation    finds definition and extract doc
 C-c C-w  sather-which-class      show name of current class
 M-.      edit-definitions        finds first definition and edit it;
                                  use M-, to find next definition;
                                  use prefix arg to find current definition;
 M-?      edit-callers            finds first caller or user and edit it;
                                  use M-, to find next caller/userdefinition
 M-TAB    sather-complete-symbol  complete a symbol in place (twice lists)

If there is no current tags table, all related commands will find one called
TAGS in the current directory or if there is none, they will prompt for
creating one. For an explicit creation, cf.

 M-x sather-tags                create a Sather TAGS table

Additional to the usual tags-query-replace the following commands can be
used for consistent replacements where multiple string pairs are
specified in a replacement buffer:

 M-x tags-multiple-query-replace-from-buffer
                                for each of pairs do tags-query-replace
 M-x sather-tags-multiple-replace-from-buffer 
                                for each of pairs replace only if sather
                                identifier, do not ask for confirmation
                
Sather-tags simplicity and efficiency relies on a proper indentation of
these top-level features in Sather source files, it will grep all lines with
identifiers anchored to the left or following exactly three spaces(!).  To
switch to a different tags table cf. the Eamcs Tags command
visit-tags-table.  Also note that Emacs Tags tables can be concatenated. So
it is advisable to produce disjoint tables for disjoint directories and
concatenate them as needed.



                       INTERFACES TO SATHER TOOLS


Compiler and error editing:

 C-c C-s               compile (press control while typing `cs')
 C-c C-, == C-c ,      next-error, positions into file & line as listed 
                       in compiler error output.

 M-x edit-errors       restart editing compiler error messages for example 
                       after restoring previously saved ones from file.

Debugger:

 M-x sdb               starts the debugger. Cf. sdb-mode documentation
 M-x lineno-mode       minor line number mode for use with sdb.


Eiffel conversion:

 M-x convert-eiffel    context-free conversion of an Eiffel buffer to 
                       somewhat close to Sather.



                        WRITING DOCUMENTATION

Documentation commands take a timestamp and the userid to produce creation
and/or modification logs.

 C-c h                 doc-header, inserts a file header 
 C-c m                 doc-modification, inserts a change log

The documentation format is simple and can be customized and/or switched off
easily. Use <help> v doc <complete> to find variables controling
documentation.  We recommend placing very brief logs in the file header and
more verbose logs in a separate change log file via the Emacs add-log or
some other facility.

See also the predefined abbreviations (M-x list-abbrevs) that help keeping
inline documentation in a standard format."

  (interactive)
  (kill-all-local-variables)
  (use-local-map sather-mode-map)
  (setq major-mode 'sather-mode)
  (setq mode-name "Sather")
  (set-syntax-table sather-mode-syntax-table)
  (make-local-variable 'matching-identifiers-alist)
  (setq matching-identifiers-alist
	'((class . (search-forward-is-end))
	  (is . (expand-to-definition-end))
	  (constant . (search-terminating-semicolon))
	  (shared . (search-terminating-semicolon))
	  (alias . (search-terminating-semicolon))
	  (if . end)
	  (loop . end)
	  (until . (search-forward-loop-end))
	  (switch . end)
	  (debug . end) 
	  (assert . end)))
  (make-local-variable 'indent-line-function)
  (if (boundp 'sather-mode-abbrev-table)
      (setq local-abbrev-table sather-mode-abbrev-table))
  (setq indent-line-function 'sather-indent-line)
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "--+[ \t]*")
  (make-local-variable 'comment-start)
  (setq comment-start "--")
  (make-local-variable 'file-property-list)
  (setq file-property-list nil)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "^$\\|" page-delimiter))
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'require-final-newline)
  (setq require-final-newline t)
  (run-hooks 'sather-mode-hooks))

(defvar doc-file-summary-sather
  (list nil
	"* FUNCTION:
*
* CLASSES:
* 
* REQUIRED FILES:
*
* RELATED FILES:
*
")
  "* A list (PATHNAME STRING) specifying the doc-header template to use for
summarizing a file in sather-mode. If PATHNAME is non-nil then this file
will be included.  Otherwise STRING is used. If NIL, the file summary will
be omitted.")

(defvar doc-copyright-note-sather
  (list nil 
	      "
 COPYRIGHT NOTICE: This code is provided \"AS IS\" WITHOUT ANY WARRANTY
 and is subject to the terms of the SATHER LIBRARY GENERAL PUBLIC
 LICENSE contained in the file: \"sather/doc/license.txt\" of the Sather
 distribution. The license is also available from ICSI, 1947 Center
 St., Suite 600, Berkeley CA 94704, USA.
")
   "* A list (PATHNAME STRING) specifying the copyright note to include in a
sather file.  If PATHNAME is non-nil then this file will be included.
Otherwise STRING is used. If NIL, the note will be omitted completely.")

(defun sather-doc-file-summary ()
  (make-local-variable 'doc-file-summary)
  (make-local-variable 'doc-copyright-note)
  (setq doc-copyright-note doc-copyright-note-sather
	doc-file-summary doc-file-summary-sather))

(if (not (memq 'sather-doc-file-summary sather-mode-hooks))
    (push 'sather-doc-file-summary sather-mode-hooks))

(defun edit-errors ()
  "Parse the current buffer as error messages.  This makes a list of error
descriptors, compilation-error-list.  For each source-file, line-number pair in
the buffer, the source file is read in, and the text location is saved in
compilation-error-list.  The function next-error, assigned to C-c , (and C-x `),
takes the next error off the list and visits its location."
  (interactive)
  (require 'compile)
  (let ((buffer (get-buffer "*compilation*")))
    (cond (buffer 
	   (switch-to-buffer buffer)
	   (beginning-of-buffer)
	   (compilation-forget-errors)
	   (setq compilation-error-list t)
	   (setq compilation-error-message "No more errors")
	   (next-error))
	  (t (beep)
	     (message "There is no buffer named *compilation*.")))))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Syntax: thing marking and movement.
;;; Thing = syntactic constructs
;;; independent of but meant to work with thing.el / sky-mouse.el commands.
;;; Drop the cursor at an arbitrary position in the text. How much 
;;; do we know about where we are without parsing from the beginning?

(defun search-terminating-semicolon (here begin)
  "HERE is a buffer point, BEGIN a keyword starting HERE. 
Finds matching semicolon."
  (while (and (re-search-forward ";" nil t) 
	      (save-excursion (backward-char 1)	
			      ;; eol is not considered comment by EMACS
			      (in-comment-p (point)))))
  (cons here (point)))
  
(defun search-forward-is-end (here begin)
  "HERE is a buffer point, BEGIN a keyword starting HERE. point is
past the keyword BEGIN. 
Finds the next top-level occurrence of `is', matches it with the corresponding
end and returns the respective boundaries from HERE to the end found.
Point is ends up after the end."
  (search-forward-matching here begin 'is)
  (search-forward-matching (point) 'is 'end)
  (cons here (point)))
  
(defun expand-to-definition-end (here begin)
  "HERE is a buffer point, BEGIN a keyword starting HERE.
point is past BEGIN.
Find the definition beginning and end and return the boundaries found.
Point is expected to end up after the end."
  (let ((end (save-excursion (search-forward-matching here begin 'end) (point)))
	(begin (save-excursion (goto-char here) 
			       (skip-definition-head-backward) (point))))
    (goto-char end)
    (cons begin end)))

(defun search-forward-loop-end (here begin)
  "HERE is a buffer point, BEGIN a keyword starting HERE.
point is past BEGIN. Called after `until' skips to the next occurrence of 
`loop' and includes it."
  (search-forward-matching here begin 'loop)
  (search-forward-matching (point) 'loop 'end)
  (cons here (point)))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Care when using forward-matching-exp. The current binding of matching list
;;; makes 'is' expand definition. For this it also looks backward to find 
;;; the beginning! Since 'is' is not a recursive construct, better use
;;; forward-matching-exp or search-forward-matching inside bodies only.
;;; Skip-feature takes care:
  
(defun skip-feature ()
  "Position to the end of the current feature. Point is before or at the beginning."
  (interactive)
  (skip-definition-head)
  (cond ((looking-at ";") (forward-char 1))
	(t (forward-char 2)
	   (search-forward-matching (- (point) 2) 'is 'end))))
 
(defun skip-definition-head ()
  "Position to the next top-level 'is' or ';'. Stop before it."
  (interactive)
  (skip-layout)
  (while (not (or (eobp) (looking-at "\\(is[ \t\n]\\|;\\)")))
    (forward-sexp-ignore-comments) (skip-layout) 
    (if (looking-at "\\.") (forward-char 1))
    ))

(defun skip-definition-head-backward ()
  "From before a top-level 'is' position backward to the beginning."
  (interactive)
  (backward-sexp-ignore-comments)	; at least a name
  (while (not (or 
	       (bobp)
	       (looking-at "\\(class\\|private\\)[ \t\n]")
	       (cond ((save-excursion (forward-sexp) (skip-layout)
				      (looking-at ";"))
		      (forward-sexp) (skip-layout) (forward-char 1)
		      (skip-layout)
		      t)
		     ((save-excursion (forward-sexp) (skip-layout)
				   (looking-at "[.][ \t\n]*;"))
		      (search-forward ";") (skip-layout) t))
	       (if (looking-at "is[ \t\n]") ; class is
		   (progn (forward-char 2) (skip-layout) t))))
    (backward-sexp-ignore-comments)))
    
(defun skip-keywords ()
  (while (and (not (eobp))
	      (looking-at "\\(constant\\|private\\|shared\\|alias\\)[ \t\n]"))
    (forward-sexp-ignore-comments) (skip-layout)))

(defun skip-keywords-backward ()
  (let (prec)
    (while (and (not (bobp))
		(save-excursion
		  (backward-sexp-ignore-comments)
		  (setq prec (point))
		  (looking-at "\\(constant\\|private\\|shared\\|alias\\)[ \t\n]")))
      (goto-char prec))))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; If we do not know where we are anchors are top-level keywords.

(defun beginning-of-class (&optional no-error)
  "Position to the class begin preceding point. Returns t if class was 
found, otherwise an error is signalled. If the optional argument
NO-ERROR is t, the function returns nil if no class is found."
  (interactive)
  (cond ((re-search-backward "^[ \t]*class[ \t\n]" nil t)
	 (beginning-of-line) (skip-layout) t)
	(no-error nil)
	(t (error "No class preceding point."))))

(defun beginning-of-top-level (&optional limit)
  "Position to the class, constant, shared, alias or private definition preceding point."
  (interactive)
  (let ((p (point)))
    (cond ((re-search-backward 
	    "^[ \t]*\\(class\\|constant\\|private\\|shared\\|alias\\)[ \t\n]" nil t)
	   (beginning-of-line) (skip-layout)
	   (if (not (looking-at "^[ \t]*class"))
	       (skip-keywords-backward)))
	  (t (error "No preceding top-level definition.")))))

(defun true-top-level-feature-p ()
  "t if we are in front of feature, i.e. we are behind top-level 'end', 
class 'is' or ';' and there is a top-level is,constant,private,shared,alias 
in front-of us, without one of the three endings in between. (Feature 
arguments ';' are not top-level)."
  (interactive)
  (and (not (in-quoted-string-p (point)))
       (save-excursion (backward-sexp-ignore-comments)
		       (or (looking-at "end[ \t\n;]") ; we were behind an end
			   (and (looking-at "is[ \t\n]") ; perhaps class is
				(save-excursion
				  (skip-definition-head-backward)
				  (looking-at "class[ \t\n]")))
			   (progn (forward-sexp) (skip-layout)
				  (looking-at ";"))))
       (let (found begin)
	 (save-excursion
	   (skip-layout) (setq begin (point))
	   (while (not (or (eobp) found (looking-at ";")))
	     (if (looking-at "\\(is\\|constant\\|private\\|shared\\|alias\\)[ \t\n]") 
		 (setq found t)
	       (progn (forward-sexp-ignore-comments) (skip-layout)))))
	 (if found (goto-char begin)))))

(defun private-feature-p ()
  (save-excursion
    (beginning-of-feature)
    (let ((kind (collect-feature-kind))) (memq 'private kind))))


;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Fast orientation
;;;
;;; A number of tools want to understand whether the current line
;;; is a top-level line. There is some heuristics involved here since 
;;; we do want to avoid parsing, for instance during indentation or 
;;; tags generation. For a true orientation, out of the middle of code we have
;;; find a top-level keyword as an anchor and parse in forward direction.
;;; Moving the cursor further down in the source in constant increments
;;; will gradually slow down all of the sather-mode's command if the 
;;; cursor does not pass over another top-level command. This is 
;;; is undesirable.
;;; Much like the whole Emacs lisp mode relies on parens at the left
;;; margin, we can use knowledge of top-level indenation. Assume 
;;; sather-indent is either 2 or 3. Then we can grep for good canditate 
;;; lines (exclusively, or just to find anchors from where we start 
;;; reasoning).

(defvar sather-top-level-egrep-pattern "'^(class[ \t]+[A-Z]|( |  |   )([a-z]|[A-Z]))'"
  "* The egrep pattern for recognizing whether a Sather line is top-level,
i.e. one of the starting lines of a class or feature definition.")

(defvar sather-top-level-re-pattern "\\(^\\|^ \\|^  \\|^   \\)[a-zA-Z]"
  "* The re pattern for recognizing whether a Sather line is top-level,
i.e. one of the starting lines of a class or feature definition.")

(defun s-top-level-p ()
  "t if current-line is likely the beginning or end of a Sather definition."
  (save-excursion
    (beginning-of-line)
    (looking-at sather-top-level-re-pattern)))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 
(defvar fast-top-level-treshold 1500
  "* Number of characters considered acceptable to do an exact parse
to find a feature definition. If we do not find a top-level keyword (class,
private,constant,shared,alias) within that range we use some heuristics to find
a top-level definition close by.")

(defun fast-beginning-of-top-level () 
  "Position on a top-level point (before a feature or class definition) preceding
the current point. Tradeoff speed and distance. If it is too far, callers
may have to search a lot. If it takes too long to find a close one, callers may
prefer to search.  Three methods are used. The first that brings us definitely
in front of a preceding feature withing a distance of fast-top-level-treshold
\(user variable) will end the search.  At first, look for a top-level keyword at
line begin.  Next, look at properly indented top-level lines.  Finally search
backward for 'is'."
  (interactive)
  (let ((limit (- (point) fast-top-level-treshold))
	(begin (save-excursion (beginning-of-top-level) (point))))
    (cond ((< limit begin) ;;(message "Anchor within limit.")
	   )
	  ;; search for 'is' is about 4 times faster than top-level
	  ;; if almost everything is routines; is the case most of the time.
	  ((save-excursion
	     (if (and (beginning-of-preceding-routine limit) (< limit (point)))
		 (setq begin (point))))
	   ;;(message "Found anchor 'is'.")
	   )
	  ((save-excursion 
	     (if (and (beginning-of-top-level-indent limit) (< limit (point)))		  
		 (setq begin (point))))
	   ;;(message "Found anchor format.")
	   )
	  (t ;;(message "Far anchor.")
	   ))
    (goto-char begin)))

(defun beginning-of-top-level-indent (&optional limit)
  "Find a top level routine beginning backwards within the 
distance of fast-top-level-treshold looking only at
top-level indented lines. Start outside of comment."
  (let (found)
    (save-excursion
      (while (and (not (bobp))
		  (re-search-backward sather-top-level-re-pattern limit t)
		  (not (if (true-top-level-feature-p) (setq found (point)))))))
    (if found (goto-char found))))

(defun beginning-of-preceding-routine (&optional limit)
  "Find a routine beginning before point."
  (let (found)
    (while (and (not (bobp))
		(setq found (re-search-backward "[ \t]is[ \t\n]" limit t))
		(or (in-comment-p (point)) (in-quoted-string-p (point)))))
    (if found (prog1 t (skip-definition-head-backward)))))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Movement over features. 

(defun beginning-of-feature (&optional)
  "Position to the feature beginning preceding point. If there is none
position to the class begin."
  ;; find class begin.
  (interactive)
  (let ((p (point)) last def)
    (fast-beginning-of-top-level) 
    (cond ((looking-at "class") (setq def (point))
	   (skip-definition-head)
	   (forward-word 1) (skip-layout)	; skip 'is'
	   ))	  
    (while (< (point) p)
      (setq last (point))
      (skip-feature) (skip-layout))
    (goto-char (or last def))))    

(defun end-of-feature ()
  "Position to the feature end following point."
  (interactive)
  (let ((p (point)))
    (if (eobp) (error "Past last class."))
    (if (in-comment-p p) (end-of-language-line))
    (skip-layout)
    (cond ((looking-at "class[ \t\n]")
	   (skip-definition-head) (forward-word 1) (skip-feature))
	  ;; avoid parsing if possible
	  ((or (eq last-command 'end-of-feature)
	       (eq last-command 'beginning-of-feature)
	       (looking-at "\\(private\\|constant\\|shared\\|alias\\)[ \t\n]")
	       (true-top-level-feature-p))
	   (skip-feature))
	  (t (message "I am lost. Looking backward to parse forward to end.")
	     (beginning-of-feature)
	     (cond ((looking-at "class")
		    (skip-definition-head) (forward-word 1)))
	     (skip-feature)
	     (if (not (< p (point))) (skip-feature))))))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Auxiliaries to test and pick up parsed pieces

(defun in-feature-head ()		; called to detect where we are
  "Boundaries if in head, ends before 'is'. 
Care: use in syntactically correct code, not for indent."
  (interactive)
  (save-excursion
    (let* ((p (point))
	   (begin (progn (beginning-of-feature) (point)))
	   (end (progn (skip-definition-head) (point)))) 
      (if (and (<= begin p) (<= p end)) (cons begin end)))))

(defvar sather-addon-feature-kinds '(deferred ignored fails)
  "* A list of routine names that declare properties of features by convention.
Feature kinds in this list will be indicated with various documentation functions
when they appear as the first call in a routine body.")

(defun collect-feature-kind ()
  "Collect feature properties and return list of them, moving point over them.
Assumes it is looking at the first of them.
Also by convention, detect whether the first identifier in the body is
one of the user variable sather-addon-feature-kinds. By default:
\"deferred\", \"ignored\" or \"fails\"."
  (interactive)
  (let (key keys)
    (while (progn (setq key (intern (sather-symbol-after-point)))
		  (memq key '(alias private constant shared)))
      (push key keys)
      (forward-sexp-ignore-comments))
    (save-excursion (skip-definition-head)
		    (cond ((looking-at "is") 
			   (forward-char 2)
			   (skip-layout)
			   (setq key (intern (sather-symbol-after-point)))
			   (if (memq key sather-addon-feature-kinds)
			       (push key keys)))))
    keys))
     
(defun feature-type ()
  "Positioned in front of the feature head incl. keywords like 
alias/private/shared/constant find out whether routine/attribute etc."
  (interactive)
  (let ((kind (collect-feature-kind)))
    (cond ((memq 'constant kind) 'constant)
	  ((memq 'shared kind) 'shared)
	  ((memq 'alias kind) 'alias)
	  (t (skip-definition-head)
	     (if (looking-at "is") 'routine 'attribute)))))

(defun collect-feature-names ()		
  "Collect the name(s if multiple feature definitions) and move forward
over them. Assumes we are looking at the first name."
  (let (sym names)
    (push (prog1 (sather-symbol-after-point) (forward-sexp-ignore-comments))
	  names)
    (while (looking-at "[ \t]*,")
      (skip-chars-forward "[ \t,]")
      (push (prog1 (sather-symbol-after-point) (forward-sexp-ignore-comments))
	    names))
    (reverse names)))

(defun collect-feature-signature ()
  "Collect the arguments and type of the feature(s). Include `is'
or a terminating `;' to make the distinction between attribute and 
routine. Cursor is assumed to be in front of signature. Move cursor
past signature."
  (buffer-substring (point)
		    (progn (skip-definition-head)
				 (if (looking-at ";") (forward-char 1)
				   (forward-word 1))
				 (point))))

(defun in-class-head ()
  "Boundaries if we are in a class head."
  (interactive)
  (let ((p (point))
	begin end)
    (save-excursion
      (cond ((or (looking-at "^[ \t]*class[ \t\n]")
		 (progn (forward-word 1) ; allow on 'class'
			(if (not (eobp)) (forward-char 1))
			(re-search-backward "^[ \t]*class[ \t\n]" nil t)))
	     (skip-chars-forward " \t")	; what we looked/searched for.
	     (setq begin (point))
	     (forward-char 5)		; behind 'class'
	     (search-forward-matching (- (point) 5) 'class 'is)
	     (setq end (point))
	     (if (and (<= begin p) (<= p end))
		 (cons begin end)))))))

(defun s-starts-with-class-p ()
  "True if line starts with class."
  (save-excursion
    (beginning-of-line)
    (looking-at "^[ \t]*class[ \t\n]")))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Lexical syntax

(defun sather-symbol-before-point (&optional in-comment)
  "Find symbol before point. At the begin of parentheses, step out."
  (interactive)
  (save-excursion 
    (skip-chars-backward " \t(")    
    (if in-comment (backward-sexp)
      (backward-sexp-ignore-comments))
    (while (not (or (= (char-syntax (char-after (point))) ?w)
		    (bobp)))		
      (skip-chars-backward " \t(")
      (if in-comment (backward-sexp)
	(backward-sexp-ignore-comments)))
    (if (bobp)
	""
      (buffer-substring (point) (progn (forward-sexp) (point))))))
	
(defun sather-symbol-after-point ()
  "Find symbol after point. Don't step out of parentheses. But maybe step in?"
  (interactive)
  (save-excursion
    (if (= (char-syntax (char-after (point))) ?\()
	(skip-chars-forward " \t("))
    (let ((end (progn (forward-sexp) (point)))
	  (begin (progn (backward-sexp) (point))))
      (buffer-substring begin end))))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Language constructs: Templates. Macros such as reader/writers.

(defvar *class-of-interest* nil)
(defun record-interest (class symbol)
  (setq class (or class (if (and symbol (classp symbol)) symbol)))
  (if (and class (assoc (intern class) tags-class-parent-list))
      (setq *class-of-interest* class)))

(defun sather-which-class (&optional no-msg)
  "Returns the class name of the class preceding point, if any.
With prefix argument 0, also print the name of the current feature if applicable.
When called from program and the first argument NO-MSG = t, suppress display."
  (interactive "P")
  (let (name)
    (save-excursion
      (cond ((beginning-of-class t)
	     (forward-word 1) (skip-layout)
	     (setq name 
		   (buffer-substring (point) (save-excursion (forward-sexp) (point)))))))
    ;; display if not one
    (if (not (eq no-msg t))	
	(cond ((and no-msg (zerop no-msg))
	       (let ((fname (save-excursion (beginning-of-feature)
					    (buffer-substring (point) 
							      (progn (end-of-line) (point))))))
		 (message (format "Class: %s, Feature: %s" name fname))))
	      (name (message (format "Class: %s" name)))
	      (t (error "No class preceding point."))))
    name))

(defun s-open-line ()
  "On line beginning, stay, else pass to end and newline."
  (cond ((not (or (empty-line-p)
		  (= (point) (save-excursion (beginning-of-line) (point))))) ;begin
	 (end-of-line) (insert "\n"))))

(defun sather-class (&optional ARG)
  "Insert a 'class' template. If this is the first class in a file a file
header is inserted using the command doc-header. 
With prefix arg 0 the file header is suppressed.
With other prefix arg, the file header prompts for user identification."
  (interactive "P")
  ;;conditionally insert doc header
  (s-open-line)
  (if (not (sather-which-class t))
      (cond ((null ARG) (doc-header nil))
	    ((= ARG 0) t)
	    (t (doc-header t))))
  (insert-template '("Class: " . upcase) 
		   "class " 'NAME " is" 'TAB 
		   "\n" 'HOT-SPOT
		   "\nend; -- class " 'NAME 'TAB "\n\n"
		   '(fill-to fill-column doc-separator-pattern "--") ; deliberately hard-wired
		   ))

(defun sather-routine ()
  "Insert a 'routine' template."
  (interactive)
  (insert-template "Routine name: "
		   'NAME  "(" 'HOT-SPOT ") is"
		   "\n-- " 'TAB
		   "\nend; -- " 'NAME))

(defun sather-if ()
  "Insert an 'if' template."
  (interactive)
  (insert-template nil "if" 'HOT-SPOT " then\nelse " 'TAB "\nend; -- if"))

(defun sather-when ()
  "Insert an 'when' template."
  (interactive)
  (insert-template nil "when" 'HOT-SPOT " then ;"))

(defun sather-except ()
  "Insert an 'except' template."
  (interactive)
  (insert-template nil "except (" 'HOT-SPOT ") then ;"))

(defun sather-loop ()
  "Insert a 'loop' template."
  (interactive)
  (insert-template nil "until" 'HOT-SPOT " loop \nend; -- loop"))

(defun sather-until ()
  "Insert a common 'until' template."
  (interactive)
  (insert-template "Loop variable: "
		   'NAME ":INT := 0;" 'TAB 
		   "\nuntil " 'NAME " >= limit loop" 'HOT-SPOT
		   "\n" 'NAME " := " 'NAME "+1;" 'TAB 
		   "\nend; -- loop"))

(defun sather-while ()
  "Insert a common 'while' template."
  (interactive)
  (insert-template nil
		   "done:BOOL := false;" 'TAB 
		   "\nuntil done loop" 'HOT-SPOT
		   "\nend; -- loop"))

(defun sather-until ()
  "Insert a common 'until' template."
  (interactive)
  (insert-template "Loop variable: "
		   'NAME ":INT :=" 'HOT-SPOT "0; until " 'NAME " = limit loop\n"
		   'NAME " := " 'NAME "+1;" 'TAB
		   "\nend; -- loop"))

(defun sather-switch ()
  "Insert a 'switch' template."
  (interactive)
  (insert-template nil 
		   "switch" 'HOT-SPOT "\nwhen then else " 'TAB 
		   "\nend; -- switch"))

(defun sather-debug ()
  "Insert a 'debug' template."
  (interactive)
  (insert-template nil "debug" 'HOT-SPOT " end;"))

(defun sather-assert ()
  "Insert a 'assert' template."
  (interactive)
  (insert-template nil "assert" 'HOT-SPOT " end;"))

(defun sather-extended-attribute ()
  "Expects a one-line commented attribute declaration of the form:

 <access> name : type [ := <init-expr> ] ; [ -- <comment>]

where access is either the keyword 'private', 'readable', 'writable', or
'public'. 
The command generates a corresponding sequence of Sather declarations.
The init-expr is copied into an initialization routine named init_<class>
where class is the current class. If this is not present it will be created."
  (interactive)
  (let (from keyw colonpt asgmntpt semicolpt name type type-end init-expr
	     attr reader writer)
    (beginning-of-line)
    (save-restriction
      (narrow-to-region (point) (save-excursion (end-of-line) (point)))
      (skip-chars-forward " \t")
      (if (not (looking-at "--")) 
	  (error "Not an extended attribute declaration."))
      (forward-char 2) (skip-chars-forward "- \t")
      (setq from (point))
      (setq keyw (buffer-substring from (progn (forward-sexp) (point))))
      (cond ((equal keyw "readable") (setq keyw 'readable))
	    ((equal keyw "writable") (setq keyw 'writable))
	    ((equal keyw "public") (setq keyw 'public))
	    ((equal keyw "private") (setq keyw 'private))
	    (t (error "Expected keyword 'private', 'readable', 'writable' or 'public'.")))
      (skip-chars-forward " \t") (setq from (point))
      (setq colonpt (if (re-search-forward ":" nil t) (1- (point))))
      (if (not colonpt) (error "Cannot find \":\"."))
      (setq asgmntpt (if (re-search-forward ":=" nil t) (- (point) 2)))
      (setq semicolpt (if (re-search-forward ";" nil t) (1- (point))))
      (if (not semicolpt) (error "Cannot find \";\"."))
      )
    (if (not asgmntpt) (setq asgmntpt semicolpt))
    ;; get everything maintaining white space.
    (setq name (buffer-substring from colonpt) ;trailing white optional
	  type (buffer-substring colonpt 
				 ;; no trailing white after type
				 (save-excursion (goto-char colonpt)
						 (forward-sexp)
						 (point)))
	  init-expr (if (/= asgmntpt semicolpt)
			(buffer-substring asgmntpt semicolpt)) ; includes :=, trailing option
	  attr (format "priv_%s" name)
	  reader name
	  writer (format "set_%s" name))  
    (end-of-line) (sather-return)
    ;;; attribute
    (insert 
     (cond ((eq keyw 'private) (format "private %s%s;" name type))
	   ((eq keyw 'public) (format "%s%s;" name type))
	   (t (format "private %s%s;" attr type))))
    (if (not (memq keyw '(private public)))
	(progn (sather-return) (sather-def-reader attr type reader)))
    (if (eq keyw 'writable) (sather-def-writer attr type writer))
    (if init-expr 
	(sather-add-init (if (memq keyw '(private public))
			     name attr) init-expr))))

;; separate definitions, so user can use them in other commands.
(defun sather-def-reader (name type reader)
  "Inserts an attribute reader. NAME and TYPE define the attribute.
READER is the name of the reader function."
  (insert (format "%s%s is res := %s end; -- public reader" reader type name))
  (sather-return))

(defun sather-def-writer (name type writer)
  "Inserts an attribute writer. NAME and TYPE define the attribute.
WRITER is the name of the writer function."
  (insert (format "%s(x%s) is %s := x end; -- public writer" writer type name))
  (sather-return))

(defun sather-add-init (&optional name init)
  "Inserts an initialization for attribute name in the routine init_X
where X is the lower-case name of the current class. If init_X does not exist
somewhere following point, it is created as a parameterless routine."
  (interactive "P")
  (let ((class (sather-which-class t)) here)
    (if class
	(setq class (downcase class))
      (error "Cannot find current class."))
    (cond ((re-search-forward (format "^[ \t]*init_%s" class) nil t)
	   (search-forward-is-end (save-excursion (backward-sexp)
						  (point))
				  'init_)
	   (backward-sexp)
	   (skip-layout))	   
	  (t 
	   (sather-return)
	   (insert-template nil
			    (format "\ninit_%s is -- init %s attributes proper"
				    class (upcase class)) 
			    'TAB
			    "\n-- (simulates attribute initializers)"
			    'TAB				    
			    (format "\nend; -- init_%s" class))
	   (beginning-of-line 1) (sather-indent-line)))
    (insert (format "%s %s;" name init))
    (sather-indent-line)   
    (sather-return)))


;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Indentation
;;;

(defvar visible-indentation nil
  "*Indentation feedback, used by slow commands like beautify-buffer.")

(defconst sather-indent 3
  "*This variable gives the indentation in Sather-mode")

(defconst sather-comment-col 32
  "*This variable gives the desired comment column for comments to 
the right of text.")

;;; Constant indent at top level. Search/parse only if we don't see
;;; we we are.
(defun sather-indent-line ()
  "Indent the current line as Sather code."
  (interactive)
  (if visible-indentation (sit-for 0))
  (let ((pt (point))
	(col (s-current-indentation))
	goal)
    (save-excursion			; template editing relies on no
					; sideeffect on cursor with 'TAB.
      (beginning-of-line) 
      (setq goal 
	    (cond ((eobp) nil)
		  ((s-explicit-indent) nil) ; leave alone
		  ;; short cut top-level multi-line comment
		  ((and (looking-at "\\(^\n\\|--\\)")	
			(save-excursion (beginning-of-line 0)
					(looking-at "--"))) nil)
		  ((s-starts-with-class-p) 0)
		  ((comment-line-p) (s-comment-indent))
		  ;; subsequent expect nonwhite
		  ((progn (skip-chars-forward " \t") (in-class-head)) 0)
		  ((looking-at "\\(private\\|constant\\|shared\\|alias\\)[ \t\n]")
		   sather-indent)
		  ((s-multi-line-cont-p) (s-get-multi-line-indent))
		  ((s-block-cont-p)	; begins with block-cont keyword
		   (s-get-block-indent)) ; indent same as block
		  ((looking-at "[.:+-/*]") ; line starts with operator
					; but do not make open paren an operator!
					; (BAR::foo).gee should indent like BAR::foo.gee
		   (let ((ch (char-after (point))))
		     (save-excursion	; look at previous line
		       (beginning-of-line 0) (skip-chars-forward " \t")
		       (cond ((= (char-after (point)) ch) (current-indentation))
			     ((looking-at "[.:(+-/*]")
			      (+ (current-indentation) sather-indent))
			     (t (+ (current-indentation) (* 2 sather-indent)))))))
		  (t (+ sather-indent (s-get-block-indent)))) ; indent one 
	    ))
    (if (and goal (/= goal col))
	(save-excursion
	  (beginning-of-line) (delete-horizontal-space) (indent-to goal))))
  (skip-chars-forward " \t"))

(defun sather-return ()
  "Return and Sather indent the new line."
  (interactive)
  (newline)
  (sather-indent-line))

(defun s-current-indentation ()
  "Returns current line indentation."
  (save-excursion
    (beginning-of-line) (skip-chars-forward " \t") (current-indentation)))

(defun previous-non-blank-indent ()
  "Indent value of previous non-blank line."
  (save-excursion (beginning-of-line) (backward-non-blank-line)
		  (s-current-indentation)))

(defun relative-indent ()
  "Indent value one level relative to previous non-blank line."
  (+ (previous-non-blank-indent) sather-indent))

(defun relative-undent ()
  "Undent value one level relative to previous non-blank line."
  (- (previous-non-blank-indent) sather-indent))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;; A line is one of the following:
;;    blank 
;;    just a comment
;;    top-level:  it starts a defintion, i.e. it starts with 
;;                class,private,shared,constant,alias or it starts with an identifier
;;                indented at most by 3. When visiting a file written by
;;                a different user we may also want the logic to work. The other
;;                user may have written the file with a slightly different value
;;                of sather-indent, such as 1 or 2.
;;    block-cont: starts with end, elsif, else, loop (after until), when, then
;;    block-head: ends with is, or starts with if, until, loop (not after
;;                until), switch, debug, or assert
;;    multi-line-cont: previous line ends with operator [.,*/+-<>=]
;;    none of the above

(defun s-prev-until-p ()
  "True if there is an unmatched 'until' before us."
  (save-excursion
    (re-search-backward
     "\\(^\\|[ \t]\\)\\(until\\|loop\\|end\\|is\\)[ \t;\n]" nil t)
    (goto-char (match-beginning 2))
    (cond ((or (in-comment-p (point))
	       (in-quoted-string-p (point)))
	   (s-prev-until-p))	;keep looking
	  ((looking-at "until") t) ;found it
	  (t nil)))		;failed
  )

(defun sather-comment ()
  "Edit a comment on the line. If one exists, reindents it and moves to it, 
otherwise creates one. Gets rid of trailing blanks, puts one space between
comment header comment text, leaves point at front of comment. If comment is
alone on a line it reindents relative to surrounding text. If it is before
any code, it is put at the line beginning.  Uses the variable 
sather-comment-col to set goal start on lines after text."
  (interactive)
  (cond ((comment-line-p)		; just a comment on the line
	 (beginning-of-line)
	 (delete-horizontal-space)
	 (indent-to (s-comment-indent))
	 (forward-char 2) (delete-horizontal-space) (insert " "))
	((comment-on-line-p) ;comment already at end of line
	 (cond ((s-ends-with-end-p)	; end comments come immediately
		(end-of-language-line)
		(delete-horizontal-space)
		(insert " ")
		(forward-char 2)
		(cond ((looking-at "[ \t]")
		       (delete-horizontal-space)
		       (insert " "))))
	       (t
		(end-of-language-line)
		(delete-horizontal-space)
		(if (< (current-column) sather-comment-col)
		    (indent-to sather-comment-col)
		  (insert " "))
		(forward-char 2)
		(cond ((looking-at "[ \t]")
		       (delete-horizontal-space)
		       (insert " "))))))
	((empty-line-p)	;put just a comment on line
	 (beginning-of-line)
	 (delete-horizontal-space)
	 (indent-to (s-comment-indent))
	 (insert "-- "))
	((s-ends-with-end-p)	;end comments come immediately
	 (end-of-line) (delete-horizontal-space) (insert " -- "))
	(t			;put comment at end of line
	 (end-of-line)
	 (delete-horizontal-space)
	 (if (< (current-column) sather-comment-col)
	     (indent-to sather-comment-col)
	   (insert " "))
	 (insert "-- "))))

(defun s-ends-with-end-p ()
  "t if line ends with 'end' or 'end;' and an optional comment."
  (save-excursion
    (let ((p (save-excursion (beginning-of-line) (point))))
      (end-of-language-line)
      (backward-sexp)
      (and (<= p (point))
	   (looking-at "end[ \t\n;-]")))))

(defun s-block-cont-p ()
  "t if line continues the indentation of enclosing block. Begins with end,
elsif, else, loop (after until), when, except or then."
  (save-excursion
    (beginning-of-line)
    (if (looking-at "^[ \t]*loop[ \t\n]")
	(s-prev-until-p)
      (looking-at 
       "^[ \t]*\\(end\\|elsif\\|else\\|when\\|except\\|then\\)[ ;\t\n]"))))

(defun backward-non-blank-line ()
  "Moves point to previous line excluding blank lines. 
Returns t if successful, nil if not."
  (beginning-of-line)
  (re-search-backward "^[ \t]*[^ \t\n]" nil t))

(defun s-comment-indent ()
  "Return indentation for a comment line."
  (save-excursion
    (let ((in (s-get-block-indent))
	  (prev-is-blank
	   (save-excursion (and (= (forward-line -1) 0) (empty-line-p)))))
      (if (or (and prev-is-blank (= in 0)) ; move to prev line if there is one
	      (not (backward-non-blank-line))) 
	  0				;early comments start to the left
	(cond ((comment-line-p)         ;is a comment, same indentation
	       (s-current-indentation))
	      (t                          ;otherwise indent once
	       (+ sather-indent (s-current-indentation))))))))

(defun s-get-block-indent ()
  "Return the outer indentation of the current block. Returns 0 or less if 
it can't find one."
  (save-excursion
    (let ((p (save-excursion (beginning-of-line) (point))))
      (if (and (s-goto-block-head)
	       (not (and (looking-at "\\(constant\\|private\\|shared\\|alias\\)[ \t]*[-\n]")
			 (= p (save-excursion (beginning-of-line 2) (point))))))
	  (current-indentation)
	0))))

(defun s-goto-block-head ()
  "Move point to the block head that would be paired with an end at point.
Return nil if none. Looks for first unpaired is, if, until, loop (not after
until), switch, debug, or assert."
  (let ((depth 1))
    (while (and (> depth 0)
		;; Search for start of keyword
		(re-search-backward
		 "\\(^\\|[ \t]\\)\\(if\\|loop\\|until\\|switch\
\\|is\\|debug\\|assert\\|end\\|constant\\|shared\\|alias\\|private\\|class\\)[ \t;\n(]" nil t))
      (goto-char (match-beginning 2))
      (cond ((in-comment-p (point)) 
	     ;; skip it to speed up further search
	     (skip-layout-backward) (forward-char 1)) ;backward search wants one char after patt
	    ((in-quoted-string-p (point)) nil)
	    ;; and other apparent top level lines.
            ((looking-at "end")		;end of block
	     (setq depth (1+ depth))
	     ;; Heuristics to do later to further speed up:
	     ;; Emacs built-in indent-region runs TAB (indent-line) command
	     ;; forward on region lines. This will run many times back over the
	     ;; same code in Sather to find a block beginning resulting in
	     ;; slowmo indentation.  Below we have several heuristics built in.
	     ;; Here we assume that reverse finding an "end" as the first
	     ;; keyword on a line, we are in a well-indented environment.
	     ;; Justing looking straight up from the "e" of the "end" will lead
	     ;; us to the beginning (if/switch...)  or a continuation
	     ;; (else/when...) of the current block. We can assume that most often
	     ;; it is the beginning. So we can expect a speedup by going there
	     ;; testing whether it is the beginning being able to skip the whole block
	     ;; rapidly. Else we do the normal nesting level counting.
	     )
	    ((looking-at "is") (setq depth (1- depth))
	     (skip-definition-head-backward)
	     ;; on a class "is" this brings is in front of the "class" keyword.
	     ;; We do a short cut here, as there may be unbalanced "end"s that would
	     ;; otherwise force us to run back through all the file. Classes are
	     ;; not nested. Therefore we can stop here.
	     (cond ((= depth 0));; we do not have to do anything special
		   ((looking-at "class[ \t\n]") (setq depth 0))
		   ;; There is also a need to avoid that a line indentation
		   ;; of a line somewhere of the end of a long class is forced
		   ;; to search backward to the beginning of a class through
		   ;; all nesting levels of procedures, 
		   ;; to find that the indentation is 0. The previous shortcut
		   ;; was to go straigth to the beginning of a class when we
		   ;; see an "is". If blocks use "is" now, this does not work
		   ;; anymore.
		   ;;((not (or (= depth 0) (looking-at "class[ \t\n]")))
		   ;; ;; top-level
		   ;; (beginning-of-class)
		   ;; (setq depth 0))
		   ((s-top-level-p) 
		    (beginning-of-class) (setq depth 0))
		   )
	     )
	    ((looking-at "class") (setq depth 0)) ; hit the top
	    ((looking-at "\\(private\\|constant\\|shared\\|alias\\)")
	     (skip-definition-head-backward)
	     ;; cf. short cut comments above
	     (cond ((= depth 0))
		   ((looking-at "class[ \t\n]") (setq depth 0))
		   ((s-top-level-p) 
		    (beginning-of-class) (setq depth 0))))
	    ((and (looking-at "loop") (s-prev-until-p))
	     nil)			;ignore the loop, wait till the until
	    (t (setq depth (1- depth))))) ;head of block
    (if (<= depth 0) t)			;check whether we hit top of file
    ))


;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Explicit and multi-line indentation

(defun s-explicit-indent () 
  "t, if this is an explicit indentation, i.e. comment-start
followed by non-blank."
  (save-excursion 
    (end-of-language-line)
    (cond ((eolp) nil)
	  (t (forward-char 2) (not (looking-at "[ \t]"))))))

(defun previous-non-blank-line-end ()
  "Go to last char of previou non-blank line, not in comment, not blank.
nil if impossible."
  (backward-non-blank-line) (end-of-language-line) 
  (skip-chars-backward " \t" (point-min))
  (cond ((not (bobp)) (backward-char 1) t)))
    
(defun s-multi-line-cont-p ()
  "previous line ends with operator."
   (save-excursion
    (and (previous-non-blank-line-end)
	 (or (looking-at "[:.,*-+/=><(]") ; allow also "foo("       
	     (and (looking-at ";") (progn  (beginning-of-line)
					   (> (s-unbalanced-exp-p) 0)))
	     (if (= (char-syntax (char-after (point))) ?w)
		 (progn (backward-word 1)
			(looking-at "\\(and\\|or\\|not\\)")))))))
  
(defun s-get-multi-line-indent ()
  "Previous line ends with operator. (For now treat `,' like all others.)
indent,undent or align relative to previous line."
  (cond ((looking-at "end_") (relative-undent))
	((save-excursion 
	   (backward-non-blank-line) (skip-chars-forward " \t")
	   ;; beginning of multi-line expression?
	   (looking-at "begin_"))
	 (relative-indent))
	(t ;; the various alternative according to s-multi-line-cont-p
	 (let ((p (point))
	       (previous-indent (previous-non-blank-indent)))
	   (goto-char p)
	   (previous-non-blank-line-end)
	   (cond ((or 
		   ;; multi-line boolean expression  and/or
		   (and (= (char-syntax (char-after (point))) ?w)
			(progn (backward-word 1)
			       (looking-at "\\(and\\|or\\|not\\)")))
		   ;; multi-line 'when' enumeration
		   (and (looking-at ",") 
			(save-excursion (and (re-search-backward "\\(when\\|alias\\|except\\|(\\)" nil t)
					     (looking-at "\\(when\\|alias\\)"))))
		   (and (looking-at "[=<>]") (not (progn (backward-char 1) (looking-at ":=")))))
		  (if (re-search-backward 
		       "\\(:=\\|^[ \t]*\\(is\\|end\\|if\\|elsif\\|until\\|when\\|except\\|alias\\|assert\\)\\)" 
		       nil t)
		      (cond ((looking-at "^[ \t]*\\(if\\|elsif\\|until\\|when\\|except\\|assert\\|alias\\)")
			     (goto-char (1+ (match-end 1))) (current-column))
			    ((looking-at ":=") (+ (current-column) 3))
			    (t previous-indent))
		    previous-indent))
		 ((looking-at ",")	; middle arglist
		  ;;(message "args")
		  (+ (if (progn (beginning-of-line) 
				(re-search-forward ":=" p t))
			 (+ 1 (current-column))
		       (current-indentation))
		     (* sather-indent (s-unbalanced-exp-p))))
		 ((looking-at "(")	; begin open arglist
		  ;;(message "open")
		  (+ (current-indentation) sather-indent))
		 ((looking-at ":")
		  ;;(message ":::")
		  (+ (current-indentation) sather-indent sather-indent))
		 ((save-excursion (beginning-of-line) 
				  (re-search-forward ":=" p t))
		  (+ (current-column) 1))
		 (t (goto-char p) previous-indent))))))

(defun s-unbalanced-exp-p ()
  "n if this line from point to end is unbalanced n levels. n is pos or neg."
  (let (end done (level 0) (p (point)))
    (save-excursion
      (end-of-language-line)
      (setq end (point))
      (goto-char p)
      (while (not (or (= (point) end) done))
	(cond ((looking-at ":=") (setq done t))
	      ((looking-at "(") (setq level (1+ level)))
	      ((looking-at ")") (setq level (1- level)))
	      ((looking-at "\"") 
	       (forward-sexp) (backward-char 1)))	
	(forward-char 1))
      level)))


;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; TAGS lookup
;;;

(defun sather-tag-at-point (prompt default)
  "Return the Sather symbol at or left of point."
  (interactive)
  (require-sather-tags-completion)
  (let* ((p (point))
	 (m (mark))
	 (sym (save-excursion		
		(let ((ch (char-after (point))))
		  (if (or (null ch)
			  (looking-at "\n")
			  (not (memq (char-syntax ch) '(?w ?_))))
		      (backward-word 1))
		  (thing-symbol (point)))))
	 (from (car sym))
	 (to (cdr sym)))
    (if (and m			; is empty when Emacs starts up
	     (/= p m) (<= from p) (<= p to) (<= from m) (<= m to))
	(setq from (min p m) to (max p m)))
    (completing-read-tag prompt (or default (buffer-substring from to)))))

(defun completing-read-tag (prompt default &optional class-only)
  (require-sather-tags-completion)
  (let ((sym (completing-read 
	      (format "%s (Default %s): " prompt default)
	      tags-completion-obarray
	      (if class-only 'sather-class-tag-p 'sather-tag-p))))
    (if (string-equal sym "") (setq sym (or default "")))
    (if (sather-tag-p (intern sym)) sym
      (error "%s: not a known Sather tag." sym))))

(defvar sather-class-feature-separator "::")

(defun edit-definitions (&optional ARG symbol default class) 
  "Find the definitions of SYMBOL, visit the file and position to the cursor to
the first definition. DEFAULT is a string that is used to prompt for SYMBOL. The
region marked or else the symbol under point is the default to look up.  To
continue searching for next definition, use command \\[tags-loop-continue].
If CLASS is given, look only for a definition in CLASS. 

With a prefix arg (first argument when called from a program), restrict to the
possibly inherited definition in a class to be prompted for. The current class
or the last (interesting) one is used if CLASS is nil.

The facility is based on the Emacs TAGS functions (cf. sather-tags for more
details). If there is no current TAGS table, the command offers to find or
create one."
  (interactive "P")
  (if (null symbol) 
      (setq symbol (sather-tag-at-point "Edit Definitions" default)))
  (cond ((or class (and ARG (not (classp symbol))))
	 (setq class 
	       (or class (completing-read-tag "Class" 
					      (or *class-of-interest*
						  (sather-which-class t))
					      t))))
	((classp symbol) (setq class nil)))
  (record-interest class symbol)
  (condition-case err
      (progn (sather-find-tag class symbol)
	     (if class
		 (message "%s%s%s from %s" 
			  class sather-class-feature-separator symbol
			  (sather-which-class t))))
    (error (error "No symbol %s%s." (if class (format "%s::" class) "") 
		  symbol))))

(defun edit-this-sather-definition (&optional arg)
  "Like edit-definition but fills in all arguments with defaults,
so as to ease kbd macro or mouse use. The symbol pointed to is looked up.
The class is the last interesting class, i.e. the one used with
edit-definition last. If the command is used the first time,
it chooses the current class in Sather source code and, in a hierarchy
buffer, the closest class preceding point."
  (interactive)
  (require-sather-tags-completion)
  (let ((symbol (sather-symbol-after-point)))
    (edit-definitions 
     0 symbol nil
     (if (not (classp symbol))
	 (or *class-of-interest* 
	     (condition-case err	; allow click in hierarchy buffer
		 (sather-which-class t)
	       (error nil))
	     ;; find class before point
	     (and (re-search-backward "[A-Z]" nil t)
		  (sather-symbol-before-point)))))))
  
(defun s-search-forward-symbol (name) 
  (search-next-sather-symbol name t t t t))

(defun search-next-sather-symbol (name &optional def call comment string)
  "Find the next caller in current buffer. Return nil if current file does
not contain one so this can run under tags-loop-continue.
DEF CALL and COMMENT are booleans telling whether or not corresponding
occurrences of symbols are to be included."
  (let (found point-found)
    (while (and (not found)
		(re-search-forward (format "\\<%s\\>" name) nil t))
      (setq point-found (- (point) (length name)))
      (if (and (string-equal (sather-symbol-before-point t) name)
	       (cond ((in-comment-p point-found)
		      (and comment (looking-at "'")
			   (= (char-after (1- point-found)) ?`)))
		     ((in-quoted-string-p (point))
		      (and string (looking-at "'")
			   (= (char-after (1- point-found)) ?`)))
		     (t (if (in-feature-head) def call))))
	  (setq found t)))
    found))

(defun edit-callers (&optional string)
  "Find all callers or users of a feature or class symbol at point.  Visit the
file and position to the cursor to the first caller.  The command prompts for
the name.  The region marked or else the symbol under point is the default to
look up.
To continue searching for next caller, use command \\[tags-loop-continue].
The facility is based on the Emacs TAGS functions (cf. sather-tags for more
details). If there is no current TAGS table, the command offers to find or
create one."
  (interactive)
  (setq tags-loop-form (list 'search-next-sather-symbol
			     (sather-tag-at-point "Edit Callers" string)
			     nil			;def
			     t))
  (tags-loop-continue t))

(defun classp (string)
  (string-equal string (upcase string)))

(defvar *sather-foreign-class-name* "C")

(defun sather-find-tag (next &optional symbol)
  "For NEXT = nil, look up the first definition of SYMBOL.
    NEXT = t, look up the next definition of SYMBOL.
    NEXT = some class, look up the possibly inherited definition of SYMBOL in class.
    SYMBOL is optional if NEXT = t."
  (cond ((eq next t) (find-tag "" t)
	 (if (null symbol) (setq symbol last-tag)))
	((and (stringp next) 
	      (string-equal next *sather-foreign-class-name*))	; names in class C are unique
	 (sather-find-tag nil symbol))
	(next				; position to defining class
	 (sather-find-tag nil (symbol-name (defining-class 
					     (intern symbol)
					     (intern next))))
	 ;; there find symbol, must be next	      
	 (setq last-tag symbol)		; next will find this
	 (sather-find-tag t symbol))
	(t (let ((comment-start "--"))
	     (find-tag-other-window symbol))))
  ;; found ... 'cause otherwise the find-tags signal error
  (let* ((classp (classp symbol))
	 (pat 
	  (if classp
	      (format "class[ \n]*%s[ \t\n{-]" symbol)
	    (format 
	     ;; accept multidefs like x,y:REAL;
	     "\\(private[ \t]\\)?\\(\\(shared\\|alias\\|constant\\)[ \t]\\)?\\(.*,[ \t]*\\)*%s[ ,:\t(;]"
	     symbol symbol))))
    (while (progn (skip-chars-forward " \t") (not (looking-at pat))) (find-tag "" t))
    ;; found right one or error
    (cond ((looking-at pat)
	   (if (not classp) (sather-which-class)
	     (message "Found Class: %s" symbol))
	   (setq tags-loop-form '(sather-find-tag t)))
	  (t ;;find-tag should throw if it fails, so we should not end up here.
	   (error "Sather-find-tag internal error.")))
    ;; in case we run under tags-loop-continue return t
    t))

;;; Improve the tags facility by having it offer completion for
;;; the file names in the tag file.
(defun list-tags (&optional string)  
  "Display list of tags in FILE from the current tags table. 
FILE should not contain a directory spec unless it has one in the tag table.
Use space to see the possible completions."
  (interactive)
  (if (null string) 
      (setq string (choose-item 
		    (mapcar '(lambda (x) (cons x x)) (tag-table-files))
		    "List tags (in file, space to complete): " )))
  (with-output-to-temp-buffer "*Tags List*"
    (princ "Tags in file ")
    (princ string)
    (terpri)
    (save-excursion
      (visit-tags-table-buffer)
      (goto-char 1)
      (search-forward (concat "\f\n" string ","))
      (forward-line 1)
      (while (not (or (eobp) (looking-at "\f")))
	(princ (buffer-substring (point)
				 (progn (skip-chars-forward "^\177")
					(point))))
	(terpri)
	(forward-line 1)))
      ;; make is sensitive to Sather commands
      (set-buffer (get-buffer "*Tags List*"))
      (sather-mode)))

(defun sather-dictionary ()
  "List all known tags in alphabetic order. User variable fill-column 
controls line breaking."
  (interactive)
  (switch-to-buffer (get-buffer-create "*Tags Dictionary*"))
  (delete-region (point-max) (point-min))
  (insert "Sather Tags Dictionary:\n")
  (mapcar '(lambda (x) (insert (format "%s " x))
	     (if (> (current-column) (or fill-column 60))
		 (insert "\n")))
	  tags-completion-obarray)
  (beginning-of-buffer)
  (sather-mode))

(defun sather-apropos (string)
  "Display list of all Sather tags in tag table REGEXP matches.
The class of a symbol is included."
  (interactive "sSather apropos (regexp): ")
  (require-sather-tags)
  (let ((pat (concat "\\(class[ \t\n]\\|" string "\\)")) ; also stop at a class begin
	class seen)
    (with-output-to-temp-buffer "*Tags List*"
      (princ "Tags matching regexp ")
      (prin1 string)
      (terpri)
      (save-excursion
	(visit-tags-table-buffer)
	(goto-char 1)
	(while (re-search-forward pat nil t)
	  (backward-sexp)
	  (cond ((looking-at "class[ \t\n]")
		 (forward-char 6) (skip-chars-forward " \t\n")
		 ;; record it 
		 (setq class 
		       (buffer-substring (point)
					 (save-excursion
					   (re-search-forward "[ {\177]")
					   (1- (point))))))
		((looking-at "\\(private\\|constant\\|shared\\|alias\\)[ \t\n]")
		 (forward-sexp))
		((save-excursion (beginning-of-line 0) (looking-at "")) ; file name
		 (forward-sexp))
		((re-search-forward string (save-excursion (re-search-forward "[:,=; ({-\177]")
							   (point)) t)
		 ;; not just matchting "class" like in FOO_CLASS
		 (let ((name (buffer-substring (progn (backward-sexp) (point))
					       (progn (re-search-forward "[:,=; ({-\177]")
						      (1- (point))))))
		   (cond ((classp name)
			  (let ((sym (intern name))) ; avoid mention parent twice
			    (cond ((memq sym seen))
				  (t (push sym seen) (princ name) (terpri)))))
			 ;; for others assume class prefixes disambiguate
			 (t (princ (format "%s%s%s" class sather-class-feature-separator name))
			    (terpri)))))
		(t (forward-sexp)))))
      ;; make is sensitive to Sather commands
      (set-buffer (get-buffer "*Tags List*"))
      (sather-mode)
      )))

(defun tags-multiple-query-replace-from-buffer (buffer)
  "Use BUFFER, prompted for when invoked interactively, as spec for 
multiple tags-query-replace. Each line in BUFFER is a pair of strings 
to query-replace in all tag files."
  (interactive "bBuffer containing replace spec: ")
  (tags-multiple-query-replace-loop buffer 'regular-query-replace-fn))

(defun sather-tags-multiple-replace-from-buffer (buffer ARG) 
"Use BUFFER, prompted for when invoked interactively, as spec for multiple
replacements. Each line in BUFFER is a pair of strings to replace in all tag
files. All definitions, calls and references are replaced silently. Moreover
occurrences in comment and quoted strings are replaced silently provided they
appear between single-quotes like in `crt'.
With a prefix arg will prompt for continuation with every new pair."
  (interactive "bBuffer containing replace spec: \nP")
  (tags-multiple-query-replace-loop buffer 'sather-silent-replace-fn ARG))

(defun regular-query-replace-fn (from to)
  (and (save-excursion (re-search-forward from nil t))
       ;; replace all in current file, cf. replace.el 
       (not (perform-replace from to t t nil))))

(defun sather-silent-replace-fn (from to)
  (switch-to-buffer (current-buffer))
  (while (s-search-forward-symbol from)
    (delete-region (point) (- (point) (length from)))
    (insert to)))

(defun tags-multiple-query-replace-loop (buffer replace-fn &optional prompt)
  (let (from to error)
    (save-excursion 
      (set-buffer buffer)
      (beginning-of-buffer)
      (while (not (eobp))
	(setq from (read (current-buffer))
	      to (read (current-buffer)))
	(skip-chars-forward " \t\n")
	(if (and error prompt (not (yes-or-no-p 
				 (format "Continue replacing %s by %s? "
					 from to))))
	    (error "Multiple replacement aborted."))
	;; new pair
	(setq tags-loop-form (list replace-fn from to))
	(condition-case what 
	    (tags-loop-continue t)
	  (error			; what is bound  to the error
	   (setq error what)))
	(set-buffer buffer))))
  (save-some-buffers))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; TAGS COMPLETION

(defvar tags-completion-obarray nil)
(defvar tags-class-parent-list nil)
(defvar tags-class-definition-list nil)

;; make sure we get a 'position function, don't want to load cl.
(defun s-position (elt vector)
  (let (found (i 0) (len (length vector)))
    (while (and (not found) (< i len))
      (setq found (if (eq (elt vector i) elt) t)
	    i (1+ i)))
    (if found i)))
(if (not (fboundp 'position)) (fset 'position (symbol-function 's-position)))

(defun sather-tag-p (tag) (position tag tags-completion-obarray))
(defun sather-class-tag-p (tag) (and (position tag tags-completion-obarray)
				     (classp (symbol-name tag))))
  
(defun sather-complete-symbol ()
  "Perform completion on Sather symbol preceding point.
That symbol is compared against the symbols from the current Sather tags
table and any additional characters determined by what is there
are inserted."				
  ;;locals upto is? No point and click, they are close.
  (interactive)
  (require-sather-tags-completion)
  (let* ((end (point))
	 (beg (save-excursion
		(backward-sexp 1)
		(point)))
	 (pattern (buffer-substring beg end))
	 (sobarray tags-completion-obarray)
	 (predicate (function sather-tag-p))		;why is this necessary?
	 (completion (try-completion pattern sobarray predicate)))
    (cond ((eq completion t))
	  ((null completion)
	   (message "Can't find completion for \"%s\"" pattern)
	   (ding))
	  ((not (string= pattern completion))
	   (delete-region beg end)
	   (insert completion))
	  (t
	   (message "Making completion list...")
	   (let ((list (all-completions pattern sobarray predicate)))
	     (with-output-to-temp-buffer "*Tags Completion*"
	       (display-completion-list list)
	       ;; make is sensitive to Sather commands
	       (set-buffer (get-buffer "*Tags Completion*"))
	       (sather-mode)
	       ))
	   (message "Making completion list...%s" "done")))))

;;; Make sure the completion array is already computed.
;;; Save it, so we spend time once only. Also make sure
;;; we recognize when the file is recomputed.
;;; While sather-apropos looks for all matches and prints them with class qualifier,
;;; in-buffer completion of qualified name can simply complete class name and then
;;; feature name. Also due to inheritance other combinations than the ones
;;; of origin of def would be relevant in general. So we put in unqualified names
;;; only.

(defun require-sather-tags-completion ()	
  "Check whether tags completion info is available. If necessary compute it."
  (require-sather-tags)
  (let (symbols attributes sym name classdef class parents defs classp)
    (if (not tags-completion-obarray)
	(save-excursion
	  (visit-tags-table-buffer)
	  (setq tags-class-parent-list nil
		tags-class-definition-list nil)
	  (beginning-of-buffer)
	  (while (< (point) (point-max))
	    (cond ((looking-at "") (beginning-of-line 3))
		  ((looking-at "^[ \t]*class ")
		   (forward-word 1)
		   (cond (class		; save previously collected def if any
			  (push (cons class parents) tags-class-parent-list)
				(push (cons class defs) tags-class-definition-list)))
		   (setq classdef t	;; next round we know class starts
			 class nil parents nil defs nil attributes nil))
		  ;; skip keywords
		  ((looking-at "[ \t]*\\(private\\|shared\\|alias\\|constant\\)[ \t]")
		   (forward-word 1))
		  (t (skip-chars-forward " \t")
		     (let ((begin (point)) done)
		       (while (not done)
			 (re-search-forward "[:,=;({ \t\177]")
			 (backward-char 1) ; skip char found
			 ;; there may be white space
			 (skip-chars-backward " \t")
			 (cond ((not (= begin (point)))
				(setq name (buffer-substring begin (point))
				      sym (intern name)
				      classp (classp name))
				(if (not (memq sym symbols)) (push sym symbols))
				(cond (classdef (setq classdef nil class sym))
				      (classp (push (list sym) defs)
					      (push sym parents))
				      (t (push sym defs)))))
			 (skip-chars-forward " \t")
			 (cond ((looking-at "\177") (setq done t))
			       (t (forward-char 1) 
				  (skip-chars-forward " \t")
				  (setq begin (point))))))
		     (beginning-of-line 2))))
	  (cond (class
		 (push (cons class parents) tags-class-parent-list)
		 (push (cons class defs) tags-class-definition-list)))
	  (setq symbols (sort symbols (function string-lessp)))
	  (setq tags-completion-obarray  (apply 'vector symbols))))))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; SATHER TAGS CREATION (setq tags-file-name nil) (setq tags-completion-array nil)
 
;; redefine to visit-tags-table to reset our completion info
(defun visit-tags-table (file)
  "Tell tags commands to use tag table file FILE.
FILE should be the name of a file created with the `etags' program.
A directory name is ok too; it means file TAGS in that directory."
  (interactive (list (read-file-name "Visit tags table (Default TAGS): "
				     default-directory
				     (concat default-directory "TAGS")
				     t)))
  (setq file (expand-file-name file))	;already substituted
  (if (file-directory-p file) (setq file (concat file "TAGS")))
  (setq tag-table-files nil
	tags-file-name file
	tags-completion-obarray nil)
  (let ((compl-file (expand-file-name "TAGS.compl" (file-name-directory file))))
    (if (file-exists-p compl-file) (load compl-file))))	

(defun require-sather-tags ()
  "If there is no current tags table, find or create one."
  (let* ((fn (expand-file-name "TAGS" default-directory)))
    (cond (tags-file-name t)
	  ((and (file-exists-p fn)
		(y-or-n-p (format "Visit tags table %s? " fn)))
	   (visit-tags-table fn))
	  ((and (file-exists-p ".sather")
		(y-or-n-p (format "%s Sather TAGS from .sather? "
				  (if (file-exists-p "TAGS") "Overwrite "
				    "No TAGS file. Create "))))
	   (save-excursion (sather-tags ".sather")))
	  (t (error "No TAGS table. Cf. sather-tags and visit-tags-table commands.")))))

(defun sather-tags (&optional dot-sather) 
  "Produces a Sather tags table from the .sather file (by default the
.sather file at the current directory) and visits it.  The tags file
is named TAGS and saved under the current directory.  All files found
under the '(source-files)' option of the .sather file are considered,
not following '(include)' options. The system's sather file called
sys_dot_sather is either defined in the .sather file (sather_home), by
the user variable sather-home-directory or the environment variable
SATHER_HOME -- in that order. In the distribution .emacs the
sather-home-directory is typically initialized by the value of
SATHER_HOME, if this is defined by the user, or to the installation
directory.

The TAGS file can be used with all commands of the Emacs tags facility
but also there are specific sather commands, cf. the mode
documentation (\\[describe-mode]) for more information.

The TAGS file includes only top-level definitions, i.e. classes, there
features heads and their parents. The efficiency and simplicity of its
production rely on the use of egrep with properly indented Sather
source files. We egrep all lines with identifiers anchored to the left
or following upto three spaces(!).  The Sather mode's understanding of
top-level features can be modified by the user variables
sather-top-level-egrep-pattern and sather-top-level-re-pattern."
  
  (interactive)
  (let* ((comment-start "--")
	 (old-tags-buffer (get-buffer "TAGS"))
	 (old-tags-buffer1 (get-buffer "*Sather Tags*"))
	 (dir default-directory)
	 (fn (expand-file-name "TAGS" dir))
	 (cfn (expand-file-name "TAGS.compl" dir))
	 files)
    (if old-tags-buffer (kill-buffer old-tags-buffer))
    (if old-tags-buffer1 (kill-buffer old-tags-buffer1))
    ;; prompt if argument missing
    (setq dot-sather (or dot-sather 
			 (read-file-name "TAGS for file (Default: .sather): "
				     default-directory
				     (concat default-directory ".sather")
				     t)))
    (let* ((dot-sather (expand-file-name dot-sather))
	   (dot-sather-home (dot-sather-sather-home dot-sather))
	   (sather-home-directory ;; let .sather file overwrite
	    (if dot-sather-home
		(s-expand-file-name dot-sather-home
				    (file-name-directory dot-sather))
	      sather-home-directory))
	   (sys-dot-sather
	    (if dot-sather-home
		(expand-file-name "sys/sys_dot_sather" sather-home-directory))))
      (setq files 
	    (append (if dot-sather-home (dot-sather-source-files sys-dot-sather))
		    (dot-sather-source-files dot-sather)))
      (save-window-excursion
	(set-buffer (get-buffer-create "*Sather Tags*"))
	(delete-region (point-min) (point-max))
	;; tag them in packages of N, but make sure that we always got 2
	;; the shell-command seems to blow up when its string representation
	;; is lengthy
	(let ((pack 20) next-files files-text done len)
	  (setq len (length files))
	  (while (not done)
	    (cond ((<= len pack) (setq next-files (reverse files) files nil done t))
		  ;; make sure at least 2 of them, egrep doesn't give filenames
		  ;; otherwise
		  ((= len (1+ pack)) (setq next-files 
					   (list (cadr files)
						 (car files))
					   files (cddr files)))
		  (t;; where is subseq?
		   (setq next-files nil)
		   (dotimes (i pack)	; make a ten pack
			    (push (car files) next-files)
			    (setq files (cdr files)))))
	    (setq files-text (mapconcat 'identity next-files " "))
	    (message "Scanning %s files. %s files left."
		     (length next-files) (setq len (length files)))
	    (shell-command
	     (format "egrep -n %s %s" sather-top-level-egrep-pattern files-text)
	     t)
	    ))
	(trim-sather-tags dir)
	(write-file fn))
      (if (file-exists-p cfn) (delete-file cfn)) ;avoid visiting old completion info
      (visit-tags-table fn)
      (setq tags-completion-obarray nil)
      (message "Computing inheritance lookup tables ...")
      (save-excursion
	(require-sather-tags-completion))
      (message "Saving inheritance lookup tables ...")
      (save-excursion
	(set-buffer (get-buffer-create "*Sather Tags Completion*"))
	(delete-region (point-min) (point-max))
	(insert (format "(setq tags-class-parent-list\n   '%s)\n\n" 
			tags-class-parent-list))
        (insert (format "(setq tags-class-definition-list\n    '%s)\n\n" 
			tags-class-definition-list))
	(insert (format "(setq tags-completion-obarray\n  (apply 'vector '%s))\n"
			(mapcar 'identity tags-completion-obarray)))
	(write-file cfn))
      )))

(defun search-dot-sather-option (option-string)
  "Find .sather option like `(source_files)'. Points ends up behind."
  (beginning-of-buffer)
  (sather-mode)
  (let (done)
    (while (not (or done (not (search-forward option-string nil t))))
      (if (not (in-comment-p (1- (point)))) (setq done t)))
    done))

;;; .sather (sather_home) precedes sahter-home-directory precedes SATHER_HOME var.
;;; Note that sather/etc/.emacs or user's .emacs for that matter can use
;;; therefore (defvar sahter-home-directory (or (getenv "SATHER_HOME") "somepath"))
;;; to shadow the installation path.
(defun dot-sather-sather-home (file)
  (let (dir)
    (save-excursion 
      (find-file file)
      (cond ((search-dot-sather-option "(sather_home)")
	     (skip-layout) 
	     (buffer-substring (point)
			       (progn (skip-chars-forward "^ \t\n")
				      (point))))
	    (t ; otherwise understand the setup of the distribution
	     (or (and (boundp 'sather-home-directory)
		      sather-home-directory)
		 (getenv "SATHER_HOME")))
	     nil))))

;;TEST: (dot-sather-sather-home "/usr/local/src/sather/etc/test/.sather")
;; (dot-sather-sather-home "~bilmes/sather/debugger/.sather")
;; (dot-sather-sather-home "~/sather/ui/.sather")


(defun dot-sather-source-files (file)
  (let (files done (directory (file-name-directory file)))
    (save-excursion
      (find-file file)
      (or 
       (not (search-dot-sather-option "(source_files)"))
       (while (not done)
	 (skip-layout)
	 ;; in front of a pathname or end?
	 (cond ((or (looking-at "(") (eobp));; other option or end
		(setq done t))
	       (t 
		(push (s-expand-file-name	; allow ~, ., .., $ in pathnames
		       (buffer-substring (point)
					 (progn (skip-chars-forward "^ \t\n")
						(point)))
		       directory)
		      files)))))
      files)))

;;; The expansion of $SATHER_HOME by cs may work differently.
;;; This variable is treated in a special way. Perhaps the user can
;;; simply set-variable sather-home-directory to achieve this effect.
;;; Note that expand-file-name does not treat ../ properly when leading in filename.
(defun s-expand-file-name (fname &optional dir)
  (if (equal (substring fname 0 2) "..")
      (expand-file-name (concat dir (substitute-in-file-name fname)))
    (expand-file-name (substitute-in-file-name fname) dir)))

;;; TEST: (s-expand-file-name "$SATHER_HOME/foo/bar.sa")
;;; TEST: (s-expand-file-name "$ZIPPY_HOME/foo/bar.sa")
;;; TEST: (substitute-in-file-name "$ZIPPY_HOME/foo/bar.sa")

;;; TEST: (dot-sather-source-files "/usr/local/src/sather/etc/test/.sather")
;;; TEST: (dot-sather-source-files "~bilmes/sather/debugger/.sather")

(defun trim-sather-tags (pwd)  
  "Produces an Emacs TAGS table from the output of egrep.
Currently we assume that egrep was run on at least two files, i.e. file
names must be present."
  (interactive)
  (let (curr-file curr-file-buf
		  (curr-buf (current-buffer))
		  file fbeg nobeg noend lno)
    (beginning-of-buffer)
    (setq curr-file
	  (buffer-substring (point) (1- (save-excursion (search-forward ":") (point)))))
    (insert (format "\n%s,\n" curr-file))
    (save-excursion (find-file (concat pwd "/" curr-file))
		    (setq curr-file-buf (current-buffer)))
    (message "Computing indexes into %s" curr-file)
    (while (and (setq fbeg (point))
		(setq nobeg (and (search-forward ":" nil t) (point)))
		(setq noend (and (search-forward ":" nil t) (point))))
      (cond ((or (looking-at " No such file") ; yup, can happen
		 (looking-at " Not a directory") 
		 (looking-at		; widow keywords and end's
		  "\\(   \\|  \\| \\|\\)\\(end\\|shared\\|alias\\|constant\\|private\\)[ \t]*[;\n-]")
		 (looking-at		; widow keywords and end's
		  "\\(   \\|  \\| \\|\\)except[ \t]*"))
	     (delete-region fbeg
			    (save-excursion (end-of-line) 
					    (if (looking-at "\n")
						(forward-char 1))
					    (point))))
	    (t 
	     (setq file (buffer-substring fbeg (1- nobeg))
		   lno (buffer-substring nobeg (1- noend)))
	     (delete-region fbeg noend)
	     (cond ((not (string-equal curr-file file))
		    (setq curr-file file)
		    (message "Computing indexes into %s" file)
		    (save-excursion (find-file (concat pwd "/" curr-file))
				    (setq curr-file-buf (current-buffer)))
		    (let ((p (point)))
		      (save-excursion (search-backward "") ;is there
				      (end-of-line 2)
				      (insert (format "%d" (1- (- p (point)))))))
		    (insert (format "\n%s,\n" curr-file))))
	     (trim-stags-line curr-file-buf lno))))
    (let ((p (point)))
      (save-excursion (search-backward "") ;is there
		      (end-of-line 2)
		      (insert (format "%d" (1- (- p (point)))))))
    ))

(defun trim-stags-line (&optional curr-file-buf lno)
  (interactive)
  ;; We are at the beginning of a line. Some constructs are lists that
  ;; continue with "defining" identifiers in subsequent lines.
  ;; egrep just collects the start. alias and attribute declarations for instance
  ;; are separated by comma. If the language-line ends with a comma we note it.
  (switch-to-buffer (current-buffer))
  (let* (bol eol multi-def-cont-p next-line 
	     (next t)
	     (file-lno (car (read-from-string lno))) ; source line no
	     (last-file-lno file-lno)	; initially the same
	     (file-chno (save-excursion (set-buffer curr-file-buf) ; source point
					(goto-line file-lno) (point))))
    ;; at least one, may many defs in multi-line list
    (while next
      ;; assumes we are at beginning of line (filestuff stripped off)
      ;; so line should look like source line.
      (setq eol (save-excursion (end-of-language-line) (point)) bol (point))
      ;; don't go into feature body; also avoid nesting levels
      ;; at least parm types must be listed vertically.
      (save-excursion
	(if (re-search-forward "\\([({:]\\|[ \t]is[ \t]\\)" eol t)
	    (setq eol (min eol (point)))))
      ;; does this look like a line that has a continuation?
      (setq multi-def-cont-p 
	    (and (not (re-search-forward "(" eol t));; unless in routine signature
		 (progn (goto-char (1- eol)) (looking-at ","))))
      ;; go to from where we cut-to-end-of-line
      (goto-char eol)
      (if (cond ((re-search-backward "(" bol t) ;; catch function 
		 (goto-char bol) (re-search-forward "(" bol t) (backward-char 1))
		((re-search-backward ":.*:=" bol t)) ; attribute def with init expr
		;; last attr with type or alias
		((re-search-backward "[:=]" bol t))
		;; or last elem in list, try only after(!) looking for = (foo = bar, ba = bu,)
		((re-search-backward "," bol t)))
	  ;; in backward search skip
	  (skip-layout-backward)
	(progn ;; give up multi defs in one line and find first
	  (goto-char bol)
	  (if (re-search-forward "[ \t]*\\([=,:;({]\\|[ \t]+is[ \t\n]\\)" (1+ eol) t)
	      (goto-char (match-beginning 0))
	    ;; what now? take whole line
	    (goto-char (1- eol)))))
      (forward-char 1);; delete to real eol (incl. lament, bol does not)
      (delete-region (point) (save-excursion (end-of-line) (point)))
      ;; get source char number 
      (save-excursion (set-buffer curr-file-buf)
		      (goto-line file-lno)
		      (setq file-chno (point)))
      (end-of-line) 
      (insert (format "\177%s,%d" file-lno file-chno))
      (forward-char 1);; beginning of next line
      (setq next multi-def-cont-p);; continue if there is a subsequent def line
      (cond (multi-def-cont-p;; keep treating successor line the same way
	     ;; insert line and find whether it has a successor too
	     (save-excursion 
	       (set-buffer curr-file-buf)
	       ;; at beginning of the line we did
	       (beginning-of-line 2)
	       (skip-layout);; don't include lament (layout and comment)
	       (beginning-of-line 1)
	       (setq file-chno (point)
		     next-line (buffer-substring file-chno
						 (save-excursion
						   (beginning-of-line 2)
						   (point)))
		     last-file-lno file-lno)
	       (goto-line file-lno);; we were at goal (file-chno) but must account for lament
	       (while (< (point) file-chno)
		 (beginning-of-line 2) (setq file-lno (1+ file-lno))))
	     (insert next-line)
	     (if next (beginning-of-line 0)))))))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Documentation

(defvar parents-done)
(defvar features-done)
(defvar parents-lines)

;;; Some of the doc stuff presents Sather in multi-style, if available
;;; Adapt to Epoch 4 style handling
(setq sky-Epoch-3-compat (and running-epoch
			      (> (length epoch::version) 6) ;;#+Epoch-3
			      (string-equal (substring epoch::version 0 7) 
					    "Epoch 3")))

(setq sky-Epoch-4-0-beta-ff-compat (and running-epoch
					(> (length epoch::version) 14) ;;#+Epoch-4.0-beta
					(not (string-equal (substring epoch::version 0 7)
							   "Epoch 3"))
					(not (string-equal (substring epoch::version 0 15) 
							   "Epoch 4.0 Alpha"))))
(cond (sky-Epoch-4-0-beta-ff-compat ;;##+Epoch-4.0-beta
       ;; the terminology changed from button to zones
       ;; no chance to try, hope it works (hws)
       (defun add-button (&rest args) (apply (function add-zone args)))
       (defun delete-button (&rest args) (apply (function delete-zone args)))
       (defun button-start (&rest args) (apply (function zone-start args)))
       (defun button-end (&rest args) (apply (function zone-end args)))
       (defun button-list (&rest args) (apply (function zone-list args)))
       (defun clear-buttons (&rest args) (apply (function clear-zones args)))))

(cond (sky-Epoch-3-compat
       
       (defun sather-style (font &optional attr)
	 "Make a font-style attribute for font or change it to font."
	 (let* ((attr (or attr (reserve-attribute)))
		(style (make-style)))
	   (set-style-foreground style (foreground))
	   (set-style-background style (background))
	   (set-style-font style font)
	   (set-attribute-style attr style)
	   attr)))
      
      (t
       (defun sather-style (font &optional style)
	 "Make a font-style attribute for font or change it to font."
	 (cond (style (set-style-font style font))
	       (t (setq style (make-style))
		  (set-style-foreground style (foreground))
		  (set-style-background style (background))
		  (set-style-font style font)
		  ))
	 style)))

(defvar sather-multi-style-support
  (and (boundp 'running-epoch) running-epoch)
  "* Documentation makes use of multi-style optionally (under Epoch for instance).
Set to nil, if this features costs unacceptable time during browsing.")

;; (font 5) ;any int makes Epoch segfault (3.2)
(defun s-set-font (str &optional scr)
  (if (stringp str) 
      (font str scr)
    (font nil scr)))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 
(defvar sather-font-list nil  
  "* Sather fonts for beautifying text.  Sather highlighting styles are
defined as the values of the user variables S-emphasize-style,
S-quote-style, and, S-comment-style which are used to markup sources if
styles are supported (effective only under Epoch).  You may define these
styles in advance in the .emacs file.  If no styles are defined, font styles
are created from Sather-font-list consisting of four elements:

    (normal-font emphasize-font quote-font comment-font).

For instance:

Width Note (Normal Emphasize Quote Comment)

6 Cursive  (\"6x13\" \"6x13bold\" \"*clean-medium-r*-12-*\" \"*clean-medium-i*-12-*\")
6 Cursive  (\"6x13\" \"6x13bold\" \"*courier-bold-o*-10-*\" \"*courier-medium-o*-10-*\")
6 Straight (\"6x13\" \"6x13bold\" \"*courier-bold-r*-10-*\" \"*courier-medium-r*-10-*\")

7 Straight (\"7x13\" \"7x13bold\" 
            \"*lucidatypewriter-bold*-12-*\" \"*lucidatypewriter-medium*-12-*\")
7 Cursive  (\"7x13\" \"7x13bold\" \"*courier-bold-o*-12-*\" \"*courier-medium-o*-12-*\")

Cf. user variable sather-multi-style-support and command sather-select-styles.")


(cond 
 (sather-multi-style-support

  ;; consume resources once only
  ;; Once the dust around Epoch 4 has settled and fewer sites use Epoch 3
  ;; find all *-attr and delete them.
  (defvar S-emphasize-attr (if sky-Epoch-3-compat (reserve-attribute)))
  (defvar S-quote-attr (if sky-Epoch-3-compat (reserve-attribute)))
  (defvar S-comment-attr (if sky-Epoch-3-compat (reserve-attribute)))
       
  (defvar S-normal-font nil)
  (defvar S-emphasize-style nil)
  (defvar S-quote-style nil)
  (defvar S-comment-style nil)))

(defun s-courier-font (width height)
  (let ((name "*courier-%s-%s*-%s-*"))
    (list width (format name "medium" "r" height)
	  (format name "bold" "r" height)
	  (format name "bold" "o" height)
	  (format name "medium" "o" height))))

(defvar size-fonts-alist 
  '((6 "6x13" "6x13bold" "*clean-medium-r*-12-*" "*clean-medium-i*-12-*")
    (7 "7x13" "7x13bold" "*lucidatypewriter-medium*-12-*" "*courier-medium-o*-12-*")
    (8 "8x13" "8x13bold" "*lucidatypewriter-medium*-12-*" "*courier-medium-o*-12-*")))

;; Respect user supplied font specs if there are ones for 9,10,11

(if (not (assq '9 size-fonts-alist)) (push (s-courier-font 9 14) size-fonts-alist))
(if (not (assq '10 size-fonts-alist)) (push (s-courier-font 10 17) size-fonts-alist))
(if (not (assq '11 size-fonts-alist)) (push (s-courier-font 11 18) size-fonts-alist))

;(push (s-courier-font 9 14) size-fonts-alist)
;(push (s-courier-font 10 17) size-fonts-alist)
;(push (s-courier-font 11 18) size-fonts-alist)

(defvar startup t)

(defun sather-select-styles (&optional width)
  "Select font styles for markup based on the size of the current font.
With \\[universal-arg] choose different width fonts. 0 starts with width 6."
  (interactive "P")
  (if (not sather-multi-style-support) (error "Multi-style support not installed."))
  (if width (setq width (+ width 6)))
  (setq width (or width (cadr (font))) ; size of current font
	width (max 6 width)
	width (min 11 width))
  (let ((spec (cdr (assoc width size-fonts-alist))))
    (cond (startup (setq startup nil))
	  (t (message "%s" spec)))
    (setq S-normal-font (car spec)
	  S-emphasize-style (sather-style (cadr spec) S-emphasize-attr)
	  S-quote-style (sather-style (caddr spec) S-quote-attr)
	  S-comment-style (sather-style (cadddr spec) S-comment-attr)))
  (s-set-font S-normal-font))

;; Don't overwrite .emacs style setting
(cond ((and sather-multi-style-support (not S-comment-style))
       (sather-select-styles)))		; respect initial font size
;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
 
(defun s-emphasize (string)
  (if sather-multi-style-support
      (format "`B`%s'B'" string)
    string))

(defun s-emphasize-quote (string)
  (if sather-multi-style-support
      (format "`I`%s'I'" string)
    string))

(defun touchup-documentation (buffer)
  (save-excursion (set-buffer buffer) (sather-mode)) ; make mouse sensitive
  (if sather-multi-style-support
      (save-excursion 
	(let (style)
	  (set-buffer buffer)
	  ;; Italics: for everything between inclusive `' and []
	  (beginning-of-buffer)
	  (while (re-search-forward "[`[]" nil t)
	    (let ((begin (1- (point)))
		  (end (progn (re-search-forward "[]']" nil t) (point))))
	      (if end (add-button begin end S-quote-style))))
	  ;; escape: `B` ... 'B'
	  (beginning-of-buffer)    
	  (while (re-search-forward "`[IB]`" nil t)
	    (delete-region (- (point) 3) (point))
	    (let ((begin (point))
		  (end (progn (re-search-forward "'[IB]'" nil t)
			      (setq style
				    (if (= (char-after (- (point) 2)) ?B)
					S-emphasize-style S-quote-style))
			      (delete-region (- (point) 3) (point))
			      (point))))
	      (if end (add-button begin end style))))))))

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

(defun sather-parents (&optional ARG class default)
  "Looks up the parents and ancestors (parents' parents) of CLASS, from the
current Sather tags table (cf. \\[sather-tags]).  The command prompts for the
CLASS using DEFAULT, if the class is nil.  The region marked or else the symbol
under point is the default to look up.

With a prefix arg (first argument if called from program) of
  0 also include inherited features,
  1 distinguish attributes and the kinds in general (slow),
  2 only include attributes (slow)."
  (interactive "P")
  (require-sather-tags-completion)
  (if (null class) 
      (setq class (sather-tag-at-point "Parents of" default)))
  (let* ((parent-buffer (concat "*" class " hierarchy*"))
	 (parents-lines 1)
	 parents-done 
	 features-done
	 (feature-display (cond ((equal ARG 0) 'features)
			       ((equal ARG 1) 'attributes-distinct)
			       ((equal ARG 2) 'attributes-only))))
    (cond ((and (classp class)
		(assoc (intern class) tags-class-parent-list))
	   (record-interest class nil)
	   (with-output-to-temp-buffer parent-buffer
	     (princ (format "Ancestors of class %s in precedence order%s:\n" 
			    class
			    (cond ((null feature-display) "")
				  ((eq feature-display 'attributes-only) " (incl. attributes)")
				  (t " (incl. features)"))))
	     (describe-parents (intern class) 0
			       (cond ((equal ARG 0) 'features)
				     ((equal ARG 1) 'attributes-distinct)
				     ((equal ARG 2) 'attributes-only)))
	     (princ (format "\n\nClass precedence list of class %s:\n\n" class))
	     (princ (mapconcat '(lambda (x) (format "%s" (car x)))
			       (reverse parents-done) ", "))
	     (save-excursion (set-buffer parent-buffer)
			     (end-of-buffer)
			     (fill-paragraph nil)
			     (end-of-buffer))
	     (princ (format "\n\nDirect descendents of class %s:\n\n" class))
	     (princ (mapconcat 'symbol-name
			       (children-list (intern class))
			       ", "))
	     (save-excursion (set-buffer parent-buffer)
			     (end-of-buffer)
			     (fill-paragraph nil)
			     (beginning-of-buffer))
	     (touchup-documentation parent-buffer)))
	  (t (error "%s is not a class." class)))))

(defun parents-this-class (&optional arg)
  "Like sather-parents but fills in all arguments with defaults,
so as to ease kbd macro or mouse use. The symbol pointed to is looked up.
With the mouse, multi-clicks can be used to input a prefix arg.
For the meaning of prefix arguments cf. `sather-parents'."
  (interactive)
  (require-sather-tags-completion)
  (let ((class (sather-symbol-after-point)))
    (if (not (classp class)) (error "Not a class."))
    (sather-parents (if multi-click-hint (- multi-click-hint 2)) class)))

(defun describe-parents (class indent &optional include-features)
  (let* ((parents (cdr (assoc class tags-class-parent-list)))
	 (root (null parents))
	 (defs (cdr (assoc class tags-class-definition-list)))
	 (previous (assoc class parents-done))
	 (pline (cdr previous))
	 features)
    (princ (format (format "%3d  %%%ds" parents-lines indent) " "))
    (princ (s-emphasize (symbol-name class)))
    (cond (previous
	   (if (and root (not include-features))
	       (princ (format "  (cf. %d above)" pline))
	     (princ (format "  ... (cf. %d above)" pline))))
	  (root
	   (push (cons class parents-lines) parents-done)
	   (if (and include-features defs) (describe-own-definitions indent class 
								     (reverse defs)
								     include-features)))
	  (t (push (cons class parents-lines) parents-done)
	     (dolist (def defs)
		     (cond ((and include-features (symbolp def))
			    (push def features))
			   ((consp def)	;parent
			    (cond ((and include-features features)
				   (describe-own-definitions indent class features 
							     include-features)
				   (setq features nil)))
			    (terpri) (setq parents-lines (1+ parents-lines))
			    (describe-parents (car def) (+ indent sather-indent) 
					   include-features))))
	     (cond ((and include-features features)
		    (describe-own-definitions indent class features include-features)
		    (setq features nil)))))))

(defun children-list (class)
  (let (dlist)
    (dolist (classdef tags-class-parent-list)
	    (if (memq class (cdr classdef)) (push (car classdef) dlist)))
    dlist))

(defun defining-class (symbol class)
  (do* ((defs (cdr (assoc class tags-class-definition-list)) (cdr defs))
	(def (car defs) (car defs))
	done
	dclass)
       (done dclass)
       (cond ((null defs) (setq done t))
	     ((eq symbol def) (setq dclass class done t))
	     ((consp def) 
	      (if (setq dclass (defining-class symbol (car def)))
		  (setq done t))))))

(defun describe-own-definitions (indent class defs &optional include-features)
  (let ((space (format (format "%%%ds" indent) " "))
	attr kind x-buff x-point)
    (if (not (save-excursion
		       (visit-tags-table-buffer)
		       (beginning-of-buffer)
		       (re-search-forward (format "^class[ \t]*%s" class) nil t)))
	;; class not tagged, no sense in distinction.
	(setq include-features features))
    (dolist 
     (x defs)
     (cond ((not (memq x features-done))
	    (push x features-done) (setq x (symbol-name x))
	    ;; determine kind'o attribute if needed later
	    (if (not (eq include-features 'features))
		(condition-case err
		    (save-window-excursion
		      (setq last-tag x)
		      (sather-find-tag t x)
		      (setq x-buff (current-buffer)
			    x-point (point))
		      (skip-keywords-backward)
		      (setq kind (save-excursion (reverse (collect-feature-kind))) 
			    attr (feature-type)))
		  (error "Tag %s not found. Maybe run sather-tags again?" x)))
	    (cond ((eq include-features 'attributes-only) ; care to not count up lines 
		   (cond ((not (eq attr 'routine))		       
			  (setq parents-lines (1+ parents-lines))
			  (terpri)			  
			  (princ (concat (format "%3d     " parents-lines) space))
			  (if (eq attr 'attribute)
			      (princ (format "%s" x))
			    (princ (s-emphasize-quote (format "%s" x))))
			  ;; add type and init
			  (princ 
			   (if (eq attr 'alias) ; interpret
			       (save-excursion 
				 (set-buffer x-buff) (goto-char x-point)
				 (re-search-forward x nil t) ; after this name
				 (buffer-substring 
				  (point)
				  (progn (re-search-forward "\\([,;]\\|--\\)" nil t)
					 (1- (point)))))
			     (save-excursion 
			       (set-buffer x-buff) (goto-char x-point)
			       (re-search-forward ":" nil t) (backward-char 1)
			       (buffer-substring (point)
						 (progn (skip-definition-head) (point))))))
			  (if kind
			      (princ (format "   [%s]"
					     (mapconcat 'symbol-name kind " ")))))))
		  (t (setq parents-lines (1+ parents-lines))
		     (terpri)
		     (princ (concat (format "%3d     " parents-lines) space))
		     (cond ((or (eq include-features 'features)				
			     (and (eq attr 'routine) (null kind)))
			    (princ x))
			   ((eq attr 'routine) ;; non-null kind
			    (princ (format "%s   [%s]" x (mapconcat 'symbol-name kind " "))))
			   ((eq attr 'alias)
			    (princ (format "* %s   [%s]" x (mapconcat 'symbol-name kind " "))))
			   (t (princ "*** ")
			      (princ (s-emphasize-quote 
				      (format "%s   [%s%s%s]" x 
					      (mapconcat 'symbol-name kind " ")
					      (if (null kind) "" " ")
					      "attribute"))))))))))))

;; New version, don't use list, but buffers.

(defvar doc-buffer)
(defvar sather-undocumented " Undocumented.\n")

(defun goto-sather-tag ()
  "Find the current sather tag point is on in TAGS file and proceed 
to the next line in the TAGS file."
  (interactive)
  (let (file startpos linebeg string)
    (visit-tags-table-buffer)
    (beginning-of-line) (search-forward "\177")
    (setq file (expand-file-name (file-of-tag)
				 (file-name-directory tags-file-name)))
    (setq linebeg (buffer-substring (1- (point))
				    (save-excursion (beginning-of-line) (point))))
    (search-forward ",")
    (setq startpos (read (current-buffer)))
    ;; visit file and position to the right place according to the tags logic.
    (find-file file)
    (widen)
    (let ((offset 1000)
	  found
	  (pat (concat "^" (regexp-quote linebeg))))
      (or startpos (setq startpos (point-min)))
      (goto-char startpos)
      (while (not (or found (bobp)))
	(setq found (and (re-search-forward pat (+ startpos offset) t)
			 (or (not string)
			     (string-equal string (sather-symbol-before-point)))))
	(cond ((not found)
	       (setq offset (* 2 offset))
	       (goto-char (- startpos offset)))))
      (or found (re-search-forward pat nil t)))))

;;; The system seems to be fast enough to find the documentation
;;; for a single symbol from several files. Since there are the sources
;;; always why should be bother with generated doc files and their
;;; consistency with code. However basis for paper/info doc is needed and
;;; a compact file to go
(defun sather-documentation (&optional arg string default)
  "Show the documentation of a Sather SYMBOL. The command prompts for the name
using DEFAULT, if SYMBOL is nil.  The region marked or else the symbol under
point is the default to look up.

With prefix arg
  0 create a documentation file for the current buffer,
  1 create documentation files for all files in tag table,
  2 selectively create documentation files for all files in tag table.
  3 create documentation for all classes in tag table in alphabetic order 
    using document-tag-table-classes."
  ;; 4 texinfo documentation file for the current Sather tags table
  (interactive "P")
  (cond ((null arg) (describe-sather-tag string default))
	((zerop arg) (document-current-buffer))
	((= arg 1) (document-tag-table-files))
	((= arg 2) (document-tag-table-files nil t))
	((= arg 3) (document-tag-table-classes))
	((= arg 4) (tags-texinfo-document))))

(defun document-this-sather-tag (&optional arg)
  "Like sather-documentation but fills in all arguments with defaults,
so as to ease kbd macro or mouse use. The symbol pointed to is looked up."
  (interactive)
  (require-sather-tags-completion)
  (describe-sather-tag (sather-symbol-after-point)))

(defun describe-sather-tag (&optional string default)
  (interactive)
  (save-excursion
    (if (null string)
	(setq string (sather-tag-at-point "Documentation of" default)))
    (let* ((doc-buffer  (concat "*DOC-" string "*"))
	   file startpos linebeg)
      (visit-tags-table-buffer)
      (sather-mode)			; forward-sexp etc. is based on syntax
      (beginning-of-buffer)
      (with-output-to-temp-buffer doc-buffer
	(while (re-search-forward string nil t)
	  (cond ((string-equal string (sather-symbol-before-point))
		 (cond ((save-excursion (beginning-of-line 0) (looking-at ""))
			;; string occurs in file name
			)
		       ((goto-sather-tag) (backward-char 1)
			(skip-definition-head-backward)
			(cond ((and (classp string) (looking-at "[ \t]*class[ \t\n]"))
			       (describe-class string))
			      ((classp string))	; skip parents
			      (t (describe-feature string
						   nil sather-class-feature-separator)))
			(visit-tags-table-buffer))
		       (t (princ (format "%s not found in %s. TAGS file not up to date.\n" 
					 string file))
			  (visit-tags-table-buffer))))))
	(touchup-documentation doc-buffer)))))

(defun princ-line ()
  (princ (buffer-substring (point) 
			   (progn (end-of-line) 
				  (forward-char 1) 
				  (point)))))

(defun describe-public-features (class)
  (let (tag file)
    (visit-tags-table-buffer)
    (setq file (expand-file-name (file-of-tag)
				 (file-name-directory tags-file-name)))
    (princ (format "\nPublic definitions of class %s: \n\n" class))
    (while (and (not (looking-at "\\([ \t]*class[ \t]\\|\\)"))
		(not (eobp)))
      (cond ((classp (setq tag (sather-symbol-after-point))) ;skip
	     (princ (format "Inherits from: %s\n\n" tag))			   
	     (beginning-of-line 2))
	    ((goto-sather-tag)		; skips forward in TAGS file
	     (backward-char 1)
	     (skip-definition-head-backward)
	     (describe-feature nil t nil) ; name public-only disambiguate
	     (visit-tags-table-buffer))
	    (t (visit-tags-table-buffer)
	       (princ (format "%s not found in %s. TAGS file not up to date." 
			      tag file)))))))
  
(defun describe-class (name)
  ;; this is called on a tag position. And we are looking at "[ \t]*class[ \t\n]".
  (re-search-forward "class[ \t]")	; after
  (skip-layout) 
  (cond ((not (looking-at name)) 
	 (sit-for 0)
	 (error "Unexpected format or TAGS not up to date.")))
  (search-forward name)
  (let* ((signature (collect-feature-signature))
	 (key (concat (s-emphasize name) signature))
	 (kind " [class]")
	 (documentation (concat "\n  " (class-documentation)))
	 (minlen (+ (length key) (length kind))))
    (princ key)
    (cond ((< minlen fill-column)
	   (princ (format (format "%%%ds" (- fill-column minlen)) " "))))
    (princ kind)
    (princ (format "\n%s\n" documentation))
    ;; in the tags table we are past the class entry on the first feature
    (describe-public-features name)))

(defun describe-feature (&optional name public-only disambiguate)
  "Describe current feature. point is supposed to look at line beginning before name.
If PUBLIC-ONLY, do it only if the feature is public.
If DISAMBIGUATE is non-nil it is a string to be used in concatenating 
class and feature name."
  (skip-definition-head-backward)
  (let ((kind (collect-feature-kind))
	(class (sather-which-class t))
	names
	signature documentation key minlen)
    (cond ((not (and public-only (memq 'private kind)))
	   (let ((names (collect-feature-names))
		 (signature (collect-feature-signature))
		 (documentation (feature-documentation)))
	     (if name (setq names (list name)))
	     (dolist (name names)
		     (setq name 
			   (s-emphasize
			    (if disambiguate (format "%s%s%s" class disambiguate name)
			      name)))
		     (setq key (concat name signature))
		     (setq kind (reverse kind))
		     (if (and (not public-only) (not (memq 'private kind)))
			 (setq kind (cons 'public kind)))
		     (if kind (setq kind (mapconcat '(lambda (x) (format "%s" x)) 
						    kind
						    " ")))
		     (setq documentation (concat "\n  " documentation))
		     (princ key)
		     (cond (kind
			    (setq kind (format " [%s]" kind))
			    (setq minlen (+ (length key) (length kind)))
			    (cond ((< minlen fill-column)
				   (princ (format (format "%%%ds" 
							  (- fill-column minlen)) 
						  " "))))
			    (princ kind)))
		     (princ (format "%s\n" documentation))))))))

(defun class-documentation () (feature-documentation t))

(defun feature-documentation (&optional accept-leading-blank-lines)
  "Extract feature documentation immediately following 'is', or, if end is on first line,
continguous comment lines following 'end'."
  (let ((is-point (point))
	combegin comend documentation)
    (cond ((or (comment-on-line-p)
	       (save-excursion 
		 (beginning-of-line 2)
		 (if accept-leading-blank-lines
		     (skip-chars-forward " \t\n")
		   (skip-chars-forward " \t"))
		 (looking-at "--")))
	   (search-forward "--")
	   (while (in-quoted-string-p (point))
	     (search-forward "--"))
	   (setq combegin (point))))
    ;; collect contiguous comment lines
    (while combegin
      (goto-char combegin)
      (push (buffer-substring combegin 
			      (progn (beginning-of-line 2) (point)))
	    documentation)
      (skip-chars-forward " \t")
      (setq combegin (if (looking-at "--") (+ (point) 2) nil)))
    (setq documentation
	  (if documentation (mapconcat 'identity (reverse documentation)
				       "  ")
	    sather-undocumented))))	  

;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; Report generation

(defvar formal-comment-start "-- -" 
  ;;"\\(-- -\\|--\\*\\*\\*\\)"
  "* regular expression defining top-level comment that is to be included
in class documentation generated by document-current-buffer and friends.")

(defun document-current-buffer (&optional dir)
  "Generic command for report generation. Winds itself forward over
the buffer collecting comment lines and relevant definitions."
  (interactive)
  (let* ((bname (buffer-file-name (current-buffer)))
	 (prompt-on-write (if dir nil t))
	 (dir (or dir (file-name-directory bname)))
	 (fname (file-name-nondirectory bname))
	 (filename (concat (substring fname 0 (- (length fname) 2)) "doc"))
	 (doc-buffer (get-buffer-create "*DOC*"))
	 doc-string)
    (save-excursion			; clear buffer
      (switch-to-buffer-other-window doc-buffer)
      (widen) (delete-region (point-min) (point-max)))
    (save-excursion
      (beginning-of-buffer)
      (collect-contiguous-comment)
      (while (not (eobp)) (document-top-level)))
    (save-excursion
      (set-buffer doc-buffer)
      (if prompt-on-write
	  (write-file (expand-file-name 
		       (read-file-name (format "Save docfile to (Default %s): "
					      filename)
				      default-directory
				      (concat default-directory filename))
		       default-directory))
	(write-file (expand-file-name filename
				      dir))))))

(defun document-top-level ()
  (interactive)
  (let ((begin (point))
	saw-comment)
    (cond ((comment-line-p)		; top-level comment
	   (if (and formal-comment-start
		    (looking-at formal-comment-start))
	       (insert-doc (current-line) "\n"))
	   (beginning-of-line 2))
	  ((empty-line-p) (beginning-of-line 2)) ; empty 
	  ;; class 'braces'
	  ((looking-at "[ \t]*class[ \t\n]") ; class open
	   (insert-doc "\n")
	   (skip-definition-head)
	   (forward-char 2) (skip-chars-forward " \t")
	   (insert-doc (buffer-substring begin (point)))
	   (or (and (collect-optional-comment) (setq saw-comment t))
	       (progn (if (eolp) (insert-doc "\n")) nil))
	   (cond ((not saw-comment)
		  (skip-chars-forward " \t\n")
		  (collect-contiguous-comment))))
	  ((looking-at "end[		;\t\n]")       ; class close
	   (insert-doc 
	    (buffer-substring (point) 
			      (progn (forward-char 4) 
				     (skip-chars-forward " \t\n")
				     (point))))
	   (or (collect-optional-comment)
	       (progn (if (eolp) (insert-doc "\n")) nil))
	   (beginning-of-line 2))
	  ;; features
	  (t (collect-next-definition)))))

(defun collect-next-definition ()
  "Collect the defining lines plus top documentation for the definition starting
at point. Return nil if this is a private feature."
  (interactive)
  (skip-layout)
  (beginning-of-line)
  (let ((begin (point)))
    (cond ((memq 'private (collect-feature-k
			   ind)) (skip-feature)
	   (skip-feature)
	   (beginning-of-line 2))
	  (t
	   (skip-definition-head)
	   (cond ((looking-at ";") (end-of-line)
		  (insert-doc (buffer-substring begin (point)) "\n")
		  (beginning-of-line 2) (skip-chars-forward " \t")
		  (if (looking-at comment-start) (collect-contiguous-comment)))
		 (t			; routine
		  (save-excursion 
		    (forward-char 2) (skip-chars-forward " \t")
		    (insert-doc (buffer-substring begin (point)))
		    (or (collect-optional-comment)
			(not (eolp))
			(progn (insert-doc "\n") (beginning-of-line 2)))
		    (skip-chars-forward " \t")
		    (collect-contiguous-comment))
		  (skip-feature)
		  (cond ((= begin (save-excursion (beginning-of-line) (point)))
			 (skip-chars-forward " \t")
			 (or (collect-optional-comment)
			     (not (eolp))
			     (progn (insert-doc "\n")
				    (beginning-of-line 2))))
			(t nil))))))))

(defun collect-contiguous-comment ()
  (while (looking-at comment-start)
    (insert-doc (current-line) "\n")
    (beginning-of-line 2) (skip-chars-forward " \t")))

(defun collect-optional-comment ()
  (cond ((looking-at comment-start) 
	 (insert-doc 
	  (buffer-substring (point) 
			    (progn (beginning-of-line 2) (point))))
	 (skip-chars-forward " \t")
	 (collect-contiguous-comment)
	 t				; newline is out
	 )))

(defun insert-doc (&rest strings)
  (save-excursion (set-buffer "*DOC*") (apply 'insert strings)))

(defun current-line ()
  (buffer-substring (save-excursion (beginning-of-line) (point))
		    (save-excursion (end-of-line) (point))))

(defun document-tag-table-files (&optional directory prompt)
  "Create documentation for all tag table files in DIRECTORY.
If PROMPT is non-nil ask for confirmation for each file."
  (interactive)
  (require-sather-tags)
  (let ((files (tag-table-files)))
    (setq directory 
	  (or directory
	      (read-file-name "Doc directory: "
			      default-directory
			      default-directory
			      t)))
    (if (file-exists-p directory)
	(dolist (file files)		       
		(cond ((or (null prompt)
			   (y-or-n-p (format "Document file %s " file)))
		       (find-file file)
		       (document-current-buffer directory))))
      (error "Directory %s does seem to exist." directory))))

(defvar doc-tag-table-classes-exclude "TEST"
  "* Regular expression. If it occurs in a class name, that class is not
included in the documentation by document-tag-table-classes. C classes are
always excluded.")

(defun tags-find-class (name)
  (visit-tags-table-buffer)
  (beginning-of-buffer)
  (cond ((re-search-forward (format "^[ \t]*class[ \t\n]*%s[ \t\n{-]" name) nil t)
	 (goto-sather-tag)
	 (if (> (point) (point-min)) 
	     (backward-char 1)
	   (if (not (looking-at "^[ \t]*class[ \t\n]*%s[ \t\n{-]"))
	       (error "No class in tag file named %s. Tag table out of date?" name))))
	(t (error "No class in tag file named %s." name))))

(defun document-tag-table-classes (&optional selective)
  "Produce documentation for the classes in the tag file in alphabetic order.
With \\[univeral-arg] non-nil, you can select particular classes in a yes-or-no
dialogue. Cf. also the user variable doc-tag-table-classes-exclude which
allows to control exclusion nby regular expression."
  (interactive "P")
  ;; Called by emacs batch. So don't support required arguments.
  (require-sather-tags)
  (let (sorted
	(doc-buffer "*DOC-TAGS-Classes*")
	from to
	(sather-multi-style-support nil)) ; too slow on large files.
    ;; filter
    (mapcar '(lambda (x)
	       (let* ((sym (car x)))
		 (if (not (eq sym 'C)) (push (symbol-name sym) sorted))))
	    tags-class-parent-list)
    (with-output-to-temp-buffer doc-buffer
      (setq sorted (sort sorted 'string-lessp))
      (princ "Classes (excluding class C):\n\n")
      (princ (mapconcat 'identity sorted ", ")) (princ ".")
      (set-buffer (get-buffer "*DOC-TAGS-Classes*"))
      (fill-paragraph nil)
      (end-of-buffer)
      (princ "\n")
      (princ (format "Excluding matches to doc-tag-table-classes-exclude =\n\"%s\"\n\n"
		     doc-tag-table-classes-exclude))
      (princ "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n")
      (dolist (class sorted)
	      (tags-find-class class)
	      (setq to (point) from (progn (backward-sexp 1) (point)))
	      (cond ((and doc-tag-table-classes-exclude
			  (not (equal doc-tag-table-classes-exclude ""))
			  (not (equal doc-tag-table-classes-exclude "\\(\\)"))
			  (save-excursion 
			    (re-search-forward doc-tag-table-classes-exclude to t)))
		     ;; must be excluded
		     t)
		    ((and selective (not (y-or-n-p (format "Include class %s? " class))))
		     t)
		    (t;; else print
		     (message "Describing %s" class)
		     (beginning-of-line)
		     (describe-class class)
		     (princ "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n")
		     )))
      (if sather-multi-style-support
	  (touchup-documentation doc-buffer)))))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Uglification

(defvar sather-contracting-specials "[-+/*\\().,;:={}<>\n]")

(defun uglify-region ()
  "Removes all 'unnecessary spacing' in the current region. Preserve current
indentation."
  (interactive)
  (let ((beg (region-beginning)) (end (region-end))
	;; we are left of kwd and have deleted horizontal space there.
	(left-of-kwd-pat (concat sather-keyword-pattern "[ \t\n;{(]"))
	;; we are right of kwd and have deleted horizontal space there.
	(right-of-kwd-pat (concat "[ \t]" sather-keyword-pattern "\\([.-]\\|(\\|{\\)")))
    (save-restriction
      (save-excursion
	(narrow-to-region beg end)
	(goto-char (point-min))
	(skip-layout);; to first non comment / non white space
	;; preserve indentation
	(if (looking-at sather-contracting-specials) (forward-char 1))
	;; remove horizontal spacing
	(while (and (< (point) (point-max))
		    (re-search-forward sather-contracting-specials (point-max) t))
	  (backward-char 1)		;looking-at char
	  (cond ((in-quoted-string-p (point)) (forward-char 1))
		((looking-at "\n") (skip-layout);;preserve indentation
		 (if (looking-at sather-contracting-specials)
		     (forward-char 1)))
		;; non-standalone comment
		((looking-at "--") 
		 (delete-horizontal-space)
		 (cond ((save-excursion
			  (or (and (<= (+ (point-min) 3) (point))
				   (progn (backward-char 3) (looking-at " is")))
			      (and (<= (+ (point-min) 5) (point))
				   (progn (backward-char 5) (looking-at " end;"))
				   )))
			(insert " "))
		       (t (sather-comment)))
		 (end-of-line))
		(t (delete-horizontal-space) ; left of special
		   ;; avoid and() or() ... if() ... if.3 = 4 etc
		   (if (save-excursion (backward-word 1)
				       (if (< (point-min) (point)) 
					   (backward-char 1))
				       (looking-at right-of-kwd-pat))
		       (insert " "))
		   (forward-char 1) 
		   (delete-horizontal-space) ; right of special
		   (cond ((looking-at "--")) ; stay here, we see it again.
			 ((looking-at left-of-kwd-pat) (insert " ")) 
			 (t 
			  (backward-char 1)
			  (cond ((looking-at "[)}][a-zA-z\n]")
				 (forward-char 1)
				 (insert " "))
				(t (forward-char 1)))))))))
      (setq end (point-max)))		; may have moved
    (sather-unmark-region beg end)))

(defvar sather-separator-list
  '(":=" "=" "<" ">" "<=" ">=" "/=" "*" "+" "-" "/")
  "* Beautify-region surrounds tokens in this list by white space.")

(defvar sather-leader-list nil
  "* Beautify-region inserts whitespace in front of these tokens.
Try '(\"(\" \"{\" \"}\").")
   
(defvar sather-trailer-list '(";" ":")
  "* Beautify-region inserts whitespace after these tokens.
Maybe you want to include \",\", too?")

(defun string-> (a b) (string-lessp b a))

(defun beautify-region ()
  "Add 'necessary spacing' in the current region. The addition of spacing
is controlled by the user variables sather-leader-list, sather-separator-list
and sather-trailer-list. There are a few exceptions resulting from the
Sather lexical syntax: '::' '--' and REAL literals are recognized."
  (interactive)
  (let ((m (region-beginning)) (p (region-end)))
    (save-excursion
      (save-restriction
	(narrow-to-region m p)
	(goto-char (point-min))
	;; for prefixes '<' vs '<=' et al.: match longer one first so as to not 
	;; introduce space someplace in the middle.
	(let* ((leader-list (sort (append sather-leader-list
					  (copy-sequence sather-separator-list)) 'string->))
	       (trailer-list (sort (append sather-separator-list
					   (copy-sequence sather-trailer-list)) 'string->))
	       (lpat (concat "\\(" (mapconcat 'identity leader-list
					      "\\|")	"\\)"))
	       (tpat (concat "\\(" (mapconcat 'identity trailer-list
					      "\\|") "\\)")))
	  ;; add leader space
	  (let (patend)
	    (if (not (equal lpat "\\(\\)"))
		(while (re-search-forward lpat (point-max) t)
		  (setq patend (point))
		  (goto-char (match-beginning 0))
		  (setq patend (- patend (point)))
		  (cond ((or (looking-at "--")
			     (save-excursion (forward-char 1)
					     (in-comment-p (point))))
			 (end-of-line))
			((in-quoted-string-p (point)) (re-search-forward "\"" nil t))
			;; not in float format 1.0e-1
			((and (looking-at "-")
			      (> (point) (+ 2 (point-min)))
			      (or (= (char-after (1- (point))) ?e)
			          (= (char-after (1- (point))) ?E))
			      (let ((char (char-after (- (point) 2))))
				(or (and (<= ?0 char)
					 (<= char ?9))
				    (= char ?.))))
			 (forward-char 1))
			(t 
			 (if (save-excursion
			       (if (< (point-min) (point)) (backward-char 1))
			       (not (looking-at "[' \t\n]")))
			     (insert " "))
			 (forward-char patend))))))
	  ;; add trailer space
	  (goto-char (point-min))
	  (if (not (equal tpat "\\(\\)"))
	      (while (and (re-search-forward tpat (point-max) t)
			  (< (point) (point-max)))
		(cond ((or (looking-at "--")
			   (save-excursion (forward-char 1)
					   (in-comment-p (point))))
		       (end-of-line))
		      ((in-quoted-string-p (point)) (re-search-forward "\"" nil t))
		      ((looking-at tpat)) ; don't do it in the middle of -- ::
		      ((and (> (point) (1+ (point-min)))
			    (save-excursion (backward-char 2)
					    (looking-at "\\(::\\|'\\)"))))
		      ((and (= (char-after (1- (point))) ?-)
			    (> (point) (+ 2 (point-min)))
			    ;; not in float format 1.0e-1
			    ;; trust leader insertion for float
			    ;; and allow dyadic '-' only between identifiers.
			    (not (and (= (char-after (- (point) 2)) ? )
				      (or 
				       (memq (char-syntax (char-after (- (point) 3))) 
					     '(?w ?_))
				       (= (char-after (- (point) 3)) ?\)))))))
		      ((not (looking-at "[ \t\n]"))
		       (insert " ")))))
	  )
	(setq p (point-max)))		; may have changed
      (if sather-multi-style-support
	  (sather-markup-region m p)))))

(defun indent-buffer ()
  "Indent the whole buffer."
  (interactive)
  (let ((visible-indentation t))
    (save-excursion 
      (indent-region (point-min) (point-max) nil))))

(defun uglify-buffer (&optional ARG)
  "Uglify the whole buffer by removing 'unnecessary' horizontal white space
and removing any keyword hightlighting (if this is supported at all).
Do not change indentation.
With non-nil prefix argument only remove highlighting."
  (interactive "P")
  (if ARG (sather-unmark-buffer)
    (save-excursion 
      (mark-whole-buffer)
      (if sather-multi-style-support
	  (sather-unmark-buffer))		; faster than what uglify does to marks
      (uglify-region))))

(defun beautify-buffer (&optional indent)
  "Beautify the whole buffer by adding 'necessary' horizontal white space
and highlighting keywords if this is possible.  Do not change indentation.
With prefix arg (first arg when called from program)
  0 also indent the buffer, 1 highlight keywords, only."
  (interactive "P")
  (cond ((or (null indent) (= indent 0))
	 (save-excursion
	   (if sather-multi-style-support (sather-unmark-buffer))
	   (if (and indent (= indent 0)) (indent-buffer)) ; do it first, is faster
	   (mark-whole-buffer)
	   (beautify-region)))
	((= indent 1) (sather-markup-buffer))))
 
;;;- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
;;; MARKUP under Epoch

(defvar sather-keyword-pattern
  "\\(and\\|assert\\|break\\|class\\|constant\\|debug\\|else\\|elsif\\|end\\|if\\|is\\|loop\\|not\\|or\\|private\\|return\\|shared\\|alias\\|switch\\|then\\|until\\|when\\|except\\)"
  "* Regular expression matching Sather keywords.")

(defvar sather-reserved-identifier-pattern
  "\\(asize\\|asize1\\|asize2\\|asize3\\|asize4\\|copy\\|extend\\|false\\|new\\|res\\|self\\|true\\|type\\|void\\|initialize\\)"
  "* Regular expression that uniquely matches Sather reserved identifiers.")
;;;don't include these right now. Get's too scattered with all uppercase.
;;;the lowercase ones seem ok also to warn not to redefine them.
;;;|ARRAY|ARRAY2|ARRAY3|ARRAY4|BOOL|C|CHAR|DOUBLE|ERR|FILE|IN|INT|OB|OUT|REAL|SELF_TYPE|STR|STR_SCAN|SYS|UNDEFINE|UNIX

(defun sather-unmark-region (from to)
  (let ((m (mark)) (p (point)))
    (cond (sather-multi-style-support
	   (dolist (button (button-list))
		   (if (or (and (<= from (button-end button))
				(<= (button-end button) to))
			   (and (<= from (button-start button))
				(<= (button-start button) to)))
		       (delete-button button)))
	   (if (and (memq 'sky-mouse features) (fboundp 'mark-region))
	       (if m (mark-region m p)))))))

(defun s-rexp-markup-region (rexp style from to)
  "Markup all full-word matches of REXP not in comment by changing to STYLE.
The changae is restricted to the region defined by the points FROM and TO."
  (if (not sather-multi-style-support) (error "Multi-style support not installed."))
  (let ((rexp-pat (concat "[-+*/(),.;:= \t\n]" rexp "[-+*/(),.;:= \t\n]"))
	(case-fold-search nil))
    (save-excursion
      (goto-char from)
      (while (re-search-forward rexp-pat to t)	
	(let ((beg (match-beginning 1)) (end (match-end 1)))
	  (cond ((not (or (in-comment-p beg) (in-quoted-string-p end)))
		 (add-button beg end style))))))))

(defun sather-markup-region (from to)
  "In the current region, markup all reserved words and predefined identifiers."
  (interactive "r")
  (if (not sather-multi-style-support) (error "Multi-style support not installed."))
  (s-rexp-markup-region sather-keyword-pattern S-emphasize-style from to)
  (s-rexp-markup-region sather-reserved-identifier-pattern S-quote-style from to)
  (sather-markup-comment from to)
  )

(defun sather-markup-comment (from to)
  "In the current region, markup all comment."
  (interactive "r")
  (if (not sather-multi-style-support) (error "Multi-style support not installed."))
  (save-excursion
    (goto-char from) (beginning-of-line)
    (while (< (point) to)
      (end-of-language-line)
      (cond ((looking-at "--")
	     (add-button (point) (progn (end-of-line) (point)) S-comment-style)))
      (beginning-of-line 2))))

(defun sather-unmark-buffer ()
  "Removes all highlighting from region except region marking (if multi-style supported)."
  (interactive)
  (if (not sather-multi-style-support) (error "Multi-style support not installed."))
  (let ((m (mark)) (p (point)))
    (clear-buttons)
    (if (and (memq 'sky-mouse features) (fboundp 'mark-region))
	(if m (mark-region m p)))))

(defun sather-markup-buffer ()
  "Highlight all Sather keywords in buffer (if multi-style supported)."
  (interactive)
  (if (not sather-multi-style-support) (error "Multi-style support not installed."))
  (sather-unmark-buffer)
  (sather-markup-region (point-min) (point-max)))


;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Eiffel to Sather conversion
;;;

(defmacro within-context-between (r &rest body)
  (` (let ((from-s (, (car r)))
	   (to-s (,(cadr r)))
	   from-pt to-pt)
       (goto-char (point-min))
       (skip-comment)
       (while (re-search-forward from-s (point-max) t)
	 (setq from-pt (match-beginning 0))
	 (when (not (in-comment-p (match-beginning 0)))
	       (re-search-forward to-s (point-max) t)
	       (when (setq to-pt (point))
		     (save-excursion
		       (save-restriction 
			 (narrow-to-region from-pt to-pt)
			 (,@ body)))))))))

(defmacro within-context (p &rest body)
  (` (progn (goto-char (point-min))
	    (skip-comment)
	    (while (re-search-forward (, p) (point-max) t)
	      (when (not (in-comment-p (match-beginning 0)))
		    (save-excursion
		      (save-restriction
			(narrow-to-region (match-beginning 0) (match-end 0))
			(,@ body))))))))

(defmacro when-containing (patt &rest body)
  (` (when (progn (goto-char (point-min)) (re-search-forward (, patt) (point-max) t))
	   (,@ body))))
(defmacro unless-containing (patt &rest body)
  (` (when (progn (goto-char (point-min)) (not (re-search-forward (, patt) (point-max) t)))
	   (,@ body))))

(defmacro for-symbols-between (group &rest body)
  (` (within-context (, (concat (car group) "[ \t\n]*[A-Za-z_{}]*[ \t\n]*" (cadr group)))
		      (,@ body))))

(defun trim-feature-signatures (convert)
  "For all features in the current buffer trim the signature by making all type
symbols uppercase, and removing unnecessary blanks."
  (interactive "P")
  (within-context-between ("(" ")")
    (unless-containing "::\\|\""
      (when-containing ":"
	   (message "Trimming arguments  \"%s\"" 
		    (buffer-substring (point-min) (point-max)))
	   (for-symbols-between (":" "")
				(upcase-region (1+ (point-min)) (point-max))
				(replace-all-strings '((": " ":")))
				(if convert
				  (replace-all-strings '((":" ":$")))))
	   (replace-all-strings '((", " ",")))
	   (within-context ";[A-Za-z_]" (replace-all-strings '((";" "; ")))))))
  (for-symbols-between ("[A-Za-z_)]:" "is")
      (message "Trimming result-type  \"%s\"" 
	       (buffer-substring (point-min) (point-max)))
      (upcase-region (+ 2 (point-min)) (- (point-max) 3))
      (replace-all-strings '((": " ":")))
      (if convert
	  (replace-all-strings '((":" ":$")))))
  (for-symbols-between ("[A-Za-z_]:" ";")
      (message "Trimming attribute type  \"%s\"" 
	       (buffer-substring (point-min) (point-max)))
      (upcase-region (+ 2 (point-min)) (1- (point-max)))
      (when convert
	    (goto-char (point-min))
	    (re-search-forward "[A-Za-z_]:[ \t\n]*" (point-max) t)
	    (insert "$")))
  (for-symbols-between ("[A-Za-z_]:" ":=")
      (message "Trimming attribute type  \"%s\"" 
	       (buffer-substring (point-min) (point-max)))
      (upcase-region (+ 2 (point-min)) (- (point-max) 2))
      (when convert
	    (goto-char (point-min))
	    (re-search-forward "[A-Za-z_]:[ \t\n]*" (point-max) t)
	    (insert "$")))
  (when convert
	(replace-all-strings '(("$BOOL" "BOOL")
			       ("$REAL" "REAL")
			       ("$INT" "INT")
			       ("$DOUBLE" "DOUBLE")))))

(defun upcase-class-names () 
  (interactive)
  (for-symbols-between ("class" "is")
		       (upcase-region (+ 5 (point-min)) (- (point-max) 2)))
  (for-symbols-between ("[ \t]" "::")
		       (upcase-region (point-min) (point-max))))

(defun trim-= ()
  "Make assignments and comparisons to read as   foo := bar."
  (interactive)
  (replace-all-strings '((" := " ":=") (" /= " "/=")
			 (":=" " := ") ("/=" " /= ")))
  (within-context "\\([A-Za-z_0-9)]\\|\\]\\)=" ;   alpha-digit-close equals
		  (goto-char (point-min)) (replace-string "=" " = ")))

;; don't modify if not necessary
(defun maybe-upcase-region (from to)
  (goto-char from)
  (re-search-forward "[A-Z \t]*" to t)
  (if (not (and (= (match-beginning 0) from) (= (match-end 0) to)))
      (upcase-region from to)))
      
(defun convert-eiffel ()
  "Converts the current buffer to Sather. It is assumed that the current
buffer contains one or more Eiffel classes.

The following transformations are performed:

 1. Replacement of keywords, such as 'inspect' -> 'switch'.
 2. Replacement of identifiers, such as 'BOOLEAN' -> 'BOOL'.
 3. Replacement of special sequences, such as '[' -> '{' or '?=' -> ':='.
 4. Replacement of Eiffel constructs by the corresponding Sather constructs.
    For instance loops, locals etc. have a syntactically different structure. 
 4. Deletion of parameter constraints in parameterized classes.
 5. Commenting of class interfaces (you have to revisit them and
    reintroduce inheritance, but typically all the export, rename
    and define declarations have to be interpreted to get it right).
 6. Transformation of pre- and postconditions into legal Sather assertions
    of the form 'assert (pre) ... end;', 'assert (post) ... end;'.
 7. Commenting of invariants and variants constructs.
 8. Conversion of create calls to assigning create calls.
 9. All class identifiers are changed to uppercase.
10. ',' and ';' separated signatures are changed to be acceptable to Sather.

The following deficiencies are known: 

1-3. A few replacements are run as query-replace to make sure, identifiers
     are not replaced accidentidally, like comment text in 'current' -> 'self'.
6. Assertions that continue over a number of lines are treated improperly
   sometimes. The 'end;' may end up in the wrong place.
7. Only the first line of invariants is commented.
8-9. The types of local declarations may not always end up to be uppercase.
10. Factorized types in argument lists like in foo(x,y:BAR, g:BAZ)
   may not always be treated correctly."

  (interactive)
  (let ((case-fold-search t))
    (replace-all-words
     '(("BOOLEAN" "BOOL")
       ("deferred class" "class")
       ("check" "assert")
       ("CHARACTER" "CHAR")
       ("Result" "res") 
       ("like Current" "SELF_TYPE")
       ("Current" "self" query)
       ("INTEGER" "INT")
       ("inspect" "switch")
       ("STRING" "STR")))
    (replace-all-strings 
     '(("\\.Void" " = void")
       ("io\\.new_line" "OUT::nl")
       ("io\\.putstring" "OUT::s")
       ("io\\.putreal" "OUT::r")
       ("io\\.putint" "OUT::i")
       ("\\.create" " := CHECK_create")
       ;("Create" "crt" query)
       ("\\[" "{")
       ("\\]" "}")
       ("?=" ":=")
       ("\\^" ".pow " query)
       ("end\n" "end;\n")
       ("end --" "end; --"))))
  (within-context-between ("class" "feature")
     (within-context-between ("{" "}")	;delete parameter constraints
	(within-context-between ("->" "}")
	   (delete-region (point-min) (1- (point-max)))))
     (replace-all-strings '((" }" "}")))
     (comment-region-lines
      (save-excursion (goto-char (point-min)) ;before "class"
		      (end-of-line) (insert " is")
		      (beginning-of-line 2) ;next line
		      (point))
      (point-max)))
  (within-context "require .*;"
		  (replace-all-strings '(("require" "assert (pre)") 
					 (";" " end;"))))
  (within-context "ensure .*;"
		  (replace-all-strings '(("ensure" "assert (post)") 
					 (";" " end;"))))
  (within-context-between ("invariant" ";")
			  (replace-all-strings 
			   '(("invariant" "-- invariant"))))
  (within-context-between ("variant" "until")
			  (replace-all-strings
			   '(("variant" "-- variant"))))
  (within-context "local [.]*[ \t\n]+do" 
		  ;;make sure all local decl. end with semicol
		  (re-search-backward "[ \t\n]+do")
		  (insert ";"))
  (within-context-between ("until" "loop")
			  (if (and (re-search-backward ";" (point-min) t)
				   (looking-at ";[ \t\n]*loop"))
			      (delete-region (point) (1+ (point))))
			  )
  (delete-all-strings '("do" "local" "from"))
  (within-context-between ("(" ")") 
     (unless-containing "::\\|\""
	(when-containing ":" (replace-all-strings '(("," ";"))))))
  (trim-feature-signatures t)
  (trim-=)
  )

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Mini choice facilities that will eventually go away
;;; or be realized based on Emacs 19 functions and/or external UI servers.
;;; We want our software to go to this small portable interface when
;;; choices are presented to the user.
;;; On a tty Emacs it can use the minibuffer, or an electric mode like
;;; the electric buffer list.
;;; Under X it can bringup panels and menus.

(defun choose-item (item-list &optional label default)
  "Present a single choice titled LABEL and return the item chosen from
ITEM-LIST, a list of conses the car of which is a string, or an obarray
 (see try-completion)."
  (interactive)
  (if (null label) (setq label "Choices: "))
      (completing-read
       (if default
	   (format "%s (Default %s): " default)
	 label)
       item-list))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(autoload 'sdb "sdb-mode" "Sdb mode" t nil)

(autoload 'lineno-mode "lineno" "Lineno mode" t nil)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Sather specific mouse commands

(defun install-sather-mode-mouse-map ()
  (make-local-variable 'sather-mode-mouse-map)
  (setq sather-mode-mouse-map sather-mouse-map)
  (mouse-use-local-map sather-mode-mouse-map))

(cond (sather-mouse-p

       (defvar sather-mouse-map (create-mouse-map mouse::global-map))

       ;; Macro keys: M-S

       (defmouse sather-mouse-map "M-S-Left" 
	 'mouse-set-point 'edit-this-sather-definition)
       (defmouse sather-mouse-map "M-S-Middle" 
	 'mouse-set-point 'document-this-sather-tag)
       (defmouse sather-mouse-map "M-S-Right" 
	 'mouse-set-point 'parents-this-class)

       ;; Activate mode mouse map

       (if (not (memq 'install-sather-mode-mouse-map
		      sather-mode-hooks))
	   (push 'install-sather-mode-mouse-map sather-mode-hooks))

       ))
