;;; -*- Mode: Emacs-Lisp;  -*-
;;; File: scroll-mouse.el
;;; Author: Heinz Schmidt (hws@ICSI.Berkeley.EDU)
;;; Copyright (C) International Computer Science Institute, 1991
;;;
;;; COPYRIGHT NOTICE: This code is provided "AS IS" WITHOUT ANY WARRANTY.
;;; It is subject to the terms of the GNU EMACS GENERAL PUBLIC LICENSE
;;; described in a file COPYING in the GNU EMACS distribution or to be obtained
;;; from Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: minor mouse scrolling mode
;;;*
;;;* RELATED PACKAGES: sky-mouse
;;;*
;;;* HISTORY:
;;;* Last edited: Mar  8 14:37 1992 (hws)
;;;* Created: Wed Jan 30 23:51:17 1991 (hws)
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(require 'sky-mouse)

(provide 'scroll-mouse "scrollm")

;;;
;;; Cursor glyphs under Epoch
;;;
(defvar v-scrollbar-glyph 116)		;cursor-sb-v-double-arrow
(defvar v-scroll-down-glyph 106)	;cursor-sb-down-arrow
(defvar v-scroll-up-glyph 114)	        ;cursor-sb-up-arrow
(defvar v-scroll-percent-glyph 112)     ;cursor-sb-right-arrow
(defvar v-scroll-drag-glyph 60)		;cursor-hand2
;;;
;;; Mouse map
;;;

(defvar mouse-scroll-map (create-mouse-map))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; SCROLLING MOUSE COMMANDS

;;;
;;; Relative scrolling
;;;

(defmouse mouse-scroll-map "Left" 'mouse-set-point
  '(lambda (arg) "This line to top." (recenter 0) (message "Line to top"))
  v-scroll-up-glyph)

(defmouse mouse-scroll-map "Middle" 
  '(lambda (arg) "Scroll dragging." 
     (mouse-set-point arg) (message "Drag scroll..."))
  'scroll-point-to-mouse
  v-scroll-drag-glyph)

(defmouse mouse-scroll-map "Right" 'mouse-set-point
  '(lambda (arg) "This line to bottom." 
     (recenter (- (window-height) 2)) (message "Line to bottom"))
  v-scroll-down-glyph)

;;;
;;; Pagewise scrolling
;;;

(defmouse mouse-scroll-map "S-Left" 'mouse-ignore
  '(lambda (arg) "Next page." (message "Page up.") (scroll-up)) v-scroll-up-glyph)

(defmouse mouse-scroll-map "S-Middle" 'mouse-set-point
  'goto-percent-linewise v-scroll-percent-glyph)

(defmouse mouse-scroll-map "S-Right" 'mouse-ignore
  '(lambda (arg) "Previous page." (message "Page down.") (scroll-down)) 
  v-scroll-down-glyph)

;;;
;;; Absolute scrolling
;;;

(defmouse mouse-scroll-map "M-Left" 'mouse-ignore
  '(lambda (arg) "End of buffer." 
     (end-of-buffer) 
     (message "All up.")) v-scroll-up-glyph)

(defmouse mouse-scroll-map "M-Right" 'mouse-ignore
  '(lambda (arg) "Beginning of buffer." 
     (beginning-of-buffer)
     (message "All down."))  
  v-scroll-down-glyph)

;;;
;;; Mouse help
;;;

(defmouse mouse-scroll-map "M-Middle" 'mouse-help-scrolling 'mouse-ignore)
(defmouse mouse-scroll-map "M-S-Left" 'mouse-help-scrolling 'mouse-ignore)
(defmouse mouse-scroll-map "M-S-Middle" 'mouse-help-scrolling 'mouse-ignore)
(defmouse mouse-scroll-map "M-S-Right" 'mouse-help-scrolling 'mouse-ignore)

;;;
;;; Control stops the mode
;;;

(defmouse mouse-scroll-map "C-Left" 'mouse-ignore 'mouse-scroll-mode)
(defmouse mouse-scroll-map "C-Middle" 'mouse-ignore 'mouse-scroll-mode)
(defmouse mouse-scroll-map "C-Right" 'mouse-ignore 'mouse-scroll-mode)

;;;
;;; Minor mode
;;;

(defvar mouse-scroll-mode nil)
(setq-default mouse-scroll-mode nil)

(push '(mouse-scroll-mode " Scroll") minor-mode-alist)

(defun mouse-scroll-mode (&optional arg)
  "Minor mouse scrolling mode, regular invocation by C-Right, changes mouse to
scrolling device:


      |           Left              Middle             Right
----------------------------------------------------------------------
      |
  -   |        Line to top        Drag Scroll     Line to bottom
      |
  S   |          Page up          Page percent       Page down
      |
  M   |          All up              Help            All down
      |          
 M-S  |           Help               Help              Help
      |
  C   |       End scrolling     End scrolling     End scrolling
      |
"
  (interactive)
  (cond ((not mouse-scroll-mode)
	 ;; save state of local mouse map and feedback and set it to scrolling
	 (if (not (boundp 'buffer-save-local-mouse-map))
	     (make-local-variable 'buffer-save-local-mouse-map))
	 (setq buffer-save-local-mouse-map (mouse-local-map))
	 (mouse-use-local-map mouse-scroll-map)
	 (when (boundp 'mouse-buffer-cursor-glyph)
	       (if (not (boundp 'buffer-save-cursor-glyph))
		   (make-local-variable 'buffer-save-cursor-glyph))
	       (setq buffer-save-cursor-glyph mouse-buffer-cursor-glyph
		     mouse-buffer-cursor-glyph v-scrollbar-glyph)
	       (cursor-glyph mouse-buffer-cursor-glyph))
	 (setq mouse-scroll-mode t)
	 (message "Mouse scrolling mode."))
	(t 
	 (mouse-use-local-map buffer-save-local-mouse-map)
	 (when (boundp 'mouse-buffer-cursor-glyph)
	       (setq mouse-buffer-cursor-glyph buffer-save-cursor-glyph)
	       (cursor-glyph mouse-buffer-cursor-glyph))
	 (setq mouse-scroll-mode nil)
	 (message "End scrolling mode."))))

(defun mouse-help-scrolling (&optional arg)
  "Provide short summary of minor mouse scrolling mode."
  (describe-mode)
  (describe-function 'mouse-scroll-mode))
