/* execute opcodes */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#ifndef lint
static char SccsId[] = "@(#)opcodes.c	1.13 2/12/90";
#endif /* lint */

#include "fools.h"
#include "utils.h"
#include "codegen.h"
#include "cont.h"

struct opcode_info_s opcodeInfo[] = {
    /*	nops	name */
    {	0,	"code_term" }, /* dummpy opcode */

#ifdef __STDC__
#define OPCODE(code, nops) { nops, #code },
#else
#define OPCODE(code, nops) { nops, "code" },
#endif

#include "opcodes.i"

};

/* check if Proc should be traced
 *
 * Type is either TRACE_ENTRY or TRACE_EXIT */
#ifdef DEBUG
#define TRACEP(Proc, Type) (checkCond((Proc), (Type)) || debugTrace)
#else
#define TRACEP(Proc, Type) (checkCond((Proc), (Type)))
#endif /* DEBUG */

/* call primitive function with n args from the stack */
#define CALL_PRIM(prim, n)\
    (*DATA((prim), cfunc, primInst))(stackAddr(argStack, (n)), (n))

#define STACKSIZE 4096
Stack argStack; /* stack */

/* initializer */
void opcodeInit()
{
    argStack = stackNew(STACKSIZE);
}

/* forward declarations */
static Obj buildFrame();
static void popFrame(), traceExit(), errorLocal(), codeArgCheck();

/* reset codeExec registers (refers to local variables in codeExec) */
#define CODE_RESET()\
{\
    sf = (stackFrame_t *)listPeek(ccState->alist);\
    if (frame = sf->frame)\
	fixed = DATA(frame, fixed, frameInst);\
    vec = sf->vec;\
}

/* export codeExec registers (refers to local variables in codeExec) */
#define CODE_EXPORT() (sf->vec = vec)
    
/* Execute byte codes from cvec within the scope defined by frame. */
Obj codeExec(cvec, frame)
     Obj cvec, frame;
{
    Obj arg0, arg1, *fixed;
    int tmp0, tmp1;
    bcode *vec;
    opcode op;
    stackFrame_t *sf;
    Callback_t cb;

    codeBegin(&cb, cvec, frame);
    CODE_RESET();

    for (;;) {
	op = codeNext(vec).op;
#ifdef DEBUG
	if (debugCode)
	    (void)fprintf(stderr, "executing opcode %s\n", opcodeName(op));
	opcodeInfo[ (int)op ].count++;
#endif

	switch (op) {

	case code_local:
	    tmp0 = codeNext(vec).num;
	    if ((arg0 = fixed[ tmp0 ]) == (Obj)NULL) errorLocal(tmp0, frame);
	    objLink(arg0);
	    stackPush(arg0, argStack);
	    break ;

	case code_unbind:
	    arg0 = codeNext(vec).obj;
	    ASSERT(CLASS(arg0) == Binding);
	    arg1 = objCar(arg0); /* symbol of binding */

	    do {
		if ((arg0 = objCdr(arg0)) == (Obj)NULL)
		    errorPrint(BadSymbol, "%O in %O", arg1, frame);
	    } while (CLASS(arg0) == Binding);
	    objLink(arg0);
	    stackPush(arg0, argStack);
	    break ;

	case code_access:
	    tmp0 = codeNext(vec).num; /* links */
	    tmp1 = codeNext(vec).num; /* offset */
	    arg0 = frame;
	    while (--tmp0 >= 0) {
		arg0 = DATA(arg0, parent, frameInst);
		ASSERT(arg0 != (Obj)NULL && CLASS(arg0) ==  Frame);
	    }
	    ASSERT(tmp1 < DATA(arg0, numfixed, frameInst));
	    arg1 = DATA(arg0, fixed, frameInst)[ tmp1 ];
	    if (arg1 == (Obj)NULL) errorLocal(tmp1, arg0);
	    objLink(arg1);
	    stackPush(arg1, argStack);
	    break ;

	case code_tail_call:
	    arg0 = stackPop(argStack);
	    tmp0 = codeNext(vec).num;
	tail_call:
	    if (CLASS(arg0) != User) goto normal_call;
	    codeArgCheck(arg0, tmp0, TRUE);

	    popFrame(frame, tmp0);
	    objUnlink(sf->proc);

	    /* replace previous stack frame */
	    if (sf->frame = frame = buildFrame(tmp0, arg0))
		fixed = DATA(frame, fixed, frameInst);
	    sf->proc = arg0;
	    vec = DATA(objCode(arg0), vec, codevecInst);
	    break ;

	case code_call:
	    arg0 = stackPop(argStack);
	    tmp0 = codeNext(vec).num;
	normal_call:
	    codeArgCheck(arg0, tmp0, FALSE);

	    if (CLASS(arg0) == User) {
		/* make new activation record */
		sf->vec = vec;
		sf = NEW_FRAME();
		listPush((Ptr)sf, ccState->alist);
		if (sf->frame = frame = buildFrame(tmp0, arg0))
		    fixed = DATA(frame, fixed, frameInst);
		sf->proc = arg0;
		vec = DATA(objCode(arg0), vec, codevecInst);
	    }
	    else if (CLASS(arg0) == Continuation) {
		arg1 = stackPop(argStack);
		--DATA(arg0, rc, basicInst);
		CODE_EXPORT();
		contResume(arg0);
		CODE_RESET();
		stackPush(arg1, argStack);
	    }
	    else {
		ASSERT(CLASS(arg0) == Prim);

		ccState->prim = arg0;
		ccState->argc = tmp0;
		arg1 = (*DATA(arg0, cfunc, primInst))
		    (stackAddr(argStack, tmp0), tmp0);
		ccState->prim = (Obj)NULL;
		objLink(arg1);
		objUnlink(arg0);
		while (--tmp0 >= 0)
		    objUnlink(stackPop(argStack));
		stackPush(arg1, argStack);

		if (TRACEP(arg0, TRACE_EXIT))
		    traceExit(arg0, arg1);
	    }
	    break ;
    
	case code_push:
	    arg0 = codeNext(vec).obj;
	    objLink(arg0);
	    stackPush(arg0, argStack);
	    break ;

	case code_pop:
	    objUnlink(stackPop(argStack));
	    break ;

	case code_call_return:
	    if (TRACEP(sf->proc, TRACE_EXIT))
		traceExit(sf->proc, stackPeek(argStack));

	    popFrame(frame, 1);
	    objUnlink(sf->proc);

	    /* pop new activation */
	    ASSERT(listLength(ccState->alist) > 1);
	    FREE_FRAME(listPop(ccState->alist));
	    sf = (stackFrame_t *)listPeek(ccState->alist);
	    if (frame = sf->frame)
		fixed = DATA(frame, fixed, frameInst);
	    vec = sf->vec;

	    break ;

	case code_branch:
	    tmp0 = codeNext(vec).num;
	    if ((arg0 = stackPop(argStack)) == FalseSymb)
		vec += tmp0;
	    objUnlink(arg0);
	    break ;

	case code_branch_save:
	    tmp0 = codeNext(vec).num;
	    if (stackPeek(argStack) == FalseSymb)
		vec += tmp0;
	    break ;

	case code_tbranch_save:
	    tmp0 = codeNext(vec).num;
	    if (stackPeek(argStack) != FalseSymb)
		vec += tmp0;
	    break ;

	case code_goto:
	    /* unconditional branch */
	    vec += codeNext(vec).num;
	    break ;

	case code_cfunc:
	    tmp0 = vec[1].num;
	    arg0 = (*vec[0].fcn)(stackAddr(argStack, tmp0), tmp0, frame);
	    vec += 2;
	    objLink(arg0);
	    while (--tmp0 >= 0)
		objUnlink(stackPop(argStack));
	    stackPush(arg0, argStack);
	    break ;
	case code_cfuncN:
	    tmp0 = vec[1].num;
	    tmp1 = vec[2].num;
	    arg0 = (*vec[0].fcn)(stackAddr(argStack, tmp0), tmp0,
				 frame, vec + 3);
	    vec += tmp1 + 3;
	    objLink(arg0);
	    while (--tmp0 >= 0)
		objUnlink(stackPop(argStack));
	    stackPush(arg0, argStack);
	    break ;

	case code_call_cc:
	    arg0 = stackPop(argStack);
	    CODE_EXPORT();
	    contStart();
	    stackPush(arg0, argStack);
	    CODE_RESET();
	    break ;

	    /* apply */
	case code_apply:
	case code_tail_apply:
	    arg1 = stackPop(argStack); /* arglist */
	    arg0 = stackPop(argStack); /* proc */
	    tmp0 = flattenList(arg1, arg0);
	    if (op == code_apply) goto normal_call;
	    else goto tail_call;

	case code_replace:
	    intCheck();
	    tmp0 = codeNext(vec).num;
	    fixed += tmp0;
	    while (--tmp0 >= 0) {
		objUnlink(*--fixed);
		*fixed = stackPop(argStack);
	    }
	    break ;

	case code_halt:
	    /* halt execution */
	    CODE_EXPORT();
	    if (arg0 = contContinue()) {
		codeEnd(&cb);
		return arg0;
	    }
	    CODE_RESET();
	    if (TRACEP(curCont, TRACE_EXIT))
		traceExit(curCont, stackPeek(argStack));
	    break ;
	}
    }
}

#ifdef DEBUG
void opcodeCount()
{
    int i;
    char *name;

    (void)puts("\nCALLS\tOPCODE");
    for (i = 0; i < sizeof (opcodeInfo) / sizeof (*opcodeInfo); i++) {
	if (name = opcodeInfo[i].name)
	    (void)printf("%5d\t%s\n", opcodeInfo[i].count, name);
    }
}
#endif /* DEBUG */

/* push items in list into the stack and return the number of items
 *
 * if list is not a proper list then an error occurs */
static int flattenList(list, proc)
     Obj list, proc;
{
    int args = 0;
    Obj obj, ptr = list;

    while (CLASS(ptr) == Pair) {
	objLink(obj = objCar(ptr));
	stackPush(obj, argStack);
	++args;
	ptr = objCdr(ptr);
    }
    if (ptr != NilSymb) {
	stackPush(list, argStack);
	stackPush(proc, argStack);
	errorPrint(Other, "Cannot apply %O to %O", list, proc);
    }
    objUnlink(list);
    return args;
}

/* find out how many activations for proc are on the stack */
static int activationLevel(proc, depthp)
     Obj proc;
     int *depthp;
{
    int acts = 0, depth = 0;
    stackFrame_t *sf;

    LIST_FOR_EACH(ccState->alist, item, {
	sf = (stackFrame_t *)item;

	if (sf->proc) {
	    ASSERT(objIsClass(sf->proc, Proc));
	    if (sf->proc == proc) acts++;
	    depth++;
	}
    });

    if (depthp) *depthp = depth;
    return acts;
}

/* Print trace-entry info for proc called with argc args. */
static void traceEntry(proc, argc, tail)
     Obj proc;
     int argc;
     Boolean tail;
{
    Obj *argv;
    int acts, depth;

    acts = activationLevel(proc, &depth);
    if (!tail) {
	acts++; /* include traced proc */
	depth++;
    }
    else if (acts == 0) acts = 1;

    (void)printf("%*d%c", depth, acts, (tail ? '*' : ' '));

    argv = stackAddr(argStack, argc);
    if (DATA(proc, name, procInst))
	(void)printf("(%s", DATA(proc, name, procInst));
    else objPrintf("(%O", proc);
    while (--argc >= 0) objPrintf(" %O", *argv++);
    (void)puts(")");
}

/* Print trace-exit info for proc returning with res. */
static void traceExit(proc, res)
     Obj proc, res;
{
    int acts, depth;

    acts = activationLevel(proc, &depth);
    if (CLASS(proc) != User) {
	depth++;
	acts++;
    }
    (void)printf("%*d ", depth, acts);

    if (DATA(proc, name, procInst))
	(void)printf("%s", DATA(proc, name, procInst));
    else objPrint(proc, stdout);
    objPrintf(" returns %O\n", res);
}

/* do various checking things for proc */
static void codeArgCheck(proc, num, tail)
     Obj proc;
     int num;
     Boolean tail;
{
    int numargs = DATA(proc, numargs, procInst);

    if (!objIsClass(proc, Proc)) {
	stackPush(proc, argStack); /* push proc on stack so it gets GCed */
	errorPrint(BadProc, "%O", proc);
    }
    else if (checkCond(proc, OPTARG)) {
	if (numargs > num) {
	    stackPush(proc, argStack);
	    errorPrint(BadArgs, "to %O (expects at least %d)", proc, numargs);
	}
    }
    else {
	if (numargs != num) {
	    stackPush(proc, argStack);
	    errorPrint(BadArgs, "to %O (expects %d)", proc, numargs);
	}
    }

    if (TRACEP(proc, TRACE_ENTRY))
	traceEntry(proc, num, tail);
    intCheck();
}

/* remove locals of frame from stack */
static void popFrame(frame, args)
     Obj frame;
     int args;
{
    if (checkCond(frame, ZAP)) {
	objUnlink(frame); /* no locals on the stack */
    }
    else {
	Obj *fixed, *base;
	int numfixed;

	fixed = DATA(frame, fixed, frameInst);
	numfixed = DATA(frame, numfixed, frameInst);
	if (DATA(frame, rc, basicInst) > 1)
	    saveFrame(frame, FALSE);
	objUnlink(frame);
	if (numfixed > 0) {
	    base = stackAddr(argStack, args);
	    while (--args >= 0)
		*fixed++ = *base++;
	    stackAdj(argStack, numfixed);
	}
    }
}
     
/* return a frame for proc containing the args from stack */
static Obj buildFrame(argc, proc)
     int argc;
     Obj proc;
{
    int nargs, nopt, nint, nf;
    Obj *fixed, frame;

    nargs = DATA(proc, numargs, procInst); /* normal args */
    nopt = argc - nargs; /* optional args */
    nf = DATA(proc, numfixed, userInst); /* length of fixed vec */
    nint = nf - nargs; /* internal defs */

    frame = objFrame(proc);
    objLink(frame);
    fixed = DATA(frame, fixed, frameInst) = stackAddr(argStack, argc);

    if (nint > 0) {
	if (checkCond(proc, OPTARG)) {
	    Obj optarg, pair;

	    --nint; /* account for optarg slot */
	    fixed = stackPtr(argStack);

	    /* build the optional arguments into a list */
	    optarg = NilSymb;
	    while (--nopt >= 0) {
		pair = gcNew(Pair);
		DATA(pair, car, pairInst) = *--fixed;
		objSetCdr(pair, optarg);
		optarg = pair;
	    }
	    objLink(optarg);
	    *fixed++ = optarg;
	}
	else fixed += nargs;

	/* fill remaining internal slots with undefined value */
	while (--nint >= 0) *fixed++ = (Obj)NULL;

	stackAdj(argStack, argc - nf);
    }

    return frame;
}

/* local parameter in fixed[offset] is undefined */
static void errorLocal(offset, frame)
     int offset;
     Obj frame;
{
    Obj fp;

    ASSERT(offset < DATA(frame, numfixed, frameInst));
    fp = DATA(frame, formals, frameInst);
    while (--offset > 0) fp = objCdr(fp);
    errorPrint(BadSymbol, "%O in %O", objCar(fp), frame);
}
