; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SCHEME-INTERNAL; -*-
; File core.lisp / Copyright (c) 1989 Jonathan Rees / See file COPYING

;;;; Pseudoscheme runtime system

(lisp:in-package "SCHEME-INTERNAL")

(import '(scheme-hacks:make-photon
	  scheme-hacks:environment-marker
	  scheme-hacks:qualified-symbol?
	  scheme-hacks:intern-renaming-perhaps
	  scheme-hacks:find-symbol-renaming-perhaps
	  scheme-hacks:lisp-package))

(defvar scheme-package scheme-hacks:scheme-package)
(defvar scheme-readtable scheme-hacks:scheme-readtable)

(defvar *target-context* nil)
(defvar *unique-id* 0)

(defun generate-unique-id () (incf *unique-id*))

; Miscellaneous objects

(defvar unspecified (make-photon "#{Unspecified}"))
(defvar unassigned  (make-photon "#{Unassigned}"))

(defvar eof-object
  (if (find-package "PSEUDOSCHEME")
      ;; Temporary hack for coexistence with old versions of Pseudoscheme!
      (intern "EOF-OBJECT" (find-package "PSEUDOSCHEME"))
      (make-photon "#{EOF-object}")))

;(defvar true-photon  (make-photon "#T"))
;(defvar false-photon (make-photon "#F"))

; PROCEDURE?

(defparameter closures-might-be-conses?
  (or (consp (eval '#'(lambda (x) x)))	;VAX LISP 2.1
      (consp (let ((g (gensym)))
	       (eval `(progn (defun ,g () 0) #',g)))) ;Symbolics
      (consp (compile nil '(lambda (x) x))) ;just for kicks
      (consp (funcall (compile nil '(lambda (x) ;VAX LISP 2.2
				      #'(lambda () (prog1 x (incf x)))))
		      0))))

(defun procedure? (obj)
  (and (functionp obj)
       (not (symbolp obj))
       (or (not (consp obj))
	   closures-might-be-conses?)))

; Global environments are implemented as packages, for now.

(defstruct (environment (:constructor construct-environment (id package))
			(:predicate environment?)
			(:print-function print-environment)
			(:copier nil))
  id
  package
  (uid (generate-unique-id)))

(defun make-environment (id &optional (package (find-environment-package id)))
  (let* ((env (construct-environment id package))
	 (marker (environment-marker package)))
    (proclaim `(special ,marker))
    (setf (symbol-value marker) env)
    env))

(defun print-environment (environment stream escape?)
  (declare (ignore escape?))
  (let ((*package* scheme-package))
    (format stream "#{Environment ~S.~S}"
	    (environment-id environment)
	    (environment-uid environment))))

(defun find-environment-package (id)
  (let* ((pkg-name (if (stringp id) id (symbol-name id)))
	 (probe (find-package pkg-name)))
    (cond (probe
	   (warn "Using existing package for environment ~S."
		 id)
	   probe)
	  (t
	   (make-package pkg-name :use (list lisp-package))))))
	 
(defun package-environment (package)
  (symbol-value (scheme-hacks:environment-marker package)))

(defun environment-define! (env name val)
  (when (qualified-symbol? name)
    (error "ENVIRONMENT-DEFINE! of non-Scheme symbol ~S" name))
  (let* ((pkg (environment-package env))
	 (sym (intern-renaming-perhaps (symbol-name name) pkg)))
    (if (eq (symbol-package sym) pkg)
	(progn (setf (symbol-value sym) val)
	       (set-function-from-value sym name))
	(error "Cannot define ~S in ~S, it is imported from ~S"
	       name
	       env
	       (package-environment (symbol-package sym))))))

(defun environment-lookup (env name)
  (when (qualified-symbol? name)
    (cerror "proceed as if ~S were a Scheme symbol"
	    "ENVIRONMENT-LOOKUP of non-Scheme symbol ~S" name))
  (let ((sym (find-symbol-renaming-perhaps (symbol-name name)
					   (environment-package env))))
    (if (and sym (get sym 'defined))
	sym
	nil)))

(defun environment-ref (env name)
  (let ((sym (environment-lookup env name)))
    (if sym
	(symbol-value sym)
	(error "Variable ~S is unbound in ~S" name env))))

;+++ Danger, ENVIRONMENT-SET! can lose if name is of the form *foo*.
;+++ Fix later.

(defun environment-set! (env name val)
  (let ((sym (environment-lookup env name)))
    (if sym
	(set!-aux env name val sym)
	(error "Variable ~S is unbound in ~S" name env))))

; Auxiliary for SET!

; The "CL-sym" argument is redundant with env and name, but is
; provided for the sake of speed (avoids a run-time package lookup).

(defun set!-aux (env name value CL-sym)
  (declare (ignore env))     ;Vestigial
  (case (get CL-sym 'defined)
    ((:assignable))
    ((:not-assignable)
     (cerror "Assign it anyhow"
	     "Variable ~S isn't supposed to be SET!"
	     (or name CL-sym)))
    ((nil)
     (unless (qualified-symbol? name)   ;(set! foo:bar ...)
       (warn "SET! of undefined variable ~S" (or name CL-sym)))))
  (setf (symbol-value CL-sym) value)
  (if (procedure? value)
      (setf (symbol-function CL-sym) value)
      (fmakunbound CL-sym))
  unspecified)

; A "context" is a structure in which values are stored (i.e., an
; environment), together with a meta-environment that gives
; information to the compiler.  For now, macrologies are also
; included, although that doesn't quite feel right.

; We cannot create the objects used for translating the translator
; until the translator exists.  The context will get filled in later.

(defstruct (context (:constructor
		      construct-context (id env &optional macrologies meta-env package))
		    (:predicate context?)
		    (:copier nil)
		    (:print-function print-context))
  id
  package				;for use by TRANSLATE-FILE
  macrologies
  meta-env
  env
  (uid (generate-unique-id)))

(defun make-context (id env &optional macrologies
				      meta-env
				      (package (environment-package env)))
  (construct-context id env macrologies meta-env package))

(defun print-context (context stream escape?)
  (declare (ignore escape?))
  (let ((*package* scheme-package))
    (format stream "#{Context ~S.~S}" (context-id context) (context-uid context))))

(defun initialize-context! (context macrologies meta-env &optional package)
  (setf (context-macrologies context) macrologies)
  (setf (context-meta-env    context) meta-env)
  (when package
    (setf (context-package     context) package)))

; ----- Prelude on all translated files

(defmacro begin-translated-file ()
  `(progn 'compile    ;???
	  (eval-when (eval compile)
	    (begin-compile-of-translated-file))
	  (begin-load-of-translated-file)))

(defparameter cl-readtable (copy-readtable nil))

(defun begin-compile-of-translated-file ()
  (declare (special *target-context*))
  ;; This is called at EVAL COMPILE times.  The package will initially be
  ;; the Scheme package, and the readtable will be the roadblock
  ;; readtable.
  (setq *package* (context-package *target-context*))
  (setq *readtable* cl-readtable))

(defun begin-load-of-translated-file ()
  ;; This is called at EVAL LOAD times.
  ;; Do nothing, for now, but this hook may come in handy later.
  )

; Auxiliaries for top-level DEFINE

(defun set-value-from-function (CL-sym &optional name) ;Follows DEFUN
  (let ((value (symbol-function CL-sym)))
;  [I can't remember why this is commented out!]
;    (when (eq (get CL-sym 'define) :not-assignable)
;      (really-set-function (symbol-value sym)) ;Revert!
;      (cerror "Define it anyhow"
;	      "Variable ~S isn't supposed to be DEFINEd." name)
;      (really-set-function value))
    (setf (symbol-value CL-sym) value)
    (after-define CL-sym name)))

(defun really-set-function (CL-sym value)
  (cond ((procedure? value)
	 (setf (symbol-function CL-sym) value))
	(t
	 (fmakunbound CL-sym))))

(defun set-function-from-value (CL-sym &optional name) ;Follows SETQ
  (let ((value (symbol-value CL-sym)))
    (really-set-function CL-sym value)
    #+Symbolics
    (scl:record-source-file-name CL-sym (if (procedure? value) 'defun 'defvar))
    (after-define CL-sym name)))

; Follows (SETQ *FOO* ...)

(defun set-forwarding-function (CL-sym &optional name)
  (setf (symbol-function CL-sym)
	#'(lambda (&rest args)
	    (apply (symbol-value CL-sym) args)))
  (after-define CL-sym name))

(defun after-define (CL-sym name)
  (declare (ignore value))		;vestigial
  (setf (get CL-sym 'defined) t)
  (when name
    (make-photon #'(lambda (port)
		     (let ((*package* scheme-package))
		       (format port "~S defined." name))))))

; EQUAL?

; Differs from Common Lisp EQUAL in that it descends into vectors.
; This is here instead of in rts.lisp because it's an auxiliary for
; open-coding MEMBER and ASSOC, and the rule is that all auxiliaries
; are in the SCHI package (not REVISED^3-SCHEME).

(defun equal? (obj1 obj2)
  (cond ((eql obj1 obj2) t)
        ((consp obj1)			;pair?
         (and (consp obj2)
	      (equal? (car obj1) (car obj2))
	      (equal? (cdr obj1) (cdr obj2))))
	((simple-string-p obj1)		;string?
	 (and (simple-string-p obj2)
	      (string= (the simple-string obj1)
		       (the simple-string obj2))))
	((simple-vector-p obj1)
	 (and (simple-vector-p obj2)
	      (let ((z (length (the simple-vector obj1))))
		(declare (fixnum z))
		(and (= z (length (the simple-vector obj2)))
		     (do ((i 0 (+ i 1)))
			 ((= i z) t)
		       (declare (fixnum i))
		       (when (not (equal? (aref (the simple-vector obj1) i)
					  (aref (the simple-vector obj2) i)))
			 (return nil)))))))
        (t nil)))
