/* Scheme implementation intended for JACAL.
   Copyright (C) 1990, 1991 Aubrey Jaffer.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

The author can be reached at jaffer@ai.mit.edu or
Aubrey Jaffer, 84 Pleasant St., Wakefield MA 01880
*/

#include <ctype.h>
#include "scm.h"

unsigned char upcase[char_code_limit];
unsigned char downcase[char_code_limit];
unsigned char lowers[]="abcdefghijklmnopqrstuvwxyz";
unsigned char uppers[]="ABCDEFGHIJKLMNOPQRSTUVWXYZ";
void init_tables()
{
	int i;
	for(i=0;i<char_code_limit;i++) upcase[i]=downcase[i]=i;
	for(i=0;i<sizeof(lowers);i++) {
		upcase[lowers[i]]=uppers[i];
		downcase[uppers[i]]=lowers[i];
	}
}

char	s_list[]="list", s_vector[]="vector", s_string[]="string";
char	OUTOFRANGE[]="Argument out of range to ",
	OVFLOW[]="numerical overflow in ";

static char	s_setcar[]="set-car!", s_setcdr[]="set-cdr!";
static char	s_length[]="length", s_append[]="append",
	s_reverse[]="reverse", s_list_ref[]="list-ref";
static char	s_memq[]="memq",s_member[]="member",
	s_assq[]="assq",s_assoc[]="assoc";
static char	s_symbol2string[]="symbol->string",
	s_str2symbol[]="string->symbol";
static char	s_inexactp[]="inexact?";
#define s_exactp (s_inexactp+2)
static char	s_zerop[]="zero?",
	s_positivep[]="positive?",s_negativep[]="negative?";
static char	s_oddp[]="odd?",s_evenp[]="even?";
static char	s_eqp[]="=",s_lessp[]="<",s_grp[]=">",
	s_lesseqp[]="<=",s_greqp[]=">=";
static char	s_abs[]="abs",
	s_quotient[]="quotient",s_remainder[]="remainder",s_modulo[]="modulo";
static char	s_gcd[]="gcd",s_lcm[]="lcm",
	s_number2string[]="number->string",s_str2number[]="string->number";

static char	s_ch_lessp[]="char<?",
	s_ch_leqp[]="char<=?",
	s_ci_eq[]="char-ci=?",
	s_ci_lessp[]="char-ci<?",
	s_ci_leqp[]="char-ci<=?";
static char	s_ch_alphap[]="char-alphabetic?",
	s_ch_nump[]="char-numeric?",
	s_ch_whitep[]="char-whitespace?",
	s_ch_upperp[]="char-upper-case?",
	s_ch_lowerp[]="char-lower-case?";
static char	s_char2int[]="char->integer",s_int2char[]="integer->char",
	s_ch_upcase[]="char-upcase",s_ch_downcase[]="char-downcase";

static char	s_st_length[]="string-length", s_make_string[]="make-string",
	s_st_ref[]="string-ref",s_st_set[]="string-set!";
static char	s_st_equal[]="string=?",s_stci_equal[]="string-ci=?",
	s_st_lessp[]="string<?",s_stci_lessp[]="string-ci<?";
static char	s_substring[]="substring",s_st_append[]="string-append";

extern char	s_make_vector[];
static char	s_ve_length[]="vector-length",
	s_ve_ref[]="vector-ref",s_ve_set[]="vector-set!";

SCM lnot(x)
SCM x;
{
	return FALSEP(x) ? BOOL_T : BOOL_F;
}
SCM booleanp(obj)
SCM obj;
{
	if (obj == BOOL_F) return BOOL_T;
	if (obj == BOOL_T) return BOOL_T;
	return BOOL_F;
}
SCM eq(x,y)
SCM x,y;
{
	if (x == y) return BOOL_T;
	else return BOOL_F;
}

SCM equal(), st_equal();

SCM vector_equal(x,y)
SCM x,y;
{
	long i;
	for(i=LENGTH(x)-1;i>=0;i--)
		if FALSEP(equal(VELTS(x)[i],VELTS(y)[i])) return BOOL_F;
	return BOOL_T;
}

SCM equal(x,y)
SCM x,y;
{
tailrecurse:
	if (x == y) return BOOL_T;
	if IMP(x) return BOOL_F;
	if IMP(y) return BOOL_F;
	if (CONSP(x) && CONSP(y)) {
		if (BOOL_F == equal(CAR(x),CAR(y))) return BOOL_F;
		x = CDR(x);
		y = CDR(y);
		goto tailrecurse;
	}
	if (TYP7(x) != TYP7(y)) return BOOL_F;
	if STRINGP(x) return (LENGTH(x) == LENGTH(y)) ? st_equal(x,y) : BOOL_F;
	if VECTORP(x) return (LENGTH(x) == LENGTH(y)) ? vector_equal(x,y) : BOOL_F;
	return BOOL_F;
}

SCM consp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return CONSP(x) ? BOOL_T : BOOL_F;
}
SCM setcar(pair,value)
SCM pair, value;
{
	ASSERT(NIMP(pair) && CONSP(pair),pair,ARG1,s_setcar);
	CAR(pair) = value;
	return UNSPECIFIED;
}
SCM setcdr(pair,value)
SCM pair, value;
{
	ASSERT(NIMP(pair) && CONSP(pair),pair,ARG1,s_setcdr);
	CDR(pair) = value;
	return UNSPECIFIED;
}
SCM nullp(x)
SCM x;
{
	return NULLP(x) ? BOOL_T : BOOL_F;
}
long ilength(sx)
SCM sx;
{
	register long i=0;
	register SCM x=sx;
	do {
		if IMP(x) return NULLP(x) ? i : -1;
		if NCONSP(x) return -1;
		x = CDR(x);
		i++;
		if IMP(x) return NULLP(x) ? i : -1;
		if NCONSP(x) return -1;
		x = CDR(x);
		i++;
		sx=CDR(sx);
	}
	while (x != sx);
	return -1;
}
SCM listp(x)
SCM x;
{
	if (ilength(x)<0) return BOOL_F;
	else return BOOL_T;
}
SCM list(objs)
SCM objs;
{
	return objs;
}
SCM length(x)
SCM x;
{
	SCM i=MAKINUM(ilength(x));
	ASSERT(i>=INUM0,i,ARG1,s_length);
	return i;
}
SCM append(args)
SCM args;
{
	SCM res = EOL;
	SCM *lloc = &res, arg;
	if IMP(args) {
		ASSERT(NULLP(args),args,ARG1,s_append);
		return res;
		}
	ASSERT(CONSP(args),args,ARG1,s_append);
	while (1) {
		arg = CAR(args);
		args = CDR(args);
		if IMP(args) {
			*lloc = arg;
			ASSERT(NULLP(args),args,ARG1,s_append);
			return res;
		}
		ASSERT(CONSP(args),args,ARG1,s_append);
		for(;NIMP(arg);arg = CDR(arg)) {
			ASSERT(CONSP(arg),args,ARG1,s_append);
			*lloc = cons(CAR(arg),EOL);
			lloc = &CDR(*lloc);
		}
	}
}
SCM reverse(lst)
SCM lst;
{
	SCM res = EOL;
	SCM p = lst;
	for(;NIMP(p);p = CDR(p)) {
		ASSERT(CONSP(p),lst,ARG1,s_reverse);
		res = cons(CAR(p),res);
	}
	ASSERT(NULLP(p),lst,ARG1,s_reverse);
	return res;
}
SCM list_ref(lst,k)
SCM lst, k;
{
	register long i;
	ASSERT(INUMP(k),k,ARG2,s_list_ref);
	i = INUM(k);
	ASSERT(i >= 0,k,ARG2,s_list_ref);
	while (i-- > 0) {
		ASSERT(NIMP(lst) && CONSP(lst),lst,ARG1,s_list_ref);
		lst=CDR(lst);
	}
	ASSERT(NIMP(lst) && CONSP(lst),lst,ARG1,s_list_ref);
	return CAR(lst);
}
SCM memq(x,lst)
SCM x,lst;
{
	for(;NIMP(lst);lst = CDR(lst)) {
		ASSERT(CONSP(lst),lst,ARG2,s_memq);
		if (CAR(lst) == x) return lst;
	}
	ASSERT(NULLP(lst),lst,ARG2,s_memq);
	return BOOL_F;
}
SCM member(x,lst)
SCM x,lst;
{
	for(;NIMP(lst);lst = CDR(lst)) {
		ASSERT(CONSP(lst),lst,ARG2,s_member);
		if (equal(CAR(lst),x) == BOOL_T) return lst;
	}
	ASSERT(NULLP(lst),lst,ARG2,s_member);
	return BOOL_F;
}
SCM assq(x,alist)
SCM x,alist;
{
	SCM tmp;
	for(;NIMP(alist);alist=CDR(alist)) {
		ASSERT(CONSP(alist),alist,ARG2,s_assq);
		tmp = CAR(alist);
		ASSERT(CONSP(tmp),alist,ARG2,s_assq);
		if (CAR(tmp) == x) return tmp;
	}
	ASSERT(NULLP(alist),alist,ARG2,s_assq);
	return BOOL_F;
}
SCM assoc(x,alist)
SCM x,alist;
{
	SCM tmp;
	for(;NIMP(alist);alist=CDR(alist)) {
		ASSERT(CONSP(alist),alist,ARG2,s_assoc);
		tmp = CAR(alist);
		ASSERT(CONSP(tmp),alist,ARG2,s_assoc);
		if (equal(CAR(tmp),x) == BOOL_T) return tmp;
	}
	ASSERT(NULLP(alist),alist,ARG2,s_assoc);
	return BOOL_F;
}

SCM symbolp(x)
SCM x;
{
	if ISYMP(x) return BOOL_T;
	if IMP(x) return BOOL_F;
	return SYMBOLP(x) ? BOOL_T : BOOL_F;
}
SCM symbol2string(s)
SCM s;
{
	if ISYMP(s) return makfromstr(ISYMCHARS(s), strlen(ISYMCHARS(s)));
	ASSERT(NIMP(s) && SYMBOLP(s),s,ARG1,s_symbol2string);
	return NAMESTR(s);
}
SCM string2symbol(s)
SCM s;
{
	ASSERT(NIMP(s) && STRINGP(s),s,ARG1,s_str2symbol);
	return intern(CHARS(s),LENGTH(s));
}

SCM numberp(x)
SCM x;
{
	if INUMP(x) return BOOL_T;
#ifdef FLOATS
	if (NIMP(x) && INEXP(x)) return BOOL_T;
#endif
	return BOOL_F;
}
#ifdef FLOATS
SCM realp(x)
     SCM x;
{
  if INUMP(x) return BOOL_T;
  if IMP(x) return BOOL_F;
  if REALP(x) return BOOL_T;
  return BOOL_F;
}
SCM intp(x)
     SCM x;
{
  double r;
  if INUMP(x) return BOOL_T;
  if IMP(x) return BOOL_F;
  if (!INEXP(x)) return BOOL_F;
  if CPLXP(x) return BOOL_F;
  r = REALPART(x);
  if (r == floor(r)) return BOOL_T;
  return BOOL_F;
}
#else /* FLOATS */
SCM numident(x)
SCM x;
{
	ASSERT(INUMP(x),x,"conversion to integer expected number","");
	return x;
}
#endif
SCM exactp(x)
SCM x;
{
	if INUMP(x) return BOOL_T;
	return BOOL_F;
}
SCM inexactp(x)
SCM x;
{
#ifdef FLOATS
	if (NIMP(x) && INEXP(x)) return BOOL_T;
#endif
	return BOOL_F;
}
SCM eqp(x,y,args)
SCM x,y,args;
{
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && INEXP(x),x,ARG1,s_eqp);
    while (1) {
      ASSERT(NIMP(y) && INEXP(y),y,ARG2,s_eqp);
      if (REALPART(x) != REALPART(y)) return BOOL_F;
      if CPLXP(x)
	if CPLXP(y)
	  if (IMAG(x) != IMAG(y)) return BOOL_F;
	  else;
	else return BOOL_F;
      else if CPLXP(y) return BOOL_F;
      if NULLP(args) return BOOL_T;
      y = CAR(args);
      args = CDR(args);
    }
  }
#else    
  ASSERT(INUMP(x),x,ARG1,s_eqp);
#endif
  while (1) {
    ASSERT(INUMP(y),y,ARG2,s_eqp);
    if ((long)x != (long)y) return BOOL_F;
    if NULLP(args) return BOOL_T;
    y = CAR(args);
    args = CDR(args);
  }
}
SCM lessp(x,y,args)
SCM x,y,args;
{
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && REALP(x),x,ARG1,s_lessp);
    while (1) {
      ASSERT(NIMP(y) && REALP(y),y,ARG2,s_lessp);
      if (REALPART(x) >= REALPART(y)) return BOOL_F;
      if NULLP(args) return BOOL_T;
      y = CAR(args);
      args = CDR(args);
    }
  }
#else    
  ASSERT(INUMP(x),x,ARG1,s_lessp);
#endif
  ASSERT(INUMP(x),x,ARG1,s_lessp);
  while (1) {
    ASSERT(INUMP(y),y,ARG2,s_lessp);
    if ((long)x >= (long)y) return BOOL_F;
    if NULLP(args) return BOOL_T;
    x = y;
    y = CAR(args);
    args = CDR(args);
  }
}
SCM greaterp(x,y,args)
SCM x,y,args;
{
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && REALP(x),x,ARG1,s_grp);
    while (1) {
      ASSERT(NIMP(y) && REALP(y),y,ARG2,s_grp);
      if (REALPART(x) <= REALPART(y)) return BOOL_F;
      if NULLP(args) return BOOL_T;
      y = CAR(args);
      args = CDR(args);
    }
  }
#else    
  ASSERT(INUMP(x),x,ARG1,s_grp);
#endif
  ASSERT(INUMP(x),x,ARG1,s_grp);
  while (1) {
    ASSERT(INUMP(y),y,ARG2,s_grp);
    if ((long)x <= (long)y) return BOOL_F;
    if NULLP(args) return BOOL_T;
    x = y;
    y = CAR(args);
    args = CDR(args);
  }
}
SCM lesseqp(x,y,args)
SCM x,y,args;
{
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && REALP(x),x,ARG1,s_lesseqp);
    while (1) {
      ASSERT(NIMP(y) && REALP(y),y,ARG2,s_lesseqp);
      if (REALPART(x) > REALPART(y)) return BOOL_F;
      if NULLP(args) return BOOL_T;
      y = CAR(args);
      args = CDR(args);
    }
  }
#else    
  ASSERT(INUMP(x),x,ARG1,s_lesseqp);
#endif
  ASSERT(INUMP(x),x,ARG1,s_lesseqp);
  while (1) {
    ASSERT(INUMP(y),y,ARG2,s_lesseqp);
    if ((long)x > (long)y) return BOOL_F;
    if NULLP(args) return BOOL_T;
    x = y;
    y = CAR(args);
    args = CDR(args);
  }
}
SCM greatereqp(x,y,args)
SCM x,y,args;
{
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && REALP(x),x,ARG1,s_greqp);
    while (1) {
      ASSERT(NIMP(y) && REALP(y),y,ARG2,s_greqp);
      if (REALPART(x) < REALPART(y)) return BOOL_F;
      if NULLP(args) return BOOL_T;
      y = CAR(args);
      args = CDR(args);
    }
  }
#else    
  ASSERT(INUMP(x),x,ARG1,s_greqp);
#endif
  ASSERT(INUMP(x),x,ARG1,s_greqp);
  while (1) {
    ASSERT(INUMP(y),y,ARG2,s_greqp);
    if ((long)x < (long)y) return BOOL_F;
    if NULLP(args) return BOOL_T;
    x = y;
    y = CAR(args);
    args = CDR(args);
  }
}

SCM zerop(z)
SCM z;
{
	if (z==INUM0) return BOOL_T;
#ifdef FLOATS
	if (NIMP(z) && INEXP(z))
	  if (REALPART(z)==0.0) return BOOL_T;
	  else return BOOL_F;
#endif
	ASSERT(INUMP(z),z,ARG1,s_zerop);
	return BOOL_F;
}
SCM positivep(x)
SCM x;
{
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && REALP(x),x,ARG1,s_positivep);
    return (REALPART(x)>0) ? BOOL_T : BOOL_F;    
  }
#else
  ASSERT(INUMP(x),x,ARG1,s_positivep);
#endif
  return (INUM(x)>0) ? BOOL_T : BOOL_F;
}
SCM negativep(x)
SCM x;
{
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && REALP(x),x,ARG1,s_negativep);
    return (REALPART(x)<0) ? BOOL_T : BOOL_F;    
  }
#else
  ASSERT(INUMP(x),x,ARG1,s_negativep);
#endif
  return (INUM(x)<0) ? BOOL_T : BOOL_F;
}
SCM oddp(n)
SCM n;
{
	ASSERT(INUMP(n),n,ARG1,s_oddp);
	return (4 & (int)n) ? BOOL_T : BOOL_F;
}
SCM evenp(n)
SCM n;
{
	ASSERT(INUMP(n),n,ARG1,s_evenp);
	return (4 & (int)n) ? BOOL_F : BOOL_T;
}
SCM absval(x)
SCM x;
{
	SCM res;
	register long z = INUM(x);
	ASSERT(INUMP(x),x,ARG1,s_abs);
	if (z<0) z = -z;
	res = MAKINUM(z);
	ASSERT(res>>2 == z,res,OVFLOW,s_abs);
	return res;
}

SCM quotient(x,y)
SCM x,y;
{
	SCM res;
	register long z;
	ASSERT(INUMP(x),x,ARG1,s_quotient);
	ASSERT(INUMP(y),y,ARG2,s_quotient);
	z = INUM(y);
	ASSERT(z,y,OVFLOW,s_quotient);
	z = INUM(x)/z;
#ifdef BADIVSGNS
	{
		long t = INUM(x)%INUM(y);
		if (t == 0) ;
		else if (t < 0)
			if (x < 0) ;
			else z--;
		else if (x < 0) z++;
	}
#endif
	res = MAKINUM(z);
	ASSERT(INUM(res) == z,res,OVFLOW,s_quotient);
	return res;
}
SCM lremainder(x,y)
SCM x,y;
{
	register long z;
	ASSERT(INUMP(x),x,ARG1,s_remainder);
	ASSERT(INUMP(y),y,ARG2,s_remainder);
	z = INUM(y);
	ASSERT(z,y,OVFLOW,s_remainder);
	z = INUM(x)%z;
#ifdef BADIVSGNS
	if (z == 0) ;
	else if (z < 0)
		if (x < 0) ;
		else z += INUM(y);
	else if (x < 0) z -= INUM(y);
#endif
	return MAKINUM(z);
}
SCM modulo(n1,n2)
SCM n1,n2;
{
	register long y,z;
	ASSERT(INUMP(n1),n1,ARG1,s_modulo);
	ASSERT(INUMP(n2),n2,ARG2,s_modulo);
	y = INUM(n2);
	ASSERT(y,n2,OVFLOW,s_modulo);
	z = INUM(n1)%y;
	return MAKINUM(y<0 ? (z>0) ? z+y : z
			   : (z<0) ? z+y : z);
}
SCM lgcd(n1,n2)
SCM n1,n2;
{
	register long u,v,k,t;
	if UNBNDP(n2) return UNBNDP(n1) ? INUM0 : n1;
	ASSERT(INUMP(n1),n1,ARG1,s_gcd);
	ASSERT(INUMP(n2),n2,ARG2,s_gcd);
	u = INUM(n1);
	if (u<0) u = -u;
	v = INUM(n2);
	if (v<0) v = -v;
	else if (0 == v) return MAKINUM(u);
	if (0 == u) return MAKINUM(v);
	for (k = 1;!(1 & ((int)u|(int)v));k <<= 1,u >>= 1,v >>= 1);
	if (1 & (int)u) t = -v;
	else {
		t = u;
b3:
		t = SRS(t,1);
	}
	if (!(1 & (int)t)) goto b3;
	if (t>0) u = t;
	else v = -t;
	if (t = u-v) goto b3;
	u = u*k;
	v = MAKINUM(u);
	ASSERT((v>>2) == u,v,OVFLOW,s_gcd);
	return v;
}
SCM llcm(n1,n2)
SCM n1,n2;
{
	SCM res;
	register long q,z;
	long x = INUM(n1);
	if UNBNDP(n2) {
		n2 = MAKINUM(1L);
		if UNBNDP(n1) return n2;
	}
	q = INUM(lgcd(n1,n2));
	if ((x == 0) || (n2 == INUM0)) return INUM0;
	q = INUM(n2)/q;
	z = x*q;
	ASSERT(z/q == x,n1,OVFLOW,s_lcm);
	if (z < 0) z = -z;
	res = MAKINUM(z);
	ASSERT((res>>2) == z,res,OVFLOW,s_lcm);
	return res;
}

extern char num_buf[];
size_t iint2str(num,rad,p)
     long num;
     int rad;
     char *p;
{
  size_t j;
  register int i=1,d;
  register long n = num;
  if (n < 0) {n = -n; i++;}
  for (n /= rad;n > 0;n /= rad) i++;
  j = i;
  n = num;
  if (n < 0) {n = -n; *p++ = '-'; i--;}
  while (i--) {
    d = n % rad;
    n /= rad;
    p[i] = d + ((d < 10) ? '0' : 'a' - 10);
  }
  return j;
}
SCM number2string(x,radix)
SCM x,radix;
{
  if UNBNDP(radix) radix=MAKINUM(10L);
  else ASSERT(INUMP(radix),radix,ARG2,s_number2string);
#ifdef FLOATS
  if NINUMP(x) {
    ASSERT(NIMP(x) && INEXP(x),x,ARG1,s_number2string);
    return makfromstr(num_buf,iflo2str(x,num_buf));
  }
#else
  ASSERT(INUMP(x),x,ARG1,s_number2string);
#endif
  return makfromstr(num_buf,iint2str(INUM(x),(int)INUM(radix),num_buf));
}

SCM istr2int(str,len,radix)
char *str;
long len;
long radix;
{
  SCM res;
  register char *p = str;
  register int c,rad = radix,i = 0;
  register long n = 0;
  if ((len) > 1) switch (p[0]) {
  case '-': case '+': i++;
  default:;}
  while(i < len) switch(c = p[i++]) {
  case DIGITS:
    c = c - '0';
    goto accumulate;
  case 'A': case 'B': case 'C': case 'D': case 'E': case 'F':
    c = c-'A'+10;
    goto accumulate;
  case 'a': case 'b': case 'c': case 'd': case 'e': case 'f':
    c = c-'a'+10;
  accumulate:
    if ((c<0)||(c>=rad)) return BOOL_F;
    res = n;
    n = n * rad - c;
    if ((n + c)/rad != res) return BOOL_F;
    continue;
  default: return BOOL_F;}
  if (p[0]!='-') n = -n;
  res = MAKINUM(n);
  if (INUM(res) != n) return BOOL_F;
  return res;
}
SCM istring2number(str,len,radix)
char *str;
long len;
long radix;
{
  char ex = 0;
  int i = 0;
  switch ((int)len) {
  case 0: return BOOL_F;
  case 1: switch (str[0]) {
  case '-': case '+': return BOOL_F;
  default:;}
  default:;}
  while (((len-i) > 2) && str[i] == '#' && ++i) switch (str[i++]) {
  case 'b': case 'B':
    radix = 2;
    break;
  case 'o': case 'O':
    radix = 8;
    break;
  case 'd': case 'D':
    radix = 10;
    break;
  case 'x': case 'X':
    radix = 16;
    break;
  case 'i': case 'I':
    ex = 2;
    break;
  case 'e': case 'E':
    ex = 1;
  }
  switch (ex) {
  case 1: return istr2int(&str[i],len-i,radix);
  case 0: {
	  SCM res=istr2int(&str[i],len-i,radix);
	  if ((res!=BOOL_F) || (radix!=10)) return res;
  }
#ifdef FLOATS
  case 2: return istr2flo(&str[i],len-i);
#endif
  }
  return BOOL_F;
}
SCM string2number(str,radix)
SCM str,radix;
{
	if UNBNDP(radix) radix=MAKINUM(10L);
	else ASSERT(INUMP(radix),radix,ARG2,s_str2number);
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_str2number);
	return istring2number(CHARS(str),LENGTH(str),INUM(radix));
}

SCM charp(x)
SCM x;
{
	return ICHRP(x) ? BOOL_T : BOOL_F;
}
SCM char_lessp(x,y)
SCM x,y;
{
	ASSERT(ICHRP(x),x,ARG1,s_ch_lessp);
	ASSERT(ICHRP(y),y,ARG2,s_ch_lessp);
	return (ICHR(x) < ICHR(y)) ? BOOL_T : BOOL_F;
}
SCM char_leqp(x,y)
SCM x,y;
{
	ASSERT(ICHRP(x),x,ARG1,s_ch_leqp);
	ASSERT(ICHRP(y),y,ARG2,s_ch_leqp);
	return (ICHR(x) <= ICHR(y)) ? BOOL_T : BOOL_F;
}
SCM chci_eq(x,y)
SCM x,y;
{
	ASSERT(ICHRP(x),x,ARG1,s_ci_eq);
	ASSERT(ICHRP(y),y,ARG2,s_ci_eq);
	return (upcase[ICHR(x)] == upcase[ICHR(y)]) ? BOOL_T : BOOL_F;
}
SCM chci_lessp(x,y)
SCM x,y;
{
	ASSERT(ICHRP(x),x,ARG1,s_ci_lessp);
	ASSERT(ICHRP(y),y,ARG2,s_ci_lessp);
	return (upcase[ICHR(x)] < upcase[ICHR(y)]) ? BOOL_T : BOOL_F;
}
SCM chci_leqp(x,y)
SCM x,y;
{
	ASSERT(ICHRP(x),x,ARG1,s_ci_leqp);
	ASSERT(ICHRP(y),y,ARG2,s_ci_leqp);
	return (upcase[ICHR(x)] <= upcase[ICHR(y)]) ? BOOL_T : BOOL_F;
}
SCM char_alphap(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_ch_alphap);
	return (isascii(ICHR(chr)) && isalpha(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char_nump(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_ch_nump);
	return (isascii(ICHR(chr)) && isdigit(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char_whitep(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_ch_whitep);
	return (isascii(ICHR(chr)) && isspace(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char_upperp(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_ch_upperp);
	return (isascii(ICHR(chr)) && isupper(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char_lowerp(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_ch_lowerp);
	return (isascii(ICHR(chr)) && islower(ICHR(chr))) ? BOOL_T : BOOL_F;
}
SCM char2int(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_char2int);
	return MAKINUM(ICHR(chr));
}
SCM int2char(n)
SCM n;
{
  ASSERT(INUMP(n),n,ARG1,s_int2char);
  ASSERT((INUM(n)>=0) && (INUM(n)<char_code_limit),n,OUTOFRANGE,s_int2char);
  return MAKICHR(INUM(n));
}
SCM char_upcase(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_ch_upcase);
	return MAKICHR(upcase[ICHR(chr)]);
}
SCM char_downcase(chr)
SCM chr;
{
	ASSERT(ICHRP(chr),chr,ARG1,s_ch_downcase);
	return MAKICHR(downcase[ICHR(chr)]);
}

SCM stringp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return STRINGP(x) ? BOOL_T : BOOL_F;
}
SCM string(chrs)
SCM chrs;
{
	SCM res;
	register char *data;
	long i = ilength(chrs);
	ASSERT(i>=0,chrs,ARG1,s_string);
	if (i == 0) return nullstr;
	res = makstr(i);
	data = CHARS(res);
	for(;NNULLP(chrs);chrs=CDR(chrs)) {
		ASSERT(ICHRP(CAR(chrs)),chrs,ARG1,s_string);
		*data++ = ICHR(CAR(chrs));
	}
	return res;
}
SCM make_string(k,chr)
SCM k,chr;
{
	SCM res;
	register char *dst;
	register long i;
	ASSERT(INUMP(k),k,ARG1,s_make_string);
	i = INUM(k);
	if (i == 0) return nullstr;
	res = makstr(i);
	dst = CHARS(res);
	if ICHRP(chr) for(i--;i>=0;i--) dst[i] = ICHR(chr);
	return res;
}
SCM st_length(str)
SCM str;
{
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_st_length);
	return MAKINUM(LENGTH(str));
}
SCM st_ref(str,k)
SCM str,k;
{
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_st_ref);
	ASSERT(INUMP(k),k,ARG2,s_st_ref);
	ASSERT(INUM(k) < LENGTH(str) && INUM(k) >= 0,k,OUTOFRANGE,s_st_ref);
	return MAKICHR(CHARS(str)[INUM(k)]);
}
SCM st_set(str,k,chr)
SCM str,k,chr;
{
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_st_set);
	ASSERT(INUMP(k),k,ARG2,s_st_set);
	ASSERT(ICHRP(chr),chr,ARG3,s_st_set);
	ASSERT(INUM(k) < LENGTH(str),k,OUTOFRANGE,s_st_set);
	CHARS(str)[INUM(k)] = ICHR(chr);
	return UNSPECIFIED;
}
SCM st_equal(s1, s2)
SCM s1, s2;
{
	register size_t i;
	register char *c1, *c2;
	ASSERT(NIMP(s1) && STRINGP(s1),s1,ARG1,s_st_equal);
	ASSERT(NIMP(s2) && STRINGP(s2),s2,ARG2,s_st_equal);
	i = LENGTH(s2);
	if (LENGTH(s1) != i) return BOOL_F;
	c1 = CHARS(s1);
	c2 = CHARS(s2);
	while(i-- != 0) if(*c1++ != *c2++) return BOOL_F;
	return BOOL_T;
}
SCM stci_equal(s1, s2)
SCM s1, s2;
{
	register size_t i;
	register unsigned char *c1, *c2;
	ASSERT(NIMP(s1) && STRINGP(s1),s1,ARG1,s_stci_equal);
	ASSERT(NIMP(s2) && STRINGP(s2),s2,ARG2,s_stci_equal);
	i = LENGTH(s2);
	if (LENGTH(s1) != i) return BOOL_F;
	c1 = (unsigned char *) CHARS(s1);
	c2 = (unsigned char *) CHARS(s2);
	while(i-- != 0) if(upcase[*c1++] != upcase[*c2++]) return BOOL_F;
	return BOOL_T;
}
SCM st_lessp(s1, s2)
SCM s1, s2;
{
	register size_t i,len;
	register unsigned char *c1, *c2;
	register int c;
	ASSERT(NIMP(s1) && STRINGP(s1),s1,ARG1,s_st_lessp);
	ASSERT(NIMP(s2) && STRINGP(s2),s2,ARG2,s_st_lessp);
	len = LENGTH(s1);
	i = LENGTH(s2);
	if (len>i) i=len;
	c1 = (unsigned char *) CHARS(s1);
	c2 = (unsigned char *) CHARS(s2);
	for(i=0;i<len;i++) {
		c = (*c1++ - *c2++);
		if (c>0) return BOOL_F;
		if (c<0) return BOOL_T;
	}
	return (len != LENGTH(s2)) ? BOOL_T : BOOL_F;
}
SCM st_leqp(s1, s2)
SCM s1, s2;
{
  return (BOOL_F ^ BOOL_T) ^ st_lessp(s2, s1);
}
SCM stci_lessp(s1, s2)
SCM s1, s2;
{
	register size_t i,len;
	register unsigned char *c1, *c2;
	register int c;
	ASSERT(NIMP(s1) && STRINGP(s1),s1,ARG1,s_stci_lessp);
	ASSERT(NIMP(s2) && STRINGP(s2),s2,ARG2,s_stci_lessp);
	len = LENGTH(s1);
	i = LENGTH(s2);
	if (len>i) i=len;
	c1 = (unsigned char *) CHARS(s1);
	c2 = (unsigned char *) CHARS(s2);
	for(i=0;i<len;i++) {
		c = (upcase[*c1++] - upcase[*c2++]);
		if (c>0) return BOOL_F;
		if (c<0) return BOOL_T;
	}
	return (len != LENGTH(s2)) ? BOOL_T : BOOL_F;
}
SCM stci_leqp(s1, s2)
SCM s1, s2;
{
  return (BOOL_F ^ BOOL_T) ^ stci_lessp(s2, s1);
}
SCM substring(str,start,end)
SCM str,start,end;
{
	long l;
	ASSERT(NIMP(str) && STRINGP(str),str,ARG1,s_substring);
	ASSERT(INUMP(start),start,ARG2,s_substring);
	ASSERT(INUMP(end),end,ARG3,s_substring);
	ASSERT(INUM(start) <= LENGTH(str),start,OUTOFRANGE,s_substring);
	ASSERT(INUM(end) <= LENGTH(str),end,OUTOFRANGE,s_substring);
	l=INUM(end)-INUM(start);
	ASSERT(l>=0,MAKINUM(l),OUTOFRANGE,s_substring);
	if (l == 0) return nullstr;
	return makfromstr(&CHARS(str)[INUM(start)],(size_t)l);
}
SCM st_append(args)
SCM args;
{
	SCM res;
	register long i=0;
	register SCM l,s;
	register char *data;
	for(l=args;NIMP(l);) {
		ASSERT(CONSP(l),l,ARG1,s_st_append);
		s = CAR(l);
		ASSERT(NIMP(s) && STRINGP(s),s,ARG1,s_st_append);
		i += LENGTH(s);
		l=CDR(l);
	}
	ASSERT(NULLP(l),args,ARG1,s_st_append);
	if (i == 0) return nullstr;
	res = makstr(i);
	data = CHARS(res);
	for(l=args;NIMP(l);l=CDR(l)) {
		s = CAR(l);
		for(i=0;i<LENGTH(s);i++) *data++ = CHARS(s)[i];
	}
	return res;
}

SCM vectorp(x)
SCM x;
{
	if IMP(x) return BOOL_F;
	return VECTORP(x) ? BOOL_T : BOOL_F;
}
SCM vector_length(v)
SCM v;
{
	ASSERT(NIMP(v) && VECTORP(v),v,ARG1,s_ve_length);
	return MAKINUM(LENGTH(v));
}
SCM vector(l)
SCM l;
{
	SCM res;
	register SCM *data;
	long i = ilength(l);
	ASSERT(i>=0,l,ARG1,s_vector);
	if (i == 0) return nullvect;
	res = make_vector(MAKINUM(i),UNSPECIFIED);
	data = VELTS(res);
	for(;NIMP(l);l=CDR(l)) *data++ = CAR(l);
	return res;
}
SCM vector_ref(v, k)
SCM v,k;
{
	ASSERT(NIMP(v) && VECTORP(v),v,ARG1,s_ve_ref);
	ASSERT(INUMP(k),k,ARG2,s_ve_ref);
	ASSERT((INUM(k) < LENGTH(v)) && (INUM(k) >= 0),
	       k,OUTOFRANGE,s_ve_ref);
	return VELTS(v)[((long) INUM(k))];
}
SCM vector_set(v,k,obj)
SCM v,k,obj;
{
	ASSERT(NIMP(v) && VECTORP(v),v,ARG1,s_ve_set);
	ASSERT(INUMP(k),k,ARG2,s_ve_set);
	ASSERT((INUM(k) < LENGTH(v)),k,OUTOFRANGE,s_ve_set);
	VELTS(v)[((long) INUM(k))] = obj;
	return UNSPECIFIED;
}

extern char s_apply[], s_map[], s_for_each[];

static iproc cxrs[] = {
	{"car",0},
	{"cdr",0},
	{"caar",0},
	{"cadr",0},
	{"cdar",0},
	{"cddr",0},
	{"caaar",0},
	{"caadr",0},
	{"cadar",0},
	{"caddr",0},
	{"cdaar",0},
	{"cdadr",0},
	{"cddar",0},
	{"cdddr",0},
	{"caaaar",0},
	{"caaadr",0},
	{"caadar",0},
	{"caaddr",0},
	{"cadaar",0},
	{"cadadr",0},
	{"caddar",0},
	{"cadddr",0},
	{"cdaaar",0},
	{"cdaadr",0},
	{"cdadar",0},
	{"cdaddr",0},
	{"cddaar",0},
	{"cddadr",0},
	{"cdddar",0},
	{"cddddr",0},
	{0,0}};

static iproc subr1s[]={
	{"not",lnot},
	{"boolean?",booleanp},
	{"pair?",consp},
	{"null?",nullp},
	{"list?",listp},
	{s_length,length},
	{s_reverse,reverse},
	{"symbol?",symbolp},
	{s_symbol2string,symbol2string},
	{s_str2symbol,string2symbol},
	{"number?",numberp},
	{"complex?",numberp},
#ifdef FLOATS
	{"real?",realp},
	{"rational?",realp},
	{"integer?",intp},
#else
	{"real?",numberp},
	{"rational?",numberp},
	{"integer?",exactp},
#endif
	{s_exactp,exactp},
	{s_inexactp,inexactp},
#ifndef FLOATS
	{"floor",numident},
	{"ceiling",numident},
	{"truncate",numident},
	{"round",numident},
#endif
	{s_zerop,zerop},
	{s_positivep,positivep},
	{s_negativep,negativep},
	{s_oddp,oddp},
	{s_evenp,evenp},
	{s_abs,absval},
	{"char?",charp},
	{s_ch_alphap,char_alphap},
	{s_ch_nump,char_nump},
	{s_ch_whitep,char_whitep},
	{s_ch_upperp,char_upperp},
	{s_ch_lowerp,char_lowerp},
	{s_char2int,char2int},
	{s_int2char,int2char},
	{s_ch_upcase,char_upcase},
	{s_ch_downcase,char_downcase},
	{"string?",stringp},
	{s_st_length,st_length},
	{"vector?",vectorp},
	{s_ve_length,vector_length},
	{"procedure?",procedurep},
	{0,0}};

static iproc subr2s[]={
	{"eq?",eq},
	{"eqv?",eq},
	{"equal?",equal},
	{"cons",cons},
#ifndef PURE_FUNCTIONAL
	{s_setcar,setcar},
	{s_setcdr,setcdr},
#endif
	{s_list_ref,list_ref},
	{s_memq,memq},
	{"memv",memq},
	{s_member,member},
	{s_assq,assq},
	{"assv",assq},
	{s_assoc,assoc},
	{s_quotient,quotient},
	{s_remainder,lremainder},
	{s_modulo,modulo},
	{"char=?",eq},
	{s_ch_lessp,char_lessp},
	{s_ci_eq,chci_eq},
	{s_ci_lessp,chci_lessp},
	{s_ch_leqp,char_leqp},
	{s_ci_leqp,chci_leqp},
	{s_st_ref,st_ref},
	{s_st_equal,st_equal},
	{s_stci_equal,stci_equal},
	{s_st_lessp,st_lessp},
	{s_stci_lessp,stci_lessp},
	{"string<=?",st_leqp},
	{"string-ci<=?",stci_leqp},
	{s_ve_ref,vector_ref},
	{0,0}};

static iproc lsubrs[]={
	{s_list,list},
	{s_append,append},
	{s_string,string},
	{s_st_append,st_append},
	{s_vector,vector},
	{0,0}};

static iproc lsubr2s[]={
	{s_eqp,eqp},
	{s_lessp,lessp},
	{s_grp,greaterp},
	{s_lesseqp,lesseqp},
	{s_greqp,greatereqp},
	{s_apply,apply},
	{s_map,map},
	{s_for_each,for_each},
	{0,0}};

static iproc subr2os[]={
	{s_number2string,number2string},
	{s_str2number,string2number},
	{s_make_string,make_string},
	{s_make_vector,make_vector},
	{0,0}};

static iproc asubrs[]={
	{s_gcd,lgcd},
	{s_lcm,llcm},
	{0,0}};

static iproc subr2xs[]={
	{"char>?",char_lessp},
	{"char-ci>?",chci_lessp},
	{"char>=?",char_leqp},
	{"char-ci>=?",chci_leqp},
	{"string>?",st_lessp},
	{"string-ci>?",stci_lessp},
	{"string>=?",st_leqp},
	{"string-ci>=?",stci_leqp},
	{0,0}};

static iproc subr3s[]={
	{s_substring,substring},
#ifndef PURE_FUNCTIONAL
	{s_st_set,st_set},
	{s_ve_set,vector_set},
#endif
	{0,0}};

void init_iprocs(subra, type)
     iproc *subra;
     int type;
{
  for(;subra->string; subra++)
    make_subr(subra->string,
	      type,
	      subra->cproc);
}

void init_subrs()
{
  init_iprocs(cxrs, tc7_cxr);
  init_iprocs(subr1s, tc7_subr_1);
  init_iprocs(subr2s, tc7_subr_2);
  init_iprocs(subr2os, tc7_subr_2o);
  init_iprocs(subr2xs, tc7_subr_2x);
  init_iprocs(lsubrs, tc7_lsubr);
  init_iprocs(lsubr2s, tc7_lsubr_2);
  init_iprocs(asubrs, tc7_asubr);
  init_iprocs(subr3s, tc7_subr_3);
}
