;
; FILE:		"format.scm"
; IMPLEMENTS:	Format function {Scheme} -- see documentation below.
; AUTHOR:	Ken Dickey, Aubrey Jaffer, Dirk Lutzebaeck
; VERSION:      1.4
; MODIFIED:	1991 Dec 17, Dirk Lutzebaeck
;               - added slashify and up/downcase support for symbols
;               - better <output-port> type check
;               - Tested with various scheme implementations
;               1991 Dec 9, Dirk Lutzebaeck
;               - Tested with SCM3c6, MIT-CScheme 7.1, Elk 1.4, UMB-Scheme 2.5,
;                 and Scheme->C 01nov91
;               - defined format-help as lambda for portability
;               1991 Dec 6, Dirk Lutzebaeck
;               - `C'-printf-like padding support
;               - format returns a formatted string if (eq? port #f)
;               - signals error for missing or too much format arguments
;               - version number added
;               - syntax errors corrected, partially reformatted
;               1991 Sept 19, Aubrey Jaffer, put in tab and formfeed support
;		and included in Scheme library.
; 		1990 April 4  Ken Dickey
; NOTES:	Imports PRETTY-PRINT (~g)
;	 	Number syntax is ieee/r4rs; r3rs syntax in comments
;		Does *not* implement ~& option

;;
;;
;;  ========
;;  FUNCTION: (FORMAT <port> <format-string> . <args>)
;;  ========
;;
;;  RESULT: returns unconsumed <args> or a string; has side effect of
;;  printing according to <format-string>.  If <port> is #t the output is
;;  to the current input port.  If <port> is #f, a formatted string is
;;  returned as the result of the call.  Otherwise <port> must be an
;;  output port.  <format-string> must be a string.  Characters are output
;;  as if the string were output by the DISPLAY function with the
;;  exception of those prefixed by a tilde (~) as follows [note that options
;;  which take arguments remove them from the argument list (they are said to
;;  be `consumed')]:
;;
;;option  mnemonic: description
;;------  ------------------------
;;    ~a  any: display the argument (as for humans) [lower case symbol output].
;;    ~A  as ~a but with upper case symbol output.
;;    ~s  slashified: write the argument (as for parsers) [lower case output].
;;    ~S  as ~s but with upper case symbol output.
;;    ~d  decimal: the integer argument is output in decimal format.
;;    ~x  hexadecimal: the integer arg. is output in lowercase hexadecimal fmt.
;;    ~X  as ~x but in upper case format.
;;    ~o  octal: the integer argument is output in octal format.
;;    ~b  binary: the integer argument is output in binary format.
;;    ~p  plural: if the arg. is greater than 1, a lower case 's' is printed.
;;    ~P  as ~p but an upper case 'S' is printed.
;;    ~c  character: the next argument is displayed as a character.
;;    ~_  space: output a space character.
;;    ~<num>_ multiple spaces: outputs <num> spaces.
;;    ~%  newline: output a newline character.
;;    ~<num>% multiple newlines: outputs <num> newline characters.
;;    ~&  freshline: unless at the begin. of a line, same as ~%, else ignored.
;;    ~|  page seperator: output a page seperator.
;;    ~~  tilde: output a tilde.
;;    ~t  tab: output a tab charcter.
;;    ~<num>t multiple tabs: outputs <num> tabs.
;;    ~g  glorify: pretty print the argument (typically an s-expression).
;;    ~?  indirection: take the next argument as a format string and consume
;;        further arguments as appropriate, then continue to process the
;;        current format string.
;;
;;    ~<num>a, ~<num>d, ~<num>x, ~<num>o, ~<num>b:
;;        padded output: the arguments output has a field width of <num> chars
;;        if <num> > 0 then padding is done with spaces from left to right
;;        if <num> < 0 then padding is done with spaces from right to left
;;        if <num> is preceded by a zero-character (#\0) then padding is done
;;        by zero characters instead of spaces (like `C'-printf)
;;        e.g. (format #f "~5a" "abc")  -> "  abc"
;;             (format #f "~-5a" "abc") -> "abc  "
;;
;;    ~<n1>.<n2>a, ~<n1>.<n2>d, ~<n1>.<n2>x, ~<n1>.<n2>o, ~<n1>.<n2>b:
;;        padded output with maximum object print width: the arguments output
;;        has a field width of <n1> chars, whereas the objects output width
;;        is not longer than <n2> (like `C'-printf); <n1> may be negative
;;        e.g. (format #f "~10.4a" "abcdef")  -> "      abcd"
;;             (format #f "~-10.4a" "abcdef") -> "abcd      "

;----- INTERPRETER SPECIFIC DEFINTIONS --------------------------------------

;(##declare (fixnum))                   ;; GAMBIT (v1.4)
;(require:require 'pretty-print)	;; Aubrey Jaffers SCM
;(define pp pretty-print)		;;  - " -

;(define (make-string len . char)       ;; T's make-string has no char option
;  (let ((c (if (null? char) " " (char->string (car char)))))
;    (let loop ((i 0) (s ""))
;      (if (= i len)
;         s
;	  (loop (+ i 1) (string-append s c))))))


;---------- FORMAT ----------------------------------------------------------

(define (FORMAT <output-port> <format-string> . <args>)
  (letrec
      ((PORT (cond
	      ((boolean? <output-port>)
	       (if <output-port> (current-output-port) #f))
	      ((output-port? <output-port>) <output-port>)
	      (else
	       (slib:error "illegal output-port `" <output-port>))))

       (OUT "")				; output string produced

       (OUT-CHAR			; appends a character to OUT
	(lambda (ch)
	  (set! out (string-append out (string ch)))))

       (OUT-STR				; appends a string to out
	(lambda (str)
	  (set! out (string-append out str))))

       (FORMAT-HELP
	(lambda (format-strg arglist)
	  (letrec
	      ((FORMAT-STRG-LEN (string-length format-strg))
	       (POS 0)			; input format str position
	       (PAD-CHAR #\ )		; padding character
	       (PAD-LEFT-TO-RIGHT #t)	; padding direction
	       (LEFT-DOT-PAD 0)		; last num. prefix

	       (NEXT-CHAR		; reads the next char from format-strg
		(lambda ()
		  (let ((ch (string-ref format-strg pos)))
		    (set! pos (+ 1 pos))
		    ch)))

	       (OUT-OBJ			; appends the string represention
		(lambda (obj pad . options) ; of obj to OUT
		  (cond
		   ((and (zero? pad) (zero? left-dot-pad))
		    (out-str (obj-to-str obj options)))
		   ((zero? left-dot-pad)
		    (let* ((field-len pad)
			   (obj-str (obj-to-str obj options))
			   (obj-len (string-length obj-str))
			   (pad-str (if (<= obj-len field-len)
					(make-string (- field-len obj-len)
						     pad-char)
					"")))
		      (if pad-left-to-right
			  (begin
			    (out-str pad-str)
			    (out-str obj-str))
			  (begin
			    (out-str obj-str)
			    (out-str pad-str)))))
		   (else
		    (let* ((field-len left-dot-pad)
			   (max-obj-len pad)
			   (obj-str (obj-to-str obj options))
			   (obj-len (string-length obj-str))
			   (tr-obj-str (if (>= obj-len max-obj-len)
					   (substring obj-str 0 max-obj-len)
					   obj-str)))
		      (set! left-dot-pad 0)
		      (apply out-obj `(,tr-obj-str ,field-len ,@options)))))))

	       (ANYCHAR-DISPATCH	; dispatches the format-strg
		(lambda (arglist)
		  (if (>= pos format-strg-len)
		      arglist		; used for ~? continuance
		      (let ((char (next-char)))
			(cond
			 ((eqv? char #\~)
			  (set! pad-char #\ )
			  (set! pad-left-to-right #t)
			  (set! left-dot-pad 0)
			  (tilde-dispatch arglist 0))
			 (else
			  (out-char char)
			  (anychar-dispatch arglist)))))))

	       (TILDE-DISPATCH
		(lambda (arglist pad)
		  (define (arg)		; return the actual argument to format
		    (if (null? arglist)	; if arglist is empty -> error
			"#[missing-format-arg]"
			(car arglist)))
		  (define (rest-args)	; returns argument rest
		    (if (null? arglist)	; if arglist is empty -> error
			(begin
			  (slib:error "no more arguments for \"" format-strg "\"")
			  '())
			(cdr arglist)))
		  (cond
		   ((>= pos format-strg-len)
		    (out-char #\~)	; tilde at end of string is just output
		    arglist)		; used for ~? continuance
		   (else
		    (case (next-char)
		      ((#\a)		; Any -- for humans
		       (out-obj (arg) pad 'downcase)
		       (anychar-dispatch (rest-args)))
		      ((#\A)		; Any -- for humans
		       (out-obj (arg) pad 'upcase)
		       (anychar-dispatch (rest-args)))
		      ((#\s)		; Slashified -- for parsers
		       (out-obj (arg) pad 'downcase 'slashify)
		       (anychar-dispatch (rest-args)))
		      ((#\S)		; Slashified -- for parsers
		       (out-obj (arg) pad 'upcase 'slashify)
		       (anychar-dispatch (rest-args)))
		      ((#\d #\D)	; Decimal
		       (out-obj (number->string (arg) 10) pad)
;;; r3rs	   (out-obj (number->string (arg) (radix 'd 's)) pad)
		       (anychar-dispatch (rest-args)))
		      ((#\x)		; Hexadecimal
		       (out-obj (string-downcase (number->string (arg) 16)) pad)
;;; r3rs	   (out-obj (number->string (arg) (radix 'x 's)) pad)
		       (anychar-dispatch (rest-args)))
		      ((#\X)		; Hexadecimal
		       (out-obj (string-upcase (number->string (arg) 16)) pad)
;;; r3rs	   (out-obj (number->string (arg) (radix 'x 's)) pad)
		       (anychar-dispatch (rest-args)))
		      ((#\o #\O)	; Octal
		       (out-obj (number->string (arg)  8) pad)
;;; r3rs	   (out-obj (number->string (arg) (radix 'o 's)) pad)
		       (anychar-dispatch (rest-args)))
		      ((#\b #\B)	; Binary
		       (out-obj (number->string (arg)  2) pad)
;;; r3rs	   (out-obj (number->string (arg) (radix 'b 's)) pad)
		       (anychar-dispatch (rest-args)))
		      ((#\c #\C)	; Character
		       (out-char (arg))
		       (anychar-dispatch (rest-args)))
		      ((#\p)		; Plural (lowercase)
		       (if (eqv? (arg) 1)
			   #f		; no action
			   (out-char #\s))
		       (anychar-dispatch (rest-args)))
		      ((#\P)		; Plural (uppercase)
		       (if (eq? (arg) 1)
			   #f		; no action
			   (out-char #\S))
		       (anychar-dispatch (rest-args)))
		      ((#\~)		; Tilde
		       (out-char #\~)
		       (anychar-dispatch arglist))
		      ((#\% #\&)	; Newline (Freshline is the same)
		       (if (zero? pad)
			   (out-char #\newline)
			   (out-str (make-string pad #\newline)))
		       (anychar-dispatch arglist))
		      ((#\_)		; Space
		       (if (zero? pad)
			   (out-char #\ )
			   (out-str (make-string pad #\ )))
		       (anychar-dispatch arglist))
		      ((#\t #\T)	; Tab -- implementation dependent
		       (if (zero? pad)
			   (out-char slib:tab)
			   (out-str (make-string pad slib:tab)))
		       (anychar-dispatch arglist))
		      ((#\|)		; Page Seperator -- impl. dependent
		       (out-char slib:form-feed)
		       (anychar-dispatch arglist))
		      ((#\g #\G)	; Pretty-print -- assumes PP available
		       (if (not (zero? pad))
			   (format-err "padding not supported with ~g")
			   (if (not port)
			       (out-str (pretty-print-to-string (arg)))
			       (begin
				 (display out port)
				 (set! out "")
				 (pretty-print (arg) port))))
		       (anychar-dispatch (rest-args)))
		      ;; {"~?" in Common Lisp is "~K" in T}
		      ((#\?)		; Indirection
		       (anychar-dispatch ;  -- take next arg as format string
			(format-help (arg) (rest-args))))
					; Note: format-help returns unused args
		      ((#\1) (tilde-dispatch arglist (+ (* pad 10) 1)))
		      ((#\2) (tilde-dispatch arglist (+ (* pad 10) 2)))
		      ((#\3) (tilde-dispatch arglist (+ (* pad 10) 3)))
		      ((#\4) (tilde-dispatch arglist (+ (* pad 10) 4)))
		      ((#\5) (tilde-dispatch arglist (+ (* pad 10) 5)))
		      ((#\6) (tilde-dispatch arglist (+ (* pad 10) 6)))
		      ((#\7) (tilde-dispatch arglist (+ (* pad 10) 7)))
		      ((#\8) (tilde-dispatch arglist (+ (* pad 10) 8)))
		      ((#\9) (tilde-dispatch arglist (+ (* pad 10) 9)))
		      ((#\0)
		       (if (and (zero? pad) pad-left-to-right)
			   (set! pad-char #\0))
		       (tilde-dispatch arglist (* pad 10)))
		      ((#\-)		; right to left padding
		       (set! pad-left-to-right #f)
		       (tilde-dispatch arglist pad))
		      ((#\.)		; object field width
		       (set! left-dot-pad pad) ; save pad
		       (tilde-dispatch arglist 0))
		      (else
		       (slib:error "unknown tilde escape "
				   (string-ref format-strg (- pos 1))))))))))

	    (anychar-dispatch arglist))))) ; end of format help

    (if (not (null? (format-help <format-string> <args>))) ; format main
	(slib:error "too much arguments for \"" <format-string> "\""))
    (if port
	(display out port)
	out)))

;------ OBJ-TO-STRING -------------------------------------------------------

;; converts an arbitrary scheme object to a string
;; options is a list which may contain the following symbols:
;;   'upcase     uppercase output string for symbols and #[...]-objects
;;   'downcase   downcase output string for symbols and #[...]-objects
;;   'slashify   slashifies output string as `write' does

(define (OBJ-TO-STR obj options)
    (cond
     ((string? obj)
      (if (memq 'slashify options)
	  (let ((obj-len (string-length obj)))
	    (string-append
	     "\""
	     (let loop ((i 0) (j 0))	; modified from Marc Feeley from pp.scm
	       (if (= j obj-len)
		   (string-append (substring obj i j) "\"")
		   (let ((c (string-ref obj j)))
		     (if (or (char=? c #\\)
			     (char=? c #\"))
			 (string-append (substring obj i j) "\\"
					(loop j (+ j 1)))
			 (loop i (+ j 1))))))))
	  obj))

     ((boolean? obj)
      (if (memq 'upcase options)
	  (if obj "#T" "#F")
	  (if obj "#t" "#f")))

     ((number? obj)
      (number->string obj 10))

     ((symbol? obj)
      (cond
       ((memq 'downcase options)
	(string-downcase (symbol->string obj)))
       ((memq 'upcase options)
	(string-upcase (symbol->string obj)))
       (else
	(symbol->string obj))))

     ((char? obj)
      (if (memq 'slashify options)
	  (string-append "#\\" (string obj))
	  (string obj)))

     ((null? obj) "()")

     ((procedure? obj)
      (if (memq 'upcase options)
	  "#[PROCEDURE]"
	  "#[procedure]"))

     ((output-port? obj)
      (if (memq 'upcase options)
	  "#[OUTPUT-PORT]"
	  "#[output-port]"))

     ((input-port? obj)
      (if (memq 'upcase options)
	  "#[INPUT-PORT]"
	  "#[input-port]"))

     ((list? obj)
      (string-append "("
		     (let loop ((obj-list obj))
		       (if (null? (cdr obj-list))
			   (obj-to-str (car obj-list) options)
			   (string-append
			    (obj-to-str (car obj-list) options)
			    " "
			    (loop (cdr obj-list)))))
		     ")"))

     ((pair? obj)
      (string-append "("
		     (obj-to-str (car obj) options)
		     " . "
		     (obj-to-str (cdr obj) options)
		     ")"))

     ((eof-object? obj)
      (if (memq 'upcase options)
	  "#[EOF-OBJECT]"
	  "#[eof-object]"))

     ((vector? obj)
      (string-append "#" (obj-to-str (vector->list obj) options)))

     (else "#[unformatable-object]")))


(define (STRING-UPCASE str)
  (let ((up-str (string-copy str)))
    (do ((i (- (string-length str) 1) (- i 1)))
	((< i 0) up-str)
      (string-set! up-str i (char-upcase (string-ref str i))))))

(define (STRING-DOWNCASE str)
  (let ((down-str (string-copy str)))
    (do ((i (- (string-length str) 1) (- i 1)))
	((< i 0) down-str)
      (string-set! down-str i (char-downcase (string-ref str i))))))

