;;; -*-Scheme-*-
;;;
;;; $Id: cache.scm,v 1.3 1993/06/16 18:58:50 cph Exp $
;;;
;;; Copyright (c) 1993 Massachusetts Institute of Technology
;;;
;;; This material was developed by the Scheme project at the
;;; Massachusetts Institute of Technology, Department of Electrical
;;; Engineering and Computer Science.  Permission to copy this
;;; software, to redistribute it, and to use it for any purpose is
;;; granted, subject to the following restrictions and understandings.
;;;
;;; 1. Any copy made of this software must include this copyright
;;; notice in full.
;;;
;;; 2. Users of this software agree to make their best efforts (a) to
;;; return to the MIT Scheme project any improvements or extensions
;;; that they make, so that these may be included in future releases;
;;; and (b) to inform MIT of noteworthy uses of this software.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4. MIT has made no warrantee or representation that the operation
;;; of this software will be error-free, and MIT is under no
;;; obligation to provide any services, by way of maintenance, update,
;;; or otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this
;;; material, there shall be no use of the name of the Massachusetts
;;; Institute of Technology nor of any adaptation thereof in any
;;; advertising, promotional, or sales literature without prior
;;; written consent from MIT in each case.

;;;; Method Caches for Scheme Object System

;;; From "Efficient Method Dispatch in PCL", Gregor Kiczales and Luis
;;; Rodriguez, Proceedings of the 1990 ACM Conference on Lisp and
;;; Functional Programming.  Parts of this code are based on the
;;; September 16, 1992 PCL implementation.

(declare (usual-integrations))
(declare (integrate-external "wrapper"))

(define-structure (cache (constructor %make-cache))
  (wrapper-index 0)
  (mask 0 read-only true)
  (limit 0 read-only true)
  (wrappers '#() read-only true)
  (values '#() read-only true)
  (overflow '()))

(define (new-cache)
  (make-cache 0 4))

(define (make-cache wrapper-index length)
  ;; LENGTH is assumed to be a power of two.
  (%make-cache wrapper-index
	       (fix:- length 1)
	       (cond ((fix:<= length 4) 1)
		     ((fix:<= length 16) 4)
		     (else 6))
	       (make-vector length #f)
	       (make-vector length #f)
	       '()))

(define-integrable (cache-length cache)
  (vector-length (cache-wrappers cache)))

(define-integrable (cache-line-wrappers cache line)
  (vector-ref (cache-wrappers cache) line))

(define-integrable (set-cache-line-wrappers! cache line wrappers)
  (vector-set! (cache-wrappers cache) line wrappers))

(define-integrable (cache-line-value cache line)
  (vector-ref (cache-values cache) line))

(define-integrable (set-cache-line-value! cache line value)
  (vector-set! (cache-values cache) line value))

(define-integrable (cache-next-line cache line)
  (if (fix:= (fix:+ line 1) (cache-length cache))
      0
      (fix:+ line 1)))

(define-integrable (cache-line-separation cache line line*)
  (let ((n (fix:- line* line)))
    (if (fix:< n 0)
	(fix:+ n (cache-length cache))
	n)))

(define (probe-cache cache wrappers)
  (let ((line (compute-primary-cache-line cache wrappers)))
    (and line
	 (let ((limit (cache-limit cache)))
	   (letrec
	       ((search-lines
		 (lambda (line i)
		   (cond ((and (cache-line-wrappers cache line)
			       (match (cache-line-wrappers cache line)))
			  (cache-line-value cache line))
			 ((fix:= i limit)
			  (search-overflow (cache-overflow cache)))
			 (else
			  (search-lines (cache-next-line cache line)
					(fix:+ i 1))))))
		(search-overflow
		 (lambda (overflow)
		   (and (not (null? overflow))
			(if (match (caar overflow))
			    (cdar overflow)
			    (search-overflow (cdr overflow))))))
		(match
		 (lambda (wrappers*)
		   (let loop ((w1 wrappers*) (w2 wrappers))
		     (and (eq? (system-pair-car w1) (system-pair-car w2))
			  (or (null? (system-pair-cdr w1))
			      (loop (system-pair-cdr w1)
				    (system-pair-cdr w2))))))))
	     (search-lines line 0))))))

(define (compute-primary-cache-line cache wrappers)
  (let ((index (cache-wrapper-index cache))
	(mask (cache-mask cache)))
    (let loop ((wrappers wrappers) (line 0) (n-adds 0))
      (cond ((null? wrappers)
	     (fix:and line mask))
	    ((wrapper-invalid? (system-pair-car wrappers))
	     #f)
	    ((fix:= n-adds wrapper-cache-number-adds-ok)
	     (loop (system-pair-cdr wrappers)
		   (fix:and (fix:+ line
				   (wrapper-ref (system-pair-car wrappers)
						index))
			    wrapper-cache-number-mask)
		   1))
	    (else
	     (loop (system-pair-cdr wrappers)
		   (fix:+ line (wrapper-ref (system-pair-car wrappers) index))
		   (fix:+ n-adds 1)))))))

(define (cache-entry-reusable? wrappers wrappers*)
  ;; True iff WRAPPERS is (1) empty, (2) contains a wrapper that is
  ;; invalid, or (3) has the same wrappers as WRAPPERS*.
  (or (not wrappers)
      (let loop ((wrappers wrappers) (wrappers* wrappers*))
	(or (null? wrappers)
	    (wrapper-invalid? (system-pair-car wrappers))
	    (and (eq? (system-pair-car wrappers) (system-pair-car wrappers*))
		 (loop (system-pair-cdr wrappers)
		       (system-pair-cdr wrappers*)))))))

(define (cache-count cache)
  (let ((length (cache-length cache)))
    (do ((line 0 (fix:+ line 1))
	 (count 0
		(if (let ((wrappers (cache-line-wrappers cache line)))
		      (and wrappers
			   (let loop ((wrappers wrappers))
			     (or (null? wrappers)
				 (and (not (wrapper-invalid?
					    (system-pair-car wrappers)))
				      (loop (system-pair-cdr wrappers)))))))
		    (fix:+ count 1)
		    count)))
	((fix:= line length) count))))

(define (probe-cache-1 cache w1)
  (and (not (fix:= (wrapper-ref w1 (cache-wrapper-index cache)) 0))
       (let ((line
	      (fix:and (wrapper-ref w1 (cache-wrapper-index cache))
		       (cache-mask cache)))
	     (match
	      (lambda (wrappers)
		(declare (integrate wrappers))
		(eq? w1 (system-pair-car wrappers)))))
	 (declare (integrate-operator match))
	 (if (and (cache-line-wrappers cache line)
		  (match (cache-line-wrappers cache line)))
	     (cache-line-value cache line)
	     (let ((limit (cache-limit cache)))
	       (let search-lines ((line (cache-next-line cache line)) (i 0))
		 (cond ((fix:= i limit)
			(let search-overflow ((entries (cache-overflow cache)))
			  (and (not (null? entries))
			       (if (match (caar entries))
				   (cdar entries)
				   (search-overflow (cdr entries))))))
		       ((and (cache-line-wrappers cache line)
			     (match (cache-line-wrappers cache line)))
			(cache-line-value cache line))
		       (else
			(search-lines (cache-next-line cache line)
				      (fix:+ i 1))))))))))

(define (probe-cache-2 cache w1 w2)
  (and (not (fix:= (wrapper-ref w1 (cache-wrapper-index cache)) 0))
       (not (fix:= (wrapper-ref w2 (cache-wrapper-index cache)) 0))
       (let ((line
	      (fix:and (fix:+ (wrapper-ref w1 (cache-wrapper-index cache))
			      (wrapper-ref w2 (cache-wrapper-index cache)))
		       (cache-mask cache)))
	     (match
	      (lambda (wrappers)
		(declare (integrate wrappers))
		(and (eq? w1 (system-pair-car wrappers))
		     (eq? w2 (system-pair-car (system-pair-cdr wrappers)))))))
	 (declare (integrate-operator match))
	 (if (and (cache-line-wrappers cache line)
		  (match (cache-line-wrappers cache line)))
	     (cache-line-value cache line)
	     (let ((limit (cache-limit cache)))
	       (let search-lines ((line (cache-next-line cache line)) (i 0))
		 (cond ((fix:= i limit)
			(let search-overflow ((entries (cache-overflow cache)))
			  (and (not (null? entries))
			       (if (match (caar entries))
				   (cdar entries)
				   (search-overflow (cdr entries))))))
		       ((and (cache-line-wrappers cache line)
			     (match (cache-line-wrappers cache line)))
			(cache-line-value cache line))
		       (else
			(search-lines (cache-next-line cache line)
				      (fix:+ i 1))))))))))

(define (probe-cache-3 cache w1 w2 w3)
  (and (not (fix:= (wrapper-ref w1 (cache-wrapper-index cache)) 0))
       (not (fix:= (wrapper-ref w2 (cache-wrapper-index cache)) 0))
       (not (fix:= (wrapper-ref w3 (cache-wrapper-index cache)) 0))
       (let ((line
	      (fix:and
	       (fix:+ (wrapper-ref w1 (cache-wrapper-index cache))
		      (fix:+ (wrapper-ref w2 (cache-wrapper-index cache))
			     (wrapper-ref w3 (cache-wrapper-index cache))))
	       (cache-mask cache)))
	     (match
	      (lambda (wrappers)
		(declare (integrate wrappers))
		(and (eq? w1 (system-pair-car wrappers))
		     (eq? w2 (system-pair-car (system-pair-cdr wrappers)))
		     (eq? w3 (system-pair-car
			      (system-pair-cdr
			       (system-pair-cdr wrappers))))))))
	 (declare (integrate-operator match))
	 (if (and (cache-line-wrappers cache line)
		  (match (cache-line-wrappers cache line)))
	     (cache-line-value cache line)
	     (let ((limit (cache-limit cache)))
	       (let search-lines ((line (cache-next-line cache line)) (i 0))
		 (cond ((fix:= i limit)
			(let search-overflow ((entries (cache-overflow cache)))
			  (and (not (null? entries))
			       (if (match (caar entries))
				   (cdar entries)
				   (search-overflow (cdr entries))))))
		       ((and (cache-line-wrappers cache line)
			     (match (cache-line-wrappers cache line)))
			(cache-line-value cache line))
		       (else
			(search-lines (cache-next-line cache line)
				      (fix:+ i 1))))))))))

(define (probe-cache-4 cache w1 w2 w3 w4)
  (and (not (fix:= (wrapper-ref w1 (cache-wrapper-index cache)) 0))
       (not (fix:= (wrapper-ref w2 (cache-wrapper-index cache)) 0))
       (not (fix:= (wrapper-ref w3 (cache-wrapper-index cache)) 0))
       (not (fix:= (wrapper-ref w4 (cache-wrapper-index cache)) 0))
       (let ((line
	      (fix:and
	       (fix:+ (fix:+ (wrapper-ref w1 (cache-wrapper-index cache))
			     (wrapper-ref w2 (cache-wrapper-index cache)))
		      (fix:+ (wrapper-ref w3 (cache-wrapper-index cache))
			     (wrapper-ref w4 (cache-wrapper-index cache))))
	       (cache-mask cache)))
	     (match
	      (lambda (wrappers)
		(declare (integrate wrappers))
		(and (eq? w1 (system-pair-car wrappers))
		     (eq? w2 (system-pair-car (system-pair-cdr wrappers)))
		     (eq? w3 (system-pair-car
			      (system-pair-cdr (system-pair-cdr wrappers))))
		     (eq? w4 (system-pair-car
			      (system-pair-cdr
			       (system-pair-cdr
				(system-pair-cdr wrappers)))))))))
	 (declare (integrate-operator match))
	 (if (and (cache-line-wrappers cache line)
		  (match (cache-line-wrappers cache line)))
	     (cache-line-value cache line)
	     (let ((limit (cache-limit cache)))
	       (let search-lines ((line (cache-next-line cache line)) (i 0))
		 (cond ((fix:= i limit)
			(let search-overflow ((entries (cache-overflow cache)))
			  (and (not (null? entries))
			       (if (match (caar entries))
				   (cdar entries)
				   (search-overflow (cdr entries))))))
		       ((and (cache-line-wrappers cache line)
			     (match (cache-line-wrappers cache line)))
			(cache-line-value cache line))
		       (else
			(search-lines (cache-next-line cache line)
				      (fix:+ i 1))))))))))

(define (fill-cache cache wrappers value)
  ;; WRAPPERS must be converted to a weak list since it will be stored
  ;; in the cache, and we don't want the cache to prevent the wrappers
  ;; from being GCed.
  (let ((wrappers (list->weak-list wrappers)))
    (or (fill-cache-if-possible cache wrappers value)
	(and (< (cache-count cache) (* (cache-length cache) .8))
	     (adjust-cache cache wrappers value))
	(expand-cache cache wrappers value))))

(define (fill-cache-if-possible cache wrappers value)
  (let ((primary (compute-primary-cache-line cache wrappers)))
    (if primary
	(let ((free (find-free-cache-line cache primary wrappers)))
	  (and free
	       (begin
		 (set-cache-line-wrappers! cache free wrappers)
		 (set-cache-line-value! cache free value)
		 cache)))
	;; If WRAPPERS contains an invalid wrapper, then do nothing
	;; and return CACHE -- the fill is no longer needed.  While
	;; other logic tries to eliminate this case, it can still
	;; happen when one of the wrappers is GCed during complex
	;; cache operations.
	cache)))

(define (adjust-cache cache wrappers value)
  ;; Try to rehash the cache.  If that fails, try rehashing with
  ;; different wrapper indexes.  Fail only when all of the wrapper
  ;; indexes have been tried and none has worked.
  (let ((length (cache-length cache)))
    (let ((new-cache (make-cache (cache-wrapper-index cache) length)))
      (letrec
	  ((fill-lines
	    (lambda (line)
	      (cond ((fix:= line length)
		     (fill-overflow (cache-overflow cache)))
		    ((try-entry (cache-line-wrappers cache line)
				(cache-line-value cache line))
		     (fill-lines (fix:+ line 1)))
		    (else
		     (try-next-wrapper-index)))))
	   (fill-overflow
	    (lambda (entries)
	      (cond ((null? entries)
		     (or (fill-cache-if-possible new-cache wrappers value)
			 (try-next-wrapper-index)))
		    ((try-entry (caar entries) (cdar entries))
		     (fill-overflow (cdr entries)))
		    (else
		     (try-next-wrapper-index)))))
	   (try-entry
	    (lambda (wrappers* value)
	      (or (cache-entry-reusable? wrappers* wrappers)
		  (fill-cache-if-possible new-cache wrappers* value))))
	   (try-next-wrapper-index
	    (lambda ()
	      (let ((index
		     (next-wrapper-index (cache-wrapper-index new-cache))))
		(and index
		     (begin
		       (set-cache-wrapper-index! new-cache index)
		       (fill-lines 0)))))))
	(fill-lines 0)))))

(define (expand-cache cache wrappers value)
  ;; Create a new cache that is twice the length of CACHE, rehash the
  ;; contents of CACHE into the new cache, and make the new entry.
  ;; Permits overflows to occur in the new cache.
  (let ((length (cache-length cache)))
    (letrec
	((fill-lines
	  (lambda (new-cache line)
	    (if (fix:= line length)
		(fill-overflow new-cache (cache-overflow cache))
		(fill-lines (maybe-do-fill new-cache
					   (cache-line-wrappers cache line)
					   (cache-line-value cache line))
			    (fix:+ line 1)))))
	 (fill-overflow
	  (lambda (new-cache overflow)
	    (if (null? overflow)
		(do-fill new-cache wrappers value)
		(fill-overflow (maybe-do-fill new-cache
					      (caar overflow)
					      (cdar overflow))
			       (cdr overflow)))))
	 (maybe-do-fill
	  (lambda (cache wrappers* value)
	    (if (cache-entry-reusable? wrappers* wrappers)
		cache
		(do-fill cache wrappers* value))))
	 (do-fill
	  (lambda (cache wrappers value)
	    (let ((primary (compute-primary-cache-line cache wrappers)))
	      (if primary
		  (let ((free (find-free-cache-line cache primary wrappers)))
		    (if free
			(begin
			  (set-cache-line-wrappers! cache free wrappers)
			  (set-cache-line-value! cache free value)
			  cache)
			(or (adjust-cache cache wrappers value)
			    (begin
			      (set-cache-overflow!
			       cache
			       (cons (cons (cache-line-wrappers cache primary)
					   (cache-line-value cache primary))
				     (cache-overflow cache)))
			      (set-cache-line-wrappers! cache primary wrappers)
			      (set-cache-line-value! cache primary value)
			      cache))))
		  cache)))))
      (fill-lines (make-cache (cache-wrapper-index cache)
			      (fix:+ length length))
		  0))))

(define (find-free-cache-line cache primary wrappers)
  ;; This procedure searches cache for a free line to hold an entry
  ;; with the given PRIMARY cache number and WRAPPERS.  Since the
  ;; entry can only be stored within (CACHE-LIMIT CACHE) lines of
  ;; PRIMARY, we either have to find a free line within that limit, or
  ;; we have to find a line with a larger primary which can be
  ;; displaced to another free line within *its* limit.
  (if (cache-entry-reusable? (cache-line-wrappers cache primary) wrappers)
      primary
      (let ((limit (cache-limit cache)))
	;; Find a line for an entry whose primary cache number is P.
	;; LINES is the sequence of entries that is waiting to be
	;; displaced into the line if we find it.
	(let pri-loop
	    ((line (cache-next-line cache primary))
	     (p primary)
	     (wrappers wrappers)
	     (lines '()))
	  (let sec-loop
	      ((line line)
	       (nsep (cache-line-separation cache p line)))
	    (cond ((fix:= line primary)
		   ;; We've scanned through the entire cache without
		   ;; finding a usable line.
		   #f)
		  ((let ((wrappers* (cache-line-wrappers cache line)))
		     (and (not (cache-entry-reusable? wrappers* wrappers))
			  (compute-primary-cache-line cache wrappers*)))
		   =>
		   (lambda (lp)
		     (let ((osep (cache-line-separation cache lp line)))
		       (cond ((fix:>= osep limit)
			      ;; This line contains an entry that is
			      ;; displaced to the limit.  [**** For
			      ;; some reason I don't understand, this
			      ;; terminates the search.]
			      #f)
			     ((or (fix:> nsep osep)
				  (and (fix:= nsep osep)
				       (= 0 (random 2))))
			      ;; The entry we're trying to place is
			      ;; further from its primary than the
			      ;; entry currently stored in this line.
			      ;; So now let's look for somewhere to
			      ;; displace the entry in this line.
			      (pri-loop (cache-next-line cache line)
					lp
					(cache-line-wrappers cache line)
					(cons line lines)))
			     (else
			      (sec-loop (cache-next-line cache line)
					(fix:+ nsep 1)))))))
		  (else
		   ;; Found a free line.  First perform all of the
		   ;; entry displacements, then return the subsequent
		   ;; free line.
		   (without-interrupts
		    (lambda ()
		      (let loop ((free-line line) (lines lines))
			(if (null? lines)
			    (begin
			      (set-cache-line-wrappers! cache free-line #f)
			      (set-cache-line-value! cache free-line #f)
			      free-line)
			    (let ((line (car lines)))
			      (set-cache-line-wrappers!
			       cache
			       free-line
			       (cache-line-wrappers cache line))
			      (set-cache-line-value!
			       cache
			       free-line
			       (cache-line-value cache line))
			      (loop line (cdr lines))))))))))))))