;\c	    Copyright (C) 1990 Pertti Kellomaki
;\c	 
;\c	 This file is part of Taurus, a parser generator producing Scheme
;\c	 
;\c	 Taurus is free software; you can redistribute it and/or modify
;\c	 it under the terms of the GNU General Public License as published by
;\c	 the Free Software Foundation; either version 1, or (at your option)
;\c	 any later version.
;\c	 
;\c	 Taurus is distributed in the hope that it will be useful,
;\c	 but WITHOUT ANY WARRANTY; without even the implied warranty of
;\c	 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;\c	 GNU General Public License for more details.
;\c	 
;\c	 You should have received a copy of the GNU General Public License
;\c	 along with Taurus; see the file COPYING.  If not, write to
;\c	 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;
;\node Internal Data Types
;\comment  node-name,  next,  previous,  up
;\chapter{Internal Data Types}
;
;This section describes the internal data types used by Taurus. The
;data types are implemented in the style of abstract data types, ie.
;constructors, selectors and recognisers for each data type are
;provided.
;
;The types are not strictly speaking abstract, since their
;representation is not protected. Instead each data type is
;implemented as a list whose \code{car} part contains a tag and the
;rest of the list contains the actual data.
;
;These procedures form the module
;
(module texpr)

;\node The Data Types Provided
;\comment  node-name,  next,  previous,  up
;\section{The Data Types Provided}
;
;\c PROPABLY NOT NEEDED
;\c node expression
;\c comment  node-name,  next,  previous,  up
;\c subsection expression
;\c 
;\c Although there is no distinct data type called expression, there are a
;\c few generic operations that can be applied to all the data types that
;\c can appear in Taurus EBNF expressions (sequence, alternative,
;\c terminal, nonterminal, iterations and action). These are:
;\c table \c ode
;\c item (expr-first-set expr)
;\c returns a list of terminal symbols that can start the given
;\c expression.
;\c item (expr-follow-set expr)
;\c returns a list of terminal symbols that can follow the given
;\c expression. 
;\c end table
;
;\node sequence
;\comment  node-name,  next,  previous,  up
;\subsection{sequence}
;
;\begin{description}
;\item[(make-sequence expressions)]
;takes a list of expressions and returns a sequence that contains the
;\code{expressions}.
;\item[(sequence? expr)]
;recognises a sequence.
;\item[(sequence-elements seq)]
;returns a list of expressions that make up the sequence.
;\end{description}
;
;\node alternative
;\comment  node-name,  next,  previous,  up
;\subsection{alternative}
;
;\begin{description}
;\item[(make-alternative choices)]
;takes a list of expressions and returns an alternative expression
;that chooses between \code{choices}.
;\item[(alternative? expr)]
;recognises an alternative.
;\item[(alternative-choices alt)]
;returns a list of expressions that the expression chooses from.
;\end{description}
;
;\node terminal
;\comment  node-name,  next,  previous,  up
;\subsection{terminal}
;
;\begin{description}
;\item[(make-terminal name)]
;constructs a terminal.
;\item[(terminal? expr)]
;recognises a terminal.
;\item[(terminal-name terminal)]
;returns the name of \code{terminal}.
;\end{description}
;
;\node nonterminal
;\comment  node-name,  next,  previous,  up
;\subsection{nonterminal}
;
;\begin{description}
;\item[(make-nonterminal name)]
;constructs a nonterminal.
;\item[(nonterminal? expr)]
;recognises a nonterminal.
;\item[(nonterminal-name nonterminal)]
;returns the name of \code{nonterminal}.
;\end{description}
;
;\node action
;\comment  node-name,  next,  previous,  up
;\subsection{action}
;
;\begin{description}
;\item[(make-action source-code)]
;constructs an action that executes \code{source-code} (semantic
;actions).
;\item[(action? expr)]
;recognises an action.
;\item[(action-source-code action)]
;returns the source code part of \code{action}.
;\end{description}
;
;\node nonzero iteration
;\comment  node-name,  next,  previous,  up
;\subsection{nonzero iteration}
;
;\begin{description}
;\item[(make-nonzero-iteration expr)]
;constructs an iteration that iterates \code{expr} one or more times.
;\item[(nonzero-iteration? expr)]
;recognises an iteration.
;\item[(iteration-iterand iterative-expr)]
;returns the expression that is iterated in \code{iterative-expr}.
;\end{description}
;
;\node zero iteration
;\comment  node-name,  next,  previous,  up
;\subsection{zero iteration}
;
;\begin{description}
;\item[(make-zero-iteration expr)]
;constructs an iteration that iterates \code{expr} zero or more times.
;\item[(zero-iteration? expr)]
;recognises an iteration.
;\item[(iteration-iterand iterative-expr)]
;returns the expression that is iterated in \code{iterative-expr}.
;\end{description}
;
;\node rule
;\comment  node-name,  next,  previous,  up
;\subsection{rule}
;
;\begin{description}
;\item[(make-rule nonterminal error-value expr)]
;constructs a grammar rule.
;\item[(rule-nonterminal rule)]
;returns the nonterminal that is being 
;defined in \code{rule}.
;\item[(rule-expr rule)]
;returns the expression part of \code{rule}.
;\item[(rule-error-value rule)]
;returns the form that the rule should evaluate and return if a syntax
;error was encountered.
;\end{description}
;
;\c PROPABLY NOT NEEDED
;\c node parser specification
;\c comment  node-name,  next,  previous,  up
;\c subsection parser specification
;\c 
;\c table \c ode
;\c item (make-parser-spec terminals nonterminals rules)
;\c constructs the internal representation of the parser specification.
;\c item (parser-terminals parser-spec)
;\c returns a list of terminals in \code{parser-spec}.
;\c item (parser-nonterminals parser-spec)
;\c returns a list of nonterminals in \code{parser-spec}.
;\c item (parser-rules parser-spec)
;\c returns a list of rules in \code{parser-spec}. Rules are returned
;\c in the order they appear in the Taurus input, ie. the starting
;\c nonterminal first.
;\c item (grammar-rule-for nonterminal parser-spec)
;\c returns the rule that defines \code{nonterminal} in \code{parser-spec}.
;\c end table
;
;\node empty
;\comment  node-name,  next,  previous,  up
;\subsection{empty}
;
;\begin{description}
;\item[(make-empty)]
;constructs the empty symbol.
;\item[(empty? expr)]
;recognises the empty symbol.
;\end{description}
;
;
;\node Implementation of Data Types
;\comment  node-name,  next,  previous,  up
;\section{Implementation of Data Types}
;
;\node sequence
;\comment  node-name,  next,  previous,  up
;\subsection{sequence}
;
;\findex{make-sequence}
(define (make-sequence elements)
  `(sequence ,@elements))

;\findex{sequence?}
(define (sequence? expr)
    (if (pair? expr)
    	(equal? (car expr) 'sequence)
	#f))

;\findex{sequence-elements}
(define sequence-elements cdr)

;\node alternative
;\comment  node-name,  next,  previous,  up
;\subsection{alternative}
;
;\findex{make-alternative}
(define (make-alternative choices)
    `(alternative ,@choices))

;\findex{alternative?}
(define (alternative? expr)
    (if (pair? expr)
    	(equal? (car expr) 'alternative)
	#f))

;\findex{alternative-choices}
(define alternative-choices cdr)

;\node terminal
;\comment  node-name,  next,  previous,  up
;\subsection{terminal}
;
;\findex{make-terminal}
(define (make-terminal name)
    `(terminal ,name))

;\findex{terminal?}
(define (terminal? expr)
    (if (pair? expr)
    	(equal? (car expr) 'terminal)
	#f))

;\findex{terminal-name}
(define (terminal-name expr)
  (cond ((equal? expr (make-empty)))
	(else (cadr expr))))
      

;\node nonterminal
;\comment  node-name,  next,  previous,  up
;\subsection{nonterminal}
;
;\findex{make-nonterminal}
(define (make-nonterminal name)
  `(nonterminal ,name))

;\findex{nonterminal?}
(define (nonterminal? expr)
    (if (pair? expr)
    	(equal? (car expr) 'nonterminal)
	#f))

;\findex{nonterminal-name}
(define nonterminal-name cadr)

;\node iterations
;\comment  node-name,  next,  previous,  up
;\subsection{iterations}
;
;\findex{make-zero-iteration}
(define (make-zero-iteration expr)
  `(zero-iteration ,expr))

;\findex{zero-iteration?}
(define (zero-iteration? expr)
    (if (pair? expr)
    	(equal? (car expr) 'zero-iteration)
	#f))

;\findex{make-nonzero-iteration}
(define (make-nonzero-iteration expr)
  `(nonzero-iteration ,expr))

;\findex{nonzero-iteration?}
(define (nonzero-iteration? expr)
    (if (pair? expr)
    	(equal? (car expr) 'nonzero-iteration)
	#f))

;\findex{iteration-iterand}
(define iteration-iterand cadr)


;\node action
;\comment  node-name,  next,  previous,  up
;\subsection{action}
;
;\findex{make-action}
(define (make-action source-code)
    `(action ,source-code))

;\findex{action?}
(define (action? expr)
    (if (pair? expr)
    	(equal? (car expr) 'action)
	#f))

;\findex{action-source-code}
(define action-source-code cadr)

;\node rule
;\comment  node-name,  next,  previous,  up
;\subsection{rule}
;
;\findex{make-rule}
(define (make-rule nonterminal error-value expr)
  `(rule ,nonterminal ,error-value ,expr))

;\findex{rule-nonterminal}
(define rule-nonterminal cadr)

;\findex{rule-error-value}
(define rule-error-value caddr)

;\findex{rule-expr}
(define rule-expr cadddr)

;\node empty
;\comment  node-name,  next,  previous,  up
;\subsection{empty}
;
;The empty symbol
;
;\findex{make-empty}
(define (make-empty) 'empty)
;\findex{empty?}
(define (empty? expr) (eq? expr 'empty))


