/* $Header: empty.c,v 3.0 88/04/18 14:34:56 jos Locked $ */
/*
 *  This file is part of the Amsterdam SGML Parser.
 *
 *  Copyright: Faculteit Wiskunde en Informatica
 *             Department of Mathematics and Computer Science
 *             Vrije Universiteit Amsterdam
 *             The Netherlands
 *
 *  Authors:   Sylvia van Egmond
 *             Jos Warmer
 */

#include "types.h"
#include "context.h"
#include "element.h"
#include "empty.h"
#include "node.h"

#ifdef DEBUG
static Bool    debug         = FALSE;
#endif

static int     changed;
static P_Group undefined_elements;
static P_Set   used;
static Bool    all_reachable = FALSE;

void undef_elems      ();
void excl_incl_defined();
void unreferenced     ();

void reach(node, elem)
P_Node     node;
P_Element  elem;
{
    P_Element  new_elem;

    if( node_type(node) == KEY ){
	if( node_key(node) == ANY ){
	    all_reachable = TRUE;
	}
    }
    if( node_type(node) == GI_NAME ){
	new_elem = element(node_gi(node));
	if( new_elem != 0 ){
	    set_add(used, element_number( node_gi(node)));
	    if( not element_done(new_elem) ){
		element_set_done(new_elem, TRUE);
		node_traverse_pre( content(new_elem), reach, new_elem);
	    }
	}
    }
}

void unreachable()
{
    P_Element  elem;
    P_Iterator iter;
    P_Element  doc_elem;

    doc_elem = element( DOCUMENT );
    element_set_done(doc_elem, TRUE);

    set_add(used, element_number(DOCUMENT));
    node_traverse_pre( content(doc_elem), reach, doc_elem );

    iter = element_iterator();
    while( elem = next_element(iter) ){
	element_set_done(elem, FALSE);
    }
}

void may_be_empty(node, elem)
P_Node    node;
P_Element elem;
{
    int        t;
    P_Iterator it;
    P_Node     n;
    int        empty     = FALSE;
    int        not_empty = FALSE;
    int        nullable  = Undecided;
    
    t = node_type(node);   
    switch( t ){
        case CONN_OR :
	    nullable = No;
            it = group_iterator(node_group(node));
            while( n = next_node(it) ){
		switch( nullable ){
		    case Undecided : if( node_nullable(n) == Yes ){
					 nullable = Yes;
				     };
				     break;
		    case No        : nullable = node_nullable(n);
				     break;
		    case Yes       : break;
		}
            }
            break;
        case CONN_AND:
        case CONN_SEQ:
        case GI :
	    nullable = Yes;
            it = group_iterator(node_group(node));
            while( n = next_node(it) ){
		switch( nullable ){
		    case Undecided : if( node_nullable(n) == No ){
					 nullable = No;
				     }
				     break;
		    case Yes       : nullable = node_nullable(n);
				     break;
		    case No        : break;
		}
            }
            break;
        case GI_START:
        case GI_END  :
	    nullable = No;
            break;
        case GI_NAME :
	    if( not element(node_gi(node)) ){
		nullable = No;
	    } else {
		nullable = node_nullable( content(element(node_gi(node))) );
	    }
            break;
        case KEY     :
	    nullable = Yes;
            break;
        default:
	    report(NO_LABEL, FATAL, 0, 0, t, "may_be_empty");
            break;
    }

    if( (node_occ(node) == OCC_OPT) or (node_occ(node) == OCC_REP) ){
	nullable = TRUE;
    }

    if( node_nullable(node) != nullable ){
        node_set_nullable(node, nullable);
	changed = TRUE;
    }
}

void calc_empty()
{
#ifdef DEBUG
    P_Element  elem;
    P_Iterator iter;
    FILE      *out;
#endif

    changed    = TRUE;
    while( changed ){
	changed = FALSE;
	elem_traverse_post(may_be_empty);
    }

#ifdef DEBUG
    if (!debug) return;
    if( not (out = fopen("empty", "w")) ){
	report(FILE_OPEN, FATAL, 0, 0, "empty");
    }
    iter = element_iterator();
    while( elem = (P_Element)iterator_next(iter) ){
	group_print(out, name_group(elem));
	switch( node_nullable(content(elem)) ){
	    case Undecided : fprintf(out, ": Undecided");  break;
	    case Yes       : fprintf(out, ": Yes      ");  break;
	    case No        : fprintf(out, ": No       ");  break;
	}
	fprintf(out, "\n");
    }
    fclose(out);
#endif
}

void clean_grammar()
{
    used       = new_set( number_of_elements()+1);
    set_add(used, 1);

    undef_elems();
    excl_incl_defined();
    unreachable();
    unreferenced();
    delete_set(used);

    fatal_report();
}

void find_undef(node, elem)
P_Node     node;
P_Element  elem;
{
    if( node_type(node) == GI_NAME ){
	if( not element(node_gi(node)) ){
	    group_add_unique(undefined_elements, node_gi(node), streq);
	}
    }
}

void undef_elems()
{
    P_Iterator   it;
    String       name;

    undefined_elements = group_create();
    elem_traverse_post(find_undef);

    it = group_iterator(undefined_elements);
    while( name = next_name(it) ){
	report(ELEM_UNDEF, FATAL, 0, 0, name);
    }
}

void excl_incl_defined()
{
    P_Iterator   it_elem;
    P_Iterator   it_ex_incl;
    P_Element    elem;
    P_Element    except_elem;
    String       ex_incl_name;

    it_elem = element_iterator();
    while( elem = next_element(it_elem) ){
	it_ex_incl = group_iterator(inclusions(elem));
	while( ex_incl_name = next_name(it_ex_incl) ){
	    except_elem = element(ex_incl_name);
	    if( except_elem == 0 ){
		report(INCL_UNDEF, FATAL, 0, 0, ex_incl_name);
	    } else {
		set_add(used, element_number(ex_incl_name));
		node_traverse_pre(content(except_elem), reach, except_elem);
	    }
	}
	it_ex_incl = group_iterator(exclusions(elem));
	while( ex_incl_name = next_name(it_ex_incl) ){
	    except_elem = element(ex_incl_name);
	    if( except_elem == 0 ){
		report(EXCL_UNDEF, FATAL, 0, 0, ex_incl_name);
	    }
	}
    }
}

void unreferenced()
{
    int  i;

    if( all_reachable ){ return; }

    for( i=1; i<=number_of_elements(); i++){
	if( not set_member(used, i) ){
	    report(ELEM_UNREACH, FATAL, 0, 0, number_to_name(i));
	}
    }
}

void clear_nullable(node, elem)
P_Node      node;
P_Element   elem;
{
    node_set_nullable(node, Undecided);
}

Bool sons_are_nullable(node)
P_Node node;
{
    P_Iterator it = group_iterator( node_group(node) );
    P_Node     child;
    Bool       answer ;
    int        t = node_type(node);

    answer = ( (t==CONN_OR) ? FALSE : TRUE);
    while( child = next_node(it) ){
	switch(t){
	    case CONN_AND:
	    case CONN_SEQ: answer = answer and (node_nullable(child)==Yes);
			   break;
	    case CONN_OR : answer = answer or (node_nullable(child)==Yes);
			   break;
	    default: report(NO_LABEL, FATAL, 0, 0, t, "son_is_nullable");
		     break;
	}
    }
    return answer;
}

void change_plus(node, elem)
P_Node    node;
P_Element elem;
{
    /*  First, convert a node that can be empty and has a plus indicator
     *  into a node with a REP inducator.
     */
    if( (node_occ(node) == OCC_PLUS) and (node_nullable(node)==TRUE) ){
	node_set_occ(node, OCC_REP);
    }

    /*  Secondly, convert a SEQ or AND group with a REP indicator into
     *  an OR group with a REP indicator
     *  iff
     *  all children of the group can be empty.
     */
    if( node_occ(node)==OCC_REP ){
	switch( node_type(node) ){
	    case CONN_SEQ:
	    case CONN_AND:
	        if( sons_are_nullable(node) ){
		    DEB1("change conn %d into OR\n", node_type(node));
		    node_set_group_type(node, CONN_OR);
		 }
	         break;
	    default: break;
        }
    }
}

void rep_node(node)
P_Node    node;
{
    P_Iterator   it;
    P_Node       child;

    switch(node_type(node)){
	case CONN_OR : node_set_occ(node, NO_OCC);
		       if( node_nullable(node) == Yes ){
			   it = group_iterator( node_group(node) );
			   while(child = next_node(it) ){
			       rep_node(child);
			   }
		       }
		       break;
	case CONN_SEQ:
	case CONN_AND: node_set_occ(node, NO_OCC);
		       if( sons_are_nullable(node) ){
			   node_set_group_type(node, CONN_OR);
			   rep_node(node);
		       }
		       break;
	case KEY     : node_set_key(node, XPCDATA);
		       break;
	case GI      : node_set_occ(node, NO_OCC);
		       break;
	case GI_START:
	case GI_END  :
	case GI_NAME : break;
	default : report(NO_LABEL, FATAL, 0, 0, node_type(node), "rep_node");
    }
}

void repeat_node(node, elem)
P_Node    node;
P_Element elem;
{
    P_Iterator   it;
    P_Node       child;

    if( (node_occ(node)==OCC_REP) and (node_type(node)==CONN_OR)
	and sons_are_nullable(node) )
    {
	it = group_iterator( node_group(node) );
	while(child = next_node(it) ){
	   rep_node(child);
	}
    }
    if( (node_occ(node)==OCC_OPT) and is_connector(node_type(node)) and
	 sons_are_nullable(node) )
    {
	node_set_occ(node, NO_OCC);
    }
}

static void convert_plus(node, elem)
P_Node     node;
P_Element  elem;
{
    element_set_busy(elem, FALSE);
    if( node_occ(node) == OCC_PLUS ){
	node_convert_plus(node);
    }
}

static void convert_plusses_and_set_busy()
{
    elem_traverse_pre(convert_plus );
}

void solve_empty()
{
    changed = FALSE;
    elem_traverse_post(change_plus);

    elem_traverse_pre(repeat_node);

    elem_traverse_post(clear_nullable);
    calc_empty();

    convert_plusses_and_set_busy();
}

void calc_status()
{
    P_Element  elem;
    P_Iterator iter;

    iter = element_iterator();
    while( elem = (P_Element)iterator_next(iter) ){
	calculate_status( content(elem) );
    }
}

#ifdef DEBUG
void debug_empty(b)
Bool b;
{
	debug = b;
}
#endif
