#include "world.h"


/**************************************************************************/
/* MACRO  **************        WritePragmas       ************************/
/**************************************************************************/
/* PURPOSE: WRITE PRAGMAS TO output.                                      */
/**************************************************************************/

#define WritePragmas(x)  if ( (x)->na )                                  \
			     fprintf( output, " %%na=%s", (x)->name );   \
			 if ( (x)->mk )                                  \
                             fprintf( output, " %%mk=%c", (x)->mark );   \
                         if ( (x)->sf )                                  \
			     fprintf( output, " %%sf=%s", (x)->file );   \
                         if ( (x)->fn )                                  \
			     fprintf( output, " %%fn=%s", (x)->funct );  \
                         if ( (x)->sl )                                  \
			     fprintf( output, " %%sl=%d", (x)->line );   \
                         fprintf( output, "\n" )


/**************************************************************************/
/* LOCAL  **************        WriteStamps        ************************/
/**************************************************************************/
/* PURPOSE: WRITE THE IF1 STAMPS IN THE STAMP TABLE TO output.            */
/**************************************************************************/

static void WriteStamps()
{
    register int i;

    for ( i = 0; i < 127; i++ )
	if ( stamps[i] != NULL )
	    fprintf( output, "C$  %c %s\n", i, stamps[i] );
}


/**************************************************************************/
/* LOCAL  **************        WriteGlobals       ************************/
/**************************************************************************/
/* PURPOSE: WRITE GLOBAL NODES IN THE IMPORT LIST TO output.              */
/**************************************************************************/

static void WriteGlobals()
{
  register PNAME i;

  for ( i = inames; i != NULL; i = i->next ) {
    if ( i->mark == ' ' )
      continue;

    fprintf( output, "I %2d", i->info->label );
    fprintf( output, " \"%s\"", i->name );

    fprintf( output, " %%mk=%c", i->mark );
    fprintf( output, "\n" );
    }
}


/**************************************************************************/
/* LOCAL  **************         WriteInfo         ************************/
/**************************************************************************/
/* PURPOSE: WRITE THE INFO NODE LIST HEADED BY ihead TO OUTPUT. A NODE IS */
/*          ONLY PRINTED IF ITS PRINT FIELD HAS NOT BEEN DISABLED.        */
/**************************************************************************/

static void WriteInfo()
{
    register PINFO i;
    register int   t;

    for ( i = ihead; i != NULL; i = i->next ) {
	if ( !(i->print) )
	    continue;

        if ( IsBasic(i) )
	    t = IF_BASIC;
        else
            t = i->type;

        fprintf( output, "T %2d %2d", i->label, t );

        switch ( i->type ) {
            case IF_FUNCTION:
	        fprintf( output, " %2d", (i->F_IN == NULL)? 0 : i->F_IN->label);
	        fprintf( output, " %2d", i->F_OUT->label );
	        break;

            case IF_STREAM:
            case IF_MULTPLE:
            case IF_ARRAY:
	        fprintf( output, " %2d   ", i->A_ELEM->label );
                break;

            case IF_UNION:
            case IF_RECORD:
	        fprintf( output, " %2d   ", i->R_FIRST->label );
                break;

            case IF_TUPLE: 
            case IF_FIELD:
            case IF_TAG:
	        fprintf( output, " %2d", i->L_SUB->label );
	        fprintf( output, " %2d", 
				 (i->L_NEXT == NULL)? 0 : i->L_NEXT->label );
					 
                break;

            case IF_UNKNOWN:
	        fprintf( output, "      " );
	        break;

	    default:
	        fprintf( output, " %2d   ", i->type - BASIC_TYPE_START );
	        break;
            }

	WritePragmas( i );
	}
}


/**************************************************************************/
/* LOCAL  **************         WriteConst        ************************/
/**************************************************************************/
/* PURPOSE: WRITE CONSTANT c WITH TYPE t TO output. IF THE CONSTANT IS    */
/*          REPRESENTED BY A NULL POINTER, THEN IT IS AN ERROR CONSTANT.  */
/*          APPROPRIATE DELIMITERS ARE DETERMINED BASED ON THE CONSTANTS  */
/*          TYPE.                                                         */
/**************************************************************************/

void WriteConst( c, t )
char  *c;
PINFO  t;
{
    if ( c == NULL ) {
        fprintf( output, " \"%s\"", ERROR_CONSTANT );
	return;
	}

    switch ( t->type ) {
        case IF_CHAR:
            fprintf( output, " \"\'%s\'\"", c );
            break;

        case IF_ARRAY:
        case IF_STREAM:
            fprintf( output, " \"\"%s\"\"", c );
            break;

        default:
            fprintf( output, " \"%s\"", c );
            break;
        }
}


/**************************************************************************/
/* LOCAL  **************       WriteNodes          ************************/
/**************************************************************************/
/* PURPOSE: PRINT THE NODES, EDGES, AND  LITERALS  OF THE  MODULE TO      */
/*          output.                                                       */
/**************************************************************************/

static void WriteNodes()
{
  register PNODE n;

  for ( n = nhead; n != NULL; n = n->next ) {
    switch ( n->type ) {
      case IFEdge:
        fprintf( output, "E %2d %2d   %2d %2d  %2d", n->snode, n->eport,
                 n->dnode, n->iport, n->info->label                   );
        break;

      case IFLiteral:
        fprintf( output, "L         %2d %2d  %2d", n->dnode, n->iport,
                 n->info->label                                     );

        WriteConst( n->CoNsT, n->info );
        break;

      case IFCBegin:
        fputc( '{', output );
        break;

      case IFCEnd:
        fprintf( output, "} %s", n->CoNsT );
        break;

      case IFIGraph:
        Error2( "WriteNodes", "IFIGraph ENCOUNTERED" );
        break;

      case IFXGraph:
        fprintf( output, "X %2d", n->info->label );
        fprintf( output, " \"%s\"", n->CoNsT );
	break;

      case IFLGraph:
        fprintf( output, "G %2d", n->info->label );
        fprintf( output, " \"%s\"", n->CoNsT );
        break;

      case IFSGraph:
        fprintf( output, "G %2d", GRAPH_LABEL );
        break;

      default:
        fprintf( output, "N %2d %2d", n->label, n->type );
        break;
      }

    WritePragmas( n );
    }
}


/**************************************************************************/
/* GLOBAL **************         If1Write          ************************/
/**************************************************************************/
/* PURPOSE: PRINT THE MONOLITH TO output. THE TYPE INFORMATION IS FOUND   */
/*          TRAVERSING ihead; THE STAMPS ARE FOUND IN ARRAY stamps; AND   */
/*          THE FUNCTION GRAPHS ARE FOUND TRAVERSING nhead.               */
/**************************************************************************/

void If1Write()
{
  WriteInfo();
  WriteStamps();
  WriteGlobals();
  WriteNodes();
}
