#include "world.h"


#define MaxLevel  100                 /* MAXIMUM NUMBER OF NESTING LEVELS */

struct level {                        /* LEVEL OCCURRENCE COUNTERS        */
    int lits;
    int edges;
    int simples;
    int comps;
    int graphs;
    };

static struct level levels[MaxLevel];     /* OCCURRENCE COUNT LEVEL STACK */
static int          maxl;                 /* MAXIMUM ENCOUNTERED LEVEL    */
static int          topl;                 /* TOP OF LEVEL STACK           */ 

static int lits;                          /* TOTAL OCCURRENCE COUNTERS    */
static int edges;
static int simples;
static int graphs;
static int comps;
static int rsum;
static int rprod;
static int rcat;
static int rleast;
static int rgreat;

static int gnodes[IF1GraphNodes];
static int snodes[IF1SimpleNodes];
static int cnodes[IF1CompoundNodes];


/**************************************************************************/
/* LOCAL  **************           Count           ************************/
/**************************************************************************/
/* PURPOSE: COUNT NODES, LITERALS, AND EDGES IN GRAPH g, PRODUCING A PER- */
/*          LEVEL SUMMARY AND TOTALS.                                     */
/**************************************************************************/

static void Count( g )
PNODE g;
{
    register PNODE n;
    register PEDGE i;

    if ( (++topl) >= MaxLevel )
	Error1( "Count: LEVEL STACK OVERFLOW" );

    if ( topl > maxl )
	maxl = topl;

    levels[topl].graphs++;

    gnodes[ g->type - IFSGraph ]++; graphs++;

    for ( i = g->imp; i != NULL; i = i->isucc )
	if ( IsConst( i ) ) {
	    lits++;
	    levels[topl].lits++;
        } else {
	    edges++;
	    levels[topl].edges++;
	    }

    for ( n = g->G_NODES; n != NULL; n = n->nsucc ) {
        for ( i = n->imp; i != NULL; i = i->isucc )
	    if ( IsConst( i ) ) {
	        lits++;
	        levels[topl].lits++;
            } else {
	        edges++;
	        levels[topl].edges++;
	        }

	if ( IsSimple( n ) ) {
	    if ( IsPeek( n ) ) 
	      continue;

            snodes[ n->type - IFAAddH ]++; simples++; levels[topl].simples++;
	    
            if ( IsReduction( n ) )
		switch ( n->imp->CoNsT[0] ) {
		    case REDUCE_SUM:
			rsum++;
			break;

		    case REDUCE_PRODUCT:
			rprod++;
			break;

		    case REDUCE_LEAST:
			rleast++;
			break;

		    case REDUCE_GREATEST:
			rgreat++;
			break;

		    case REDUCE_CATENATE:
			rcat++;
			break;

		    default:
			Error1( "ILLEGAL REDUCTION FUNCTION" );
		    }

	    continue;
	    }

	if ( IsCompound( n ) ) {
	  cnodes[ n->type - IFForall ]++; 
	  comps++; 
	  levels[topl].comps++;

	  for ( g = n->C_SUBS; g != NULL; g = g->gsucc )
            Count( g );
	  }
	}

    topl--;
}


/**************************************************************************/
/* LOCAL  **************       WriteCountInfo      ************************/
/**************************************************************************/
/* PURPOSE: WRITE COUNT VALUES TO stderr: TOTALS AND A SUMMARY BY LEVEL.  */
/**************************************************************************/

static void WriteCountInfo( msg )
char *msg;
{
    register struct level *l;
    register int           i;

    fprintf( stderr, "\n   * OCCURRENCE COUNTS %s\n\n", msg );

    fprintf( stderr, " Lits  %4d Edges %4d Smpls %4d Comps %4d Grphs %4d",
		       lits, edges, simples, comps, graphs              );

    for ( i = 0; i < IF1GraphNodes; i++ ) {
	if ( (i % 7) == 0 )
	    fprintf( stderr, "\n" );

        fprintf( stderr, " %-5.5s %4d", gnames[i], gnodes[i] );
	}

    for ( i = 0; i < IF1CompoundNodes; i++ ) {
	if ( (i % 7) == 0 )
	    fprintf( stderr, "\n" );

        fprintf( stderr, " %-5.5s %4d", cnames[i], cnodes[i] );
	}

    for ( i = 0; i < IF1SimpleNodes; i++ ) {
	if ( (i % 7) == 0 )
	    fprintf( stderr, "\n" );

        fprintf( stderr, " %-5.5s %4d", snames[i], snodes[i] );
	}

    fprintf( stderr, "\n" );

    fprintf( stderr, " RSum  %4d RProd %4d RLst  %4d RGrt  %4d RCat  %4d\n\n",
		       rsum, rprod, rleast, rgreat, rcat                    );

    fprintf( stderr, "   * SUMMARY BY LEVEL\n\n" );

    for ( i = 0; i <= maxl; i++ ) {
	l = &(levels[i]);

        fprintf( stderr, " Level %4d            Lits  %4d Edges %4d", i, 
			 l->lits, l->edges                           );

	fprintf( stderr, " Smpls %4d Comps %4d Grphs %4d\n",
		         l->simples, l->comps, l->graphs  );
        }

}


/**************************************************************************/
/* GLOBAL **************         If1Count          ************************/
/**************************************************************************/
/* PURPOSE: COUNT OCCURRENCES OF GRAPH NODES, SUBGRAPH NODES, SIMPLE      */
/*          NODES, COMPOUND NODES, LITERALS, AND EDGES IN ALL FUNCTION    */
/*          GRAPHS.  THE COUNTS ARE PRINTED TO stderr.                    */
/**************************************************************************/

void If1Count( msg )
char *msg;
{
    register struct level *l;
    register PNODE         f;
    register int           i;

    lits = edges  = simples = graphs = comps = 0;
    rsum = rgreat = rleast  = rcat   = rprod = 0;

    topl = maxl =  -1;

    for ( i = 0; i < MaxLevel; i++ ) {
	l = &(levels[i]);

	l->lits = l->edges = l->simples = l->comps = l->graphs = 0;
	}

    for ( i = 0; i < IF1GraphNodes; i++ )
	gnodes[i] = 0;

    for ( i = 0; i < IF1SimpleNodes; i++ )
	snodes[i] = 0;

    for ( i = 0; i < IF1CompoundNodes; i++ )
	cnodes[i] = 0;

    for ( f = glstop->gsucc; f != NULL; f = f->gsucc )
	Count( cfunct = f );

    WriteCountInfo( msg );
}


static void PrintLInfo( lvl, c, l )
int  lvl;
char  *c;
PNODE  l;
{
  register int i;

   for ( i = 1; i <= lvl; i++ )
     fprintf( stderr, " " );

   fprintf( stderr, "(%d) %s [%s,%s,line=%d]\n", 
	    lvl, c, l->file, l->funct, l->line );
}


static void WriteTheLMap( lvl, g )
int   lvl;
PNODE g;
{
  register PNODE n;
  register PNODE sg;

  for ( n = g->G_NODES; n != NULL; n = n->nsucc ) {
    switch ( n->type ) {
      case IFForall:
	PrintLInfo( lvl, "FORALL", n );
	WriteTheLMap( lvl + 1, n->F_BODY );
	break;

      case IFLoopB:
      case IFLoopA:
	PrintLInfo( lvl, "FOR INITIAL", n );
	WriteTheLMap( lvl + 1, n->L_BODY );
	break;

      case IFSelect:
	WriteTheLMap( lvl, n->S_CONS );
	WriteTheLMap( lvl, n->S_ALT );
	break;

      default:
	break;
      }
    }
}


void WriteLoopMap( msg )
char *msg;
{
  register PNODE f;

  for ( f = glstop->gsucc; f != NULL; f = f->gsucc ) {
    if ( IsIGraph( f ) )
      continue;

    fprintf( stderr, "\n   * LOOP MAP (%s) %s\n\n", f->G_NAME, msg );
    WriteTheLMap( 1, f );
    }
}
