#include "world.h"

static int rocnt = 0;          /* COUNT OF READ-ONLY INTERFACE AGGREGATES */
static int acnt  = 0;          /* COUNT OF INTERFACE AGGREGATES           */


/**************************************************************************/
/* GLOBAL **************     WriteInterfaceInfo    ************************/
/**************************************************************************/
/* PURPOSE: WRITE INTERFACE INFORMATION TO stderr.                        */
/**************************************************************************/

void WriteInterfaceInfo()
{
  fprintf( stderr, "\n**** INTERFACE OPTIMIZATIONS\n" );
  fprintf( stderr, " Array Input Arguments:           %d\n", acnt  );
  fprintf( stderr, " Array Read-Only Input Arguments: %d\n", rocnt );
}


/**************************************************************************/
/* LOCAL  **************     GetComponentType      ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE COMPONENT TYPE STRUCTURE FOR ARRAY i.              */
/**************************************************************************/

static PINFO GetComponentType( i )
PINFO i;
{
  register PINFO ii;

  for ( ii = i->A_ELEM; ii != NULL; ii = ii->A_ELEM )
    switch ( ii->type ) {
      case IF_INTEGER:
      case IF_REAL:
      case IF_DOUBLE:
      case IF_CHAR:
        return( ii );

      case IF_ARRAY:
        continue;

      default:
        Error2( "MIXED LANGUAGE PROGRAMMING", 
		"ILLEGAL INTERFACE COMPONENT TYPE" );
      }

  Error2( "GetComponentType", "FOR LOOP FAILURE" );
}


/**************************************************************************/
/* LOCAL  **************           GetDim          ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE DIMENSIONALITY OF ARRAY i.                         */
/**************************************************************************/

static int GetDim( i )
PINFO i;
{
  register PINFO ii;
  register int   c;

  for ( c = 0, ii = i->A_ELEM; ii != NULL; ii = ii->A_ELEM ) {
    c++;

    switch ( ii->type ) {
      case IF_INTEGER:
      case IF_REAL:
      case IF_DOUBLE:
      case IF_CHAR:
        return( c );

      case IF_ARRAY:
        break;

      default:
        Error2( "Mixed Language Programming", "ILLEGAL INTERFACE TYPE" );
      }
    }

  Error2( "GetDim", "FOR LOOP FAILURE" );
  return( -1 );
}


/**************************************************************************/
/* GLOBAL **************        GetLanguage        ************************/
/**************************************************************************/
/* PURPOSE: RETURN THE SOURCE LANGUAGE OF FUNCTION f.                     */
/**************************************************************************/

int GetLanguage( f )
PNODE f;
{
  if ( IsIGraph( f ) ) {
    if ( f->mark == 'f' ) /* NEW CANN 2/92 */
      return( FOR_FORTRAN );

    if ( f->mark == 'c' ) /* NEW CANN 2/92 */
      return( FOR_C );

    if ( f->mark == 'i' ) /* NEW CANN 2/92 */
      return( FOR_C );

    return( FOR_SISAL );
    }

  return( FOR_SISAL );
}


/**************************************************************************/
/* GLOBAL **************     BindInterfaceName     ************************/
/**************************************************************************/
/* PURPOSE: CONVERT FUNCTION NAME nm TO THE APPROPRIATE FORM FOR LANGUAGE */
/*          lang AND RETURN A COPY.                                       */
/**************************************************************************/


char *BindInterfaceName( nm, lang, mark )
char *nm;
int   lang;
char  mark;
{
  register char *p;
           char buf[100];

  if ( lang == FOR_C )
    p = LowerCase( nm, FALSE, FALSE );
  else if ( lang == FOR_FORTRAN ) {
    if ( Iupper )
      p = UpperCase( nm, IunderL, IunderR ); 
    else
      p = LowerCase( nm, IunderL, IunderR );
    }

  /* else if ( entry_point && (!standalone) ) { */

  /* NEW CANN 2/92: forF or forC AND ENTRY POINT */
  else if ( mark == 'c' || mark == 'f'  ) {
    sprintf( buf, "_%s", UpperCase( nm, TRUE, FALSE ) );
    p = CopyString( buf );
    }

  else 
    p = UpperCase( nm, TRUE, FALSE );

  return( p );
}


/**************************************************************************/
/* LOCAL  **************  PrintAbsoluteBaseAddress ************************/
/**************************************************************************/
/* PURPOSE: PRINT THE ABSOLUTE BASE ADDRESS OF ARRAY i TO output.         */
/**************************************************************************/

static void PrintAbsoluteBaseAddress( i )
PEDGE i;
{
  fprintf( output, "((%s*)(((ARRAYP)", i->info->A_ELEM->tname );
  PrintTemp( i );
  fprintf( output, ")->Base))+((ARRAYP)" );
  PrintTemp( i );
  fprintf( output, ")->LoBound" );
}


static PEDGE FindDescriptor( n, f, me )
PNODE n;
PNODE f;
int   me;
{
  register PEDGE i;
  register PINFO out;
  register int   ac;
  register int   c;
  register int   wanted;

  for ( i = n->imp; i != NULL; i = i->isucc )
    if ( i->isucc == NULL )
      break;

  for ( ac = 0, out = f->info->F_OUT; out != NULL; out = out->L_NEXT )
    if ( IsArray( out->L_SUB ) )
      ac++;

  if ( ac <= 0 )
    Error2( "FindDescriptor", "INTERFACE ARRAY OUTPUTS NOT FOUND" );

  if ( i == NULL )
    Error2( "FindDescriptor", "INTERFACE ARRAY DESCRIPTOR NOT FOUND" );

  wanted = ac - me + 1;

  for ( c = 1; i != NULL; i = i->ipred, c++ )
    if ( c == wanted )
      break;

  if ( i == NULL )
    Error2( "FindDescriptor", "INTERFACE ARRAY DESCRIPTOR NOT FOUND (2)" );

  if ( !IsArray( i->info ) )
    Error2( "FindDescriptor", "INTERFACE DESCRIPTOR NOT AN ARRAY" );

  if ( !IsInteger( i->info->A_ELEM ) )
    Error2( "FindDescriptor", "INTERFACE DESCRIPTOR NOT AN ARRAY OF INTEGER" );

  return( i );
}


/**************************************************************************/
/* LOCAL  **************    PrintInterfaceCall     ************************/
/**************************************************************************/
/* PURPOSE: PRINT INTERFACE INVOCATION n OF FUNCTION f TO output.         */
/**************************************************************************/

void PrintInterfaceCall( indent, n, f )
int   indent;
PNODE n;
PNODE f;
{
  register PEDGE i;
  register int   c;
  register int   lang;
  register PEDGE dv;
  register PINFO ct;
  register PINFO tct;
  register int   td;
  register int   d;
  register PINFO rt;
  register int   r;
  register int   ac;
  register int   cmp;
  register PEDGE e;
  register int   arr;
  register char  *op;

  fprintf( output, "/* INTERFACE MODULE CALL */\n" );

  lang = GetLanguage( f );

  switch ( lang ) {
    case FOR_C:
    case FOR_FORTRAN:
      fprintf( output, "{\n" );

      /* DECLARE STORAGE FOR PASS BY REFERENCE OPERATION */
      for ( i = n->imp->isucc; i != NULL; i = i->isucc ) {
	if ( IsArray( i->info ) ) {
          ct = GetComponentType( i->info );
	  fprintf( output, "%s *itmp%d;\n", ct->tname, i->iport );
	  continue;
	  }

	if ( lang != FOR_C )
	  fprintf( output, "%s itmp%d;\n", i->info->tname, i->iport );
	}

/* CANN NEW 3/92 */

      /* cmp == TRUE IF 2 OR MORE RETURN VALUES OR 1 ARRAY RESULT */
      cmp = (f->info->F_OUT->L_NEXT == NULL)? FALSE : TRUE;
      cmp = (IsArray(f->info->F_OUT->L_SUB))? TRUE : cmp;

      for ( arr=FALSE,r=0, rt = f->info->F_OUT; rt != NULL; rt = rt->L_NEXT ) {
	r++;

        if ( IsArray( rt->L_SUB ) ) {
          ct = GetComponentType( rt->L_SUB );
          d  = GetDim( rt->L_SUB );
	  fprintf( output, "%s *iret%d;\n", ct->tname, r );
	  arr = TRUE;
	  }
	else if ( cmp ) {
	  fprintf( output, "%s iret%d;\n", rt->L_SUB->tname, r );
	  }
	}

      if ( r > 0 )
        fprintf( output, "int iretsz;\n" );
      if ( arr )
        fprintf( output, "POINTER IArr;\n" );
/* END CANN NEW 3/92 */

      /* ASSIGN VALUES TO PASS BY REFERENCE STORAGE */
      for ( i = n->imp->isucc; i != NULL; i = i->isucc ) {
        if ( IsArray( i->info ) ) {
          if ( !IsArray( i->info->A_ELEM ) ) {
            fprintf( output, "PrepArr1( " );
            fprintf( output, " itmp%d, ", i->iport );
            PrintTemp( i );
            fprintf( output, ", %s );\n", i->info->A_ELEM->tname );
          } else {
            fprintf( output, "PrepArr2( " );
            fprintf( output, " itmp%d, ", i->iport );
            PrintTemp( i );
            tct = GetComponentType( i->info );
            td  = GetDim( i->info );
            fprintf( output, ", %s, %d, %d, ", tct->tname, td, 3+(td*5) );
	    fprintf( output, "I%s );\n", i->info->wname );
	    i->info->touch5 = TRUE;
	    }

          continue;
	  }

        if ( lang != FOR_C ) {
          fprintf( output, "itmp%d = ", i->iport );
          PrintTemp( i );
          fprintf( output, ";\n" );
          }
        }

/* CANN NEW 3/92 */
      for ( ac=0, r=0, rt = f->info->F_OUT; rt != NULL; rt = rt->L_NEXT ) {
	r++;

        /* PREPARE FOR AN ARRAY OUTPUT */
        if ( IsArray( rt->L_SUB ) ) {
	  ac++;

          ct = GetComponentType( rt->L_SUB );
          d  = GetDim( rt->L_SUB );
	  dv = FindDescriptor( n, f, ac );

	  if ( bounds || sdbx )
	    fprintf( output, "IDescriptorCheck( %d, itmp%d );\n", d,dv->iport);

	  fprintf( output, "iretsz =" );

	  for ( c = 0; c < d; c++ )
	    fprintf( output, "%s(itmp%d[%d]-itmp%d[%d]+1) ",
		     (c != 0)? "* " : " ",
                     dv->iport, 3+(c*5)+1, dv->iport, 3+(c*5)+0 ); 

          fprintf( output, ";\n" );

	  if ( c == 1 )
	    fprintf( output, "iretsz = (iretsz<=0)? 0 : iretsz;\n" );
          else
	    fprintf( output, "iretsz = (iretsz<=0)? 1 : iretsz;\n" );

          if ( n->Smark ) {
	    op = "Alloc";
            smallocs++;
          } else {
	    op = "pr_Alloc";
            pmallocs++;
            }

	  if ( c == 1 ) {
	    if ( (e = FindExport(n,r)) == NULL ) {

	      fprintf( output, 
		       "iret%d = (%s*) %s(iretsz*sizeof(%s));\n", 
		       r, ct->tname, op, ct->tname );

	    } else {
	      fprintf( output, 
		       "PrepRetArr1( IArr, iret%d, %s, iretsz, itmp%d );\n",
		       r, ct->tname, dv->iport );
	      PrintTemp( e );
	      fprintf( output, " = IArr;\n" );
	      }
	    }
	  else
	    fprintf( output, 
		     "iret%d = (%s*) %s(iretsz*sizeof(%s));\n", 
		     r, ct->tname, op, ct->tname );
	  }
        }
/* END CANN NEW 3/92 */

      /* MAKE THE CALL */
      if ( cmp )
        fprintf( output, "%s( ", n->imp->CoNsT );
      else {
        if ( n->exp != NULL ) { 
          PrintTemp( n->exp );
          fprintf( output, " = %s( ", n->imp->CoNsT );
          }
        else
          fprintf( output, "%s( ", n->imp->CoNsT );
        }

      for ( c = 1, i = n->imp->isucc; i != NULL; i = i->isucc, c++ ) {
        if ( IsArray( i->info ) )
          fprintf( output, "itmp%d", i->iport );
        else if ( lang != FOR_C )
          fprintf( output, "&itmp%d", i->iport );
        else 
          PrintTemp( i );

        if ( i->isucc != NULL )
          fprintf( output, ", " );

        if ( c % 5 == 0 && i->isucc != NULL )
          fprintf( output, "\n  " );
	}

/* CANN NEW 3/92 */
      if ( cmp ) {
        for ( r = 0, rt = f->info->F_OUT; rt != NULL; rt = rt->L_NEXT ) {
	  r++;
	  if ( IsArray( rt->L_SUB ) )
	    fprintf( output, ", iret%d", r );
          else
	    fprintf( output, ", &iret%d", r );
	  }
        }
/* END CANN NEW 3/92 */

      fprintf( output, " );\n" );

      /* FREE INPUT INTERFACE STORAGE */ 
      for ( i = n->imp->isucc; i != NULL; i = i->isucc )
        if ( IsArray( i->info ) )
          if ( IsArray( i->info->A_ELEM ) )
            fprintf( output, "DeAlloc( itmp%d );\n", i->iport );

/* CANN NEW 3/92 */
      if ( cmp ) {
        for ( ac=0, r=0, rt = f->info->F_OUT; rt != NULL; rt = rt->L_NEXT ) {
	  r++;

	  if ( (e = FindExport(n,r)) == NULL ) {
            if ( IsArray( rt->L_SUB ) ) {
	      ac++;
              fprintf( output, "DeAlloc( iret%d );\n", r );
	      }

	    continue;
	    }

          /* FREE OUTPUT INTERFACE STORAGE */ 
          if ( IsArray( rt->L_SUB ) ) {
	    ac++;

            ct = GetComponentType( rt->L_SUB );
            d  = GetDim( rt->L_SUB );

	    dv = FindDescriptor( n, f, ac );

	    if ( d == 1 )
	      continue;

	    rt->L_SUB->touch4 = TRUE;

	    PrintTemp( e );
	    fprintf( output, " = " );
	    fprintf( output, "I%s( FALSE, iret%d, itmp%d );\n", 
		     rt->L_SUB->rname, r, dv->iport );

            fprintf( output, "DeAlloc( iret%d );\n", r );
	    continue;
	    }

	  /* SAVE THE SCALAR (IF USED BY THE CALLER!) */
	  if ( (e = FindExport(n,r)) == NULL )
	    continue;

	  PrintTemp( e );
          fprintf( output, " = iret%d;\n", r );
	  }
	}
/* END CANN NEW 3/92 */

      fprintf( output, "}\n" );
      break;

    default:
      Error2( "PrintInterfaceCall", "ILLEGAL LANGUAGE TYPE" );
      break;
    }
}


/**************************************************************************/
/* LOCAL  **************      PrintWriteArray      ************************/
/**************************************************************************/
/* PURPOSE: PRINT WRITE OPERATION FOR ARRAY i WITH COMPONENT TYPE ct TO   */
/*          output.                                                       */
/**************************************************************************/

static void PrintWriteArray( indent, dst, src, dd, d, i, ct )
int    indent;
char  *dst;
char  *src;
int    dd;
int    d;
PINFO  i;
PINFO  ct;
{
  char nsrc[100];
  char ndst[100];
  char buf[100];

  PrintIndentation( indent );
  fprintf( output, "{\n" );

  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s *dbase%d;\n", ct->tname, dd );

  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s  i%d;\n", "int", dd );

  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s  dsize%d;\n", "int", dd );

  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s  lsize%d;\n", "int", dd );

  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s *p%d;\n", i->A_ELEM->tname, dd );


  PrintIndentation( indent+2 );
  fprintf( output, "dsize%d = dinfo[%d].DSize;\n", dd, dd-1 );

  PrintIndentation( indent+2 );
  fprintf( output, "lsize%d = dinfo[%d].LSize;\n", dd, dd-1 );


  PrintIndentation( indent+2 );
  fprintf( output, "dbase%d = ((%s*)%s)+(dsize%d*dinfo[%d].Offset);\n",
           dd, ct->tname, dst, dd, dd-1                               );


  PrintIndentation( indent+2 );
  fprintf( output, "PAElm( p%d, (*%s), dinfo[%d].SLow, %s );\n", 
           dd, src, dd-1, i->A_ELEM->tname                    );

  if ( d == 1 ) {
    PrintIndentation( indent+2 );
    fprintf( output, "if ( dinfo[%d].Mutable )\n", dd-1 );

    PrintIndentation( indent+4 );
    fprintf( output, "if ( p%d == dbase%d ) goto MoveOn;\n", dd, dd );
    }


  if ( d == 1 ) {
    PrintVECTOR();
    PrintASSOC();
    sprintf( buf, "dbase%d", dd );
    PrintSAFE( buf );
    }

  PrintIndentation( indent+2 );
  fprintf( output, "for ( i%d = 0; i%d < lsize%d; i%d++ ) {\n", 
           dd, dd, dd, dd                                    );


  if ( d == 1 ) {
    PrintIndentation( indent+4 );
    fprintf( output, "*dbase%d = p%d[i%d];\n", dd, dd, dd );

    PrintIndentation( indent+4 );
    fprintf( output, "dbase%d += dsize%d;\n", dd, dd );
  } else {
    sprintf( nsrc, "(&p%d[i%d])", dd, dd );
    sprintf( ndst, "dbase%d", dd );

    PrintWriteArray( indent+4, ndst, nsrc, dd+1, d-1, i->A_ELEM, ct );

    PrintIndentation( indent+4 );
    fprintf( output, "dbase%d += dsize%d;\n", dd, dd );
    }

  PrintIndentation( indent+4 );
  fprintf( output, "}\n" );

  if ( d == 1 )
    fprintf( output, "MoveOn:;\n" );

  /* PrintIndentation( indent+2 ); */
  /* fprintf( output, "SFreeArr( (*%s) );\n", src ); */

  PrintIndentation( indent );
  fprintf( output, "}\n" );
}


/**************************************************************************/
/* LOCAL  **************      PrintWriteRoutine    ************************/
/**************************************************************************/
/* PURPOSE: PRINT INTERFACE ARRAY WRITE ROUTINES FOR FUNCTION f TO output.*/
/**************************************************************************/

static void PrintWriteRoutine( i )
register PINFO i;
{
  register PINFO ct;
  register int   d;

  ct = GetComponentType( i );
  d  = GetDim( i );

  fprintf( output, "\nstatic void I%s( arr, out, info )\n", i->wname );
  fprintf( output, "%-6s  arr;\n", i->tname );
  fprintf( output, "%-6s *out;\n", ct->tname );
  fprintf( output, "%-6s *info;\n", "int" );
  fprintf( output, "{\n" );
  fprintf( output, "  DIMINFO dinfo[%d];\n", d );

  /* fprintf( output, "  Sequential = TRUE;\n" ); */

  if ( bounds || sdbx ) 
    fprintf( output, "IDescriptorCheck( %d, info );\n", d );

  fprintf( output, "  InitDimInfo( FALSE, %d, dinfo, info );\n", d );

  PrintWriteArray( 2, "out", "(&arr)", 1, d, i, ct );

  /* fprintf( output, "  Sequential = FALSE;\n" ); */

  fprintf( output, "}\n" );
}


/**************************************************************************/
/* LOCAL  **************      PrintReadArray       ************************/
/**************************************************************************/
/* PURPOSE: PRINT READ OPERATION FOR ARRAY i WITH COMPONENT TYPE ct TO    */
/*          output.                                                       */
/**************************************************************************/

static void PrintReadArray( indent, src, dst, dd, d, i, ct )
int    indent;
char  *src;
char  *dst;
int    dd;
int    d;
PINFO  i;
PINFO  ct;
{
  char nsrc[100];
  char ndst[100];
  char buf[100];

  PrintIndentation( indent );
  fprintf( output, "{\n" );

  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s *sbase%d;\n", ct->tname, dd );

  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s  i%d;\n", "int", dd );

  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s *p%d;\n", i->A_ELEM->tname, dd );


  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s  dsize%d;\n", "int", dd );

  PrintIndentation( indent+2 );
  fprintf( output, "register %-7s  lsize%d;\n", "int", dd );


  PrintIndentation( indent+2 );
  fprintf( output, "dsize%d = dinfo[%d].DSize;\n", dd, dd-1 );

  PrintIndentation( indent+2 );
  fprintf( output, "lsize%d = dinfo[%d].LSize;\n", dd, dd-1 );


  PrintIndentation( indent+2 );
  fprintf( output, "sbase%d = ((%s*)%s)+(dsize%d*dinfo[%d].Offset);\n",
           dd, ct->tname, src, dd, dd-1                               );


  if ( d == 1 ) {
    PrintIndentation( indent+2 );
    fprintf( output, "if ( dinfo[%d].Mutable ) {\n", dd-1 );

    PrintIndentation( indent+4 );
    fprintf( output, 
             "OptInitIArr( %s, p%d, lsize%d, dinfo[%d].SLow, %s, sbase%d );\n",
             dst, dd, dd, dd-1, i->A_ELEM->tname, dd                         );

    PrintIndentation( indent+4 );
    fprintf( output, "goto MoveOn;\n" );

    PrintIndentation( indent+4 );
    fprintf( output, "}\n" );
    }

  PrintIndentation( indent+2 );
  fprintf( output, "InitIArr( %s, p%d, lsize%d, dinfo[%d].SLow, %s );\n",
           dst, dd, dd, dd-1, i->A_ELEM->tname                         );


  if ( d == 1 ) {
    PrintVECTOR();
    PrintASSOC();
    sprintf( buf, "p%d", dd );
    PrintSAFE( buf );
    }

  PrintIndentation( indent+2 );
  fprintf( output, "for ( i%d = 0; i%d < lsize%d; i%d++ ) {\n", 
           dd, dd, dd, dd                                    );


  if ( d == 1 ) {
    PrintIndentation( indent+4 );
    fprintf( output, "p%d[i%d] = *sbase%d;\n", dd, dd, dd );

    PrintIndentation( indent+4 );
    fprintf( output, "sbase%d += dsize%d;\n", dd, dd );
  } else {
    sprintf( ndst, "(&p%d[i%d])", dd, dd );
    sprintf( nsrc, "sbase%d", dd );

    PrintReadArray( indent+4, nsrc, ndst, dd+1, d-1, i->A_ELEM, ct );

    PrintIndentation( indent+4 );
    fprintf( output, "sbase%d += dsize%d;\n", dd, dd );
    }

  PrintIndentation( indent+4 );
  fprintf( output, "}\n" );

  if ( d == 1 )
    fprintf( output, "MoveOn:;\n" );

  /* OUTERMOST LEVEL!!!! */
  if ( dd == 1 )
     fprintf( output, "  ((ARRAYP)arr)->Mutable = dinfo[0].Mutable;\n" );

  PrintIndentation( indent );
  fprintf( output, "}\n" );
}


/**************************************************************************/
/* LOCAL  **************       PrintReadRoutine    ************************/
/**************************************************************************/
/* PURPOSE: PRINT INTERFACE ARRAY READ ROUTINE TO output.                 */
/**************************************************************************/

static void PrintReadRoutine( i )
register PINFO i;
{
  register PINFO ct;
  register int   d;

  ct = GetComponentType( i );
  d  = GetDim( i );

  fprintf( output, "\nstatic %s I%s( ronly, in, info )\n", i->tname, i->rname );
  fprintf( output, "int ronly;\n" );
  fprintf( output, "%-6s *in;\n", ct->tname );
  fprintf( output, "%-6s *info;\n", "int" );
  fprintf( output, "{\n" );
  fprintf( output, "  DIMINFO dinfo[%d];\n", d );
  fprintf( output, "  %-7s arr;\n", i->tname );

  if ( bounds || sdbx ) 
    fprintf( output, "IDescriptorCheck( %d, info );\n", d );

  fprintf( output, "  InitDimInfo( ronly, %d, dinfo, info );\n", d );

  PrintReadArray( 2, "in", "(&arr)", 1, d, i, ct );

  fprintf( output, "  return( arr );\n" );
  fprintf( output, "}\n" );
}


/**************************************************************************/
/* GLOBAL **************       IsReadOnly          ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF AGGREGATE eport OF NODE n IS READ-ONLY.        */
/**************************************************************************/

int IsReadOnly( n, eport )
PNODE n;
int   eport;
{
  register PEDGE ee;
  register PEDGE e;
  register PNODE f;

  for ( e = n->exp; e != NULL; e = e->esucc ) {
    if ( e->eport != eport )
      continue;

    if ( e->cm == -1 || e->pm > 0 || e->wmark )
      return( FALSE );

    switch ( e->dst->type ) {
      case IFALimL:
      case IFALimH:
      case IFASize:
      case IFSaveCallParam:
      case IFSaveSliceParam:
	break;

      case IFCall:
	if ( (f = FindFunction( e->dst->imp->CoNsT )) == NULL)
	  Error2( "IsReadOnly", "FindFunction FOR CALL FAILED" );

	if ( IsIGraph( f ) )
	  if ( f->mark != 's' ) /* CANN NEW 2/92 */
	    break;

	return( FALSE );

      case IFLoopPoolEnq:
      case IFOptLoopPoolEnq:
	if ( (f = FindFunction( e->dst->usucc->G_NAME )) == NULL)
	  Error2( "IsReadOnly", "FindFunction FOR SLICE BODY NAME FAILED" );

	if ( !IsReadOnly( f, e->iport ) )
	  return( FALSE );

	break;

      case IFOptAElement:
	if ( e->info->type != IF_PTR )
	  break;

	if ( !IsReadOnly( e->dst, 1 ) )
	  return( FALSE );

	break;

      case IFAElement:
	if ( IsBasic( e->info->A_ELEM ) )
	  break;
      case IFGetArrayBase:
      case IFAssign:
	if ( !IsReadOnly( e->dst, 1 ) )
	  return( FALSE );

	break;

      case IFForall:
	if ( IsExport( e->dst->F_GEN, e->iport ) != NULL )
	  return( FALSE );
	if ( IsExport( e->dst->F_RET, e->iport ) != NULL )
	  return( FALSE );

	if ( !IsReadOnly( e->dst->F_BODY, e->iport ) )
	  return( FALSE );

	break;

      case IFSelect:
        if ( IsExport( e->dst->S_TEST, e->iport ) != NULL )
          return( FALSE );

        if ( IsExport( e->dst->S_CONS, e->iport ) != NULL )
	  if ( !IsReadOnly( e->dst->S_CONS, e->iport ) )
	    return( FALSE );

        if ( IsExport( e->dst->S_ALT, e->iport ) != NULL )
	  if ( !IsReadOnly( e->dst->S_ALT, e->iport ) )
	    return( FALSE );

        break;

      case IFLoopA:
      case IFLoopB:
	if ( IsExport( e->dst->L_INIT, e->iport ) != NULL )
	  return( FALSE );
	if ( IsExport( e->dst->L_TEST, e->iport ) != NULL )
	  return( FALSE );
	if ( IsExport( e->dst->L_RET,  e->iport ) != NULL )
	  return( FALSE );

	if ( !IsReadOnly( e->dst->L_BODY, e->iport ) )
	  return( FALSE );

	break;

      default:
	return( FALSE );
      }
    }

  return( TRUE );
}


/**************************************************************************/
/* LOCAL  **************       PrintReadOp         ************************/
/**************************************************************************/
/* PURPOSE: PRINT INTERFACE READ OPERATION FOR f (NAMED nm) TO output.    */
/**************************************************************************/

static void PrintReadOp( nm, f, lang )
char  *nm;
PNODE  f;
int    lang;
{
  register PINFO  i;
  register int    c;
  register int    ronly;
           char   buf[100];

  for ( c = 1, i = f->info->F_IN; i != NULL; i = i->L_NEXT, c++ ) {
    sprintf( buf, "args->In%d", c );
    ronly = FALSE;

    switch( i->L_SUB->type ) {
      case IF_INTEGER:
      case IF_REAL:
      case IF_DOUBLE:
        fprintf( output, "  %s = *In%d;\n", buf, c );
        break;

      case IF_ARRAY:
	acnt++;

	/* OPTIMIZE READ-ONLY TRANSFERS INTO SISAL */
	if ( IsReadOnly( f, c ) ) {
	  ronly = TRUE;
	  rocnt++;
	  }

	if ( bind && ronly ) {
	  fprintf( output, "  if ( _a_%s->In%d == NULL )\n", nm, c );
	  PrintIndentation( 2 );
	  }

        fprintf( output, "  %s = I%s( %s, In%d, InI%d );\n", buf,
			 i->L_SUB->rname, (ronly)? "TRUE" : "FALSE",
			 c, c                                       );
        break;

      default:
        break;
      }
    }
}


/**************************************************************************/
/* LOCAL  **************       PrintWriteOp        ************************/
/**************************************************************************/
/* PURPOSE: PRINT INTERFACE WRITE OPERATION FOR f TO output.              */
/**************************************************************************/

static void PrintWriteOp( f, lang )
PNODE f;
int   lang;
{
  register PINFO  i;
  register int    c;
           char   buf[100];

  for ( c = 1, i = f->info->F_OUT; i != NULL; i = i->L_NEXT, c++ ) {
    sprintf( buf, "args->Out%d", c );

    switch( i->L_SUB->type ) {
      case IF_INTEGER:
      case IF_REAL:
      case IF_DOUBLE:
        fprintf( output, "  *Out%d = %s;\n", c, buf );
        break;

      case IF_ARRAY:
        fprintf( output, "  I%s( %s, Out%d, OutI%d );\n", i->L_SUB->wname, 
                 buf, c, c                                             );
        break;

      default:
        break;
      }
    }
}


/**************************************************************************/
/* LOCAL  **************    PrintInterfaceHeader   ************************/
/**************************************************************************/
/* PURPOSE: PRINT INTERFACE FUNCTION HEADER FOR f WITH NAME nm TO output. */
/**************************************************************************/

static void PrintInterfaceHeader( nm, f, lang )
char  *nm;
PNODE  f;
int    lang;
{
  register PINFO  i;
  register int    c;
  register PINFO  ii;

  fprintf( output, "\nstatic %s *_a_%s;\n", f->info->sname, nm );
  fprintf( output, "static int _f_%s = TRUE;\n", nm );

  fprintf( output, "\nvoid %s(", nm );

  for ( c = 1, i = f->info->F_IN; i != NULL; i = i->L_NEXT, c++ ) {
    switch( i->L_SUB->type ) {
      case IF_INTEGER:
      case IF_REAL:
      case IF_DOUBLE:
        fprintf( output, " In%d,", c );
        break;

      case IF_ARRAY:
        fprintf( output, " In%d, InI%d,", c, c );
        break;

      default:
        Error2( "Mixed Language Programming:", "ILLEGAL INTERFACE INPUT TYPE" );
      }

    if ( (c % 5) == 0 )
      fprintf( output, "\n  " );
    }

  if ( ((c-1) % 5) != 0 )
    fprintf( output, "\n  " );

  for ( c = 1, i = f->info->F_OUT; i != NULL; i = i->L_NEXT, c++ )  {
    switch( i->L_SUB->type ) {
      case IF_INTEGER:
      case IF_REAL:
      case IF_DOUBLE:
        fprintf( output, " Out%d", c );

        if ( i->L_NEXT != NULL )
          fprintf( output, "," );

        break;

      case IF_ARRAY:
        fprintf( output, " Out%d, OutI%d", c, c );

        if ( i->L_NEXT != NULL )
          fprintf( output, "," );

        break;

      default:
        Error2( "Mixed Language Programming:", "ILLEGAL OUTPUT TYPE" );
      }

    if ( (c % 5) == 0 && (i->L_NEXT != NULL) )
      fprintf( output, "\n  " );
    }

  fprintf( output, " )\n" );

  /* PRINT FUNCTION HEADER ARGUMENT DECLARATIONS */
  for ( c = 1, i = f->info->F_IN; i != NULL; i = i->L_NEXT, c++ ) 
    switch( i->L_SUB->type ) {
      case IF_INTEGER:
      case IF_REAL:
      case IF_DOUBLE:
        fprintf( output, "%-6s *In%d;\n", i->L_SUB->tname, c );
        break;

      case IF_ARRAY:
        ii = GetComponentType( i->L_SUB );

        fprintf( output, "%-6s *In%d;\n", ii->tname, c );
        fprintf( output, "%-6s *InI%d;\n", "int", c ); 
        break;

      default:
        Error2( "Mixed Language Programming:", "ILLEGAL OUTPUT TYPE" );
      }

  for ( c = 1, i = f->info->F_OUT; i != NULL; i = i->L_NEXT, c++ ) 
    switch( i->L_SUB->type ) {
      case IF_INTEGER:
      case IF_REAL:
      case IF_DOUBLE:
        fprintf( output, "%-6s *Out%d;\n", i->L_SUB->tname, c );
        break;

      case IF_ARRAY:
        ii = GetComponentType( i->L_SUB );

        fprintf( output, "%-6s *Out%d;\n", ii->tname, c );
        fprintf( output, "%-6s *OutI%d;\n", "int", c ); 
        break;

      default:
        Error2( "Mixed Language Programming:", "ILLEGAL OUTPUT TYPE" );
      }
}


/**************************************************************************/
/* GLOBAL **************  PrintInterfaceUtilities  ************************/
/**************************************************************************/
/* PURPOSE: PRINT INTERFACE UTILITIES TO output.                          */
/**************************************************************************/

void PrintInterfaceUtilities()
{
  register PINFO i;

  for ( i = ihead; i != NULL; i = i->next ) {
    if ( i->touch4 ) {
      PrintReadRoutine( i );
      i->touch4 = FALSE;
      }

    if ( i->touch5 ) {
      PrintWriteRoutine( i );
      i->touch5 = FALSE;
      }
    }
}


/**************************************************************************/
/* GLOBAL **************      PrintInterface       ************************/
/**************************************************************************/
/* PURPOSE: PRINT INTERFACE FUNCTION FOR SISAL FUNCTION f.                */
/**************************************************************************/

void PrintInterface( f )
PNODE f;
{
  register PINFO i;
  register int   c;
  register int   lang;
  register char *nm;

  lang = (f->mark == 'c')? FOR_C : FOR_FORTRAN;

  /* MARK THE READ ROUTINES */
  for ( i = f->info->F_IN; i != NULL; i = i->L_NEXT ) {
    if ( !IsArray( i->L_SUB ) )
      continue;

    i->L_SUB->touch4 = TRUE;
    }

  /* MARK THE WRITE ROUTINES */
  for ( i = f->info->F_OUT; i != NULL; i = i->L_NEXT ) {
    if ( !IsArray( i->L_SUB ) )
      continue;

    i->L_SUB->touch5 = TRUE;
    }

  /* SKIP LEADING 2 UNDERSCORES-------v */
  /* nm = BindInterfaceName( &(f->G_NAME[2]), lang, FALSE ); */
  nm = BindInterfaceName( &(f->G_NAME[2]), lang, 's' ); /* NEW CANN 2/92 */

  PrintInterfaceHeader( nm, f, lang );

  fprintf( output, "{\n" );
  fprintf( output, "  register %s *args;\n\n", f->info->sname );

  fprintf( output, "#ifdef CInfo\n" );
  fprintf( output, "  SaveCopyInfo;\n" );
  fprintf( output, "#endif\n" );

  fprintf( output, "#ifdef FInfo\n" );
  fprintf( output, "  SaveFlopInfo;\n" );
  fprintf( output, "#endif\n" );

  if ( bind ) {
    fprintf( output, "  if ( _f_%s ) {\n", nm );
    PrintIndentation( 2 );
    }

  PrintIndentation( 2 );
  fprintf( output, "_a_%s = (%s*) Alloc( sizeof( %s ) );\n", 
		   nm, f->info->sname, f->info->sname       );

  if ( bind ) {
    for ( c = 1, i = f->info->F_IN; i != NULL; i = i->L_NEXT, c++ )
      switch( i->L_SUB->type ) {
        case IF_ARRAY:
          fprintf( output, "    _a_%s->In%d = NULL;\n", nm, c );
	  break;

	default:
	  break;
        }

    fprintf( output, "  }\n" );
    }

  fprintf( output, "  args = _a_%s;\n", nm );

  if ( gdata )
    fprintf( output, "  InitGlobalData();\n" );

  PrintReadOp( nm, f, lang );

  if ( sdbx ) {
    fprintf( output, "SdbxCurrentFunctionList = MyFunctionList;\n" );
    fprintf( output, "SdbxMonitor( SDBX_ESTART );\n" );
    }

  fprintf( output, "  %s( (POINTER) args );\n", f->G_NAME );

  if ( sdbx ) {
    fprintf( output, "SdbxMonitor( SDBX_ESTOP );\n" );
    }

  PrintInputDeallocs( nm, 2, f );

  PrintWriteOp( f, lang );

  PrintOutputDeallocs( 2, f );

  if ( !bind )
    fprintf( output, "  DeAlloc( (POINTER) args );\n" );

  if ( bind )
    fprintf( output, "  _f_%s = FALSE;\n", nm );

  fprintf( output, "}\n" );
}
