#include "world.h"


/**************************************************************************/
/* MACRO  **************        WritePragmas       ************************/
/**************************************************************************/
/* PURPOSE: WRITE NAME, MARK, AND SOURCE LINE PRAGMAS FOLLOWED BY EOLN TO */
/*          output.                                                       */
/**************************************************************************/

#define WritePragmas(x)  if ( (x)->na )                                  \
			     fprintf( output, " %%na=%s", (x)->name );   \
			 if ( (x)->umk )                                 \
                             fprintf( output, " %%mk=@" );               \
			 if ( (x)->mk )                                  \
                             fprintf( output, " %%mk=%c", (x)->mark );   \
                         if ( (x)->sf )                                  \
			     fprintf( output, " %%sf=%s", (x)->file );   \
                         if ( (x)->fn )                                  \
			     fprintf( output, " %%fn=%s", (x)->funct );  \
                         if ( (x)->sl )                                  \
			     fprintf( output, " %%sl=%d", (x)->line );   \
                         fprintf( output, "\n" )


/**************************************************************************/
/* STATIC **************      AssignNewKports      ************************/
/**************************************************************************/
/* PURPOSE: ASSIGN NEW K PORT NUMBERS, STARTING WITH 1, TO THE IMPORTS OF */
/*          COMPOUND NODE c AND ADJUST ALL REFERENCES. THE NEXT LEGAL     */
/*          PORT NUMBER IS RETURNED.                                      */
/**************************************************************************/

static int AssignNewKports( c )
PNODE c;
{
    register PNODE g;
    register PEDGE i;
    register int   p = 1;

    for ( i = c->imp; i != NULL; i = i->isucc, p++ )
	if ( i->iport != p ) {
            for ( g = c->C_SUBS; g != NULL; g = g->gsucc )
                ChangeExportPorts( g, i->iport, -p );

            i->iport = -p;
	    }

    return( p );
}


/**************************************************************************/
/* STATIC **************      AssignNewLports      ************************/
/**************************************************************************/
/* PURPOSE: ASSIGN NEW L PORT NUMBERS, STARTING WITH p, TO THE IMPORTS    */
/*          OF LOOP l'S INITIAL SUBGRAPH AND ADJUST ALL REFERENCES. THE   */
/*          NEXT LEGAL PORT NUMBER IS RETURNED.                           */
/**************************************************************************/

static int AssignNewLports( p, l )
int   p;
PNODE l;
{
    register PEDGE i;

    for ( i = l->L_INIT->imp; i != NULL; i = i->isucc, p++ )
	if ( i->iport != p ) {
            ChangeExportPorts( l->L_TEST, i->iport, -p );
            ChangeExportPorts( l->L_BODY, i->iport, -p );
            ChangeExportPorts( l->L_RET,  i->iport, -p );
	    ChangeImportPorts( l->L_BODY, i->iport, -p );

            i->iport = -p;
	    }

    return( p );
}


/**************************************************************************/
/* STATIC **************      AssignNewMports      ************************/
/**************************************************************************/
/* PURPOSE: ASSIGN NEW M PORT NUMBERS, STARTING WITH p, TO THE IMPORTS OF */
/*          FORALL f'S GENERATE SUBGRAPH AND ADJUST ALL REFERENCES.  THE  */
/*          NEXT LEGAL PORT NUMBER IS RETURNED.                           */
/**************************************************************************/

static int AssignNewMports( p, f )
int   p;
PNODE f;
{
    register PEDGE i;

    for ( i = f->F_GEN->imp; i != NULL; i = i->isucc, p++ )
	if ( i->iport != p ) {
            ChangeExportPorts( f->F_BODY, i->iport, -p );
            ChangeExportPorts( f->F_RET,  i->iport, -p );

            i->iport = -p;
	    }

    return( p );
}


/**************************************************************************/
/* STATIC **************      AssignNewTports      ************************/
/**************************************************************************/
/* PURPOSE: ASSIGN NEW T PORT NUMBERS, STARTING WITH p, TO THE IMPORTS OF */
/*          FORALL f'S BODY SUBGRAPH AND ADJUST ALL REFERENCES. THE NEXT  */
/*          LEGAL PORT NUMBER IS RETURNED.                                */
/**************************************************************************/

static int AssignNewTports( p, f )
int   p;
PNODE f;
{
    register PEDGE i;

    for ( i = f->F_BODY->imp; i != NULL; i = i->isucc, p++ )
	if ( i->iport != p ) {
            ChangeExportPorts( f->F_RET, i->iport, -p );
            i->iport = -p;
	    }

    return( p );
}


/**************************************************************************/
/* STATIC **************    AssignNewLoopTports    ************************/
/**************************************************************************/
/* PURPOSE: ASSIGN NEW T PORT NUMBERS, STARTING WITH p, TO THE IMPORTS OF */
/*          LOOP l'S BODY SUBGRAPH AND ADJUST ALL REFERENCES. THE NEXT    */
/*          LEGAL PORT IS RETURNED.                                       */
/**************************************************************************/

static int AssignNewLoopTports( p, l )
int   p;
PNODE l;
{
    register PEDGE i;

    for ( i = l->L_BODY->imp; i != NULL; i = i->isucc )
        if ( !IsImport( l->L_INIT, i->iport ) ) {
            if ( i->iport != p ) {
                ChangeExportPorts( l->L_TEST, i->iport, -p );
                i->iport = -p;
		}

            p++;
            }

    return( p );
}


/**************************************************************************/
/* STATIC **************      AssignNewRports      ************************/
/**************************************************************************/
/* PURPOSE: ASSIGN NEW R PORT NUMBERS, STARTING WITH 1, TO THE IMPORTS OF */
/*          SUBGRAPH g AND ADJUST ALL REFERENCES IN THE EXPORT LIST OF    */
/*          THE COMPOUND NODE TO WHICH g BELONGS.                         */
/**************************************************************************/

static void AssignNewRports( g )
PNODE g;
{
    register PEDGE i;
    register int   p = 1;

    for ( i = g->imp; i != NULL; i = i->isucc, p++ )
	if ( i->iport != p ) {
            ChangeExportPorts( g->G_DAD, i->iport, -p );
            i->iport = -p;
	    }
}


/**************************************************************************/
/* LOCAL  **************     FixConstantImports    ************************/
/**************************************************************************/
/* PURPOSE: CHECK FOR AND UNDO THE SIDE EFFECTS OF CONSTANT FOLDING:      */
/*          DOUBLE_REAL CONSTANTS IN REAL FORMAT AND SIGNED ARITHMETIC    */
/*          CONSTANTS (NOT ACCEPTED BY LLNL SOFTWARE). A SIGNED CONSTANT  */
/*          IS CONVERTED INTO A NEG NODE WHOSE IMPORT IS THE CONSTANT'S   */
/*          POSITIVE. THE NEW NODE IS GIVEN LABEL ++lab. THE  LAST        */
/*          ASSIGNED LABEL IS RETURNED.  A NULL CONSTANT DEFINES AN ERROR */
/*          VALUE.                                                        */
/**************************************************************************/

static int FixConstantImports( n, lab )
PNODE n;
int   lab;
{
    register PEDGE  i;
    register PEDGE  ii;
    register char  *p;
    register PNODE  neg;

    for ( i = n->imp; i != NULL; i = i->isucc ) {
	if ( !IsConst( i ) || (i->CoNsT == NULL) )
	    continue;

	if ( IsDouble( i->info ) )
	    for ( p = i->CoNsT; *p != '\0'; p++ )
		if ( (*p == 'E') || (*p == 'e') )
		    *p = 'd';

	if ( IsReal( i->info ) )
	    for ( p = i->CoNsT; *p != '\0'; p++ )
		if ( (*p == 'D') || (*p == 'd') )
		    *p = 'e';

	if ( IsArithmetic( i->info ) && (!sgnok) )
	    if ( i->CoNsT[0] == '-' ) {
		neg = NodeAlloc( ++lab, IFNeg );
		ii  = EdgeAlloc( NULL, CONST_PORT, neg, 1 );

		ii->info = i->info;
		ii->CoNsT = &(i->CoNsT[1]);

		LinkImport( neg, ii );

		i->CoNsT = NULL;
		i->eport = 1;

		LinkExport( neg, i );
		LinkNode( (IsGraph(n))? n : n->npred, neg );
		}
	}

    return( lab );
}


/**************************************************************************/
/* STATIC **************      AssignNewLabels      ************************/
/**************************************************************************/
/* PURPOSE: ASSIGN NEW LABELS TO THE NODES OF GRAPH g AND ASSIGN NEW PORT */
/*          NUMBERS TO THE K, M, T, L, AND R PORTS OF ALL COMPOUND NODES  */
/*          SUCH THAT NODE LABELS BEGIN WITH 1 (ASSIGNED IN INCREMENTS OF */
/*          ONE) AND PORT NUMBERS OBAY THE FOLLOWING:                     */
/*                                                                        */
/*          FORALL:  K < M < T  AND K AND R PORTS  START AT 1 BY 1        */
/*          LOOPA :  K < L < T  AND K AND R PORTS  START AT 1 BY 1        */
/*          LOOPB :  K < L      AND K AND R PORTS  START AT 1 BY 1        */
/*                                                                        */
/*          NOTE: CONSTANT FOLDING SIDE EFFECTS ARE UNDONE.               */
/**************************************************************************/

static void  AssignNewLabels( g )
register PNODE g;
{
    register int   lab;
    register PNODE n;
    register int   p;

    FixConstantImports( g, 0 );

    for ( lab = 0, n = g->G_NODES; n != NULL; n = n->nsucc ) {
	lab = FixConstantImports( n, lab );

	n->label = ++lab;

        if ( !IsCompound( n ) )
	    continue;

        p = AssignNewKports( n );

	switch ( n->type ) {
	    case IFTagCase:
		for ( g = n->C_SUBS; g != NULL; g = g->gsucc )
		    AssignNewRports( g );

		break;

	    case IFSelect:
		n->S_TEST->imp->iport = 1;                   /* B PORT NUMBER */

		AssignNewRports( n->S_ALT );
		AssignNewRports( n->S_CONS );
		break;

	    case IFForall:
		AssignNewTports( AssignNewMports( p, n ), n );
		AssignNewRports( n->F_RET );
                break;

	    case IFLoopA:
		n->L_TEST->imp->iport = 1;                   /* B PORT NUMBER */

                AssignNewLoopTports( AssignNewLports( p, n ), n );
		AssignNewRports( n->L_RET );
		break;

	    case IFLoopB:
		n->L_TEST->imp->iport = 1;                   /* B PORT NUMBER */

                AssignNewLports( p, n );
		AssignNewRports( n->L_RET );
		break;
	    }
        }
}


/**************************************************************************/
/* LOCAL  **************        WriteStamps        ************************/
/**************************************************************************/
/* PURPOSE: WRITE THE IF1 STAMPS IN THE STAMP TABLE TO output.            */
/**************************************************************************/

static void WriteStamps()
{
    register int i;

    for ( i = 0; i < 127; i++ )
	if ( stamps[i] != NULL )
	    fprintf( output, "C$  %c %s\n", i, stamps[i] );
}


/**************************************************************************/
/* LOCAL  **************         WriteInfo         ************************/
/**************************************************************************/
/* PURPOSE: WRITE THE INFO NODE LIST HEADED BY ihead TO OUTPUT.           */
/**************************************************************************/

static void WriteInfo()
{
    register PINFO i;
    register int   t;

    for ( i = ihead; i != NULL; i = i->next ) {
        if ( IsBasic(i) )
	    t = IF_BASIC;
        else
            t = i->type;

        fprintf( output, "T %2d %2d", i->label, t );

        switch ( i->type ) {
            case IF_FUNCTION:
	        fprintf( output, " %2d", (i->F_IN == NULL)? 0 : i->F_IN->label);
	        fprintf( output, " %2d", i->F_OUT->label );
	        break;

            case IF_STREAM:
            case IF_MULTPLE:
            case IF_ARRAY:
	    case IF_BUFFER:
	        fprintf( output, " %2d   ", i->A_ELEM->label );
                break;

            case IF_UNION:
            case IF_RECORD:
	        fprintf( output, " %2d   ", i->R_FIRST->label );
                break;

            case IF_TUPLE: 
            case IF_FIELD:
            case IF_TAG:
	        fprintf( output, " %2d", i->L_SUB->label );
	        fprintf( output, " %2d", 
				 (i->L_NEXT == NULL)? 0 : i->L_NEXT->label );
					 
                break;

            case IF_UNKNOWN:
	        fprintf( output, "      " );
	        break;

	    default:
	        fprintf( output, " %2d   ", i->type - BASIC_TYPE_START );
	        break;
            }

        WritePragmas( i );
	}
}


/**************************************************************************/
/* LOCAL  **************         WriteConst        ************************/
/**************************************************************************/
/* PURPOSE: WRITE A CONSTANT TO output.  IF THE CONSTANT IS REPRESENTED   */
/*          BY A NULL POINTER, THEN IT IS AN ERROR CONSTANT.  ELSE THE    */
/*          APPROPRIATE DELIMITERS ARE DETERMINED BASED ON THE CONSTANTS  */
/*          TYPE. THE PORT NUMBERS MAY BE NEGATIVE.                       */
/**************************************************************************/

void WriteConst( c )
PEDGE c;
{
    fprintf( output, "L         %2d %2d  %2d", c->dst->label,
                     abs( c->iport ), c->info->label         );

    if ( c->CoNsT == NULL ) {
        fprintf( output, " \"%s\"", ERROR_CONSTANT );
	return;
	}

    if ( IsDefArrayBuf( c->dst ) && (c->isucc == NULL) ) {
	fprintf( output, " \"%s\"", c->CoNsT );         /* BUFFER LITERAL */
	return;
	}

    switch ( c->info->type ) {
        case IF_CHAR:
            fprintf( output, " \"\'%s\'\"", c->CoNsT );
            break;

        case IF_ARRAY:
        case IF_STREAM:
            fprintf( output, " \"\"%s\"\"", c->CoNsT );
            break;

        default:
            fprintf( output, " \"%s\"", c->CoNsT );
            break;
        }
}


/**************************************************************************/
/* MACRO  **************         WriteEdge         ************************/
/**************************************************************************/
/* PURPOSE: WRITE EDGE e TO output. THE PORT NUMBERS MAY BE NEGATIVE.     */
/**************************************************************************/

#define WriteEdge(e)   fprintf( output, "E %2d %2d   %2d %2d  %2d",        \
	                                e->src->label, abs( e->eport ),    \
                                        e->dst->label, abs( e->iport ),    \
					e->info->label               )


/**************************************************************************/
/* LOCAL  **************        WriteImports       ************************/
/**************************************************************************/
/* PURPOSE: WRITE NODE n's IMPORTS TO output.                             */
/**************************************************************************/

static void WriteImports( n )
PNODE n;
{
    register PEDGE  i;

    for ( i  = n->imp; i != NULL; i = i->isucc ) {
	if ( IsConst( i ) )
	    WriteConst( i );
	else
            WriteEdge( i );

	WritePragmas( i );
        }
}


/**************************************************************************/
/* LOCAL  **************        WriteNodes         ************************/
/**************************************************************************/
/* PURPOSE: PRINT THE NODES OF GRAPH g AND THEIR IMPORT EDGES TO output.  */
/*          NODE RELABELING AND COMPOUND NODE PORT RENUMBERING IS DONE    */
/*          BEFORE PRINTING.                                              */
/**************************************************************************/

static void WriteNodes( g )
PNODE g;
{
    register PNODE  n;
    register PNODE  sg;
    register PALIST l;

    switch ( g->type ) {
        case IFLGraph:
	    fprintf( output, "G %2d", g->G_INFO->label );
	    fprintf( output, " \"%s\"", g->G_NAME );
	    break;

	case IFIGraph:
	    fprintf( output, "I %2d", g->G_INFO->label );
	    fprintf( output, " \"%s\"", g->G_NAME );
	    break;

	case IFXGraph:
	    fprintf( output, "X %2d", g->G_INFO->label );
	    fprintf( output, " \"%s\"", g->G_NAME );
	    break;

	case IFSGraph:
	    fprintf( output, "G %2d", g->label );
            break;
	}

    WritePragmas( g );

    AssignNewLabels( g );
    /* WriteImports( g ); */

    for ( n = g->G_NODES; n != NULL; n = n->nsucc ) {
        switch ( n->type ) {
	    case IFLoopA:
	    case IFLoopB:
	    case IFTagCase:
	    case IFForall:
	    case IFSelect:
	        fprintf( output, "{ Compound %2d %2d\n", n->label, n->type );

	        for ( sg = n->C_SUBS; sg != NULL; sg = sg->gsucc )
	            WriteNodes( sg );

                fprintf( output, "} %2d %2d %2d   ", n->label, 
				  n->type, n->C_SCNT        );

	        for ( l = n->C_ALST; l != NULL; l = l->next )
                    fprintf( output, " %d", l->datum );

	        break;

            default:
	        fprintf( output, "N %2d %2d", n->label, n->type );
            }

        WritePragmas( n );
	WriteImports( n );
	}

    WriteImports( g );
}


/**************************************************************************/
/* GLOBAL **************         If2Write          ************************/
/**************************************************************************/
/* PURPOSE: PRINT ALL IF2 TYPES, STAMPS, AND FUNCTION GRAPHS TO output.   */
/*          THE TYPE INFORMATION IS FOUND TRAVERSING THE BINARY TREE      */
/*          itree; THE STAMPS ARE FOUND IN ARRAY stamps; AND THE FUNCTION */
/*          GRAPHS ARE FOUND TRAVERSING glstop. NODES ARE RELABELED AND   */
/*          ALL PORTS OF COMPOUND NODES INCLUDING THEIR SUBGRAPHS ARE     */
/*          RENUMBERED. PORT NUMBER AND NODE LABEL CONVENTIONS MAY HAVE   */
/*          BEEN VIOLATED DURING GRAPH OPTIMIATION. IT IS EASIER TO FIX   */
/*          THIS HERE!  THE SIDE EFFECTS OF CONSTANT FOLDING ARE UNDONE   */
/*          (SIGNED CONSTANTS ARE ALLOWED IF sgnok IS TRUE).              */
/**************************************************************************/

void If2Write()
{
    register PNODE f;

    WriteInfo();
    WriteStamps();

    for ( f = glstop->gsucc; f != NULL; f = f->gsucc )
	WriteNodes( cfunct = f );
}
