#include "world.h"

#define BUFFER_SIZE      200

#define NextToken        token = fgetc( input )

#define IsSpace(x)       ( ((x) == ' ') || ((x) == '\t') )
#define IsEoln(x)        ( (x) == '\n' )
#define IsWhite(x)       ( IsSpace(x) || IsEoln(x) )
#define IsDigit(x)       ( ((x) >= '0') && ((x) <= '9') )
#define IsEqual(x)       ( (x) == '=' )

#define EatLine          while ( !IsEoln( token ) ) NextToken; NextToken; line++
#define EatSpaces        while ( IsSpace( token ) ) NextToken

#define WHITE_CHARS      NULL

static int  token = ' ';
static char buffer[BUFFER_SIZE];

int    line   = 1;
int    maxint = 1;
PRAGS  pragmas;


/**************************************************************************/
/* LOCAL  **************         ReadError         ************************/
/**************************************************************************/
/* PURPOSE: PRINT msg TO stderr ALONG WITH THE LINE NUMBER OF THE LAST    */
/*          LINE READ AND ABORT EXECUTION.                                */
/**************************************************************************/

static void ReadError( msg )
char *msg;
{
    fprintf( stderr, "%s: -E (line %d) %s\n", program, line, msg );
    Stop( ERROR );
}


/**************************************************************************/
/* LOCAL  **************        ReadLineId         ************************/
/**************************************************************************/
/* PURPOSE: READ AND RETURN A LINE IDENTIFIER.  IT MAY BE PROCEEDED BY    */
/*          ONE OR MORE EOLN, TAB, OR BLANK CHARACTERS.  WHEN DONE, token */
/*          WILL CONTAIN THE CHARACTER FOLLOWING THE RETURNED IDENTIFIER. */
/**************************************************************************/

static int ReadLineId() 
{ 
    register int id;

    while ( IsWhite( token )  ) {
        if ( IsEoln( token ) ) 
	    line++; 

	NextToken;
	}

    if ( (id = token) != EOF )
	NextToken;

    return( id ); 
}


/**************************************************************************/
/* LOCAL  **************        ReadInteger        ************************/
/**************************************************************************/
/* PURPOSE: READ A UNSIGNED INTEGER, OPTIONALLY PROCEEDED BY ONE OR MORE  */
/*          BLANK OR TAB CHARACTERS.  WHEN DONE, token WILL CONTAIN THE   */
/*          CHARACTER FOLLOWING THE INTEGER. INPUT LINE BOUNDARIES ARE    */
/*          NOT CROSSED. A RUNNING RECORD OF THE LARGEST INTEGER READ IS  */
/*          KEPT IN maxint.                                               */
/**************************************************************************/

static int ReadInteger()
{
    register int val = 0;

    EatSpaces;

    if ( !IsDigit( token ) )
        ReadError( "INTEGER EXPECTED" );

    while ( IsDigit( token ) ) {
        val = (val * 10)  + (token - '0');
        NextToken;
        }

    if ( maxint < val )
	maxint = val;

    return( val );
}


/**************************************************************************/
/* LOCAL  **************        ReadString         ************************/
/**************************************************************************/
/* PURPOSE: READ A STRING TERMINATED BY delim AND RETURN A COPY. THE      */
/*          TRMINATOR  WHITE_CHARS REPRESENTS EITHER EOLN, TAB, OR BLANK. */
/*          THE DELIMITER IS NOT KEPT AS PART OF THE RETURNED STRING.     */
/*          WHEN DONE, token WILL CONTAIN THE CHARACTER FOLLOWING THE     */
/*          TERMINATION CHARACTER UNLESS IT IS EOLN. THE BACKSLASH        */
/*          CHARACTER (\) ACTS AS AN ESCAPE UNLESS THE STRING IS TO END   */
/*          WITH WHITE_CHAR.  AT NO TIME IS A LINE BOUNDARY BROKEN.       */
/**************************************************************************/

static char *ReadString( delim )
char delim;
{
    register int  idx   = 0;
    register char prev1 = NULL;
    register char prev2 = NULL;

    for (;;) {
	if ( delim == WHITE_CHARS ) {
	    if ( IsWhite( token ) )
		break;
            }
        else if ( (delim == token) && ((prev1 != '\\') || (prev2 == '\\')) )
            break;

	if ( IsEoln( token ) )
	    ReadError( "ILLEGAL STRING SYNTAX--UNEXPECTED EOLN" );

        buffer[idx++] = token;

	if ( idx >= BUFFER_SIZE )
	    ReadError( "ReadString BUFFER OVERFLOW" );

	prev2 = prev1;
	prev1 = token;
	NextToken;
        }

    buffer[idx] = NULL;

    if ( !IsEoln( token ) )
	NextToken;

    return( CopyString( buffer ) );
}


/**************************************************************************/
/* LOCAL  **************        ReadLiteral        ************************/
/**************************************************************************/
/* PURPOSE: READ A LITERAL DELIMITED BY DOUBLE QUOTES OPTIONALY PROCEEDED */
/*          BY ONE OR MORE TAB OR BLANK CHARACTERS (BUT CONTAINED ON ONE  */
/*          LINE). THE LITERAL ITSELF CAN BE A STRING DELIMIED BY DOUBLE  */
/*          QUOTES, SINGLE QUOTES, OR NO QUOTES.  A COPY OF THE LITERAL   */
/*          MINUS ANY DELIMITERS (OUTER OR INNER) IS RETURNED. WHEN DONE, */
/*          token WILL CONTAINS  THE CHARACTER  FOLLOWING  THE  FINAL     */
/*          DELIMITER.  LINE BOUNDARIES ARE NOT BROKEN.  IF A LITERAL IS  */
/*          NOT FOUND, ReadError IS CALLED IF abort IS TRUE.  THE LITERAL */
/*          "error" IS SPECIAL, REPRESENTED INTERNALLY AS A NULL POINTER. */
/**************************************************************************/

static char *ReadLiteral( abort )
int abort;
{
    register char *s;

    EatSpaces;

    if ( token != '\"' ) {
        if ( abort ) 
	    ReadError( "LITERAL EXPECTED" );

        return( NULL );
	}

    NextToken;

    switch ( token ) {
	case '\n':
	    ReadError( "UNEXPECTED EOLN" );

        case '\"':
	    NextToken;

	    s = ReadString( '\"' );

	    if ( token != '\"' )
		ReadError( "LITERAL DELIMITER EXPECTED" );

            NextToken;
	    break;

	case '\'':
	    NextToken;
	    s = ReadString( '\'' );

	    if ( token != '\"' )
		ReadError( "LITERAL DELIMITER EXPECTED" );

            NextToken;
	    break;

	default:
	    s = ReadString( '\"' );

	    if ( strcmp( s, ERROR_CONSTANT ) == 0 ) {
		/* free( s ); */
		s = NULL;
		}

	    break;
	}

    return( s );
}


/**************************************************************************/
/* LOCAL  **************         ReadStamp         ************************/
/**************************************************************************/
/* PURPOSE: READ A STAMP AND ENTER IT IN STAMP TABLE.  THE STAMP SYNTAX   */
/*          IS C$ FOLLOWED BY ONE OR MORE TAB OR BLANK CHARACTERS UP TO A */
/*          NON-EOLN CHARACTER (THE STAMP).  THE REMAINDER OF THE LINE IS */
/*          STAMP DOCUMENTATION. WHEN DONE, BARING ANY TROUBLE, token WILL*/
/*          CONTAIN THE EOLN CHARACTER.  IT IS ASSUMED THAT THE CALLER    */
/*          RECOGNIZED THE C OF C$ AND TOKEN CONTAINS THE $?.  IF A $ IS  */
/*          NOT SEEN, THE LINE IS A COMMENT TO BE DISCARDED BY THE CALLER.*/
/**************************************************************************/

static void ReadStamp()
{
    char stamp;

    if ( token == '$' ) {
	NextToken;
	EatSpaces;

	stamp = token;

	if ( IsEoln( token ) )
	    ReadError( "ILLEGAL STAMP SYNTAX" );

        NextToken;
        AddStamp( stamp, ReadString( '\n' ) ); 
	}
}


/**************************************************************************/
/* LOCAL  **************        ReadPragmas        ************************/
/**************************************************************************/
/* PURPOSE: READ NAME, SOURCE FILE, FUNCTION NAME, SOURCE LINE, MARK AND  */
/*          REFERENCE COUNT PRAGMAS: %na=name, %sf=file, %fn=function,    */
/*          %mk=MARKS, %sr=VALUE, %pm=VALUE, %cm=VALUE, %pl=VALUE, AND    */
/*          %cc=COST. OTHERS ARE SILENTLY IGNORED.  WHEN DONE, token WILL */
/*          CONTAIN THE EOLN CHARACTER.  COST IS A DOUBLE.                */
/**************************************************************************/

static void ReadPragmas()
{
    register char *name;

    InitPragmas( &pragmas );

    for (;;) {
        EatSpaces;

	if ( IsEoln( token ) ) 
	    break;

	if ( token != '%' ) {
	    NextToken;
	    continue;
	    }

        NextToken;

	switch ( token ) {
            case 'm':
		NextToken;

		if ( token == 'k' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

	            switch ( token ) {
			case 's': case 'c': case 'f': case 'i': case 'e':
		            pragmas.mark = token;
		            break;

			case 'V':
			    break;

			case 'I':
		            pragmas.imark = TRUE;
		            break;

			case 'P':
		            pragmas.pmark = TRUE;
		            break;

			case 'R':
	                    pragmas.rmark1 = RMARK;
		            break;

			case 'r':
		            pragmas.rmark1 = rMARK;
		            break;

			case 'O':
	                    pragmas.omark1 = TRUE;
		            break;

			case 'D':
	                    pragmas.dmark = TRUE;
		            break;

			case 'L':
	                    pragmas.lmark = TRUE;
		            break;

			case 'E':
	                    pragmas.emark = TRUE;
		            break;

			case 'F':
		            pragmas.fmark = TRUE;
			    break;

			case 'C':
		            pragmas.cmark = TRUE;
			    break;

			case '@':
			    pragmas.umark = TRUE;
			    break;

			case 'N':
			    pragmas.nmark = TRUE;
			    break;

			case 'B':
		            pragmas.bmark = TRUE;
			    break;

			case 'W':
			    pragmas.wmark = TRUE;
			    break;

			case 'S':
			    pragmas.smark = TRUE;
			    break;

                        default:
			    ReadError( "ILLEGAL mk PRAGMA MARK" );
			}

		    NextToken;
		    }

		break;

            case 'p':
		NextToken;

		if ( token == 'm' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL pm PRAGMA" );

		    pragmas.pm   = atoi( name );
		    break;
		    }

		if ( token == 'l' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL pl PRAGMA" );

		    pragmas.pl   = atoi( name );
		    }

                break;

            case 'c':
		NextToken;

		if ( token == 'm' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL sr PRAGMA" );

		    pragmas.cm   = atoi( name );
		    break;
		    }

		if ( token == 'c' ) {
		    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL cc PRAGMA" );

                    pragmas.ccost = atof( name );
                    }

                break;

            case 'f':
		NextToken;

		if ( token == 'n' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL fn PRAGMA" );

		    pragmas.funct = name;
                    }

                break;

            case 'n':
		NextToken;

		if ( token == 'a' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL na PRAGMA" );

		    pragmas.name = name;
                    }

                break;

            case 's':
		NextToken;

		if ( token == 'l' ) {
		    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    pragmas.line = ReadInteger();
		    break;
                    }

		if ( token == 'f' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL sf PRAGMA" );

		    pragmas.file = name;
		    break;
                    }

		if ( token == 'r' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL sr PRAGMA" );

		    pragmas.sr   = atoi( name );
		    }
		    
                break;

	    default:
                break;
            }
        }       
}


/**************************************************************************/
/* LOCAL  **************       ReadAssocList       ************************/
/**************************************************************************/
/* PURPOSE: READ AND RETURN A COMPOUND NODE's ASSOCIATION LIST CONTAINING */
/*          cnt UNSINGED INTEGERS.                                        */
/**************************************************************************/

static PALIST ReadAssocList( cnt )
register int cnt;
{
    register PALIST l = AssocListAlloc( ReadInteger() );

    while ( (--cnt) > 0 )
        l = LinkAssocLists( l, AssocListAlloc( ReadInteger() ) );

    return( l );
}


/**************************************************************************/
/* GLOBAL **************          If2Read          ************************/
/**************************************************************************/
/* PURPOSE: READ IF2 INFORMATION FROM input AND BUILD ITS INTERNAL FORM.  */
/*          A PARTIAL SYNTAX CHECK IS MADE.  IN GENERAL SEMANTIC AND      */
/*          SYNTACTIC CORRECTNESS IS ASSUMED.                             */
/**************************************************************************/

void If2Read()
{
    register int     label, type,  cnt;
    register int     snode, eport, dnode, iport;
    register int     ref1,  ref2;
    register char   *s;
    register PALIST  l;

    EnterScope();

    for (;;) {
        switch ( ReadLineId() ) {
            case 'C':
	        ReadStamp();
                break;

	    case 'D':
		snode = ReadInteger();
		dnode = ReadInteger();

		MakeAde( snode, dnode );
		break;

            case 'E':
                snode = ReadInteger();
                eport = ReadInteger();
                dnode = ReadInteger();
                iport = ReadInteger();
                type  = ReadInteger();

		ReadPragmas();
                MakeEdge( snode, eport, dnode, iport, type );
                break;

            case 'L':
                dnode = ReadInteger();
                iport = ReadInteger();
                label = ReadInteger();
                s     = ReadLiteral( TRUE );

		ReadPragmas();
                MakeConst( dnode, iport, label, s );
                break;

            case 'G':
	        type = ReadInteger();
		s    = ReadLiteral( FALSE );

		ReadPragmas();

                if ( s == NULL )
                    MakeGraph( IFSGraph, type, NULL );
                else
		    MakeGraph( IFLGraph, type, s );

                break;

            case 'I':
                type = ReadInteger();
                s    = ReadLiteral( TRUE );

                ReadPragmas();
                MakeGraph( IFIGraph, type, s );
                break;

            case 'X':
                type = ReadInteger();
                s    = ReadLiteral( TRUE );

		ReadPragmas();
                MakeGraph( IFXGraph, type, s );
                break;

            case 'N':
                label = ReadInteger();
                type  = ReadInteger();
    
		ReadPragmas();
                MakeNode( label, type );
                break;

            case 'T':
                label = ReadInteger();
                type  = ReadInteger();
    
                if ( type != IF_UNKNOWN ) {
                    ref1 = ReadInteger();

		    switch ( type ) {
			case IF_FIELD:
			case IF_FUNCTION:
			case IF_TAG:
			case IF_TUPLE:
                            ref2 = ReadInteger();
			    break;

                        default:
                            ref2 = 0;
			    break;
			}
                    }

                ReadPragmas();
                MakeInfo( label, type, ref1, ref2 );
                break;

            case '{':
	        EnterScope();
                break;

            case '}':
                label = ReadInteger();
                type  = ReadInteger();
                cnt   = ReadInteger();
		l     = ReadAssocList( cnt );

                ReadPragmas();

		    /* PERFORM SCOPE EXIT FOR COMPOUND NODE */
                MakeCompound( label, type, cnt, l );

                break;

            case EOF:
		goto Done;

            default:
                ReadError( "ILLEGAL LINE IDENTIFIER" );
            }

        EatLine;
        }

Done:
    ExitScope();
}
