#include "world.h"

#define BUFFER_SIZE      200

#define NextToken        token = fgetc( input )

#define IsSpace(x)       ( ((x) == ' ') || ((x) == '\t') )
#define IsEoln(x)        ( (x) == '\n' )
#define IsWhite(x)       ( IsSpace(x) || IsEoln(x) )
#define IsDigit(x)       ( ((x) >= '0') && ((x) <= '9') )
#define IsEqual(x)       ( (x) == '=' )

#define EatLine          while ( !IsEoln( token ) ) NextToken; NextToken; line++
#define EatSpaces        while ( IsSpace( token ) ) NextToken

#define WHITE_CHARS      NULL

static int  token;                             /* CURRENT INPUT CHARACTER */
static int  line;                /* INPUT LINE OF CURRENT INPUT CHARACTER */

static char buffer[BUFFER_SIZE];

PRAGS pragmas;                           /* PRAGMAS OF CURRENT INPUT LINE */


/**************************************************************************/
/* LOCAL  **************         ReadError         ************************/
/**************************************************************************/
/* PURPOSE: PRINT msg TO stderr ALONG WITH THE LINE NUMBER OF THE LAST    */
/*          LINE READ AND ABORT EXECUTION.                                */
/**************************************************************************/

static void ReadError( msg )
char *msg;
{
    fprintf( stderr, "%s: -E (line %d) %s\n", program, line, msg );
    Stop( ERROR );
}


/**************************************************************************/
/* LOCAL  **************        ReadLineId         ************************/
/**************************************************************************/
/* PURPOSE: READ AND RETURN A LINE IDENTIFIER.  IT MAY BE PROCEEDED BY    */
/*          ONE OR MORE EOLN, TAB, OR BLANK CHARACTERS.  WHEN DONE, token */
/*          WILL CONTAIN THE CHARACTER FOLLOWING THE RETURNED IDENTIFIER. */
/**************************************************************************/

static int ReadLineId() 
{ 
    register int id;

    while ( IsWhite( token )  ) {
        if ( IsEoln( token ) ) 
	    line++; 

	NextToken;
	}

    if ( (id = token) != EOF )
	NextToken;

    return( id ); 
}


/**************************************************************************/
/* LOCAL  **************        ReadInteger        ************************/
/**************************************************************************/
/* PURPOSE: READ A UNSIGNED INTEGER, OPTIONALLY PROCEEDED BY ONE OR MORE  */
/*          BLANK OR TAB CHARACTERS.  WHEN DONE, token WILL CONTAIN THE   */
/*          CHARACTER FOLLOWING THE INTEGER. INPUT LINE BOUNDARIES ARE    */
/*          NOT CROSSED.                                                  */
/**************************************************************************/

static int ReadInteger()
{
    register int val = 0;

    EatSpaces;

    if ( !IsDigit( token ) )
        ReadError( "INTEGER EXPECTED" );

    while ( IsDigit( token ) ) {
        val = (val * 10)  + (token - '0');
        NextToken;
        }

    return( val );
}


/**************************************************************************/
/* LOCAL  **************        ReadString         ************************/
/**************************************************************************/
/* PURPOSE: READ A STRING TERMINATED BY delim AND RETURN A COPY. THE      */
/*          TRMINATOR  WHITE_CHARS REPRESENTS EITHER EOLN, TAB, OR BLANK. */
/*          THE DELIMITER IS NOT KEPT AS PART OF THE RETURNED STRING.     */
/*          WHEN DONE, token WILL CONTAIN THE CHARACTER FOLLOWING THE     */
/*          TERMINATION CHARACTER UNLESS IT IS EOLN. THE BACKSLASH        */
/*          CHARACTER (\) ACTS AS AN ESCAPE UNLESS THE STRING IS TO END   */
/*          WITH WHITE_CHAR.  AT NO TIME IS A LINE BOUNDARY BROKEN.       */
/**************************************************************************/

static char *ReadString( delim )
char delim;
{
    register int  idx   = 0;
    register char prev1 = NULL;
    register char prev2 = NULL;

    for (;;) {
	if ( delim == WHITE_CHARS ) {
	    if ( IsWhite( token ) )
		break;
            }
        else if ( (delim == token) && ((prev1 != '\\') || (prev2 == '\\')) )
            break;

	if ( IsEoln( token ) )
	    ReadError( "ILLEGAL STRING SYNTAX--UNEXPECTED EOLN" );

        buffer[idx++] = token;

	if ( idx >= BUFFER_SIZE )
	    ReadError( "ReadString BUFFER OVERFLOW" );

	prev2 = prev1;
	prev1 = token;
	NextToken;
        }

    buffer[idx] = '\0';

    if ( !IsEoln( token ) )
	NextToken;

    return( CopyString( buffer ) );
}


/**************************************************************************/
/* LOCAL  **************        ReadLiteral        ************************/
/**************************************************************************/
/* PURPOSE: READ A LITERAL DELIMITED BY DOUBLE QUOTES OPTIONALY PROCEEDED */
/*          BY ONE OR MORE TAB OR BLANK CHARACTERS (BUT CONTAINED ON ONE  */
/*          LINE). THE LITERAL ITSELF CAN BE A STRING DELIMIED BY DOUBLE  */
/*          QUOTES, SINGLE QUOTES, OR NO QUOTES.  A COPY OF THE LITERAL   */
/*          MINUS ANY DELIMITERS (OUTER OR INNER) IS RETURNED. WHEN DONE, */
/*          token WILL CONTAINS  THE CHARACTER  FOLLOWING  THE  FINAL     */
/*          DELIMITER.  LINE BOUNDARIES ARE NOT BROKEN.  IF A LITERAL IS  */
/*          NOT FOUND, ReadError IS CALLED IF abort IS TRUE.  THE LITERAL */
/*          "error" IS SPECIAL, REPRESENTED INTERNALLY AS A NULL POINTER. */
/**************************************************************************/

static char *ReadLiteral( abort )
int abort;
{
    register char *s;

    EatSpaces;

    if ( token != '\"' ) {
        if ( abort ) 
	    ReadError( "LITERAL EXPECTED" );

        return( NULL );
	}

    NextToken;

    switch ( token ) {
	case '\n':
	    ReadError( "UNEXPECTED EOLN" );

        case '\"':
	    NextToken;

	    s = ReadString( '\"' );

	    if ( token != '\"' )
		ReadError( "LITERAL DELIMITER EXPECTED" );

            NextToken;
	    break;

	case '\'':
	    NextToken;
	    s = ReadString( '\'' );

	    if ( token != '\"' )
		ReadError( "LITERAL DELIMITER EXPECTED" );

            NextToken;
	    break;

	default:
	    s = ReadString( '\"' );

	    if ( strcmp( s, ERROR_CONSTANT ) == 0 ) {
		/* free( s ); */
		s = NULL;
		}

	    break;
	}

    return( s );
}


/**************************************************************************/
/* LOCAL  **************         ReadStamp         ************************/
/**************************************************************************/
/* PURPOSE: READ A STAMP AND ENTER IT IN STAMP TABLE.  THE STAMP SYNTAX   */
/*          IS C$ FOLLOWED BY ONE OR MORE TAB OR BLANK CHARACTERS UP TO A */
/*          NON-EOLN CHARACTER (THE STAMP).  THE REMAINDER OF THE LINE IS */
/*          STAMP DOCUMENTATION. WHEN DONE, BARING ANY TROUBLE, token WILL*/
/*          CONTAIN THE EOLN CHARACTER.  IT IS ASSUMED THAT THE CALLER    */
/*          RECOGNIZED THE C OF C$ AND TOKEN CONTAINS THE $?.  IF A $ IS  */
/*          NOT SEEN, THE LINE IS A COMMENT TO BE DISCARDED BY THE CALLER.*/
/**************************************************************************/

static void ReadStamp()
{
    register char stamp;
    register int  kind;

    if ( token == '$' ) {
	NextToken;
	EatSpaces;

	stamp = token;

	if ( IsEoln( token ) )
	    ReadError( "ILLEGAL STAMP SYNTAX" );

	NextToken;
        AddStamp( stamp, ReadString( '\n' ) ); 
	return;
	}

  /* CHECK IF COMMENT STRING DEFINES A NAME TABLE ENTRY OF FORM: CE$name, */
  /* CF$name, OR CC$name.  THE LATTER (C FUNCTION NAME) IS THE DEFAULT    */
  /* UNRESOLVED FUNCTION TYPE. */

    switch ( kind = token ) {
      case 'C':
      case 'F':
      case 'E':
	NextToken;

	if ( token != '$' )
	  return;

	NextToken;

	if ( IsEoln( token ) )
	    return;

	switch ( kind ) {
	  case 'E':
	    PlaceInEntryTable( LowerCase( ReadString( '\n' ) ) );
	    break;

	  case 'F':
	    PlaceInFortranTable( LowerCase( ReadString( '\n' ) ) );
	    break;

	  default:
	    PlaceInCTable( LowerCase( ReadString( '\n' ) ) );
	    break;
          }

      default:
	break;
      }
}


/**************************************************************************/
/* LOCAL  **************        ReadPragmas        ************************/
/**************************************************************************/
/* PURPOSE: READ NAME, FILE, LINE, AND MARK PRAGMAS: %na=name, %sf=file,  */
/*          %sl=line, %fn=funct, AND %mk=mark.  ALL OTHERS ARE SILENTLY   */
/*          IGNORED.                                                      */
/*                                                                        */
/*          THE NAME (name), FILE (file), AND FUNCTION (funct) MUST       */
/*          MUST CONTAIN AT LEAST ONE NON-WHITE CHARACTER; THE LINE       */
/*          (line) MUST BE AN UNSIGNED NUMBER; AND THE MARK MUST BE A     */
/*          SINGLE NON-WHITE CHARACTER. WHEN DONE, token WILL CONTAIN     */
/*          THE EOLN CHARACTER.                                           */
/**************************************************************************/

static void ReadPragmas()
{
    register char *name;
    register int   line;

    InitPragmas( &pragmas );

    for (;;) {
        EatSpaces;

	if ( IsEoln( token ) ) 
	    break;

	if ( token != '%' ) {
	    NextToken;
	    continue;
	    }

        NextToken;

	switch ( token ) {
            case 'm':
		NextToken;

		if ( token == 'k' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

		    if ( IsWhite( token ) )
			ReadError( "ILLEGAL mk PRAGMA" );

		    pragmas.mk   = TRUE;
		    pragmas.mark = token;

		    NextToken;
                    }

		break;

            case 'f':
		NextToken;

		if ( token == 'n' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL fn PRAGMA" );

		    pragmas.funct = name;
		    pragmas.fn    = TRUE;
                    }

                break;

            case 'n':
		NextToken;

		if ( token == 'a' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL na PRAGMA" );

		    pragmas.na   = TRUE;
		    pragmas.name = name;
                    }

                break;

            case 's':
		NextToken;

		if ( token == 'l' ) {
		    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    line = ReadInteger();

		    pragmas.sl   = TRUE;
		    pragmas.line = line;
		    break;
                    }

		if ( token == 'f' ) {
                    NextToken;

		    if ( !IsEqual( token ) )
			continue;

                    NextToken;

                    name = ReadString( WHITE_CHARS );

                    if ( *name == NULL )
			ReadError( "ILLEGAL sf PRAGMA" );

		    pragmas.sf   = TRUE;
		    pragmas.file = name;
                    }

                break;

	    default:
                break;
            }
        }       
}


/**************************************************************************/
/* LOCAL  **************       ReadAssocList       ************************/
/**************************************************************************/
/* PURPOSE: READ AND RETURN A COMPOUND NODE's ASSOCIATION LIST. WHEN DONE */
/*          token WILL CONTAIN EITHER THE '%' CHARACTER OF THE FIRST      */
/*          PRAGMA OR EOLN.                                               */
/**************************************************************************/

static char *ReadAssocList()
{
    register int idx = 0;

    for (;;) {
	if ( token == '%' || IsEoln( token ) )
	    break;

        buffer[idx++] = token;

	if ( idx >= BUFFER_SIZE )
	    ReadError( "ReadAssocList BUFFER OVERFLOW" );

	NextToken;
        }

    buffer[idx] = '\0';

    return( CopyString( buffer ) );
}


/**************************************************************************/
/* GLOBAL **************          If1Read          ************************/
/**************************************************************************/
/* PURPOSE: READ IF1 INFORMATION FROM input AND BUILD ITS INTERNAL FORM.  */
/*          A PARTIAL SYNTAX CHECK IS MADE.  IN GENERAL SEMANTIC AND      */
/*          SYNTACTIC CORRECTNESS IS ASSUMED.                             */
/**************************************************************************/

void If1Read()
{
    register int     label, type;
    register int     snode, eport, dnode, iport;
    register int     ref1,  ref2;
    register char   *s;

    token = ' ';
    line  = 1;

    for (;;) {
        switch ( ReadLineId() ) {
            case 'C':
	        ReadStamp();
                break;

            case 'E':
                snode = ReadInteger();
                eport = ReadInteger();
                dnode = ReadInteger();
                iport = ReadInteger();
                type  = ReadInteger();

		ReadPragmas();

                MakeEdge( snode, eport, dnode, iport, type );
                break;

            case 'L':
                dnode = ReadInteger();
                iport = ReadInteger();
                type  = ReadInteger();
                s     = ReadLiteral( TRUE );

		ReadPragmas();

                MakeConst( dnode, iport, type , s );
                break;

            case 'G':
	        type  = ReadInteger();
		s     = ReadLiteral( FALSE );

		ReadPragmas();

                if ( s == NULL )
                    MakeGraph( IFSGraph, type, NULL );
                else
		    MakeGraph( IFLGraph, type, s );

                break;

            case 'I':
                type  = ReadInteger();
                s     = ReadLiteral( TRUE );

                ReadPragmas();

                MakeGraph( IFIGraph, type, s );
                break;

            case 'X':
                type  = ReadInteger();
                s     = ReadLiteral( TRUE );

		ReadPragmas();

                MakeGraph( IFXGraph, type, s );
                break;

            case 'N':
                label = ReadInteger();
                type  = ReadInteger();

		ReadPragmas();

                MakeNode( label, type );
                break;

            case 'T':
                label = ReadInteger();
                type  = ReadInteger();
    
                if ( type != IF_UNKNOWN ) {
                    ref1 = ReadInteger();

		    switch ( type ) {
			case IF_FIELD:
			case IF_FUNCTION:
			case IF_TAG:
			case IF_TUPLE:
                            ref2 = ReadInteger();
			    break;

                        default:
                            ref2 = 0;
			    break;
			}
                    }

                ReadPragmas();

                MakeInfo( label, type, ref1, ref2 );
                break;

            case '{':
		ReadPragmas();
	        MakeCompound( IFCBegin, NULL );
                break;

            case '}':
		s = ReadAssocList();

		ReadPragmas();
                MakeCompound( IFCEnd, s );
                break;

            case EOF:
		goto Done;

            default:
                ReadError( "ILLEGAL LINE IDENTIFIER" );
            }

        EatLine;
        }

Done:
    return;
}
