#include "world.h"

#define MaxClass      2000           /* MAX NUMBER OF EQUIVALENCE CLASSES */

static int   lclass;                 /* LAST EQUIVALENCE CLASS IN htable  */
static PINFO htable[MaxClass];       /* EQUIVALENCE CLASS HEAD POINTERS   */
static PINFO ttable[MaxClass];       /* EQUIVALENCE CLASS TAIL POINTERS   */


/**************************************************************************/
/* LOCAL  **************      SameEquivClass       ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF a and b ADDRESS TYPES IN THE SAME EQUIVALENCE  */
/*          CLASS.                                                        */
/**************************************************************************/

#define EquivClass(x) (((x) == NULL)? -1 : (x)->eid)

static int SameEquivClass( a, b )
PINFO a;
PINFO b;
{
    return( (EquivClass( a->info1 ) == EquivClass( b->info1 )) &&
	    (EquivClass( a->info2 ) == EquivClass( b->info2 )) );
}


/**************************************************************************/
/* LOCAL  **************    CreateNewEquivClass    ************************/
/**************************************************************************/
/* PURPOSE: CREATE A NEW EQUIVALENCE CLASS WHOSE FIRST ENTRY IS r.        */
/**************************************************************************/

static void CreateNewEquivClass( r )
PINFO r;
{
    if ( lclass++ >= MaxClass )
	Error1( "CreateNewEquivClass: OUT OF EQUIVALENCE CLASSES" );

    r->mnext = NULL;
    r->eid   = lclass;

    htable[ lclass ] = ttable[ lclass ] = r;
}


/**************************************************************************/
/* LOCAL  **************   RemoveFromEquivClass    ************************/
/**************************************************************************/
/* PURPOSE: REMOVE ENTRY x FROM THE EQUIVALENCE CLASS CONTAINING o.       */
/**************************************************************************/

static void RemoveFromEquivClass( o, x )
PINFO o;
PINFO x;
{
    if ( x == ttable[ o->eid ] )
        ttable[ o->eid ] = o;

    o->mnext = x->mnext;
}


/**************************************************************************/
/* LOCAL  **************      AddToEquivClass      ************************/
/**************************************************************************/
/* PURPOSE: ADD n TO THE END OF THE EQUIVALENCE CLASS HEADED BY r.        */
/**************************************************************************/

static void AddToEquivClass( r, n )
PINFO r;
PINFO n;
{
    register int c;

    c = r->eid;

    n->mnext = NULL;
    n->eid   = c;

    ttable[ c ]->mnext = n;
    ttable[ c ] = n;
}


/**************************************************************************/
/* LOCAL  **************     InitEquivClasses      ************************/
/**************************************************************************/
/* PURPOSE: PLACE ALL SYMBOL TABLE TYPES INTO ONE THEIR APPROPRIATE       */
/*          EQUIVALENT CLASSES. WHEN DONE, lclass + 1 ADDRESSES THE NEXT  */
/*          AVAILABLE SLOT IN THE EQUIVALENCE CLASS TABLE.                */
/**************************************************************************/

static void InitEquivClasses()
{
    register PINFO i;
    register int   c;

    for ( c = 0; c < MaxClass; c++ ) {
        htable[ c ] = NULL;
        ttable[ c ] = NULL;
        }

    for ( i = ihead; i != NULL; i = i->next ) {
	i->mnext = NULL;
	i->eid   = i->type;

	if ( htable[ i->eid ] == NULL ) {
            htable[ i->eid ] = ttable[ i->eid ] = i;
	    continue;
	    }

        ttable[ i->eid ]->mnext = i;
        ttable[ i->eid ] = i;
	}

    lclass = IF_NONTYPE;
}


/**************************************************************************/
/* LOCAL  **************        GatherOthers       ************************/
/**************************************************************************/
/* PURPOSE: REMOVE MEMBER r AND ALL ITS EQUIVALENT SUCESSORS FROM THIER   */
/*          EQUIVALENCE CLASS AND PLACE THEM IN A NEW CLASS.  NODE p IS   */
/*          THE PREDECESSOR OF r.                                         */
/**************************************************************************/

static void GatherOthers( p, r )
PINFO p;
PINFO r;
{
    register PINFO pm;

    /* MARK ALL ENTRIES EQUIVALENT TO REPRESENTATIVE r */

    for ( pm = r->mnext; pm != NULL; pm = pm->mnext )
        if ( SameEquivClass( r, pm ) )
	    pm->label = -(pm->label);

    /* REMOVE ALL MARKED ENTRIES */

    RemoveFromEquivClass( p, r );
    CreateNewEquivClass( r );

    while ( p->mnext != NULL ) {
        pm = p->mnext;

        if ( pm->label < 0 ) {
            pm->label = -(pm->label);

            RemoveFromEquivClass( p, pm );
            AddToEquivClass ( r, pm );
	    }
        else
            p = pm;
	}
}


/**************************************************************************/
/* LOCAL  **************        PointToHead        ************************/
/**************************************************************************/
/* PURPOSE: MODIFY THE LABEL OF EACH EQUIVALENCE CLASS MEMBER, EXCEPT THE */
/*          FIRST, TO THAT OF THE FIRST (THE ONLY ONE TO BE PRINTED).     */
/**************************************************************************/

static void PointToHead()
{
    register PINFO m;
    register PINFO r;
    register int   c;

    for ( c = 0; c <= lclass; c++ )
        if ( htable[c] != NULL ) {
            r = htable[c];

            for ( m = r->mnext; m != NULL; m = m->mnext ) {
	        m->label = r->label;
	        m->print = FALSE;
	        }
	    }
}


/**************************************************************************/
/* GLOBAL **************         SmashTypes        ************************/
/**************************************************************************/
/* PURPOSE: GIVE STRUCTURALLY EQUIVALENT TYPES THE SAME LABEL. ONLY ONE   */
/*          TYPE FROM EACH EQUIVALENT CLASS IS MARKED FOR PRINTING.       */
/*                                                                        */
/* NOTE:    ORIGINALLY WRITTEN IN PASCAL BY sks AT LLNL:  1/10/83.        */
/**************************************************************************/

void SmashTypes()
{
    register int   chgd = TRUE;
    register int   c;
    register PINFO p;
    register PINFO m;
    register PINFO r;

    InitEquivClasses();

    while ( chgd ) {
        chgd = FALSE;

        for ( c = 0; c <= lclass; c++ ) {
	    r = htable[c];
	    p = r;

	    if ( p == NULL )
	        m = NULL;
	    else
	        m = p->mnext;

	    while ( m != NULL )
	        if ( SameEquivClass( r, m ) ) {
		    p = m;
		    m = p->mnext;
	        } else {
		    GatherOthers( p, m );
		    chgd = TRUE; m = NULL;
		    }
	    }
        }    

    PointToHead();
}
