#include "world.h"

#define MaxClass      2000           /* MAX NUMBER OF EQUIVALENCE CLASSES */

static int   lclass;                 /* LAST EQUIVALENCE CLASS IN htable  */
static PINFO htable[MaxClass];       /* EQUIVALENCE CLASS HEAD POINTERS   */
static PINFO ttable[MaxClass];       /* EQUIVALENCE CLASS TAIL POINTERS   */


/**************************************************************************/
/* LOCAL  **************   AdjustInfoReferences    ************************/
/**************************************************************************/
/* PURPOSE: ADJUST ALL INFO REFERENCES IN GRAPH g SO THEY ADDRESS THE     */
/*          FIRST MEMBER OF EACH EQUIVALENCE CLASS.                       */
/**************************************************************************/

static void AdjustInfoReferences( g )
PNODE g;
{
  register PNODE sg;
  register PNODE n;
  register PEDGE i;

  for ( n = g; n != NULL; n = n->nsucc ) {
    if ( n->info != NULL )
      if ( n->info->fmem != NULL )
        n->info = n->info->fmem;

    for ( i = n->imp; i != NULL; i = i->isucc )
      if ( i->info != NULL )
	if ( i->info->fmem != NULL )
	  i->info = i->info->fmem;

    /* BUG FIX cann 7/15/91 */
    if ( IsCompound( n ) || (n->type >= IFFirstSum && n->type <= IFFirstAbsMax))
      for ( sg = n->C_SUBS; sg != NULL; sg = sg->gsucc )
	AdjustInfoReferences( sg );
    }
}


/**************************************************************************/
/* LOCAL  **************      SameEquivClass       ************************/
/**************************************************************************/
/* PURPOSE: RETURN TRUE IF a and b ADDRESS TYPES IN THE SAME EQUIVALENCE  */
/*          CLASS.                                                        */
/**************************************************************************/

#define EquivClass(x) (((x) == NULL)? -1 : (x)->eid)

static int SameEquivClass( a, b )
PINFO a;
PINFO b;
{
    return( (EquivClass( a->info1 ) == EquivClass( b->info1 )) &&
	    (EquivClass( a->info2 ) == EquivClass( b->info2 )) );
}


/**************************************************************************/
/* LOCAL  **************    CreateNewEquivClass    ************************/
/**************************************************************************/
/* PURPOSE: CREATE A NEW EQUIVALENCE CLASS WHOSE FIRST ENTRY IS r.        */
/**************************************************************************/

static void CreateNewEquivClass( r )
PINFO r;
{
    if ( lclass++ >= MaxClass )
	Error1( "CreateNewEquivClass: OUT OF EQUIVALENCE CLASSES" );

    r->mnext = NULL;
    r->eid   = lclass;

    htable[ lclass ] = ttable[ lclass ] = r;
}


/**************************************************************************/
/* LOCAL  **************   RemoveFromEquivClass    ************************/
/**************************************************************************/
/* PURPOSE: REMOVE ENTRY x FROM THE EQUIVALENCE CLASS CONTAINING o.       */
/**************************************************************************/

static void RemoveFromEquivClass( o, x )
PINFO o;
PINFO x;
{
    if ( x == ttable[ o->eid ] )
        ttable[ o->eid ] = o;

    o->mnext = x->mnext;
}


/**************************************************************************/
/* LOCAL  **************      AddToEquivClass      ************************/
/**************************************************************************/
/* PURPOSE: ADD n TO THE END OF THE EQUIVALENCE CLASS HEADED BY r.        */
/**************************************************************************/

static void AddToEquivClass( r, n )
PINFO r;
PINFO n;
{
    register int c;

    c = r->eid;

    n->mnext = NULL;
    n->eid   = c;

    ttable[ c ]->mnext = n;
    ttable[ c ] = n;
}


/**************************************************************************/
/* LOCAL  **************     InitEquivClasses      ************************/
/**************************************************************************/
/* PURPOSE: PLACE ALL SYMBOL TABLE TYPES INTO THEIR APPROPRIATE           */
/*          EQUIVALENT CLASSES. WHEN DONE, lclass + 1 ADDRESSES THE NEXT  */
/*          AVAILABLE SLOT IN THE EQUIVALENCE CLASS TABLE.                */
/**************************************************************************/

static void InitEquivClasses()
{
    register PINFO i;
    register int   c;

    for ( c = 0; c < MaxClass; c++ ) {
        htable[ c ] = NULL;
        ttable[ c ] = NULL;
        }

    for ( i = ihead; i != NULL; i = i->next ) {
	i->mnext = NULL;
	i->eid   = i->type;

	if ( htable[ i->eid ] == NULL ) {
            htable[ i->eid ] = ttable[ i->eid ] = i;
	    continue;
	    }

        ttable[ i->eid ]->mnext = i;
        ttable[ i->eid ] = i;
	}

    lclass = TYPE_STOP; /* LAST LEGAL IF2 TYPE (see if2.h) */
}


/**************************************************************************/
/* LOCAL  **************        GatherOthers       ************************/
/**************************************************************************/
/* PURPOSE: REMOVE MEMBER r AND ALL ITS EQUIVALENT SUCESSORS FROM THIER   */
/*          EQUIVALENCE CLASS AND PLACE THEM IN A NEW CLASS.  NODE p IS   */
/*          THE PREDECESSOR OF r.                                         */
/**************************************************************************/

static void GatherOthers( p, r )
PINFO p;
PINFO r;
{
    register PINFO pm;

    /* MARK ALL ENTRIES EQUIVALENT TO REPRESENTATIVE r */

    for ( pm = r->mnext; pm != NULL; pm = pm->mnext )
        if ( SameEquivClass( r, pm ) )
	    pm->label = -(pm->label);

    /* REMOVE ALL MARKED ENTRIES */

    RemoveFromEquivClass( p, r );
    CreateNewEquivClass( r );

    while ( p->mnext != NULL ) {
        pm = p->mnext;

        if ( pm->label < 0 ) {
            pm->label = -(pm->label);

            RemoveFromEquivClass( p, pm );
            AddToEquivClass ( r, pm );
	    }
        else
            p = pm;
	}
}


/**************************************************************************/
/* LOCAL  **************        PointToHead        ************************/
/**************************************************************************/
/* PURPOSE: MODIFY THE LABEL OF EACH EQUIVALENCE CLASS MEMBER, EXCEPT THE */
/*          FIRST, TO THAT OF THE FIRST. ALSO SET THE fmem OF EACH MEMBER */
/*          TO ADDRESS THE FIRST.                                         */
/**************************************************************************/

static void PointToHead()
{
    register PINFO m;
    register PINFO r;
    register int   c;

    for ( c = 0; c <= lclass; c++ )
        if ( htable[c] != NULL ) {
            r = htable[c];

            for ( m = r->mnext; m != NULL; m = m->mnext ) {
	        m->label = r->label;
	        m->fmem  = r;
	        }
	    }
}


/**************************************************************************/
/* GLOBAL **************         SmashTypes        ************************/
/**************************************************************************/
/* PURPOSE: GIVE STRUCTURALLY EQUIVALENT TYPES THE SAME LABEL. THEN       */
/*          RELABEL ALL SYMBOL TABLE ENTRIES TO APPEAR IN THE TYPE TABLE. */
/*          NOTE: ORIGINALLY WRITTEN IN PASCAL BY sks AT LLNL:  1/10/83.  */
/*          ONLY THE FIRST MEMBER OF EACH EQUIVALENCE CLASS IS KEEPED IN  */
/*          THE SYMBOL TABLE.  THE IF2 GRAPH INFO POINTERS ARE ADJUSTED   */
/*          ACCORDINGLY.                                                  */
/**************************************************************************/

void SmashTypes()
{
    register int   chgd = TRUE;
    register int   c;
    register PINFO p;
    register PINFO m;
    register PINFO r;
    register PINFO ppred;
    register PINFO ps;
    register PNODE f;

    InitEquivClasses();

    while ( chgd ) {
        chgd = FALSE;

        for ( c = 0; c <= lclass; c++ ) {
	    r = htable[c];
	    p = r;

	    if ( p == NULL )
	        m = NULL;
	    else
	        m = p->mnext;

	    while ( m != NULL )
	        if ( SameEquivClass( r, m ) ) {
		    p = m;
		    m = p->mnext;
	        } else {
		    GatherOthers( p, m );
		    chgd = TRUE; m = NULL;
		    }
	    }
        }    

    PointToHead();

    /* RELABEL ALL NODES TO APPEAR IN THE TYPE TABLE                      */

    for ( c = 0, p = ihead; p != NULL; p = p->next )
	switch ( p->type ) {
	    case IF_TUPLE:
	    case IF_FUNCTION:
	    case IF_UNKNOWN:
	    case IF_NONTYPE:
	    case IF_MULTPLE:
	    case IF_BUFFER:
		continue;

	    default:
		/* ONLY ADJUST LABEL OF FIRST EQUIVALENCE CLASS MEMBER    */
		if ( p->fmem == NULL )
		    p->label = ++c;

                break;
            }

    /* IN EACH CLASS, ADJUST THE LABEL OF EACH MEMBER TO THAT OF THE 1ST  */
    for ( p = ihead; p != NULL; p = p->next )
	if ( p->fmem != NULL )
	    p->label = p->fmem->label;

    for ( f = glstop->gsucc; f != NULL; f = f->gsucc )
      AdjustInfoReferences( f );

    /* FIX integer AND VARIOUS OTHER SPECIAL TYPES */
    if ( integer != NULL )
      if ( integer->fmem != NULL )
	integer = integer->fmem;

    if ( ptr_integer != NULL )
      if ( ptr_integer->fmem != NULL )
	ptr_integer = ptr_integer->fmem;

    if ( ptr_real != NULL )
      if ( ptr_real->fmem != NULL )
	ptr_real = ptr_real->fmem;

    if ( ptr_double != NULL )
      if ( ptr_double->fmem != NULL )
	ptr_double = ptr_double->fmem;

    if ( ptr != NULL )
      if ( ptr->fmem != NULL )
	ptr = ptr->fmem;

    /* ONLY KEEP THE FIRST MEMBER OF EACH EQUIVALENCE CLASS IN THE SYMBOL */
    /* TABLE.                                                             */
    for ( ppred = NULL, p = ihead; p != NULL; p = ps ) {
      if ( p->info1 != NULL )
	if ( p->info1->fmem != NULL )
	  p->info1 = p->info1->fmem;

      if ( p->info2 != NULL )
	if ( p->info2->fmem != NULL )
	  p->info2 = p->info2->fmem;

      ps = p->next;

      if ( p->fmem == NULL ) {
	p->mnext = NULL;
	ppred    = p;
	continue;
	}

      if ( ppred == NULL )
	ihead = ps;
      else
	ppred->next = p->next;

      p->type = IF_UNKNOWN; /* HOPEFULLY, THIS WILL HELP UNCOVER BUGS! */
      }

    itail = ppred;
}
