#include "SC.h"
#include "SCLib.h"
#include "Debug.h"

int     GettingCmd = 0;

#define Padding(n)	(TwentySpaces+n)
static char	*TwentySpaces = "                    ";

/* If we have a character, return it.  If not, fill the buffer and */
/* return the new current character. */
#define CC(Prompt)	(DebugBuffer.Buf_Buffer[DebugBuffer.Buf_idx]\
               ?DebugBuffer.Buf_Buffer[DebugBuffer.Buf_idx]\
               :(DebugFillBuffer(Prompt),\
                 DebugBuffer.Buf_Buffer[DebugBuffer.Buf_idx]\
                 )\
               )

/* If not at the end of the buffer just step the pointer, otherwise */
/* refill the buffer */
#define NextC(Prompt)	{\
                   if(DebugBuffer.Buf_Buffer[DebugBuffer.Buf_idx]) {\
                      DebugBuffer.Buf_idx++;\
                   } else {\
                      DebugFillBuffer(Prompt);\
        }\
}

#define EOB	((char)(NULL))

/* Return pointer to current buffer index */
#define DebugBufferPtr() ( &DebugBuffer.Buf_Buffer[DebugBuffer.Buf_idx] ) 
#define EmptyDebugBuffer() { DebugBuffer.Buf_idx = 0; DebugBuffer.Buf_Buffer[0] = EOB; }

typedef int	((*IFunction)());
     typedef struct {
       char		*name;	
       IFunction	Command;
       char		*shorthelp;
       char		**longhelp;
     } CommandEntry;
#define NoCommand	{ NULL,UnknownCommand,NULL,NULL }
#define ACommand(name,f,shorthelp,help) { name,f,shorthelp,help }
     
     static	int	HelpCommand();
     static	int	UnknownCommand();
     
     /* ------------------------------------------------------------ */
     static	int	DebugAgain();
     static	int	DebugBreak();
     static	int	DebugContinue();
     static	int	DebugDelete();
     static	int	DebugEnter();
     static	int	DebugFile();
     static	int	DebugGarbage();
     static	int	DebugInclude();
     static	int	DebugList();
     static	int	DebugNew();
     static	int	DebugMode();
     static	int	DebugPrint();
     static	int	DebugQuit();
     static	int	DebugRun();
     static	int	DebugSet();
     static	int	DebugTrace();
     static	int	DebugView();
     static	int	DebugWhere();
     /* ------------------------------------------------------------ */
     char	*AHelp[] = {
       "again",
       "  Rerun from the last user entry point.  For instance, if the",
       "  last run command was ``debug-0> run 10;'', then the again",
       "  command would do a run 10.",
       NULL
       };
     char	*BHelp[] = {
       "break option...",
       "  Set a breakpoint.  A number of ways are provided:",
       "    break error",
       "      Break when a node produces an error value.",
       "    break function NAME1 NAME2 ...",
       "      Break upon entering function NAME",
       "    break graphend ( N1 | NAME1 ) (N2 | NAME2) ...",
       "      Break as the specified graph ends.  If NAME is specified,",
       "      then the break occurs at the end of the named function.",
       "      If a line number N is specified, then the break occurs after",
       "      the subgraph around that point.",
       "    break line N1 N2 ...",
       "      Break at the line specified by N.",
       "    break name NAME1 {N1} NAME2 {N2} ...",
       "      Break after the NAME is generated.  If N is also specified,",
       "      only those operations at line N which generate NAME will",
       "      cause a break",
       "    break operation N1 N2 ...",
       "      Break before the specified intermediate code operation.",
       "      Use the ``list code'' or the ``list opsat'' commands to",
       "      get operation numbers.",
       NULL
       };
     
     char	*CHelp[] = {
       "continue",
       "  Exit debug mode, and continue program ",
       NULL
       };
     
     char	*DHelp[] = {
       "delete BP0 {BP1 ...}",
       "delete breakpoints BP0 {BP1 ...}",
       "delete views VIEW0 {VIEW1 ...}",
       "delete alias AL0 {AL1 ...}", 
       "  Delete one (or several) breakpoints or views by ID.  These",
       "  id's are given when a breakpoint is set or can be found by",
       "  using the ``list breakpoints'' or ``list views'' commands.",
       "  Using the ``delete *'' command will cause ALL breakpoints",
       "  to be deleted.",
       NULL
       };
     
     char	*EHelp[] = {
       "enter ENTRYNAME {ARGS... | < INPUTFILE}",
       "  See the ``run'' command for more information.  Executes a arbitrary",
       "  function rather than the default entry.",
       NULL
       };
     
     char	*FHelp[] = {
       "file NAME {ascii} {binary} {names} {reset} {init string}",
       "  Setup a view file.  Any of the options can be unset by prepending",
       "  a !.  The defaults are: ascii  names init \"\"",
       "  Setting binary (or !ascii) is not currently supported.",
       "  Setting !names keeps the view displayer from listing names (or",
       "  comment strings (see help view for more info).  The reset",
       "  keyword causes the file to be closed and reopened for writing.",
       "  The init keyword must be the last keyword.  It is followed by",
       "  strings that will be written to the file the first time it is",
       "  written to",
       NULL
       };
     
     char	*GHelp[] = {
       "garbage",
       "Calling ``garbage'' will explicitly compact the heap by",
       "removing  unreachable  values.  Aside from keeping your",
       "working set size small, it  won't  change  any  runtime",
       "behavior.",
       NULL
       };
     
     char	*HHelp[] = {
       "help { COM... }",
       "  Give overview of all commands or a full description of several.",
       "  The conventions used in describing command syntax are keywords in",
       "  lower case, {} indicates optional, | indicates alternation, () for",
       "  grouping, ... indicates repetition.  All commands and keywords can",
       "  be abrieviated to their unique letter, and are case insensitive.",
       "  I.e.  ``Break name foo'' is equivalent to ``b n foo''.",
       "  Names, line numbers, and operation numbers refer, by",
       "  default, to the current module being executed (the first if no",
       "  execution has taken place).  The names and numbers can be more",
       "  completely specified by using a qualified identifier of the",
       "  form MODULE`FUNCTION:(NAME | N).  Single and double quotes can",
       "  be used to group arguments in a command line.  This, for instance",
       "  is the only way to access values like \"old i\" or to use multiple",
       "  words in a view comment.",
       NULL
       };
     
     char	*IHelp[] = {
       "include FILE {FILE...}",
       "  Read commands from the named files",
       NULL
       };
     
     char	*LHelp[] = {
       "list aliases",
       "  List the commands created by the user using NEW",
       "list breakpoints",
       "  List the active breakpoints by breakpoint number.",
       "list code ( NAME | N )",
       "  List the intermediate code for a named function or for graphs or",
       "  compound instructions at a line number.",
       "list entries {MODULENAME}",
       "  List the defined entry points in either all modules or the specified",
       "  one.",
       "list files",
       "  List the active view output files and their current attributes.",
       "  See the ``file'' command for help with files.",
       "list modules",
       "  List the modules defined for this program.",
       "list names FUNCTIONNAME",
       "  List the value names that are defined in specified function.",
       "list opsat N",
       "  List the intermediate code operations that are defined at the",
       "  specified line number.",
       "list sysvars",
       "  List the internally defined, user settable variables and their values.",
       "  See the ``set'' command for information on these values.",
       "list views",
       "  List the variable views that are active.",
       NULL
       };
     char	*MHelp[] = {
       "mode {linestep | opstep | normal}",
       "  Set the run mode.  In linestep, the debugger will run until the",
       "  source line changes (or until a breakpoint is found).  In opstep,",
       "  the debugger will execute one intermediate code operation.  In",
       "  normal mode, the debugger runs until the end of code or a breakpoint.",
       NULL
       };
     char	*NHelp[] = {
       "new 'pattern' 'expansion' (within quotes or double quotes)",
       "  The new command creates an alias to one or more commands. Arguments",
       "  can be passed to the expansion string using the $ sign. A $ sign in",
       "  the pattern string specifies the location of an argument. A $# (#=1-9)",
       "  indicates the argument to be substitute in the expansion string. Up to",
       "  9 arguments can be specified. Examples:",
       "    new 'step' 'mode opstat;c;mode normal';",
       "    new 'stop at $ or $' 'break line $1;break name $2';",
       "    new 'stop around $' 'break function $1;break graphend $1';",
       NULL
       };
     char	*PHelp[] = {
       "print name {name...}",
       "  Print the value associated with one or more names.  To access",
       "  names outside the current function, use a qualified name like",
       "  ``print f:a''",
       NULL
       };
     char	*QHelp[] = {
       "quit",
       "  Exit debugger and terminate program",
       NULL
       };
     char	*RHelp[] = {
       "run {args... | < filename}",
       "  Run the program using the default entry.",
       "  The arguments are passed into the entry and the outputs reported.",
       "  Input can be redirected from a file by using the redirect mark <",
       "  With no arguments, the inputs are read from debugger input.",
       NULL
       };
     char	*SHelp[] = {
       "set VARNAME VALUE",
       "  Set the indicated name to the value given, i.e. ``set PrintMax 3''",
       "  Variable names must be fully specified.  System vars include:",
       "    Name           DefaultValue    Meaning",
       "    PRINTMAX          0            Max # of elements to print in an array",
       "                                   0 means print all of them.",
       "    DISPLAYBYNAME     False        True => Display names in break ops",
       "                                   False => Display values in break ops",
       "    NODESTEP          False        True => break at every IF1 op",
       "                                   False => don't break at ops",
       "    LINESTEP          False        True => break at each line",
       "                                   False => don't break each line",
       "  Note that some of these variables are affected by commands like",
       "  ``mode''.",
       NULL
       };
     char	*THelp[] = {
       "trace ...",
       "  See break.  All arguments are the same.  The only difference is",
       "  the computation will not stop for trace points.  This is useful",
       "  if a view has been set up.  The view will be reported, but",
       "  execution will not stop.",
       NULL
       };
     char	*VHelp[] = {
       "view {@N} {> FILENAME} {%COMMENT} NAME...",
       "  A view is a peek at the value of a variable whenever the computation",
       "  reaches a break or trace point.  The scope of the name is fully",
       "  dynamic, i.e. A lookup will occur for the name in the context of",
       "  the breakpoint, so ``view X'' could refer to different variables",
       "  depending on where the break occurs.  The @N syntax is used to specify",
       "  that a view refers only to the specific breakpoint identifier N.  The",
       "  > FILENAME sytax specifies that the view is to printed to an output",
       "  file instead of the screen (see the ``file'' command for information",
       "  on how to set file parameters).  The %COMMENT syntax applies to the",
       "  NAME immediately following the comment.  It is used to provide a",
       "  textual message in place of the value's name when it is printed.",
       "  Examples:",
       "  View the value of X at each breakpoint.",
       "    debug-0> view x;",
       "  View the value of X in function F",
       "    debug-0> view f:x;",
       "  Trace the value of X to a file",
       "    debug-0> view > outfile x;",
       "  Trace the value of X to a file without the name",
       "    debug-0> view > outfile x;",
       "    debug-0> file outfile !names;",
       "  View the value of X only at breakpoint id 2",
       "    debug-0> view @2 x;",
       "  View the value of X with a message",
       "    debug-0> view \"%The value of X=\" x;",
       NULL
       };
     char	*WHelp[] = {
       "where {N}",
       "  Dumps the call frame stack.  With optional argument, it reports",
       "  the last N frames.",
       NULL
       };
     
     /* ------------------------------------------------------------ */
     CommandEntry CommandTable[] = {
       /* a */	ACommand("again",DebugAgain,"Run again",AHelp),
                /* b */	ACommand("break",DebugBreak,"Set Breakpoints",BHelp),
                /* c */	ACommand("continue",DebugContinue,"Continue execution",CHelp),
                /* d */	ACommand("delete",DebugDelete,"Delete breakpoint",DHelp),
                /* e */	ACommand("enter",DebugEnter,"Run any function",EHelp),
                /* f */	ACommand("file",DebugFile,"Set up view file info",FHelp),
                /* g */	ACommand("garbage",DebugGarbage,"Empty the garbage can",GHelp),
                /* h */	ACommand("help",HelpCommand,"Help or ?",HHelp),
                /* i */	ACommand("include",DebugInclude,"read a command file",IHelp),
                /* j */	NoCommand,
                /* k */	NoCommand,
                /* l */	ACommand("list",DebugList,"List values/status",LHelp),
                /* m */	ACommand("mode",DebugMode,"Set run mode",MHelp),
                /* n */	ACommand("new",DebugNew,"Create new command",NHelp),
                /* o */	NoCommand,
                /* p */	ACommand("print",DebugPrint,"Print a value",PHelp),
                /* q */	ACommand("quit",DebugQuit,"Exit program",QHelp),
                /* r */	ACommand("run",DebugRun,"Run default function",RHelp),
                /* s */	ACommand("set",DebugSet,"Set values",SHelp),
                /* t */	ACommand("trace",DebugTrace,"Set Tracepoints",THelp),
                /* u */	NoCommand,
                /* v */	ACommand("view",DebugView,"View values at breakpoints",VHelp),
                /* w */	ACommand("where",DebugWhere,"Show call frame",WHelp),
                /* x */	NoCommand,
                /* y */	NoCommand,
                /* z */	NoCommand
                };
#define ShowKeywords(List)    {int i; char message[100]; for (i=0;i<26;i++) {if ( List[i] ) {(void)sprintf(message,"  %s\n",List[i]); DebugMessage(message);}}}
     char	*BreakKeywords[] = {
       /* a */	NULL,
                /* b */	NULL,
                /* c */	NULL,
                /* d */	NULL,
                /* e */	"error",
                /* f */	"function",
                /* g */	"graphend",
                /* h */	NULL,
                /* i */	NULL,
                /* j */	NULL,
                /* k */	NULL,
                /* l */	"line",
                /* m */	NULL,
                /* n */	"name",
                /* o */	"operation",
                /* p */	NULL,
                /* q */	NULL,
                /* r */	NULL,
                /* s */	NULL,
                /* t */	NULL,
                /* u */	NULL,
                /* v */	NULL,
                /* w */	NULL,
                /* x */	NULL,
                /* y */	NULL,
                /* z */	NULL};
     char	*DeleteKeywords[] = {
       /* a */	"aliases",
                /* b */	"breakpoints",
                /* c */	NULL,
                /* d */	NULL,
                /* e */	NULL,
                /* f */	NULL,
                /* g */	NULL,
                /* h */	NULL,
                /* i */	NULL,
                /* j */	NULL,
                /* k */	NULL,
                /* l */	NULL,
                /* m */	NULL,
                /* n */	NULL,
                /* o */	NULL,
                /* p */	NULL,
                /* q */	NULL,
                /* r */	NULL,
                /* s */	NULL,
                /* t */	NULL,
                /* u */	NULL,
                /* v */	"views",
                /* w */	NULL,
                /* x */	NULL,
                /* y */	NULL,
                /* z */	NULL};
     char	*ListKeywords[] = {
       /* a */	"aliases",
                /* b */	"breakpoints",
                /* c */	"code",
                /* d */	NULL,
                /* e */	"entries",
                /* f */	"files",
                /* g */	NULL,
                /* h */	NULL,
                /* i */	NULL,
                /* j */	NULL,
                /* k */	NULL,
                /* l */	NULL,
                /* m */	"modules",
                /* n */	"names",
                /* o */	"opsat",
                /* p */	NULL,
                /* q */	NULL,
                /* r */	NULL,
                /* s */	"sysvars",
                /* t */	NULL,
                /* u */	NULL,
                /* v */	"views",
                /* w */	NULL,
                /* x */	NULL,
                /* y */	NULL,
                /* z */	NULL};
     char	*ModeKeywords[] = {
       /* a */	NULL,
                /* b */	NULL,
                /* c */	NULL,
                /* d */	NULL,
                /* e */	NULL,
                /* f */	NULL,
                /* g */	NULL,
                /* h */	NULL,
                /* i */	NULL,
                /* j */	NULL,
                /* k */	NULL,
                /* l */	"linestep",
                /* m */	NULL,
                /* n */	"normal",
                /* o */	"opstep",
                /* p */	NULL,
                /* q */	NULL,
                /* r */	NULL,
                /* s */	NULL,
                /* t */	NULL,
                /* u */	NULL,
                /* v */	NULL,
                /* w */	NULL,
                /* x */	NULL,
                /* y */	NULL,
                /* z */	NULL};
     /* ------------------------------------------------------------ */
     int
       NameEq(name,s)
     char	*name;
     char	*s;
{
  char	N,S;
  
  for(;*name && *s && isgraph(*s);(name++,s++)) {
    N = islower(*name)?toupper(*name):(*name);
    S = islower(*s)?toupper(*s):(*s);
    if ( N != S ) return 0;
  }
  return 1;
}
/* ------------------------------------------------------------ */
int
CommandTablePosition(s)
     char	*s;
{
  int		Pos, length;
  char          cmd[80];
  
  /* Calculate the length of the command */
  for( length=0; *s && isgraph(s[length]); length++);
  
  /* Commands must be one letter or the whole word */
  if ( s[0] == '?' ) {
    Pos = (int)('h'-'a');
    s[0] = 'h';
  } else if ( islower(s[0]) ) {
    Pos = (int)(s[0]-'a');
  } else if ( isupper(s[0]) ) {
    Pos = (int)(s[0]-'A');
  } else {
    return (-1);
  }
  
  /* Compare the whole word when length > 1 */
  if (!CommandTable[Pos].name || (length > 1 && length != strlen(CommandTable[Pos].name)) ||
      !NameEq(CommandTable[Pos].name,s)) return(-1);
  
  return Pos;
}
/* ------------------------------------------------------------ */
char
KeywordTablePosition(Keywords,name)
     char	*Keywords[];
     char	*name;
{
  int		Pos;
  
  if ( islower(name[0]) ) {
    Pos = (int)(name[0]-'a');
    if (!Keywords[Pos] || !NameEq(Keywords[Pos],name)) Pos = (-1);
  } else if ( isupper(name[0]) ) {
    Pos = (int)(name[0]-'A');
    if (!Keywords[Pos] || !NameEq(Keywords[Pos],name)) Pos = (-1);
  } else {
    Pos = (-1);
  }
  
  return (Pos<0)?((char)(-1)):(((char)Pos)+'a');
}
/* ------------------------------------------------------------ */
static int
HelpCommand(com)
     char	*com;
{
  char		*parse[2];
  char		message[100];
  int		i;
  int		ComPos;
  char		**HelpList;
  
  DebugMessage("\n");
  
  ParseList(com,parse,1);
  
  if ( *parse[1] ) {
    /* Arguments */
    while( *parse[1] ) {
      ParseList(parse[1],parse,1); /* Get the next argument */
      
      ComPos = CommandTablePosition(parse[0]);
      if (ComPos >= 0) {
        for(HelpList = CommandTable[ComPos].longhelp;*HelpList;HelpList++) {
          DebugMessage(*HelpList);
          DebugMessage("\n");
        }
        DebugMessage("\n");
      } else {
        (void)sprintf(message,"No information on %s\n",parse[0]);
        DebugMessage(message);
      }
    }
  } else {
    /* No Arguments */
    for (i=0;i<26;i++) {
      if (CommandTable[i].name) {
        (void)sprintf(message,"%c\t%s%s%s\n",
                      (char)('a'+i),
                      CommandTable[i].name,
                      Padding(strlen(CommandTable[i].name)),
                      CommandTable[i].shorthelp);
        DebugMessage(message);
      }
    }
  }
  
  return	0;
}
/* ------------------------------------------------------------ */
void
DebugExecute(Entry,args)
     char	*Entry;
     char	*args;
{
  NodeInfo	EntryNode;
  char		EntryError[MaxCommentSize];
  int		Redirection;
  char		*filename,*p;
  char		*InputString;
  
  /* ------------------------------------------------------------ */
  /* Find the appropriate entry point */
  EntryNode = FindGraph(Entry,NULL,TRUE);
  if ( !EntryNode ) {
    DebugMessage("Entry not found\n");
    return;
  }
  
  /* ------------------------------------------------------------ */
  /* See if the file is redirected, provided with arguments, or is */
  /* going to be read from stdin.. */
  if ( *args == '<' ) {
    /* If < and > both exist, its a record argument, not redirection! */
    Redirection = (index(args,'>') == NULL);
  } else {
    Redirection = 0;
  }
  
  /* ------------------------------------------------------------ */
  /* If the command provides arguments, use them or the redirection */
  /* file. */
  if ( Redirection ) {
    /* Skip to the start of the file name */
    for(filename = args+1; *filename && isspace(*filename); filename++);
    
    /* Make sure the name is NULL terminated (no trailing blanks) */
    for(p=filename; *p; p++) if (isspace(*p)) { *p = '\0'; break; }
    
    /* Allocate a string that holds the file */
    InputString = ReadFileIntoAString(filename);
    
    /* Set it up for input */
    OpenStringInput(InputString);
  } else if ( *args ) {	/* Read from the arguments */
    OpenStringInput(args);
  }
  
  /* ------------------------------------------------------------ */
  /* Run the graph */
  GraphEntry(EntryNode,EntryError);
  if ( EntryError[0] ) DebugMessage(EntryError);
  
  /* ------------------------------------------------------------ */
  /* Close the input string if opened */
  if ( *args ) CloseStringInput();
  if ( Redirection ) Free(InputString);
  
  return;
}
/* ------------------------------------------------------------ */
static int
SetBreakOrTrace(com,BreakOrTrace)
     char	*com;
     int	BreakOrTrace;
{
  char		*parse[2];
  char		arg;
  int		BP;
  char		message[MaxCommentSize];
  
  ParseList(com,parse,1);
  
  if ( *parse[1] )  {
    ParseList(parse[1],parse,1);
    arg = KeywordTablePosition(BreakKeywords,parse[0]);
    
    /* Process all arguments in the line */
    do 
    {
      ParseList(parse[1],parse,1);
      switch ( arg ) {
        
      case 'e':			/* ERROR */
        BP = BreakError(parse[0],BreakOrTrace);
        parse[1] = NULL;
        break;
        
      case 'f':			/* FUNCTION */
        BP = BreakFunction(parse[0],BreakOrTrace);
        break;
        
      case 'g':			/* GRAPHEND */
        BP = BreakGraphEnd(parse[0],BreakOrTrace);
        break;
        
      case 'l':			/* LINE */
        BP = BreakLine(parse[0],BreakOrTrace);
        break;
        
      case 'n':			/* NAME */
        /* Included the optional line number when specified */
        if ( *parse[1] && isdigit(*parse[1]) )
        {
          parse[0] += strlen(parse[0]);
          *parse[0] = ' ';
          for( ++parse[1]; *parse[1] && isdigit(*parse[1]); ++parse[1] );
          if ( *parse[1] ) 
          {
            *parse[1] = '\0';
            while( *(++parse[1]) && isspace(*parse[1]) );
          }
        }
        
        BP = BreakName(parse[0],BreakOrTrace);
        break;
        
      case 'o':			/* OPERATION */
        BP = BreakOperation(parse[0],BreakOrTrace);
        break;
        
      default:
        BP = 0;
        DebugMessage("Invalid break argument\n");
      }
      
      /* Report the breakpoint (if any was set) */
      if (BP) {
        (void)sprintf(message,"%s [%d]\n",(BreakOrTrace)?"Trace":"Break",BP);
        DebugMessage(message);
      }
      
    } while( *parse[1]  );
    
  } else {
    DebugMessage("break requires arguments:\n");
    ShowKeywords(BreakKeywords);
  }
  
  return 0;
}
/* ------------------------------------------------------------ */
char	LastEntry[MaxCommentSize] = "";
char	SavedEntryArgs[MaxCommentSize] = "";
static int
DebugAgain(com)			/* ARGSUSED */
     char	*com;
{
  if ( LastEntry[0] ) {
    DebugExecute(LastEntry,SavedEntryArgs);
  } else {
    DebugMessage("Nothing to re-run\n");
  }
  
  return 0;
}
/* ------------------------------------------------------------ */
static int
DebugBreak(com)
     char	*com;
{
  return SetBreakOrTrace(com,0);
}
/* ------------------------------------------------------------ */
static int
DebugTrace(com)
     char	*com;
{
  return SetBreakOrTrace(com,1);
}

/* ------------------------------------------------------------ */
/* ------------------------------------------------------------ */
static int
DebugContinue(com)		/*ARGSUSED*/
     char	*com;
{
  if ( DebugLevel == 0 && StartInDebug ) {
    DebugMessage("Nothing to continue\n");
    return 0;
  } else {
    return 2;
  }
}
/* ------------------------------------------------------------ */
static int
DebugDelete(com)
     char	*com;
{
  char		*parse[2];
  char		*p;		/* Will search for first real character */
  char		key;
  
  /* Strip off the command keyword */
  ParseList(com,parse,1);
  
  /* Find out what the first real character of the arguments is... */
  for(p=parse[1];*p && !isalpha(*p) && !isdigit(*p);p++);
  
  if ( isalpha(*p) ) {
    ParseList(parse[1],parse,1);
    key = KeywordTablePosition(DeleteKeywords,parse[0]);
  } else {
    key = 'b';			/* Default to deleting break/tracepoints */
  }
  
  switch (key) {
  case 'a':                    /* Aliases */
    DeleteAlias(parse[1]);
    break;
    
  case 'b':			/* BREAKPOINTS */
    DeleteBreak(parse[1]);
    break;
    
  case 'v':			/* VIEWS */
    DeleteView(parse[1]);
    break;
    
  default:
    DebugMessage("Cannot delete those\n");
  }
}
/* ------------------------------------------------------------ */
static int
DebugInclude(com)
     char	*com;
{
  char		*parse[2];
  
  ParseList(com,parse,1);
  while(*parse[1]) {
    ParseList(parse[1],parse,1);
    IncludeFile(parse[0]);
  }
  
  return 	0;
}

/* ------------------------------------------------------------ */
static int
DebugList(com)
     char	*com;
{
  char		*parse[2];
  char		arg;
  
  ParseList(com,parse,1);
  
  if ( *parse[1] )  {
    ParseList(parse[1],parse,1);
    arg = KeywordTablePosition(ListKeywords,parse[0]);
    switch ( arg ) {
    case 'a':                  /* Aliases */
      ListAliases();
      break;
      
    case 'b':			/* BREAKPOINTS */
      ListBreakPoints(parse[1]);
      break;
      
    case 'c':			/* CODE */
      ListCode(parse[1]);
      break;
      
    case 'e':			/* ENTRIES (functions) */
      ListFunctions(parse[1]);
      break;
      
    case 'f':			/* open view FILES */
      ListFiles();
      break;
      
    case 'm':			/* MODULES */
      ListModules(parse[1]);
      break;
      
    case 'n':			/* NAMES */
      ListNames(parse[1]);
      break;
      
    case 'o':			/* OPSAT */
      ListOpsAt(parse[1]);
      break;
      
    case 's':			/* SYSVARS */
      ListSystemVars(parse[1]);
      break;
      
    case 'v':			/* VIEW variables */
      ListViews();
      break;
      
    default:
      DebugMessage("Invalid List argument\n");
    }
  } else {
    DebugMessage("List requires an argument:\n");
    ShowKeywords(ListKeywords);
  }
  
  return 0;
}

static int
DebugNew(com)
     char *com;
{
  char     *parse[4];
  char     message[MaxCommentSize];
  int      id;
  
  id = 0;
  
  /* Get the pattern and expansion */
  ParseList(com, parse,3);
  if ( *parse[1] && *parse[2] )
    id = AddAlias( parse[1], parse[2] );
  else
    DebugMessage("NEW requires a pattern and expansion string within quotes\n");
  
  if ( id ) 
  {
    (void)sprintf( message, "New command [%d] created\n",id);
    DebugMessage(message);
  }
  
  return 0;
}

static int
DebugRun(com)
     char	*com;
{
  char		*parse[2];
  
  /* Strip off the run command  */
  ParseList(com,parse,1);
  (void)strcpy(LastEntry,EntryName);/* Save the last entry for AGAIN command */
  (void)strcpy(SavedEntryArgs,parse[1]); /* Save the arguments to the run */
  
  /* Execute with the default entry name */
  DebugExecute(EntryName,parse[1]);
  return	0;
}

static int
DebugEnter(com)
     char	*com;
{
  char		*parse[2];
  
  /* Strip off the execute command and Break the arguments into entry */
  /* name and function arguments */
  ParseList(com,parse,1);
  ParseList(parse[1],parse,1);
  (void)strcpy(LastEntry,parse[0]); /* Save the last entry for AGAIN command */
  (void)strcpy(SavedEntryArgs,parse[1]); /* Save the arguments to the run */
  
  /* Execute the entry */
  DebugExecute(parse[0],parse[1]);
  
  return	0;
}

static int
DebugFile(com)
     char	*com;
{
  char		*parse[2];
  
  ParseList(com,parse,1);
  EnterFile(parse[1]);
  
  return	0;
}

static int
DebugGarbage(com)
     char	*com;
{
  GarbageCollect();
  
  return	0;
}

static int
DebugMode(com)
     char	*com;
{
  char		*parse[2];
  
  /* split off the command arg and the keyword */
  ParseList(com,parse,1);
  ParseList(parse[1],parse,1);
  
  /* Set the appropriate flags... */
  switch ( KeywordTablePosition(ModeKeywords,parse[0]) ) {
  case 'l':			/* LINESTEP */
    LineStep = TRUE;
    NodeStep = FALSE;
    break;
  case 'n':			/* OPSTEP */
    LineStep = FALSE;
    NodeStep = FALSE;
    break;
  case 'o':			/* NORMAL */
    LineStep = FALSE;
    NodeStep = TRUE;
    break;
  default:
    DebugMessage("Invalid mode\n");
  }
  return 0;
}

static int
DebugPrint(com)
     char	*com;
{
  char		*parse[2];
  
  ParseList(com,parse,1);
  while(*parse[1]) {
    /* Print to standard in ascii */
    ParseList(parse[1],parse,1);
    PrintName(parse[0],(FileEntry*)NULL,(char*)NULL);
  }
  
  return 	0;
}

static int
DebugQuit(com)
     char	*com;		/* ARGSUSED */
{
  return	1;
}

static int
DebugSet(com)
     char	*com;
{
  char		*parse[2];
  
  /* Peal out the command name from the variable name and value */
  ParseList(com,parse,1);
  
  /* Break into the name and the value */
  ParseList(parse[1],parse,1);
  
  /* Set the value */
  SetVar(parse[0],parse[1]);
  
  return 0;
}
static int
DebugView(com)
     char	*com;
{
  char		*parse[2];
  
  /* Peal out the command name from the args... */
  ParseList(com,parse,1);
  
  ViewVars(parse[1]);
  
  return 0;
}
static int
DebugWhere(com)
     char	*com;
{
  char		*parse[2];
  int		max;
  char		UnFixedName[NameSize];
  int		count;
  Environment	*Env,*F;
  char		message[MaxCommentSize];
  NodeInfo	Mod;
  
  ParseList(com,parse,1);
  
  /* Find the max number of frames to view */
  if (*parse[1]) {
    max  = atoi(parse[1]);
  } else {
    max = 0;
  }
  
  Env = TopOfEnvironmentStack();
  for(count=0;Env && (max == 0 || count < max);count++) {
    F = GetFunctionNode(Env);
    if ( !F ) break;		/* Shouldn't happen... */
    
    UnFixupName(F->NAME,UnFixedName);
    Mod = FunctionOwner(F->INFO);
    
    (void)sprintf(message," %s %s %d\n",
                  ModuleNameOf(Mod),
                  UnFixedName,
                  Env->INFO->DebugInfo.SL);
    DebugMessage(message);
    
    Env = PrevEnvironment(F);
  }
  
  return 0;
}
/* ------------------------------------------------------------ */
static int
UnknownCommand(com)
     char	*com;
{
  char		message[100];
  
  (void)sprintf(message,"Unknown command: %s\n",com);
  DebugMessage(message);
  
  return	0;
}
/* ------------------------------------------------------------ */

void
DebugFillBuffer(PromptRequired)
     SisalBoolean PromptRequired;
{
  /* Issue a prompt if needed */
  if (PromptRequired ) {
    DebugEmitPrompt(GettingCmd);
  }

  DebugBuffer.Buf_idx = 0;
  if (!fgets(DebugBuffer.Buf_Buffer,
             sizeof(DebugBuffer.Buf_Buffer),
             (DebugBuffer.Buf_FP)?(DebugBuffer.Buf_FP):(stdin))) {
    DebugBuffer.Buf_Buffer[DebugBuffer.Buf_idx] = EOB;
  }
}

void
DebugCommand(PromptRequired)
     SisalBoolean PromptRequired;
{
  char	command[MaxCommentSize];
  int	comlen;
  char	c,prev,*expansion;
  int	Quoted;
  int	DQuoted;
  int	CommandStatus;
  int	ComPos;
  
  for(;;) {
    /* Clear out the command for the next pass */
    comlen = 0;
    command[0] = '\0';
    
    /* Break out a semi-colon terminated command */
    prev = '\0';
    DQuoted = 0;
    Quoted = 0;
    for( c=CC(PromptRequired), GettingCmd = 1;( c != EOB) && 
	(prev == '\\' || Quoted || DQuoted ||
	 (command[0] == '#') || (c != ';')); c=CC(PromptRequired) ) {
      
      /* Break on end of line for comment commands */
      if ( command[0] == '#' && c == '\n') break;
      
      /* Skip processing any newlines here */
      if ( c == '\n' ) {
	GettingCmd = ( comlen != 0 );
        NextC(PromptRequired);		/* Consume and ... */
        continue;
      }
      
      /* Don't save any preceding blanks */
      if (comlen > 0 || !isspace(c)) command[comlen++] = c;
      
      /* Check to see if we're entering a quoted section */
      if ( !Quoted && prev != '\\' && c == '"' ) DQuoted = !DQuoted;
      if ( !DQuoted && prev != '\\' && c == '\'' ) Quoted = !Quoted;
      
      NextC(PromptRequired);
      prev = c;
    }
    
    /* Quit the command loop on end-of-file */
    GettingCmd = 0;
    if (CC(PromptRequired) == EOB) break;
    
    /* Consume the semi-colon  (or newline if this was a comment command). */
    NextC(PromptRequired);
    
    /* Just ignore null commands and comment commands */
    if ( comlen == 0 || command[0] == '#' ) continue;
    
    /* Terminate the string */
    command[comlen] = '\0';
    
    ComPos = CommandTablePosition(command);
    CommandStatus = 0;
    if ( ComPos < 0 ) {
      /* Execute all the commands in the expanded alias */
      if ( (expansion = MatchAlias(command)) )
      {
	(void) strncpy( command, DebugBufferPtr(), MaxCommentSize-1 );
	EmptyDebugBuffer();
	(void) strncpy( DebugBufferPtr(), expansion, MaxCommentSize-1 );
	(void) strncat( DebugBufferPtr(), command, MaxCommentSize-1 );
      }
      else
      {
	/* Abort execution when an error is found */
	CommandStatus = UnknownCommand(command);
      }
    } else {
      CommandStatus = CommandTable[ComPos].Command(command);
    }
    
    if ( CommandStatus == 1 ) {
      DebugExit("Exiting...\n");
    } else if ( CommandStatus == 2 ) {
      break;
    }
  }
}
