/***********************************************************
 *
 *  ScTool Widget, a subclass of AsciiText Widget
 *    by Jeremy Uejio uejio@lll-crg.llnl.gov
 *
 ***********************************************************/
/*
 * Copyright 1990 the Regents of the University of California.  All
 * rights reserved.  Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without fee is
 * hereby granted, provided that this copyright notice appear in all
 * copies.  See the file copyright.h for more information.
 *
 */


/*
 * ScTool.c - Source code for ScTool Widget.
 *
 * This Widget is similar to the AsciiText widget but provides some
 * routines to call a callback when <CR> is pressed.
 *
 * Date:    March 1990
 *
 * By:      Jeremy Y. Uejio
 *          LLNL
 *          uejio@lll-crg.llnl.gov
 */

#include "copyright.h"
#include <stdio.h>
/*#include <X11/copyright.h>*/
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/TextP.h>
#include <X11/Xaw/AsciiSrcP.h>
#include <X11/Xaw/AsciiSink.h>
#include <X11/Xaw/Cardinals.h>
#include "ScToolP.h"

/* the rest are needed for psuedo ttys */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/signal.h>   /* needed for kill SIGHUP */
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/stat.h>     /* needed for IsExecutable() */
#include <sgtty.h>

#define MAXMAXLINE 255

/* private routines -- forward declarations */

static void Initialize();
static void SetEndOfPrompt();
static XtActionProc Dispatch(), sc_delete(), sc_insert();

/* public routines */

void ScWriteText();
void ScClearText();
int ScToolStart();
void ScToolKill();

/* global defines for this widget */

#define offset(field) XtOffset(ScToolWidget, scTool.field)
static XtResource resources[] = {
    /* {name, class, type, size, offset, default_type, default_addr}, */
	{ XtNcommandCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
	    offset(input_callback), XtRCallback, NULL },
	{ XtNexitCallback, XtCCallback, XtRCallback, sizeof(XtCallbackList),
	    offset(exit_callback), XtRCallback, NULL },
	{ XtNalwaysCallback, XtCBoolean, XtRInt, sizeof(int),
	    offset(always_callback), XtRImmediate, (caddr_t)0 },
	{ XtNprogram, XtCProgram, XtRString, sizeof(String),
	    offset(program), XtRString, NULL },
};
#undef offset

static XtActionsRec actions[] = {
	{"Dispatch",(XtActionProc) Dispatch},
	{"sc-delete",(XtActionProc) sc_delete},
	{"sc-insert-character",(XtActionProc) sc_insert},
	{NULL,NULL}
};

static char translations[] = 
  "<Key>Delete:        sc-delete() delete-previous-character()\n\
   <Key>Return:        end-of-file() newline() Dispatch()\n\
   <Key>:              end-of-file() insert-char() sc-insert-character()\n\
   <Btn2Down>:         end-of-file() insert-selection(PRIMARY,CUT_BUFFER0) \n";

ScToolClassRec scToolClassRec = {
  { /* core fields */
    /* superclass       */      (WidgetClass) &asciiTextClassRec,
    /* class_name       */      "ScTool",
    /* widget_size      */      sizeof(ScToolRec),
    /* class_initialize */      NULL,
    /* class_part_init  */	NULL,
    /* class_inited     */      FALSE,
    /* initialize       */      Initialize,
    /* initialize_hook  */	NULL,
    /* realize          */      XtInheritRealize,
    /* actions          */      actions,
    /* num_actions      */      XtNumber(actions),
    /* resources        */      resources,
    /* num_ resource    */      XtNumber(resources),
    /* xrm_class        */      NULLQUARK,
    /* compress_motion  */      TRUE,
    /* compress_exposure*/      XtExposeGraphicsExpose,
    /* compress_enterleave*/	TRUE,
    /* visible_interest */      FALSE,
    /* destroy          */      NULL,
    /* resize           */      XtInheritResize,
    /* expose           */      XtInheritExpose,
    /* set_values       */      NULL,
    /* set_values_hook  */	NULL,
    /* set_values_almost*/	XtInheritSetValuesAlmost,
    /* get_values_hook  */	NULL,
    /* accept_focus     */      XtInheritAcceptFocus,
    /* version          */	XtVersion,
    /* callback_private */      NULL,
    /* tm_table         */      XtInheritTranslations,
    /* query_geometry	*/	XtInheritQueryGeometry
  },
  { /* Simple fields */
    /* change_sensitive	*/	XtInheritChangeSensitive
  },
  { /* text fields */
    /* empty            */      0
  },
  { /* ascii fields */
    /* empty            */      0
  },
  { /* scTool fields */
    /* interpreter      */      NULL
  }
};

WidgetClass scToolWidgetClass = (WidgetClass)&scToolClassRec;

static void Initialize(request, new)
     Widget request;
     Widget new;
{
	ScToolWidget nc = (ScToolWidget)new;

	/* width and height are not initialized by core */
	
	if (nc->core.height == 0)
	  nc->core.height = 400;
	if (nc->core.width == 0)
	  nc->core.width =400;
	nc->scTool.do_callback = 1;
	nc->scTool.always_callback = 0;
	nc->scTool.line = NULL;
	nc->scTool.start_pos = 0;
	nc->scTool.inputId = 0;
	nc->scTool.master = 0;
	nc->scTool.program = NULL;
	nc->scTool.pid = 0;
	nc->scTool.argc = 0;
	nc->scTool.argv = NULL;
	XtOverrideTranslations(nc,XtParseTranslationTable(translations));
}

static void SetEndOfPrompt(w,pos)
     ScToolWidget w;
     XawTextPosition pos;
     /* specifies the end of the prompt.  The user can't delete past this
	point */
{
	w->scTool.start_pos = pos;
}

static XtActionProc sc_delete(w,event,params,num_params)
     ScToolWidget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
     /* do not delete before scTool.start_pos. */
{
	XawTextBlock textblock;
	XawTextPosition Pos;
	
	Pos = w->text.lastPos;
	XawTextSetInsertionPoint(w,w->text.lastPos);
	if (Pos <= w->scTool.start_pos) {
		textblock.firstPos = 0;
		textblock.length = 1;
		textblock.ptr = " ";
		XawTextReplace(w,Pos,Pos,&textblock);
		XawTextSetInsertionPoint(w,Pos);
	}
}

extern errno;
#define WRITESIZE 800
#define COUNT 5

static int Write_line(sock,msg,length)
     int sock;
     char* msg;
     int length;
     /* returns -1 if write fails else returns 0.  Trys to write
	for COUNT times before returning */
{
	char debugmsg[MAXMAXLINE],errmsg[MAXMAXLINE];
	int rval,maxwrite,count;
	if (msg) {
		count = 0;
		while ((length>0) && (count < COUNT)) {
			if (length>WRITESIZE) maxwrite=WRITESIZE;
			else maxwrite=length;
			sprintf(debugmsg,
				 "Trying to send %d chars to %d: %s\n",
				 maxwrite,sock,msg);
			rval = write(sock,msg,maxwrite);
			/* check return value, if errno is EWOULDBLOCK,
			   then no data could be written so try again, but
			   don't give an error message. */
			if ((rval < 0)&&(errno!=EWOULDBLOCK)) {
				count++;
				sprintf(errmsg,
					"cannot write to socket %d, errno %d",
					sock,errno);
			} else if ((rval < 0) && (errno==EWOULDBLOCK)) {
				count++;
				sprintf(debugmsg,
					"Write blocked on socket %d",sock);
				rval = 0;
			} else {
				count = 0;
				sprintf(debugmsg,"Sent %d chars.",rval);
			}
			length = length - rval;
			msg = msg+rval;
		}
	}
	if (count!=0) {
		return -1;
	} else {
		return 0;
	}
}

#undef COUNT

static void ExecuteLine(w)
     Widget w;
{
	ScToolWidget neww;
	String string;
	XawTextPosition length;
	neww = (ScToolWidget)w;
	if (neww->scTool.line)
	  XtFree(neww->scTool.line);
	XtVaGetValues(w,XtNstring,&string,NULL);
	length = neww->text.lastPos - neww->scTool.start_pos+1;
	if (length>0) {
		neww->scTool.line=XtMalloc(length);
		neww->scTool.line=strncpy(neww->scTool.line,
				string+neww->scTool.start_pos,(int)length-1);
		neww->scTool.line[(int)length-1]=NULL;
		SetEndOfPrompt(neww,neww->text.lastPos);
		if (neww->scTool.do_callback)
		  if (neww->scTool.program) {
			  Write_line(neww->scTool.master,
				     neww->scTool.line,(int)length-1);
		  } else {
			  XtCallCallbacks(w,XtNcommandCallback,
					  neww->scTool.line);
		  }
		neww->scTool.do_callback = 1; /* reset it so that when
						 the user presses a <CR>, 
						 the callbacks are called */
 	}
	return;
}

static XtActionProc Dispatch(w,event,params,num_params)
     Widget w;
     XEvent *event; 
     String *params; 
     Cardinal *num_params; 
{ 	
	/* note: other routines call ExecuteLine() that's why it's a
	   separate routine */
	ExecuteLine(w);
 	return;
}

static void AddCharacter(w,c)
     Widget w; 
     char c; 
{ 	
	if (c=='\n')
	  ExecuteLine(w);
}

static XtActionProc sc_insert(w,event,params,num_params) 
     Widget w;
     XEvent *event; 
     String *params;
     Cardinal *num_params; 
{ 	
	char buffer[255];
 	XLookupString(event,buffer,255,NULL,NULL);
	AddCharacter(w,buffer[0]); 
}  

/* public procedures:
   
   ScWriteText(w,msg,do_callbackflag) -- 
   writes msg to ScToolWidget.  If do_callbackflag is true, then 
   the callback will be called on every <CR> in msg.
   Therefore, do not call ScWriteText within the callback because 
   ScWriteText indirectly can call the callback.
   The current line is passed to the callback in the call_data.

   ScAlwaysCallback(w,flag) --
   if flag is True, then execute the callback on every <CR> in the tool
   output.

   ScClearText(w) -- clears the text window.

   ScToolStart(w,path,argv,argc) -- starts the tool called path, with the
   arguments argv.  Note: argv[0] is usually path.  Returns the PID if
   the tool is started successfully.
*/

void ScWriteText(w,msg,do_callbackflag)
     Widget w;
     char *msg;
     int do_callbackflag;
{
	int length,newlength,maxlength;
	ScToolWidget neww;
	char textbuffer[1025];
	XawTextBlock text;
	XawTextPosition firstpos;
	XEvent event;
	XtAppContext AppCon;
	neww = (ScToolWidget)w;
	/* AppCon = XtWidgetToApplicationContext(w);
	while (XtAppPending(AppCon)==XtIMXEvent) {
		XtAppNextEvent(AppCon,&event);
		XtDispatchEvent(&event);
	} */
	if (!neww) {
		printf ("%s",msg);
		fflush(stdout);
		return;
	}
	neww->scTool.do_callback = do_callbackflag;
	text.firstPos = 0;
	text.format = FMT8BIT;
	XawTextSetInsertionPoint(w,neww->text.lastPos);
	/* character must be added to both my structures and to the text. */
	/* if do_callback then must call callback on <CR> so need to add
	   a character at a time */
	/* if (do_callbackflag||neww->scTool.always_callback) { */
	if (do_callbackflag) {
		text.length = 1;
		while (*msg) {
			/* note: must add msg to the textbuffer first
			   so that the <CR> is passed thru to AddCharacter */
			text.ptr = msg;
			firstpos = XawTextGetInsertionPoint(w);
			XawTextReplace(w,firstpos,firstpos,&text);
			XawTextSetInsertionPoint(w,firstpos+1);
			AddCharacter(w,*msg);
			msg++;
		}
	}
	/* else add the msg in 1500 character parts */
	else { 
		firstpos = XawTextGetInsertionPoint(w);
		length = strlen(msg);
		if (length>1500) {  /* if long msg then move to top first
				       before replacing text */
			XtVaSetValues(w,XtNdisplayPosition,firstpos,NULL);
		}
		maxlength = length;
		text.ptr = textbuffer;
		while (length>0) {
			newlength = 50;
			if (newlength>length) newlength = length;
			strncpy(textbuffer,msg+maxlength-length,newlength);
			textbuffer[newlength]=NULL;
			text.length=newlength;
			XawTextReplace(w,firstpos,firstpos,&text);
			length = length - newlength;
			firstpos+=newlength;
			XawTextSetInsertionPoint(w,firstpos);
		}
		SetEndOfPrompt(neww,XawTextGetInsertionPoint(w));
	}
	neww->scTool.do_callback = 1;
	/* empty the event buffer so that can trap user commands while
	   this widget is printing--actually don't need to do this because
	   XtMainLoop should poll? */
	/* while (XtAppPending(AppCon)==XtIMXEvent) {
		XtAppNextEvent(AppCon,&event);
		XtDispatchEvent(&event);
	} */
	return;
}

void ScClearText(w)
     Widget w;
{
	/* this routine might be applied to a normal asciiText widget
	   so check if it really is a tool widget */
	ScToolWidget neww;
	String dummy;
	XtVaSetValues(w,XtNstring,"",NULL);
	/* use XtGetValues to show the update */
	XtVaGetValues(w,XtNstring,&dummy,NULL);
	if (XtClass(w)==scToolWidgetClass) {
		neww = (ScToolWidget)w;
		neww->scTool.do_callback = 1;
		neww->scTool.line = NULL;
		neww->scTool.start_pos = 0;
	}
}

void ScAlwaysCallback(w,flag)
     Widget w;
     int flag;
{
	ScToolWidget neww;
	neww = (ScToolWidget)w;
	neww->scTool.always_callback = flag;
	/* reset beginning of line if setting always_callback to false
	if (!flag) {
		SetEndOfPrompt(neww,XawTextGetInsertionPoint(w));
	} */
}

static char pty[] = "/dev/pty??";
static char tty[] = "/dev/tty??";

static int openMaster()
{
    int  i, master,slave;
    char c;
    
    /* need to try to open both the master and the slave just in case
       someone is hanging on to one and not the other. */

    for (c='p'; c<='r'; c++) {
	pty[8] = c;
	for (i=0; i<=15; i++) {
	    pty[9] = "0123456789abcdef"[i];
	    tty[8] = pty[8];
	    tty[9] = pty[9];
	    if (((master = open(pty, O_RDWR)) != -1) && 
		(access(tty, R_OK|W_OK) == 0) &&
		((slave = open(tty, O_RDWR)) != -1)) {
		    close(slave);
		    return (master); 
	    }
	}
    }
    fprintf(stderr, "all ptys in use\n");
    return -1;  /* so lint doesn't complain */
}

static int openSlave(master)
{
    int slave;

    tty[8] = pty[8];
    tty[9] = pty[9];
    if ((slave = open(tty, O_RDWR)) != -1) {
	    return (slave);
    }
    else {
	close (master);
	fprintf(stderr, "open: cannot open slave pty %s", tty);
    }
    return -1; /* so lint doesn't complain */
}

#define XMASK	(0111)
/* Little function to check executable bits in file status */
static int
IsExecutable(path)
  char	*path;
{
  struct stat	FileStatus;

  if (stat(path,&FileStatus)) {
    return 0;
  } else {
    return (FileStatus.st_mode & XMASK) != 0;
  }
}
#undef XMASK

char *getenv();
/*
** GetExe(name,path)
**
** This procedure searches for the pathname of the executable in the
** name itself and then in the user's path
*/
static int
GetExe(name,path)
     char	*name;
     char	*path;
{
  int	i;
  char	*UserPath;
  char	*p;

  strcpy(path,name);
  if (IsExecutable(name)) { 
	  return True;
  }
  /* Try each directory in the user path */
  UserPath = getenv("PATH");
  if (UserPath) {
	  p = UserPath;
	  while(*p) {
		  for(i=0;*p && *p != ':';p++) path[i++] = *p;
		  path[i++] = '/';
		  strcpy(path+i,name);
		  if (IsExecutable(path)) { 
			  return True;
		  }
		  if (*p) p++;		/* Skip the colon */
	  }
  }
  
  /* Have to assume its just the default name */
  strcpy(path,name);
  return False;
}

static int PsuedoTtyFork(w,master,progname,numsimpleargs,simpleargs)
     Widget w;
     int *master;
     char* progname;
     int numsimpleargs;
     char** simpleargs;
     /* forks the program called progname and returns the master socket
	in master for a psuedo tty pair to the program and the pid of
	the program. I use vfork because it's faster and takes less
	memory.  However, it is dangerous so be careful if you modify
	this code. 
	This routine uses GetExe to check if the pathname is in the
	current path and is executable.  The pathname is modified so 
	that is contains the full path. */
{
	char errmsg[255],debugmsg[255];
	char path[255];
	int f,pid,slave,l,lb,devtty;
	struct sgttyb b;
	struct tchars tc;
	struct ltchars lc;
	struct winsize win;
	if (!GetExe(progname,path)) {
		sprintf(errmsg,"Cannot call %s\n",path);
		return 0;
	}
	devtty = open("/dev/tty",O_RDWR);
	if (devtty < 0) {
		ScWriteText(w,"Cannot open controlling tty",0);
		return 0;
	}
	*master = openMaster();
	if (*master<0) {
		return -1;
	}
	ioctl(devtty,TIOCGETP,&b);
	ioctl(devtty,TIOCGETC,&tc);
	ioctl(devtty,TIOCGLTC,&lc);
	ioctl(devtty,TIOCGETD,&l);
	ioctl(devtty,TIOCLGET,&lb);
	ioctl(devtty,TIOCGWINSZ,&win);
	close(devtty);
	pid = vfork();
	if (pid < 0) {
		ScWriteText(w,"Cannot fork process",0);
		return 0;
	}
	else if (pid) {
		sprintf(debugmsg,"forked PID=%d\n",pid);
		return pid;
	}
	else {
		/* child so execve the progname with the args */
		/* first close controlling tty */
		f=open("/dev/tty",O_RDWR);
		if (f >= 0) {
			ioctl(f,TIOCNOTTY,0);
			close(f);
		}
		slave = openSlave(*master);
		if (slave<0) {
			_exit(1);
		}
		ioctl(slave,TIOCSETC,&tc);
		ioctl(slave,TIOCSLTC,&lc);
		ioctl(slave,TIOCSETD,&l);
		ioctl(slave,TIOCLSET,&lb);
		ioctl(slave,TIOCSWINSZ,&win);
		b.sg_flags = ANYP|XTABS;
		ioctl(slave,TIOCSETP, &b);
		simpleargs[0] = path;
		simpleargs[numsimpleargs] = NULL;
		pid = getpid();
		ioctl(slave,TIOCSPGRP,&pid);
		setpgrp(0,pid);
		dup2(slave,0);
		dup2(slave,1);
		dup2(slave,2);
		close(slave);
		execv(path,simpleargs);
		sprintf(errmsg,"Cannot call %s\n",path);
		_exit(1); /* call _exit instead of exit because cleanup
			     is redundant and messes up vfork */
	}
	return 0;
}

static XtInputCallbackProc WriteToWindow(client_data,sock,id)
     caddr_t client_data;
     int *sock;
     XtInputId *id;
     /* callback for input from tool.  Reads data from sock and writes data
	to the client_data widget using ScWriteText.  
	If there is an error reading from the socket or and eof is read, 
	then the tool is killed	and the input is removed. */
{
	ScToolWidget w;
	char errmsg[MAXMAXLINE],debugmsg[MAXMAXLINE];
	char msg[MAXMAXLINE];
	int numchar;
	w = (ScToolWidget)client_data;
	if ((numchar=read(*sock,msg,MAXMAXLINE-1)) < 0) {
		/* error reading from socket */
		sleep(1);  /* kind of a wait until tool is really dead */
		if (((numchar = read(*sock,msg,MAXMAXLINE-1)) < 0)) {
			/* sprintf(debugmsg,"about to kill tool, %s\n",
				w->scTool.program); */
		        ScToolKill((Widget)w);
		} else if (numchar == 0) {
			/* sprintf(debugmsg,
			   "End of file reached on %d\n",*sock);
			sprintf(debugmsg,"about to kill tool, %s\n",
				w->scTool.program); */
		        ScToolKill((Widget)w);
		} else if (numchar > 0) {
			/* sprintf(debugmsg,
			   "Glitch in reading from %d\n",*sock);
			sprintf(errmsg,"%sData might be lost.\n",debugmsg); */
		} else {
			/* sprintf(errmsg,
				"error reading on socket %d, errno = %d\n",
				*sock,errno); */
			;
		}
	} else if (numchar == 0) {
		/* sprintf(debugmsg,"End of file reached on %d\n",*sock); */
		;
	} else {
		msg[numchar] = NULL;
		/* sprintf(debugmsg,
		   "Msg(%d) from %d: %s\n",numchar,*sock,msg); */
		ScWriteText((Widget)w,msg,0);
		/* note:  must write first, because the callbacks may
		   destroy this widget. */
		/* ALWAYS call the callback if always_callback is set */
		if (w->scTool.always_callback) {
			XtCallCallbacks(w,XtNcommandCallback,msg);
		}
	}
	return;
}

#undef MAXMAXLINE
static char* ParseHostname(pathname)
     char **pathname;
     /* if pathname is of the form hostname:path then return hostname
	and set pathname to path. */
{
	char *colonPosition,*begin;
	if (colonPosition = index(*pathname,':')) {
		begin = *pathname;
		*pathname = &colonPosition[1];
		colonPosition[0] = NULL;
		return begin;
	} else {
		return NULL;
	}
}

int ScToolStart(w,path,argc,argv)
     Widget w;
     char* path;
     int argc;
     char** argv;
     /* starts the program path with the arguments argc and argv by
	vforking and exec'ing. Argv[0] is the same as path.
	Returns the pid of the started tool. */
{
	ScToolWidget new=(ScToolWidget)w; 
	int hostargc,i;
	char **hostargv;
	char* hostname;
	char msg[255];
	/* if a program is already running, then print a message, and 
	   abort */
	if (new->scTool.program) {
		sprintf(msg,"The program, %s, is already running.\n",
			new->scTool.program);
		ScWriteText(w,msg,0);
		ScWriteText(w,"Only one program can be running in this ",0);
		ScWriteText(w,"window at one time.\n",0);
		return 0;
	}
	/* if the pathname is of the form hostname:pathname, then create
	   some new arguments and use rsh as the program to fork */
	if (hostname=ParseHostname(&path)) {
		hostargc = argc+2;
		/* add 1 because need to null terminate hostargv */
		hostargv = (char**)XtMalloc((hostargc+1)*sizeof(char*));
		hostargv[0] = "rsh";
		hostargv[1] = hostname;
		hostargv[2] = path;
		/* ParseHostname changes the pathname so must not take
		   it from argv[0] */
		for (i=1; i<argc; i++)
		  hostargv[i+2]=argv[i];
		hostargv[hostargc] = NULL;
		new->scTool.pid = PsuedoTtyFork(w,&new->scTool.master,
						hostargv[0],hostargc,hostargv);
	} else {
		new->scTool.pid = PsuedoTtyFork(w,&new->scTool.master,
						path,argc,argv);
	}
	if (new->scTool.pid > 0) {
		new->scTool.program = XtNewString(path);
		new->scTool.inputId =
		  XtAppAddInput(XtWidgetToApplicationContext(new),
				new->scTool.master,XtInputReadMask,
				WriteToWindow,new);
		/* only copy the arguments if the tool was successfully
		   executed. */
		if (new->scTool.argv) {
			for (i=0; i<new->scTool.argc; i++)
			  XtFree(new->scTool.argv[i]);
			XtFree(new->scTool.argv);
		}
		new->scTool.argc = argc;
		new->scTool.argv =
		  (char**)XtMalloc((new->scTool.argc+1)*sizeof(char*));
		for (i=0; i<argc; i++) {
			new->scTool.argv[i] = XtNewString(argv[i]);
		}
		new->scTool.argc = argc;
	}
	return new->scTool.pid;
}

char** ScToolArgv(w)
     Widget w;
     /* returns a newly malloc'ed char** list of arguments to the 
	program that is running */
{
	ScToolWidget neww;
	char **argv;
	int i;
	neww = (ScToolWidget)w;
	if (neww->scTool.argc <= 0)
	  return NULL;
	argv = (char**)XtMalloc(neww->scTool.argc*sizeof(char*));
	for (i=0; i < neww->scTool.argc; i++)
	  argv[i] = XtNewString(neww->scTool.argv[i]);
	argv[neww->scTool.argc] = NULL;
	return argv;
}

void ScToolKill(w)
     Widget w;
     /* kills the running tool */
{
	ScToolWidget neww;
	int i;
	char line[255];
	neww = (ScToolWidget)w;
	if (neww->scTool.program) {
		XtRemoveInput(neww->scTool.inputId);
		kill(neww->scTool.pid,SIGHUP);
		for (i=0; i<neww->scTool.argc; i++)
		  XtFree(neww->scTool.argv[i]);
		neww->scTool.argc = 0;
	}
	sprintf(line,"\nThe program %s, has terminated.\n",
		neww->scTool.program);
	neww->scTool.program=NULL;
	ScWriteText(neww,line,0);
	/* call the exit callback last so that get values on 
	   XtNprogram in the callback will return NULL */
	if (neww->scTool.always_callback)
	  XtCallCallbacks(w,XtNexitCallback,XtNewString(line));
}

char* ScToolProgramName(w)
     Widget w;
     /* returns the name of the tool program or NULL if no tool is running */
{
	ScToolWidget neww;
	neww = (ScToolWidget)w;
	return neww->scTool.program;
}
