/* 
 * tkCanvXYGraph.c --
 *
 *	This file implements linegraph items for canvas
 *	widgets.
 *
 * Copyright (c) 1991-1994 The Regents of the University of California.
 * Copyright (c) 1994-1996 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkGraph.c 1.39 96/03/02 17:28:06
 */

#include <stdio.h>
#include <tk.h>
#include <tkInt.h>
#include <tkPort.h>
#include <tkCanvas.h>
#include <tkCanvXYGraph.h>
#include <tkCanvDataset.h>

#ifdef USE_DMALLOC
#include<dmalloc.h>
#endif

/*
 * The structure below defines the record for each xygraph item.
 */

/*
 * Information used for parsing configuration specs:
 */
int Tk_TickcoordsParseProc _ANSI_ARGS_((ClientData clientData,
					Tcl_Interp *interp,
					Tk_Window tkwin,
					char *value,
					char *widgRec,
					int offset));
char *Tk_TickinfoPrintProc _ANSI_ARGS_((ClientData clientData,
					Tk_Window tkwin,
					char *widgRec,
					int offset,
					Tcl_FreeProc **freeProcPtr));
char *Tk_NsetsPrintProc _ANSI_ARGS_((ClientData clientData,
				    Tk_Window tkwin,
				    char *widgRec,
				    int offset,
				    Tcl_FreeProc **freeProcPtr));

int Tk_AxislineParseProc _ANSI_ARGS_((ClientData clientData,
				    Tcl_Interp *interp,
				    Tk_Window tkwin,
				    char *value,
				    char *widgRec,
				    int offset));
char *Tk_AxislinePrintProc _ANSI_ARGS_((ClientData clientData,
				    Tk_Window tkwin,
				    char *widgRec,
				    int offset,
				    Tcl_FreeProc **freeProcPtr));

int Tk_TickpositionParseProc _ANSI_ARGS_((ClientData clientData,
				    Tcl_Interp *interp,
				    Tk_Window tkwin,
				    char *value,
				    char *widgRec,
				    int offset));
char *Tk_TickpositionPrintProc _ANSI_ARGS_((ClientData clientData,
				    Tk_Window tkwin,
				    char *widgRec,
				    int offset,
				    Tcl_FreeProc **freeProcPtr));

int Tk_TickorientParseProc _ANSI_ARGS_((ClientData clientData,
				    Tcl_Interp *interp,
				    Tk_Window tkwin,
				    char *value,
				    char *widgRec,
				    int offset));
char *Tk_TickorientPrintProc _ANSI_ARGS_((ClientData clientData,
				    Tk_Window tkwin,
				    char *widgRec,
				    int offset,
				    Tcl_FreeProc **freeProcPtr));

static Tk_CustomOption stateOption = {
    (Tk_OptionParseProc *) Tk_StateParseProc,
    Tk_StatePrintProc, (ClientData) 2
};
static Tk_CustomOption tagsOption = {
    (Tk_OptionParseProc *) Tk_CanvasTagsParseProc,
    Tk_CanvasTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption dashOption = {
    (Tk_OptionParseProc *) Tk_CanvasDashParseProc,
    Tk_CanvasDashPrintProc, (ClientData) NULL
};
static Tk_CustomOption tileOption = {
    (Tk_OptionParseProc *) Tk_TileParseProc,
    Tk_TilePrintProc, (ClientData) NULL
};
static Tk_CustomOption tickcoordsOption = {
    (Tk_OptionParseProc *) Tk_TickcoordsParseProc,
    Tk_TickinfoPrintProc, (ClientData) NULL
};
static Tk_CustomOption nsetsOption = {
    (Tk_OptionParseProc *) Tk_TickcoordsParseProc,
    Tk_NsetsPrintProc, (ClientData) NULL
};
static Tk_CustomOption axislineOption = {
    (Tk_OptionParseProc *) Tk_AxislineParseProc,
    Tk_AxislinePrintProc, (ClientData) NULL
};
static Tk_CustomOption tickpositionOption = {
    (Tk_OptionParseProc *) Tk_TickpositionParseProc,
    Tk_TickpositionPrintProc, (ClientData) NULL
};
static Tk_CustomOption tickorientOption = {
    (Tk_OptionParseProc *) Tk_TickorientParseProc,
    Tk_TickorientPrintProc, (ClientData) NULL
};
static Tk_CustomOption pixelOption = {
    (Tk_OptionParseProc *) Tk_PixelParseProc,
    Tk_PixelPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-activedash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.activeDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-activefill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, activeFillColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-activeoutline", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.activeColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-activeoutlinestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.activeStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-activeoutlinetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.activeTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_BITMAP, "-activestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, activeFillStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-activetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, activeFillTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-activewidth", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(GraphItem, outline.activeWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_CUSTOM, "-dash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.dash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_PIXELS, "-dashoffset", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(GraphItem, outline.offset),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-disableddash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.disabledDash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_COLOR, "-disabledfill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, disabledFillColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-disabledoutline", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.disabledColor),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-disabledoutlinestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.disabledStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-disabledoutlinetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.disabledTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_BITMAP, "-disabledstipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, disabledFillStipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-disabledtile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, disabledFillTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-disabledwidth", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(GraphItem, outline.disabledWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_COLOR, "-fill", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, fillColor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-outline", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(GraphItem, outline.color), TK_CONFIG_NULL_OK},
    {TK_CONFIG_BITMAP, "-outlinestipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.stipple),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-outlinetile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, outline.tile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_CUSTOM, "-state", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, state),TK_CONFIG_NULL_OK,
	&stateOption},
    {TK_CONFIG_BITMAP, "-stipple", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, fillStipple), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_CUSTOM, "-tile", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, fillTile),
	TK_CONFIG_NULL_OK, &tileOption},
    {TK_CONFIG_STRING, "-updatecommand", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(Tk_Item, updateCmd), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-xmajortickinterval", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(GraphItem, xmajortickinterval),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-xminortickinterval", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(GraphItem, xminortickinterval),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-ymajortickinterval", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(GraphItem, ymajortickinterval),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-yminortickinterval", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(GraphItem, yminortickinterval),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_PIXELS, "-xmajorticksize", (char *) NULL, (char *) NULL,
	"8", Tk_Offset(GraphItem, xmajorticksize),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-xminorticksize", (char *) NULL, (char *) NULL,
	"4", Tk_Offset(GraphItem, xminorticksize),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-ymajorticksize", (char *) NULL, (char *) NULL,
	"8", Tk_Offset(GraphItem, ymajorticksize),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_PIXELS, "-yminorticksize", (char *) NULL, (char *) NULL,
	"4", Tk_Offset(GraphItem, yminorticksize),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-width", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(GraphItem, outline.width),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_DOUBLE, "-xtickorigin", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(GraphItem, xtickorigin),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-ytickorigin", (char *) NULL, (char *) NULL,
	"0", Tk_Offset(GraphItem, ytickorigin),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-xaxismin", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(GraphItem, xaxismin),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_DOUBLE, "-yaxismin", (char *) NULL, (char *) NULL,
	"0.0", Tk_Offset(GraphItem, yaxismin),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_DOUBLE, "-xaxismax", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(GraphItem, xaxismax),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_DOUBLE, "-yaxismax", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(GraphItem, yaxismax),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-xmajortickinfo", (char *) NULL, (char *) NULL,
	(char *) NULL, 'X', TK_CONFIG_DONT_SET_DEFAULT,
	&tickcoordsOption},
    {TK_CONFIG_CUSTOM, "-ymajortickinfo", (char *) NULL, (char *) NULL,
	(char *) NULL, 'Y', TK_CONFIG_DONT_SET_DEFAULT,
	&tickcoordsOption},
    {TK_CONFIG_CUSTOM, "-xminortickinfo", (char *) NULL, (char *) NULL,
	(char *) NULL, 'x', TK_CONFIG_DONT_SET_DEFAULT,
	&tickcoordsOption},
    {TK_CONFIG_CUSTOM, "-yminortickinfo", (char *) NULL, (char *) NULL,
	(char *) NULL, 'y', TK_CONFIG_DONT_SET_DEFAULT,
	&tickcoordsOption},
    {TK_CONFIG_CUSTOM, "-nsets", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_DONT_SET_DEFAULT,
	&nsetsOption},
    {TK_CONFIG_STRING, "-xtickformat", (char *) NULL, (char *) NULL,
	"", Tk_Offset(GraphItem, xtickformat), TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-ytickformat", (char *) NULL, (char *) NULL,
	"", Tk_Offset(GraphItem, ytickformat), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tickposition", (char *) NULL, (char *) NULL,
	(char *) NULL, 'x', TK_CONFIG_DONT_SET_DEFAULT,
	&tickpositionOption},
    {TK_CONFIG_CUSTOM, "-tickorientation", (char *) NULL, (char *) NULL,
	(char *) NULL, 'x', TK_CONFIG_DONT_SET_DEFAULT,
	&tickorientOption},
    {TK_CONFIG_CUSTOM, "-xaxisline", (char *) NULL, (char *) NULL,
	(char *) NULL, 'x', TK_CONFIG_DONT_SET_DEFAULT,
	&axislineOption},
    {TK_CONFIG_CUSTOM, "-yaxisline", (char *) NULL, (char *) NULL,
	(char *) NULL, 'y', TK_CONFIG_DONT_SET_DEFAULT,
	&axislineOption},
    {TK_CONFIG_COLOR, "-axislinecolor", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(GraphItem, axisline_outline.color),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-axislinewidth", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(GraphItem, axisline_outline.width),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_CUSTOM, "-tickwidth", (char *) NULL, (char *) NULL,
	"1.0", Tk_Offset(GraphItem, tick_outline.width),
	TK_CONFIG_DONT_SET_DEFAULT, &pixelOption},
    {TK_CONFIG_COLOR, "-tickcolor", (char *) NULL, (char *) NULL,
	"black", Tk_Offset(GraphItem, tick_outline.color),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-axislinedash", (char *) NULL, (char *) NULL,
	(char *) NULL, Tk_Offset(GraphItem, axisline_outline.dash),
	TK_CONFIG_NULL_OK, &dashOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */

static void		ComputeGraphBbox _ANSI_ARGS_((Tk_Canvas canvas,
			    GraphItem *graphPtr));
static int		ConfigureGraph _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateGraph _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, struct Tk_Item *itemPtr,
			    int argc, char **argv));
static void		DeleteGraph _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display));
static void		DisplayGraph _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display, Drawable dst,
			    int x, int y, int width, int height));
static int		GraphCoords _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    char **argv));
static int		GraphToPostscript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int prepass));
static int		GraphToArea _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *areaPtr));
static double		GraphToPoint _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *pointPtr));
#define ScaleGraph TkScaleGraph
void			ScaleGraph _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
#define TranslateGraph TkTranslateGraph
void			TranslateGraph _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double deltaX, double deltaY));
static double *		ComputeGraphTicks _ANSI_ARGS_((Tk_Item *itemPtr,
		    	    char axis, int *count));
static int		GraphUpdateGC _ANSI_ARGS_((Tk_Window tkwin,
			    GraphItem *graphPtr));
static int		GraphUpdateDrawable _ANSI_ARGS_((Tk_Window tkwin,
			    GraphItem *graphPtr));
static int		GraphCreateDrawable _ANSI_ARGS_((Tk_Window tkwin,
			    GraphItem *graphPtr));
/*
static int		GraphAccept _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Visitor *visitorPtr,
			    Tk_Item *itemPtr));
*/

/*
 * The structures below defines the xygraph item type
 * by means of procedures that can be invoked by generic item code.
 */

Tk_ItemType tkGraphType = {
    "lgraph",				/* name */
    sizeof(GraphItem),			/* itemSize */
    CreateGraph,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureGraph,			/* configureProc */
    GraphCoords,			/* coordProc */
    DeleteGraph,			/* deleteProc */
    DisplayGraph,			/* displayProc */
    TK_ITEM_VISITOR_SUPPORT,		/* flags */
    GraphToPoint,			/* pointProc */
    GraphToArea,			/* areaProc */
    GraphToPostscript,			/* postscriptProc */
    ScaleGraph,				/* scaleProc */
    TranslateGraph,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL,		/* nextPtr */
    (Tk_ItemBboxProc *) ComputeGraphBbox,/* bboxProc */
    Tk_Offset(Tk_VisitorType, visitLine), /* acceptProc */
    (Tk_ItemGetCoordProc *) NULL,	/* getCoordPtr */
    (Tk_ItemSetCoordProc *) NULL	/* setCoordPtr */
};

/*
 *--------------------------------------------------------------
 *
 * CreateGraph --
 *
 *	This procedure is invoked to create a new xygraph
 *	item in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new xygraph item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateGraph(Tcl_Interp *interp, Tk_Canvas canvas, Tk_Item *itemPtr, int argc, char **argv)
{
    GraphItem *graphPtr = (GraphItem *) itemPtr;
    Tcl_CmdInfo info;
    int i, j;

    if ((argc==1) || ((argc>1) && (argv[1][0] == '-')
	    && (argv[1][1] >= 'a') && (argv[1][1] <= 'z'))) {
	i = 1;
    } else {
	i = 4;
    }

    if (argc < i) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tk_PathName(Tk_CanvasTkwin(canvas)), " create ",
		argv[0], " x1 y1 x2 y2 ?options?\"",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Carry out initialization that is needed in order to clean
     * up after errors during the the remainder of this procedure.
     */

    Tk_CreateOutline(&(graphPtr->outline));
    Tk_CreateOutline(&(graphPtr->tick_outline));
    Tk_CreateOutline(&(graphPtr->axisline_outline));
    graphPtr->fillTile = NULL;
    graphPtr->activeFillTile = NULL;
    graphPtr->disabledFillTile = NULL;
    graphPtr->fillColor = NULL;
    graphPtr->activeFillColor = NULL;
    graphPtr->disabledFillColor = NULL;
    graphPtr->fillStipple = None;
    graphPtr->activeFillStipple = None;
    graphPtr->disabledFillStipple = None;
    graphPtr->fillGC = None;
    graphPtr->drawableGC = None;
    graphPtr->drawable = None;
    graphPtr->clipMask = None;
    graphPtr->configCmd = (char *) NULL;
    graphPtr->resize_id = 1;
    graphPtr->xaxismin = 0;
    graphPtr->yaxismin = 0;
    graphPtr->xaxismax = 1;
    graphPtr->yaxismax = 1;
    graphPtr->xtickorigin = 0.0;
    graphPtr->ytickorigin = 0.0;
    graphPtr->xmajortickinterval = 0.0;
    graphPtr->ymajortickinterval = 0.0;
    graphPtr->xminortickinterval = 0.0;
    graphPtr->yminortickinterval = 0.0;
    graphPtr->xaxislines = (double *)ckalloc(sizeof(double));
    graphPtr->yaxislines = (double *)ckalloc(sizeof(double));
    graphPtr->xaxislines[0] = 0.0;
    graphPtr->yaxislines[0] = 0.0;
    graphPtr->nxaxislines = 1;
    graphPtr->nyaxislines = 1;
    graphPtr->nchildren = 0;
    graphPtr->xtickformat = (char *)NULL;
    graphPtr->ytickformat = (char *)NULL;
    graphPtr->tickprops = WEST_TICK_IN|SOUTH_TICK_IN;
    graphPtr->child_arr_size = 4;
    graphPtr->children = (Tk_Item **)ckalloc(sizeof(Tk_Item *)*graphPtr->child_arr_size);
    for(j = 0; j < graphPtr->child_arr_size; j++)
	graphPtr->children[j] = (Tk_Item *)NULL;

    /*
     * Process the arguments to fill in the item record.
     */

    if ((GraphCoords(interp, canvas, itemPtr, i, argv) != TCL_OK)) {
	goto error;
    }
    if (ConfigureGraph(interp, canvas, itemPtr, argc-i, argv+i, 0)
	    == TCL_OK) {
	return TCL_OK;
    }

    error:
    DeleteGraph(canvas, itemPtr, Tk_Display(Tk_CanvasTkwin(canvas)));
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * GraphCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on xygraph.  See the user documentation
 *	for details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
GraphCoords(Tcl_Interp *interp, Tk_Canvas canvas, Tk_Item *itemPtr, int argc, char **argv)
{
    GraphItem *graphPtr = (GraphItem *) itemPtr;
    char c0[TCL_DOUBLE_SPACE], c1[TCL_DOUBLE_SPACE];
    char c2[TCL_DOUBLE_SPACE], c3[TCL_DOUBLE_SPACE];

    Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1, itemPtr->x2, itemPtr->y2);

    if (argc == 0) {
	Tcl_PrintDouble(interp, graphPtr->bbox[0], c0);
	Tcl_PrintDouble(interp, graphPtr->bbox[1], c1);
	Tcl_PrintDouble(interp, graphPtr->bbox[2], c2);
	Tcl_PrintDouble(interp, graphPtr->bbox[3], c3);
	Tcl_AppendResult(interp, c0, " ", c1, " ", c2, " ", c3,
		(char *) NULL);
    } else if ((argc == 1)||(argc == 4)) {
	char **largv = argv; 	if (argc==1) {
	    if (Tcl_SplitList(interp, argv[0], &argc, &largv) != TCL_OK) {
		if (largv != NULL) {
		    ckfree((char *) largv);
		}
		return TCL_ERROR;
	    } else if (argc != 4) {
		sprintf(c0,"%d",argc);
		Tcl_AppendResult(interp, "wrong # coordinates: expected 4, got ",
		c0, (char *) NULL);
		return TCL_ERROR;
	    }
	}
	if ((Tk_CanvasGetCoord(interp, canvas, largv[0],
 		    &graphPtr->bbox[0]) != TCL_OK)
		|| (Tk_CanvasGetCoord(interp, canvas, largv[1],
		    &graphPtr->bbox[1]) != TCL_OK)
		|| (Tk_CanvasGetCoord(interp, canvas, largv[2],
			&graphPtr->bbox[2]) != TCL_OK)
		|| (Tk_CanvasGetCoord(interp, canvas, largv[3],
			&graphPtr->bbox[3]) != TCL_OK)) {
	    if (largv!=argv) ckfree((char *) largv);
	    return TCL_ERROR;
	}
	if (largv!=argv) ckfree((char *) largv);
	ComputeGraphBbox(canvas, graphPtr);

	GraphCreateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
	GraphUpdateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
	GraphUpdateGC(Tk_CanvasTkwin(canvas), graphPtr);
    } else {
	sprintf(interp->result,
		"wrong # coordinates: expected 0 or 4, got %d",
		argc);
	return TCL_ERROR;
    }

    Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1, itemPtr->x2, itemPtr->y2);

    graphPtr->resize_id = (graphPtr->resize_id + 1) % 256;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigureGraph --
 *
 *	This procedure is invoked to configure various aspects
 *	of a xygraph item, such as its border and
 *	background colors.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Configuration information, such as colors and stipple
 *	patterns, may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureGraph(Tcl_Interp *interp, Tk_Canvas canvas, Tk_Item *itemPtr, int argc, char **argv, int flags)
{
    GraphItem	*graphPtr = (GraphItem *) itemPtr;
    XGCValues	gcValues;
    Tk_Item	*setPtr = (Tk_Item *)NULL;
    GC		newGC;
    unsigned long mask;
    Tk_Window	tkwin;
    Tk_Tile	tile;
    XColor	*color;
    Pixmap	stipple;
    Tk_State	state;
    TkCanvas	*canvasPtr = (TkCanvas *)canvas;
    Tcl_DString	configCmd;
    Pixmap	pixmap;
    int		i;

    /*
     *	cerr << "ConfigureGraph()\n";
     */
    tkwin = Tk_CanvasTkwin(canvas);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) graphPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    Tcl_DStringInit(&configCmd);
    if (graphPtr->configCmd) {
	Tcl_DStringAppend(&configCmd,graphPtr->configCmd,-1);
	Tcl_DStringAppend(&configCmd,"\n",1);
    }

    state = itemPtr->state;

    /*
     * A few of the options require additional processing, such as
     * graphics contexts.
     */

    if (graphPtr->outline.activeWidth > graphPtr->outline.width ||
	    graphPtr->outline.activeDash.number > 0 ||
	    graphPtr->outline.activeTile != NULL ||
	    graphPtr->outline.activeColor != NULL ||
	    graphPtr->outline.activeStipple != None ||
	    graphPtr->activeFillTile != NULL ||
	    graphPtr->activeFillColor != NULL ||
	    graphPtr->activeFillStipple != None) {
	itemPtr->redraw_flags |= TK_ITEM_STATE_DEPENDANT;
    } else {
	itemPtr->redraw_flags &= ~TK_ITEM_STATE_DEPENDANT;
    }

    mask = Tk_ConfigOutlineGC(&gcValues, canvas, itemPtr,
	     &(graphPtr->outline));
    if (mask) {
	gcValues.cap_style = CapProjecting;
	mask |= GCCapStyle;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    } else {
	newGC = None;
    }
    if (graphPtr->outline.gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), graphPtr->outline.gc);
    }
    graphPtr->outline.gc = newGC;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    tile = graphPtr->fillTile;
    color = graphPtr->fillColor;
    stipple = graphPtr->fillStipple;
    if (state==TK_STATE_HIDDEN) {
	ComputeGraphBbox(canvas, graphPtr);
	return TCL_OK;
    }
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (graphPtr->activeFillTile!=NULL) {
	    tile = graphPtr->activeFillTile;
	}
	if (graphPtr->activeFillColor!=NULL) {
	    color = graphPtr->activeFillColor;
	}
	if (graphPtr->activeFillStipple!=None) {
	    stipple = graphPtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->disabledFillTile!=NULL) {
	    tile = graphPtr->disabledFillTile;
	}
	if (graphPtr->disabledFillColor!=NULL) {
	    color = graphPtr->disabledFillColor;
	}
	if (graphPtr->disabledFillStipple!=None) {
	    stipple = graphPtr->disabledFillStipple;
	}
    }

    Tk_SetTileCanvasItem(graphPtr->fillTile, canvas, (Tk_Item *) NULL);
    Tk_SetTileCanvasItem(graphPtr->activeFillTile, canvas, (Tk_Item *) NULL);
    Tk_SetTileCanvasItem(graphPtr->disabledFillTile, canvas, (Tk_Item *) NULL);
    Tk_SetTileCanvasItem(tile, canvas, itemPtr);

    if ((pixmap = Tk_PixmapOfTile(tile)) != None) {
	gcValues.fill_style = FillTiled;
	gcValues.tile = pixmap;
	newGC = Tk_GetGC(tkwin, GCTile|GCFillStyle, &gcValues);
    } else if (color == NULL) {
	newGC = None;
    } else {
	gcValues.foreground = color->pixel;
	if (stipple != None) {
	    gcValues.stipple = stipple;
	    gcValues.fill_style = FillStippled;
	    mask = GCForeground|GCStipple|GCFillStyle;
	} else {
	    mask = GCForeground;
	}
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    }
    if (graphPtr->fillGC != None) {
	Tk_FreeGC(Tk_Display(tkwin), graphPtr->fillGC);
    }
    graphPtr->fillGC = newGC;

    if (graphPtr->drawableGC == None) {
	mask = GCFunction|GCForeground|GCBackground;
	gcValues.function = GXcopy;

	graphPtr->drawableGC = Tk_GetGC(tkwin, mask, &gcValues);
    }

    /*
     * Deal with the oulines for ticks and axis lines
     */

    mask = Tk_ConfigOutlineGC(&gcValues, canvas, itemPtr,
	     &(graphPtr->tick_outline));
    
    if (mask) {
	gcValues.cap_style = CapButt;
	mask |= GCCapStyle;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    } else {
	newGC = None;
    }

    if (graphPtr->tick_outline.gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), graphPtr->tick_outline.gc);
    }
    graphPtr->tick_outline.gc = newGC;


    mask = Tk_ConfigOutlineGC(&gcValues, canvas, itemPtr,
	     &(graphPtr->axisline_outline));
    
    if (mask) {
	gcValues.cap_style = CapButt;
	mask |= GCCapStyle;
	newGC = Tk_GetGC(tkwin, mask, &gcValues);
    } else {
	newGC = None;
    }

    if (graphPtr->axisline_outline.gc != None) {
	Tk_FreeGC(Tk_Display(tkwin), graphPtr->axisline_outline.gc);
    }
    graphPtr->axisline_outline.gc = newGC;

    /* Create the drawable and clipMask
     */

    GraphCreateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
    GraphUpdateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
    GraphUpdateGC(Tk_CanvasTkwin(canvas), graphPtr);
    ComputeGraphBbox(canvas, graphPtr);

    if (Tcl_DStringValue(&configCmd) && *Tcl_DStringValue(&configCmd)) {
	if (Tk_Display(tkwin) != None)
	    XFlush(Tk_Display(tkwin));
        if (Tcl_GlobalEval(canvasPtr->interp,Tcl_DStringValue(&configCmd))!=TCL_OK) {
            Tcl_AddErrorInfo(canvasPtr->interp,"\n    (command bound to graph configuration)");
            Tk_BackgroundError(canvasPtr->interp);
        }
    }
    Tcl_DStringFree(&configCmd);

    graphPtr->resize_id = (graphPtr->resize_id + 1) % 256;

    for(i = 0, setPtr = *graphPtr->children ;
	i < graphPtr->nchildren;
	setPtr = graphPtr->children[i++]){

	/*
	 * Update the set's coordinates
	 */

	(setPtr->typePtr->coordProc)(interp, canvas, setPtr, -1, (char **)NULL);
	setPtr->redraw_flags |= FORCE_REDRAW;

	((TkCanvas *)canvas)->flags |= REPICK_NEEDED;
    }

    graphPtr->resize_id = (graphPtr->resize_id + 1) % 256;

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteGraph --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a xygraph item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteGraph(Tk_Canvas canvas, Tk_Item *itemPtr, Display *display)
{
    Tk_Item *setPtr = NULL;
    Tk_Item *nextPtr = NULL;
    GraphItem *graphPtr = (GraphItem *) itemPtr;
    TkCanvas *cPtr = (TkCanvas *)canvas;
    int i;
    char **info;

    Tk_DeleteOutline(display, &(graphPtr->outline));
    Tk_DeleteOutline(display, &(graphPtr->tick_outline));
    Tk_DeleteOutline(display, &(graphPtr->axisline_outline));
    if (graphPtr->fillTile != NULL) {
	Tk_FreeTile(graphPtr->fillTile);
    }
    if (graphPtr->activeFillTile != NULL) {
	Tk_FreeTile(graphPtr->activeFillTile);
    }
    if (graphPtr->disabledFillTile != NULL) {
	Tk_FreeTile(graphPtr->disabledFillTile);
    }
    if (graphPtr->fillColor != NULL) {
	Tk_FreeColor(graphPtr->fillColor);
    }
    if (graphPtr->activeFillColor != NULL) {
	Tk_FreeColor(graphPtr->activeFillColor);
    }
    if (graphPtr->disabledFillColor != NULL) {
	Tk_FreeColor(graphPtr->disabledFillColor);
    }
    if (graphPtr->fillStipple != None) {
	Tk_FreeBitmap(display, graphPtr->fillStipple);
    }
    if (graphPtr->activeFillStipple != None) {
	Tk_FreeBitmap(display, graphPtr->activeFillStipple);
    }
    if (graphPtr->disabledFillStipple != None) {
	Tk_FreeBitmap(display, graphPtr->disabledFillStipple);
    }
    if (graphPtr->fillGC != None) {
	Tk_FreeGC(display, graphPtr->fillGC);
    }
    if (graphPtr->drawableGC != None) {
	Tk_FreeGC(display, graphPtr->drawableGC);
    }
    if (graphPtr->clipMask != None) {
	Tk_FreePixmap(display, graphPtr->clipMask);
    }
    if (graphPtr->drawable != None) {
	Tk_FreePixmap(display, graphPtr->drawable);
    }

    /*
     * Don't delete the children, just let them know that their
     * mommy is dead.
     */
    if (graphPtr->children) {
	Tk_Item *child;
	for(i=0; i < graphPtr->nchildren; i++) {
	    ((DatasetItem *)(graphPtr->children[i]))->graph = (GraphItem *)NULL;
	}
    }
    /*
     * if (((Interp *)cPtr->interp)->flags & DELETED)
     *	return;
     */

    if (Tk_CanvasTkwin(canvas) == None)
	return;

    /*
     * if (((TkWindow *)Tk_CanvasTkwin(canvas))->flags & TK_ALREADY_DEAD)
     *	return;
     */
}

/*
 *--------------------------------------------------------------
 *
 * ComputeGraphBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a xygraph.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ComputeGraphBbox(Tk_Canvas canvas, GraphItem *graphPtr)
{
    int bloat, tmp, width, i;
    int xticksize=0;
    int yticksize=0;
    double dtmp;
    double bboxtmp[4];
    Tk_State state = graphPtr->header.state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = graphPtr->outline.width;
    if (state==TK_STATE_HIDDEN) {
	graphPtr->header.x1 = graphPtr->header.y1 =
	graphPtr->header.x2 = graphPtr->header.y2 = -1;
	return;
    }
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)graphPtr) {
	if (graphPtr->outline.activeWidth>width) {
	    width = graphPtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->outline.disabledWidth>0) {
	    width = graphPtr->outline.disabledWidth;
	}
    }

    /*
     * Make sure that the first coordinates are the lowest ones.
     */

    if (graphPtr->bbox[1] > graphPtr->bbox[3]) {
	double tmp;
	tmp = graphPtr->bbox[3];
	graphPtr->bbox[3] = graphPtr->bbox[1];
	graphPtr->bbox[1] = tmp;
    }
    if (graphPtr->bbox[0] > graphPtr->bbox[2]) {
	double tmp;
	tmp = graphPtr->bbox[2];
	graphPtr->bbox[2] = graphPtr->bbox[0];
	graphPtr->bbox[0] = tmp;
    }

    for(i=0; i < 4; i++)
	bboxtmp[i] = graphPtr->bbox[i];


    /*
     * Expand the bbox to include any tickmarks outside the graph
     */

    if (graphPtr->xmajortickinterval > 0.0 &&
	graphPtr->xmajorticksize > 0.0)
	xticksize=graphPtr->xmajorticksize;
    if (graphPtr->ymajortickinterval > 0.0 &&
	graphPtr->ymajorticksize > 0.0)
	yticksize=graphPtr->ymajorticksize;

    if (graphPtr->xminortickinterval > 0.0 &&
	graphPtr->xminorticksize > graphPtr->xmajorticksize)
	xticksize=graphPtr->xminorticksize;

    if (graphPtr->yminortickinterval > 0.0 &&
	graphPtr->yminorticksize > graphPtr->xmajorticksize)
	yticksize=graphPtr->yminorticksize;

    if (graphPtr->tickprops & WEST_TICK)
	bboxtmp[0] -= xticksize;
    if (graphPtr->tickprops & NORTH_TICK)
	bboxtmp[1] -= yticksize;
    if (graphPtr->tickprops & EAST_TICK)
	bboxtmp[2] += xticksize;
    if (graphPtr->tickprops & SOUTH_TICK)
	bboxtmp[3] += yticksize;

    if (graphPtr->outline.gc == None) {
	bloat = 0;
    } else {
	bloat = (width+1)/2;
    }

    /*
     * Special note:  the rectangle is always drawn at least 1x1 in
     * size, so round up the upper coordinates to be at least 1 unit
     * greater than the lower ones.
     */

    tmp = (int) ((bboxtmp[0] >= 0) ? bboxtmp[0] + .5
	    : bboxtmp[0] - .5);
    graphPtr->header.x1 = tmp - bloat;
    tmp = (int) ((bboxtmp[1] >= 0) ? bboxtmp[1] + .5
	    : bboxtmp[1] - .5);
    graphPtr->header.y1 = tmp - bloat;
    dtmp = bboxtmp[2];
    if (dtmp < (bboxtmp[0] + 1)) {
	dtmp = bboxtmp[0] + 1;
    }
    tmp = (int) ((dtmp >= 0) ? dtmp + .5 : dtmp - .5);
    graphPtr->header.x2 = tmp + bloat;
    dtmp = bboxtmp[3];
    if (dtmp < (bboxtmp[1] + 1)) {
	dtmp = bboxtmp[1] + 1;
    }
    tmp = (int) ((dtmp >= 0) ? dtmp + .5 : dtmp - .5);
    graphPtr->header.y2 = tmp + bloat;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayGraph --
 *
 *	This procedure is invoked to draw a xygraph
 *	item in a given drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvas.
 *
 *--------------------------------------------------------------
 */

static void
DisplayGraph(Tk_Canvas canvas, Tk_Item *itemPtr, Display *display, Drawable drawable, int x, int y, int width, int height)
{
    GraphItem *graphPtr = (GraphItem *) itemPtr;
    short x1, y1, x2, y2;
    Tk_Tile fillTile;
    Pixmap fillStipple;
    Tk_State state;
    int graph_width, graph_height;
    XGCValues gcValues;
    double *cPtr;
    int count;

    graph_width = (int) (graphPtr->bbox[2] - graphPtr->bbox[0]);
    graph_height = (int) (graphPtr->bbox[3] - graphPtr->bbox[1]);
    if (graphPtr->drawable == None) {
	GraphCreateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
	GraphUpdateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
	GraphUpdateGC(Tk_CanvasTkwin(canvas), graphPtr);
    }

    /*
     * Compute the screen coordinates of the bounding box for the item.
     * Make sure that the bbox is at least one pixel large, since some
     * X servers will die if it isn't.
     */

    Tk_CanvasDrawableCoords(canvas, graphPtr->bbox[0], graphPtr->bbox[1],
	    &x1, &y1);
    Tk_CanvasDrawableCoords(canvas, graphPtr->bbox[2], graphPtr->bbox[3],
	    &x2, &y2);
    if (x2 <= x1) {
	x2 = x1+1;
    }
    if (y2 <= y1) {
	y2 = y1+1;
    }

    /*
     * Display filled part first (if wanted), then outline.  If we're
     * stippling, then modify the stipple offset in the GC.  Be sure to
     * reset the offset when done, since the GC is supposed to be
     * read-only.
     */

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    fillTile = graphPtr->fillTile;
    fillStipple = graphPtr->fillStipple;
    if (((TkCanvas *)canvas)->currentItemPtr == (Tk_Item *)graphPtr) {
	if (graphPtr->activeFillTile!=NULL) {
	    fillTile = graphPtr->activeFillTile;
	}
	if (graphPtr->activeFillStipple!=None) {
	    fillStipple = graphPtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->disabledFillTile!=NULL) {
	    fillTile = graphPtr->disabledFillTile;
	}
	if (graphPtr->disabledFillStipple!=None) {
	    fillStipple = graphPtr->disabledFillStipple;
	}
    }

    if (graphPtr->fillGC != None) {
	if ((fillTile != None) || (fillStipple != None)) {
	    Tk_CanvasSetStippleOrigin(canvas, graphPtr->fillGC);
	}
	XFillRectangle(display, drawable, graphPtr->fillGC,
		x1, y1, (unsigned int) (x2-x1), (unsigned int) (y2-y1));
	if ((fillTile != None) || (fillStipple != None)) {
	    XSetTSOrigin(display, graphPtr->fillGC, 0, 0);
	}
    }
    if (graphPtr->outline.gc != None) {

	Tk_ChangeOutlineGC(canvas, itemPtr, &(graphPtr->outline));
	XDrawRectangle(display, drawable, graphPtr->outline.gc,
		x1, y1, (unsigned) (x2-x1), (unsigned) (y2-y1));
	Tk_ResetOutlineGC(canvas, itemPtr, &(graphPtr->outline));
    }

    if (graphPtr->tick_outline.gc != None) {
	double x0, y0;
	int i;
	/*
	 * fprintf(stderr, "Tick positions:  %s\n", t);
	 * fprintf(stderr, "Tick orient:  %s\n", graphPtr->tickorientation);
	 */
	if (graphPtr->tickprops & (SOUTH_TICK)) {
	    cPtr = ComputeGraphTicks(itemPtr, 'X', &count);

	    if(graphPtr->tickprops & SOUTH_TICK_IN) {
		for(i=0; i < count; i++) {
		    x0 = cPtr[i] - graphPtr->xaxismin;
		    x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
			(graphPtr->xaxismax - graphPtr->xaxismin);
		    x0 += x1;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int)x0,
			    (int)(y2-graphPtr->xmajorticksize),
			    (int)x0,
			    y2);
		}
	    }
	    if(graphPtr->tickprops & SOUTH_TICK_OUT) {
		for(i=0; i < count; i++) {
		    x0 = cPtr[i] - graphPtr->xaxismin;
		    x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
			(graphPtr->xaxismax - graphPtr->xaxismin);
		    x0 += x1;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int)x0,
			    (int)(y2+graphPtr->xmajorticksize),
			    (int)x0,
			    y2);
		}
	    }

	    if (cPtr)
		ckfree((char *)cPtr);

	    cPtr = ComputeGraphTicks(itemPtr, 'x', &count);

	    if(graphPtr->tickprops & SOUTH_TICK_IN) {
		for(i=0; i < count; i++) {
		    x0 = cPtr[i] - graphPtr->xaxismin;
		    x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
			(graphPtr->xaxismax - graphPtr->xaxismin);
		    x0 += x1;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int)x0,
			    (int)(y2-graphPtr->xminorticksize),
			    (int)x0,
			    y2);
		}
	    }
	    if(graphPtr->tickprops & SOUTH_TICK_OUT) {
		for(i=0; i < count; i++) {
		    x0 = cPtr[i] - graphPtr->xaxismin;
		    x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
			(graphPtr->xaxismax - graphPtr->xaxismin);
		    x0 += x1;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int)x0,
			    (int)(y2+graphPtr->xminorticksize),
			    (int)x0,
			    y2);
		}
	    }

	    if (cPtr)
		ckfree((char *)cPtr);
	}

	if (graphPtr->tickprops & (NORTH_TICK)) {
	    cPtr = ComputeGraphTicks(itemPtr, 'X', &count);

	    if(graphPtr->tickprops & NORTH_TICK_IN) {
		for(i=0; i < count; i++) {
		    x0 = cPtr[i] - graphPtr->xaxismin;
		    x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
			(graphPtr->xaxismax - graphPtr->xaxismin);
		    x0 += x1;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int)x0,
			    (int)(y1+graphPtr->xmajorticksize),
			    (int)x0,
			    y1);
		}
	    }
	    if(graphPtr->tickprops & NORTH_TICK_OUT) {
		for(i=0; i < count; i++) {
		    x0 = cPtr[i] - graphPtr->xaxismin;
		    x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
			(graphPtr->xaxismax - graphPtr->xaxismin);
		    x0 += x1;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int)x0,
			    (int)(y1-graphPtr->xmajorticksize),
			    (int)x0,
			    y1);
		}
	    }

	    if (cPtr)
		ckfree((char *)cPtr);

	    cPtr = ComputeGraphTicks(itemPtr, 'x', &count);

	    if(graphPtr->tickprops & NORTH_TICK_IN) {
		for(i=0; i < count; i++) {
		    x0 = cPtr[i] - graphPtr->xaxismin;
		    x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
			(graphPtr->xaxismax - graphPtr->xaxismin);
		    x0 += x1;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int)x0,
			    (int)(y1+graphPtr->xminorticksize),
			    (int)x0,
			    y1);
		}
	    }
	    if(graphPtr->tickprops & NORTH_TICK_OUT) {
		for(i=0; i < count; i++) {
		    x0 = cPtr[i] - graphPtr->xaxismin;
		    x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
			(graphPtr->xaxismax - graphPtr->xaxismin);
		    x0 += x1;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int)x0,
			    (int)(y1-graphPtr->xminorticksize),
			    (int)x0,
			    y1);
		}
	    }

	    if (cPtr)
		ckfree((char *)cPtr);
	}

	if (graphPtr->tickprops & (WEST_TICK)) {
	    cPtr = ComputeGraphTicks(itemPtr, 'Y', &count);

	    if(graphPtr->tickprops & WEST_TICK_IN) {
		for(i=0; i < count; i++) {
		    y0 = cPtr[i] - graphPtr->yaxismin;
		    y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			    (graphPtr->yaxismax - graphPtr->yaxismin);
		    y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int) x1,
			    (int) y0,
			    (int) (x1+graphPtr->ymajorticksize),
			    (int) y0);
		}
	    }
	    if(graphPtr->tickprops & WEST_TICK_OUT) {
		for(i=0; i < count; i++) {
		    y0 = cPtr[i] - graphPtr->yaxismin;
		    y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			    (graphPtr->yaxismax - graphPtr->yaxismin);
		    y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int) x1,
			    (int) y0,
			    (int) (x1-graphPtr->ymajorticksize),
			    (int) y0);
		}
	    }

	    if (cPtr)
		ckfree((char *)cPtr);

	    cPtr = ComputeGraphTicks(itemPtr, 'y', &count);
	    if(graphPtr->tickprops & WEST_TICK_IN) {
		for(i=0; i < count; i++) {
		    y0 = cPtr[i] - graphPtr->yaxismin;
		    y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			    (graphPtr->yaxismax - graphPtr->yaxismin);
		    y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int) x1,
			    (int) y0,
			    (int) (x1+graphPtr->yminorticksize),
			    (int) y0);
		}
	    }
	    if(graphPtr->tickprops & WEST_TICK_OUT) {
		for(i=0; i < count; i++) {
		    y0 = cPtr[i] - graphPtr->yaxismin;
		    y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			    (graphPtr->yaxismax - graphPtr->yaxismin);
		    y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int) x1,
			    (int) y0,
			    (int) (x1-graphPtr->yminorticksize),
			    (int) y0);
		}
	    }

	    if (cPtr)
		ckfree((char *)cPtr);
	}

	if (graphPtr->tickprops & (EAST_TICK)) {
	    cPtr = ComputeGraphTicks(itemPtr, 'Y', &count);

	    if(graphPtr->tickprops & EAST_TICK_IN) {
		for(i=0; i < count; i++) {
		    y0 = cPtr[i] - graphPtr->yaxismin;
		    y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			    (graphPtr->yaxismax - graphPtr->yaxismin);
		    y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int) x2,
			    (int) y0,
			    (int) (x2-graphPtr->ymajorticksize),
			    (int) y0);
		}
	    }
	    if(graphPtr->tickprops & EAST_TICK_OUT) {
		for(i=0; i < count; i++) {
		    y0 = cPtr[i] - graphPtr->yaxismin;
		    y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			    (graphPtr->yaxismax - graphPtr->yaxismin);
		    y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int) x2,
			    (int) y0,
			    (int) (x2+graphPtr->ymajorticksize),
			    (int) y0);
		}
	    }

	    if (cPtr)
		ckfree((char *)cPtr);

	    cPtr = ComputeGraphTicks(itemPtr, 'y', &count);

	    if(graphPtr->tickprops & EAST_TICK_IN) {
		for(i=0; i < count; i++) {
		    y0 = cPtr[i] - graphPtr->yaxismin;
		    y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			    (graphPtr->yaxismax - graphPtr->yaxismin);
		    y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int) x2,
			    (int) y0,
			    (int) (x2-graphPtr->yminorticksize),
			    (int) y0);
		}
	    }
	    if(graphPtr->tickprops & EAST_TICK_OUT) {
		for(i=0; i < count; i++) {
		    y0 = cPtr[i] - graphPtr->yaxismin;
		    y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			    (graphPtr->yaxismax - graphPtr->yaxismin);
		    y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;

		    XDrawLine(display,
			    drawable,
			    graphPtr->tick_outline.gc,
			    (int) x2,
			    (int) y0,
			    (int) (x2+graphPtr->yminorticksize),
			    (int) y0);
		}
	    }
	}
    }

    if (graphPtr->axisline_outline.gc != None) {
	double y0, x0;
	/*
	 * Draw the axis lines
	 */
	count = graphPtr->nyaxislines;
	cPtr = graphPtr->yaxislines;
	while(count-- > 0) {
	    y0 = cPtr[count] - graphPtr->yaxismin;
	    y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
		    (graphPtr->yaxismax - graphPtr->yaxismin);
	    y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;

	    if (y0 >= y1 && y0 <= y2) {
		XDrawLine(display, drawable, graphPtr->axisline_outline.gc,
			(int) x1, (int) y0, (int) (x2), (int) y0);
	    }
	}

	count = graphPtr->nxaxislines;
	cPtr = graphPtr->xaxislines;
	while(count-- > 0) {
	    x0 = cPtr[count] - graphPtr->xaxismin;
	    x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		(graphPtr->xaxismax - graphPtr->xaxismin);
	    x0 += x1;

	    if (x0 >= x1 && x0 <= x2) {
		XDrawLine(display, drawable, graphPtr->axisline_outline.gc,
			(int)x0, (int)(y1),(int)x0, y2);
	    }
	}
    }
}

/*
 *--------------------------------------------------------------
 *
 * GraphToPoint --
 *
 *	Computes the distance from a given point to a given
 *	rectangle, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are coordPtr[0] and coordPtr[1] is inside the rectangle.  If the
 *	point isn't inside the rectangle then the return value is the
 *	distance from the point to the rectangle.  If itemPtr is filled,
 *	then anywhere in the interior is considered "inside"; if
 *	itemPtr isn't filled, then "inside" means only the area
 *	occupied by the outline.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
GraphToPoint(Tk_Canvas canvas, Tk_Item *itemPtr, double *pointPtr)
{
    GraphItem *graphPtr = (GraphItem *) itemPtr;
    double xDiff, yDiff, x1, y1, x2, y2, inc, tmp;
    double width;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = graphPtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (graphPtr->outline.activeWidth>width) {
	    width = graphPtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->outline.disabledWidth>0) {
	    width = graphPtr->outline.disabledWidth;
	}
    }

    /*
     * Generate a new larger rectangle that includes the border
     * width, if there is one.
     */

    x1 = graphPtr->bbox[0];
    y1 = graphPtr->bbox[1];
    x2 = graphPtr->bbox[2];
    y2 = graphPtr->bbox[3];
    if (graphPtr->outline.gc != None) {
	inc = width/2.0;
	x1 -= inc;
	y1 -= inc;
	x2 += inc;
	y2 += inc;
    }

    /*
     * If the point is inside the rectangle, handle specially:
     * distance is 0 if rectangle is filled, otherwise compute
     * distance to nearest edge of rectangle and subtract width
     * of edge.
     *
     */

    if ((pointPtr[0] >= x1) && (pointPtr[0] < x2)
		&& (pointPtr[1] >= y1) && (pointPtr[1] < y2)) {
	if ((graphPtr->fillGC != None) || (graphPtr->outline.gc == None)) {
	    return 0.0;
	}
	xDiff = pointPtr[0] - x1;
	tmp = x2 - pointPtr[0];
	if (tmp < xDiff) {
	    xDiff = tmp;
	}
	yDiff = pointPtr[1] - y1;
	tmp = y2 - pointPtr[1];
	if (tmp < yDiff) {
	    yDiff = tmp;
	}
	if (yDiff < xDiff) {
	    xDiff = yDiff;
	}
	xDiff -= width;
	if (xDiff < 0.0) {
	    return 0.0;
	}
	return 0.0;
    }

    /*
     * Point is outside rectangle.
     */

    if (pointPtr[0] < x1) {
	xDiff = x1 - pointPtr[0];
    } else if (pointPtr[0] > x2)  {
	xDiff = pointPtr[0] - x2;
    } else {
	xDiff = 0;
    }

    if (pointPtr[1] < y1) {
	yDiff = y1 - pointPtr[1];
    } else if (pointPtr[1] > y2)  {
	yDiff = pointPtr[1] - y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
}

/*
 *--------------------------------------------------------------
 *
 * GraphToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangle.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by graphPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
GraphToArea(Tk_Canvas canvas, Tk_Item *itemPtr, double *areaPtr)
{
    GraphItem *graphPtr = (GraphItem *) itemPtr;
    double halfWidth;
    double width;
    Tk_State state = itemPtr->state;

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }

    width = graphPtr->outline.width;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (graphPtr->outline.activeWidth>width) {
	    width = graphPtr->outline.activeWidth;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->outline.disabledWidth>0) {
	    width = graphPtr->outline.disabledWidth;
	}
    }

    halfWidth = width/2.0;
    if (graphPtr->outline.gc == None) {
	halfWidth = 0.0;
    }

    if ((areaPtr[2] <= (graphPtr->bbox[0] - halfWidth))
	    || (areaPtr[0] >= (graphPtr->bbox[2] + halfWidth))
	    || (areaPtr[3] <= (graphPtr->bbox[1] - halfWidth))
	    || (areaPtr[1] >= (graphPtr->bbox[3] + halfWidth))) {
	return -1;
    }
    /*
       Changed the following line so that the graph behaves like
       a filled box, whether it's filled or not.

       if ((graphPtr->fillGC == None) && (graphPtr->outline.gc != None)
	    && (areaPtr[0] >= (graphPtr->bbox[0] + halfWidth))
	    && (areaPtr[1] >= (graphPtr->bbox[1] + halfWidth))
	    && (areaPtr[2] <= (graphPtr->bbox[2] - halfWidth))
	    && (areaPtr[3] <= (graphPtr->bbox[3] - halfWidth))) {
	return -1;
    }
    */
    if ((areaPtr[0] <= (graphPtr->bbox[0] - halfWidth))
	    && (areaPtr[1] <= (graphPtr->bbox[1] - halfWidth))
	    && (areaPtr[2] >= (graphPtr->bbox[2] + halfWidth))
	    && (areaPtr[3] >= (graphPtr->bbox[3] + halfWidth))) {
	return 1;
    }
    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScaleGraph --
 *
 *	This procedure is invoked to rescale a xygraph
 *	item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The xygraph referred to by itemPtr is rescaled
 *	so that the following transformation is applied to all
 *	point coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

void
ScaleGraph(Tk_Canvas canvas, Tk_Item *itemPtr, double originX, double originY, double scaleX, double scaleY)
{
    GraphItem *graphPtr = (GraphItem *) itemPtr;
    Tk_Item *setPtr;
    int i;

    Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1, itemPtr->x2, itemPtr->y2);

    graphPtr->bbox[0] = originX + scaleX*(graphPtr->bbox[0] - originX);
    graphPtr->bbox[1] = originY + scaleY*(graphPtr->bbox[1] - originY);
    graphPtr->bbox[2] = originX + scaleX*(graphPtr->bbox[2] - originX);
    graphPtr->bbox[3] = originY + scaleY*(graphPtr->bbox[3] - originY);
    ComputeGraphBbox(canvas, graphPtr);

    GraphCreateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
    GraphUpdateDrawable(Tk_CanvasTkwin(canvas), graphPtr);
    GraphUpdateGC(Tk_CanvasTkwin(canvas), graphPtr);

    graphPtr->resize_id = (graphPtr->resize_id + 1) % 256;

    Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1, itemPtr->x2, itemPtr->y2);

    for(i = 0, setPtr = *graphPtr->children ;
	i < graphPtr->nchildren; 
	setPtr = graphPtr->children[i++]) {
	Tk_CanvasEventuallyRedraw(canvas, setPtr->x1, setPtr->y1, setPtr->x2,
		setPtr->y2);
	((TkCanvas *)canvas)->flags |= REPICK_NEEDED;
    }
}

/*
 *--------------------------------------------------------------
 *
 * TranslateGraph --
 *
 *	This procedure is called to move a rectangle or oval by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the rectangle or oval is offset by
 *	(xDelta, yDelta), and the bounding box is updated in the
 *	generic part of the item structure.
 *
 *--------------------------------------------------------------
 */

void
TranslateGraph(Tk_Canvas canvas, Tk_Item *itemPtr, double deltaX, double deltaY)
{
    GraphItem *graphPtr = (GraphItem *) itemPtr;
    int i;

    Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1,
	    itemPtr->x2, itemPtr->y2);

    graphPtr->bbox[0] += deltaX;
    graphPtr->bbox[1] += deltaY;
    graphPtr->bbox[2] += deltaX;
    graphPtr->bbox[3] += deltaY;
    ComputeGraphBbox(canvas, graphPtr);

    graphPtr->resize_id = (graphPtr->resize_id + 1) % 256;

    Tk_CanvasEventuallyRedraw(canvas, itemPtr->x1, itemPtr->y1,
	    itemPtr->x2, itemPtr->y2);
}

/*
 *--------------------------------------------------------------
 *
 * GraphToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	rectangle and oval items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in interp->result, replacing whatever used to be there.
 *	If no error occurs, then Postscript for the rectangle is
 *	appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
GraphToPostscript(Tcl_Interp *interp, Tk_Canvas canvas, Tk_Item *itemPtr, int prepass)
{
    char pathCmd[1024], string[512];
    GraphItem	*graphPtr = (GraphItem *) itemPtr;
    double	y1, y2;
    double	x0, y0;
    double	*cPtr=(double *)NULL;
    int		i;
    int		width;
    int		count;
    Tk_Tile	tile;
    XColor	*color;
    Pixmap	stipple;
    Tk_Tile	fillTile;
    XColor	*fillColor;
    Pixmap	fillStipple;
    Tk_State	state = itemPtr->state;

    y1 = Tk_CanvasPsY(canvas, graphPtr->bbox[1]);
    y2 = Tk_CanvasPsY(canvas, graphPtr->bbox[3]);

    /*
     * Generate a string that creates a path for the rectangle or oval.
     * This is the only part of the procedure's code that is type-
     * specific.
     */


    sprintf(pathCmd, "%.15g %.15g moveto %.15g 0 rlineto 0 %.15g rlineto %.15g 0 rlineto closepath\n",
	    graphPtr->bbox[0], y1,
	    graphPtr->bbox[2]-graphPtr->bbox[0], y2-y1,
	    graphPtr->bbox[0]-graphPtr->bbox[2]);

    if(state == TK_STATE_NULL) {
	state = ((TkCanvas *)canvas)->canvas_state;
    }
    tile = graphPtr->outline.tile;
    color = graphPtr->outline.color;
    fillTile = graphPtr->fillTile;
    fillColor = graphPtr->fillColor;
    fillStipple = graphPtr->fillStipple;
    if (((TkCanvas *)canvas)->currentItemPtr == itemPtr) {
	if (graphPtr->outline.activeTile!=NULL) {
	    tile = graphPtr->outline.activeTile;
	}
	if (graphPtr->outline.activeColor!=NULL) {
	    color = graphPtr->outline.activeColor;
	}
	if (graphPtr->activeFillTile!=NULL) {
	    fillTile = graphPtr->activeFillTile;
	}
	if (graphPtr->activeFillColor!=NULL) {
	    fillColor = graphPtr->activeFillColor;
	}
	if (graphPtr->activeFillStipple!=None) {
	    fillStipple = graphPtr->activeFillStipple;
	}
    } else if (state==TK_STATE_DISABLED) {
	if (graphPtr->outline.disabledTile!=NULL) {
	    tile = graphPtr->outline.disabledTile;
	}
	if (graphPtr->outline.disabledColor!=NULL) {
	    color = graphPtr->outline.disabledColor;
	}
	if (graphPtr->disabledFillTile!=NULL) {
	    fillTile = graphPtr->disabledFillTile;
	}
	if (graphPtr->disabledFillColor!=NULL) {
	    fillColor = graphPtr->disabledFillColor;
	}
	if (graphPtr->disabledFillStipple!=None) {
	    fillStipple = graphPtr->disabledFillStipple;
	}
    }

    /*
     * First draw the filled area of the rectangle.
     */

    if (fillColor != NULL) {
	Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	if (Tk_CanvasPsColor(interp, canvas, fillColor)
		!= TCL_OK) {
	    return TCL_ERROR;
	}
	if (fillStipple != None) {
	    Tcl_AppendResult(interp, "clip ", (char *) NULL);
	    if (Tk_CanvasPsStipple(interp, canvas, fillStipple)
		    != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (color != NULL) {
		Tcl_AppendResult(interp, "grestore gsave\n", (char *) NULL);
	    }
	} else {
	    Tcl_AppendResult(interp, "fill\n", (char *) NULL);
	}
    }

    /*
     * Now draw the outline, if there is one.
     */

    if (color != NULL) {
	Tcl_AppendResult(interp, pathCmd, "0 setlinejoin 2 setlinecap\n",
		(char *) NULL);
	if (Tk_CanvasPsOutline(canvas, itemPtr,
		&(graphPtr->outline))!= TCL_OK) {
	    return TCL_ERROR;
	}
    }

    /*
     * Now draw the tickmarks, if there are any.  Re-use the
     * pathCmd variable.
     */

    Tcl_AppendResult(interp, "newpath\n", (char *)NULL);
    pathCmd[0] = (char)NULL;

    if (graphPtr->tickprops & (SOUTH_TICK)) {
	cPtr = ComputeGraphTicks(itemPtr, 'X', &count);

	if (graphPtr->tickprops & SOUTH_TICK_IN) {
	    for(i=0; i < count; i++) {
		x0 = cPtr[i] - graphPtr->xaxismin;
		x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		    (graphPtr->xaxismax - graphPtr->xaxismin);
		x0 += graphPtr->bbox[0];

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)x0,
			(double)y2,
			(double)x0,
			(double)y2 + graphPtr->xmajorticksize);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (graphPtr->tickprops & SOUTH_TICK_OUT) {
	    for(i=0; i < count; i++) {
		x0 = cPtr[i] - graphPtr->xaxismin;
		x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		    (graphPtr->xaxismax - graphPtr->xaxismin);
		x0 += graphPtr->bbox[0];

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)x0,
			(double)y2,
			(double)x0,
			(double)y2 - graphPtr->xmajorticksize);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}

	if (cPtr)
	    ckfree((char *)cPtr);

	if (graphPtr->tickprops & SOUTH_TICK_IN) {
	    cPtr = ComputeGraphTicks(itemPtr, 'x', &count);
	    for(i=0; i < count; i++) {
		x0 = cPtr[i] - graphPtr->xaxismin;
		x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		    (graphPtr->xaxismax - graphPtr->xaxismin);
		x0 += graphPtr->bbox[0];

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)x0,
			(double)y2,
			(double)x0,
			(double)y2 + graphPtr->xminorticksize);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (graphPtr->tickprops & SOUTH_TICK_OUT) {
	    cPtr = ComputeGraphTicks(itemPtr, 'x', &count);
	    for(i=0; i < count; i++) {
		x0 = cPtr[i] - graphPtr->xaxismin;
		x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		    (graphPtr->xaxismax - graphPtr->xaxismin);
		x0 += graphPtr->bbox[0];

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)x0,
			(double)y2,
			(double)x0,
			(double)y2 - graphPtr->xminorticksize);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (cPtr)
	    ckfree((char *)cPtr);
    }

    if (graphPtr->tickprops & (NORTH_TICK)) {
	cPtr = ComputeGraphTicks(itemPtr, 'X', &count);

	if (graphPtr->tickprops & NORTH_TICK_IN) {
	    for(i=0; i < count; i++) {
		x0 = cPtr[i] - graphPtr->xaxismin;
		x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		    (graphPtr->xaxismax - graphPtr->xaxismin);
		x0 += graphPtr->bbox[0];

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)x0,
			(double)y1,
			(double)x0,
			(double)y1 - graphPtr->xmajorticksize);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (graphPtr->tickprops & NORTH_TICK_OUT) {
	    for(i=0; i < count; i++) {
		x0 = cPtr[i] - graphPtr->xaxismin;
		x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		    (graphPtr->xaxismax - graphPtr->xaxismin);
		x0 += graphPtr->bbox[0];

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)x0,
			(double)y1,
			(double)x0,
			(double)y1 + graphPtr->xmajorticksize);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}

	if (cPtr)
	    ckfree((char *)cPtr);

	if (graphPtr->tickprops & NORTH_TICK_IN) {
	    cPtr = ComputeGraphTicks(itemPtr, 'x', &count);
	    for(i=0; i < count; i++) {
		x0 = cPtr[i] - graphPtr->xaxismin;
		x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		    (graphPtr->xaxismax - graphPtr->xaxismin);
		x0 += graphPtr->bbox[0];

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)x0,
			(double)y1,
			(double)x0,
			(double)y1 - graphPtr->xminorticksize);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (graphPtr->tickprops & NORTH_TICK_OUT) {
	    cPtr = ComputeGraphTicks(itemPtr, 'x', &count);
	    for(i=0; i < count; i++) {
		x0 = cPtr[i] - graphPtr->xaxismin;
		x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		    (graphPtr->xaxismax - graphPtr->xaxismin);
		x0 += graphPtr->bbox[0];

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)x0,
			(double)y1,
			(double)x0,
			(double)y1 + graphPtr->xminorticksize);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (cPtr)
	    ckfree((char *)cPtr);
    }

    if (graphPtr->tickprops & (WEST_TICK)) {

	cPtr = ComputeGraphTicks(itemPtr, 'Y', &count);
	
	if (graphPtr->tickprops & WEST_TICK_IN) {
	    for(i=0; i < count; i++) {
		y0 = cPtr[i] - graphPtr->yaxismin;
		y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			(graphPtr->yaxismax - graphPtr->yaxismin);
		y0 = y2 + y0;

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)graphPtr->bbox[0],
			(double)y0,
			(double)graphPtr->bbox[0] + graphPtr->ymajorticksize,
			(double)y0);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (graphPtr->tickprops & WEST_TICK_OUT) {
	    for(i=0; i < count; i++) {
		y0 = cPtr[i] - graphPtr->yaxismin;
		y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			(graphPtr->yaxismax - graphPtr->yaxismin);
		y0 = y2 + y0;

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)graphPtr->bbox[0],
			(double)y0,
			(double)graphPtr->bbox[0] - graphPtr->ymajorticksize,
			(double)y0);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}

	if (cPtr)
	    ckfree((char *)cPtr);

	cPtr = ComputeGraphTicks(itemPtr, 'y', &count);

	if (graphPtr->tickprops & WEST_TICK_IN) {
	    for(i=0; i < count; i++) {
		y0 = cPtr[i] - graphPtr->yaxismin;
		y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			(graphPtr->yaxismax - graphPtr->yaxismin);
		y0 = y2 + y0;

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)graphPtr->bbox[0],
			(double)y0,
			(double)graphPtr->bbox[0] + graphPtr->yminorticksize,
			(double)y0);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (graphPtr->tickprops & WEST_TICK_OUT) {
	    for(i=0; i < count; i++) {
		y0 = cPtr[i] - graphPtr->yaxismin;
		y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			(graphPtr->yaxismax - graphPtr->yaxismin);
		y0 = y2 + y0;

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)graphPtr->bbox[0],
			(double)y0,
			(double)graphPtr->bbox[0] - graphPtr->yminorticksize,
			(double)y0);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (cPtr)
	    ckfree((char *)cPtr);
    }

    if (graphPtr->tickprops & (EAST_TICK)) {

	cPtr = ComputeGraphTicks(itemPtr, 'Y', &count);
	
	if (graphPtr->tickprops & EAST_TICK_IN) {
	    for(i=0; i < count; i++) {
		y0 = cPtr[i] - graphPtr->yaxismin;
		y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			(graphPtr->yaxismax - graphPtr->yaxismin);
		y0 = y2 + y0;

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)graphPtr->bbox[2],
			(double)y0,
			(double)graphPtr->bbox[2] - graphPtr->ymajorticksize,
			(double)y0);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (graphPtr->tickprops & EAST_TICK_OUT) {
	    for(i=0; i < count; i++) {
		y0 = cPtr[i] - graphPtr->yaxismin;
		y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			(graphPtr->yaxismax - graphPtr->yaxismin);
		y0 = y2 + y0;

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)graphPtr->bbox[2],
			(double)y0,
			(double)graphPtr->bbox[2] + graphPtr->ymajorticksize,
			(double)y0);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}

	if (cPtr)
	    ckfree((char *)cPtr);

	cPtr = ComputeGraphTicks(itemPtr, 'y', &count);

	if (graphPtr->tickprops & EAST_TICK_IN) {
	    for(i=0; i < count; i++) {
		y0 = cPtr[i] - graphPtr->yaxismin;
		y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			(graphPtr->yaxismax - graphPtr->yaxismin);
		y0 = y2 + y0;

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)graphPtr->bbox[2],
			(double)y0,
			(double)graphPtr->bbox[2] - graphPtr->yminorticksize,
			(double)y0);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (graphPtr->tickprops & EAST_TICK_OUT) {
	    for(i=0; i < count; i++) {
		y0 = cPtr[i] - graphPtr->yaxismin;
		y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			(graphPtr->yaxismax - graphPtr->yaxismin);
		y0 = y2 + y0;

		sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
			(double)graphPtr->bbox[2],
			(double)y0,
			(double)graphPtr->bbox[2] + graphPtr->yminorticksize,
			(double)y0);
		Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	    }
	}
	if (cPtr)
	    ckfree((char *)cPtr);
    }

    if (graphPtr->tick_outline.color != NULL) {
	if (graphPtr->tickprops) {
	    Tcl_AppendResult(interp, "2 setlinejoin 0 setlinecap\n",
		    (char *) NULL);
	    if (Tk_CanvasPsOutline(canvas, itemPtr,
		    &(graphPtr->tick_outline))!= TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }


    /*
    * Draw the axis lines
    */

    Tcl_AppendResult(interp, "newpath\n", (char *)NULL);
    pathCmd[0] = (char)NULL;

    count = graphPtr->nyaxislines;
    cPtr = graphPtr->yaxislines;
    while(count-- > 0) {
	y0 = cPtr[count] - graphPtr->yaxismin;
	y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
		(graphPtr->yaxismax - graphPtr->yaxismin);
	/*
	y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;
	*/
	y0 = y2 + y0;

	/*
	printf("y0, y1, y2 = %g, %g, %g\n", y0, y1, y2);
	*/
	if (y0 <= y1 && y0 >= y2 && graphPtr->axisline_outline.color != NULL) {
	    sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
		    (double)graphPtr->bbox[0],
		    (double)y0,
		    (double)graphPtr->bbox[2],
		    (double)y0);
	    Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	}
    }

    count = graphPtr->nxaxislines;
    cPtr = graphPtr->xaxislines;
    while(count-- > 0) {
	x0 = cPtr[count] - graphPtr->xaxismin;
	x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
	    (graphPtr->xaxismax - graphPtr->xaxismin);
	x0 += graphPtr->bbox[0];

	if (x0 >= graphPtr->bbox[0] && x0 <= graphPtr->bbox[2]
	    && graphPtr->axisline_outline.color != NULL) {

	    sprintf(pathCmd, "%.15g %.15g moveto %.15g %.15g lineto\n",
		    (double)x0,
		    (double)y2,
		    (double)x0,
		    (double)y1);
	    Tcl_AppendResult(interp, pathCmd, (char *) NULL);
	}
    }

    if (graphPtr->axisline_outline.color != NULL) {
	Tcl_AppendResult(interp, "2 setlinejoin 0 setlinecap\n",
		(char *) NULL);
	if (Tk_CanvasPsOutline(canvas, itemPtr,
		&(graphPtr->axisline_outline))!= TCL_OK) {
	    return TCL_ERROR;
	}
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GraphAccept --
 *
 *      This procedure is called to implement the visitor command
 *      for the Rectangle item type.
 *
 * Results:
 *      The return value is a standard Tcl result.  If an error
 *      occurs then an error message is
 *      left in interp->result, replacing whatever used
 *      to be there.
 *
 * Side effects:
 *      Depends on the visitor.
 *
 *--------------------------------------------------------------
 */
 
/*
static int
GraphAccept(Tcl_Interp *interp, Tk_Canvas canvas, Tk_Visitor *visitorPtr, Tk_Item *itemPtr)
{
    GraphItem *graphPtr = (GraphItem *) itemPtr;

    if (visitorPtr->typePtr->visitRectangle == NULL)
	return TCL_OK;

    return (*visitorPtr->typePtr->visitRectangle)(interp, canvas,
				  visitorPtr, itemPtr);
}
*/

static int
GraphUpdateGC(Tk_Window tkwin, GraphItem *graphPtr)
{
    GC newGC;
    XGCValues gcValues;
    unsigned long mask;

    if (graphPtr->clipMask == None || graphPtr->drawable == None) {
	GraphCreateDrawable(tkwin, graphPtr);
	GraphUpdateDrawable(tkwin, graphPtr);
    }

    if (graphPtr->clipMask != None && graphPtr->drawable != None) {
	mask = ~0;
	gcValues.function = GXcopy;
	gcValues.foreground = 0;
	gcValues.background = 0;
	gcValues.fill_style = FillTiled;
	gcValues.plane_mask = mask;
	gcValues.clip_x_origin = 0;
	gcValues.clip_y_origin = 0;
	gcValues.clip_mask = graphPtr->clipMask;
	gcValues.tile = graphPtr->drawable;

	newGC = Tk_GetGC(tkwin,
		(unsigned long) GCFunction|GCPlaneMask|GCClipXOrigin|
		GCClipYOrigin|GCClipMask|GCFillStyle|GCForeground|
		GCBackground|GCTile, &gcValues);

	if (graphPtr->drawableGC != None)
	    Tk_FreeGC(Tk_Display(tkwin), graphPtr->drawableGC);

	graphPtr->drawableGC = newGC;
    }
}

static int
GraphUpdateDrawable(Tk_Window tkwin, GraphItem *graphPtr)
{
    Tk_Item *setPtr;
    Display *display;
    int width, height, depth;
    XGCValues gcValues;
    GC newGC;
    int flag;
    unsigned long mask;

    display = Tk_Display(tkwin);
    width = (int) (graphPtr->bbox[2] - graphPtr->bbox[0]);
    height = (int) (graphPtr->bbox[3] - graphPtr->bbox[1]);
    depth = Tk_Depth(tkwin);

    /* Clear the drawable
     */

    gcValues.function = GXset;
    if (graphPtr->clipMask != None) {
	newGC = XCreateGC(display, graphPtr->clipMask, GCFunction, &gcValues);
	XFillRectangle(display, graphPtr->clipMask, newGC, 0, 0,
		(unsigned int)width, (unsigned int)height);
	XFreeGC(display, newGC);
    }

    if (graphPtr->drawable != None) {
	newGC = XCreateGC(display, graphPtr->drawable, GCFunction, &gcValues);
	XFillRectangle(display, graphPtr->drawable, newGC, 0, 0,
		(unsigned int)width, (unsigned int)height);
	XFreeGC(display, newGC);
    }

    return TCL_OK;
}

static int		
GraphCreateDrawable(Tk_Window tkwin, GraphItem *graphPtr)
{
    Display *display;
    Drawable d;
    int width, height, depth;
    XGCValues gcValues;
    GC newGC;

    display = Tk_Display(tkwin);
    d = RootWindowOfScreen(Tk_Screen(tkwin));
    width = (int) (graphPtr->bbox[2] - graphPtr->bbox[0]);
    height = (int) (graphPtr->bbox[3] - graphPtr->bbox[1]);
    depth = Tk_Depth(tkwin);

    /* Create a new pixmap on which to draw the data sets
     */

    if (graphPtr->drawable != None) {
	Tk_FreePixmap(display, graphPtr->drawable);
    }
    if (graphPtr->clipMask != None) {
	Tk_FreePixmap(display, graphPtr->clipMask);
    }

    if (d != (Drawable) NULL) {
	if (width <= 0) width=1;
	if (height <= 0) height=1;

	graphPtr->drawable = Tk_GetPixmap(display, d, width, height, depth);
	graphPtr->clipMask = Tk_GetPixmap(display, d, width, height, 1);
    }

    return TCL_OK;
}

/*
 * This procedure calculates the positions of the tickmarks in graph
 * coordinates and stores the results in the xc and yc variables.
 * The return value is the number of tickmarks found.
 *
 * The "axis" parameter should be used to describe which axis should
 * be used.  use 'x' for the x axis minor ticks, 'X' for the x axis major
 * ticks, 'y' for the y axis minor ticks, and 'Y' for the y axis major
 * ticks.
 */


static double *
ComputeGraphTicks(Tk_Item *itemPtr, char axis, int *count)
{
    GraphItem	*graphPtr = (GraphItem *) itemPtr;
    double	origin,
    		min,
		max,
		interval,
		temp,
		size;
    double	i;
    int		j;
    int		nticks=0;
    double	*cPtr = (double *)NULL;

    *count = 0;

    if (axis != 'x' && axis != 'y' & axis != 'X' && axis != 'Y')
	return 0;

    if (axis == 'X') {
	origin = graphPtr->xtickorigin;
	min = graphPtr->xaxismin;
	max = graphPtr->xaxismax;
	interval = graphPtr->xmajortickinterval;
	size = graphPtr->xmajorticksize;
    }
    if (axis == 'x') {
	origin = graphPtr->xtickorigin;
	min = graphPtr->xaxismin;
	max = graphPtr->xaxismax;
	interval = graphPtr->xminortickinterval;
	size = graphPtr->xminorticksize;
    }
    if (axis == 'Y') {
	origin = graphPtr->ytickorigin;
	min = graphPtr->yaxismin;
	max = graphPtr->yaxismax;
	interval = graphPtr->ymajortickinterval;
	size = graphPtr->ymajorticksize;
    }
    if (axis == 'y') {
	origin = graphPtr->ytickorigin;
	min = graphPtr->yaxismin;
	max = graphPtr->yaxismax;
	interval = graphPtr->yminortickinterval;
	size = graphPtr->yminorticksize;
    }

    if (max < min) {
	temp = min;
	min = max;
	max = temp;
    }

    if (interval > 0 && size > 0) {
	nticks = (int) ((max - min) / interval) + 4;
	if (nticks < 0) nticks = -nticks;
	if (nticks > 200) {
	    return cPtr;
	} else {

	    cPtr = (double *)ckalloc(sizeof(double) * (nticks));
	    for(j=0; j < nticks; j++)
		cPtr[j] = 0.0;

	    for(i = origin, j=0; i <= max; i+= interval) {
		if (j >= nticks) {
		    fprintf(stderr, "Fatal error!  Bad tick index: %d / %d\n",
			j, nticks);
		    abort();
		}
		if (i >= min) {
		    cPtr[j++] = i;
		}
	    }
	    for(i = origin; i >= min; i-= interval) {
		if (j >= nticks) {
		    fprintf(stderr, "Fatal error!  Bad tick index: %d / %d\n",
			j, nticks);
		    abort();
		}
		if (i <= max) {
		    cPtr[j++] = i;
		}
	    }
	    *count = j;
	}
    }

    return cPtr;
}

int
Tk_TickcoordsParseProc(ClientData clientData, Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec, int offset)
{
    return TCL_OK;
}

/*
 * This procedure prints the coordinates of the major x ticks in
 * canvas coordinates, along with the value for that tick.  The
 * coordinates of the tick denote the position of the tick along
 * the graph's outline.
 */

char *
Tk_TickinfoPrintProc(ClientData clientData, Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr)
{
    GraphItem	*graphPtr = (GraphItem *) widgRec;
    double *cPtr=(double *)NULL;
    double x0, y0;
    short x1, y1;
    short x2, y2;
    short xa, ya;
    short xb, yb;
    double min, max;
    char *res;
    char doublestr[TCL_DOUBLE_SPACE];
    char fmt[256];
    int count;
    char *format;
    Tcl_DString dsPtr;

    Tcl_DStringInit(&dsPtr);

    /*
    Tk_CanvasDrawableCoords((Tk_Canvas)tkwin, graphPtr->bbox[0], graphPtr->bbox[1],
	    &x1, &y1);
    Tk_CanvasDrawableCoords((Tk_Canvas)tkwin, graphPtr->bbox[2], graphPtr->bbox[3],
	    &x2, &y2);
    */
    
    x1 = graphPtr->bbox[0];
    y1 = graphPtr->bbox[1];
    x2 = graphPtr->bbox[2];
    y2 = graphPtr->bbox[3];

    cPtr = ComputeGraphTicks((Tk_Item *)widgRec, offset, &count);

    if (count == 0) {
	switch (offset) {
	    case 'X':
	    case 'x':
		xa = x2;
		xb = x1;
		ya = y2;
		yb = y2;
		min = graphPtr->xaxismin;
		max = graphPtr->xaxismax;
		format = graphPtr->xtickformat;
		break;

	    case 'Y':
	    case 'y': 
		xa = x1;
		xb = x1;
		ya = y1;
		yb = y2;
		min = graphPtr->yaxismin;
		max = graphPtr->yaxismax;
		format = graphPtr->ytickformat;
		break;
	    default:
		abort();
	}

	Tcl_DStringStartSublist(&dsPtr);

	sprintf(doublestr, "%g", (double)xa);
	Tcl_DStringAppendElement(&dsPtr, doublestr);

	sprintf(doublestr, "%g", (double)ya);
	Tcl_DStringAppendElement(&dsPtr, doublestr);

	if (format != NULL) {
	    sprintf(fmt, "%s", format);
	} else {
	    strcpy(fmt, "%g");
	}
	if (strncasecmp(fmt, "hh:mm:ss", 8) == 0) {
	} else {
	    sprintf(doublestr, fmt, max);
	}

	Tcl_DStringAppendElement(&dsPtr, doublestr);

	Tcl_DStringEndSublist(&dsPtr);
	Tcl_DStringStartSublist(&dsPtr);

	sprintf(doublestr, "%g", (double)xb);
	Tcl_DStringAppendElement(&dsPtr, doublestr);

	sprintf(doublestr, "%g", (double)yb);
	Tcl_DStringAppendElement(&dsPtr, doublestr);

	if (format != NULL) {
	    sprintf(fmt, "%s", format);
	} else {
	    strcpy(fmt, "%g");
	}
	sprintf(doublestr, fmt, min);

	Tcl_DStringAppendElement(&dsPtr, doublestr);

	Tcl_DStringEndSublist(&dsPtr);
    }
    while(count-- > 0) {
	switch (offset) {
	    case 'X':
	    case 'x': 
		y0 = y2;
		x0 = cPtr[count] - graphPtr->xaxismin;
		x0 *= (graphPtr->bbox[2] - graphPtr->bbox[0]) /
		    (graphPtr->xaxismax - graphPtr->xaxismin);
		x0 += x1;
		format = graphPtr->xtickformat;
		break;
	    case 'Y':
	    case 'y': 
		x0 = x1;
		y0 = cPtr[count] - graphPtr->yaxismin;
		y0 *= (graphPtr->bbox[3] - graphPtr->bbox[1]) /
			(graphPtr->yaxismax - graphPtr->yaxismin);
		y0 = y1 + (graphPtr->bbox[3] - graphPtr->bbox[1]) - y0;
		format = graphPtr->ytickformat;
		break;
	}

	Tcl_DStringStartSublist(&dsPtr);

	/*
	Tcl_PrintDouble(((TkCanvas *)tkwin)->interp, x0, doublestr);
	*/
	sprintf(doublestr, "%g", x0);
	Tcl_DStringAppendElement(&dsPtr, doublestr);

	sprintf(doublestr, "%g", y0);
	Tcl_DStringAppendElement(&dsPtr, doublestr);

	/*
	Tcl_PrintDouble(((TkCanvas *)tkwin)->interp, cPtr[count], doublestr);
	*/
	if (format != NULL) {
	    sprintf(fmt, "%s", format);
	} else {
	    strcpy(fmt, "%g");
	}
	sprintf(doublestr, fmt, cPtr[count]);
	Tcl_DStringAppendElement(&dsPtr, doublestr);

	Tcl_DStringEndSublist(&dsPtr);
    }

    *freeProcPtr = TCL_DYNAMIC;
    res = (char *)ckalloc(Tcl_DStringLength(&dsPtr) + 2);

    strcpy(res, Tcl_DStringValue(&dsPtr));

    if (cPtr)
	ckfree((char *)cPtr);
    Tcl_DStringFree(&dsPtr);
    return res;
}

/*
 * These procedures parse/print the coordinates of the major x ticks in
 * canvas coordinates, along with the value for that tick.  The
 * coordinates of the tick denote the position of the tick along
 * the graph's outline.
 */

int
Tk_AxislineParseProc(ClientData clientData, Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec, int offset)
{
    GraphItem	*graphPtr = (GraphItem *) widgRec;
    int		argc;
    int		i;
    char	**argv;
    char	*ptr=(char *)NULL;
    double	*cPtr=(double *)NULL;
    double	n;

    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK)
	return TCL_ERROR;

    if (argc > 0) {
	cPtr = (double *)ckalloc(sizeof(double) * argc);
    }

    for(i=0; i < argc; i++) {
	n = strtod(argv[i], &ptr);

	if (ptr == argv[i]) {
	    if (cPtr)
		ckfree((char *)cPtr);
	    Tcl_SetResult(interp, "Non-real values used for axis lines", TCL_VOLATILE);
	    return TCL_ERROR;
	}

	cPtr[i] = (double)n;
    }

    /*
     * Free up the space used by the previous set of axis lines
     */

    if (offset == 'x') {
	if (graphPtr->xaxislines)
	    ckfree((char *)(graphPtr->xaxislines));
	graphPtr->nxaxislines = argc;
	graphPtr->xaxislines = cPtr;
    }
    if (offset == 'y') {
	if (graphPtr->yaxislines)
	    ckfree((char *)(graphPtr->yaxislines));
	graphPtr->nyaxislines = argc;
	graphPtr->yaxislines = cPtr;
    }

    return TCL_OK;
}

char *
Tk_AxislinePrintProc(ClientData clientData, Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr)
{
    GraphItem	*graphPtr = (GraphItem *) widgRec;
    int		i;
    double	x0, y0;
    char	*res;
    char	doublestr[TCL_DOUBLE_SPACE];
    int		count;
    Tcl_DString	dsPtr;

    Tcl_DStringInit(&dsPtr);

    switch (offset) {
	case 'X':
	case 'x': 
	    for(i=0; i < graphPtr->nxaxislines; i++) {
		sprintf(doublestr, "%g", graphPtr->xaxislines[i]);
		Tcl_DStringAppendElement(&dsPtr, doublestr);
	    }
	    break;
	case 'Y':
	case 'y': 
	    for(i=0; i < graphPtr->nyaxislines; i++) {
		sprintf(doublestr, "%g", graphPtr->yaxislines[i]);
		Tcl_DStringAppendElement(&dsPtr, doublestr);
	    }
	    break;
    }

    *freeProcPtr = TCL_DYNAMIC;
    res = (char *)ckalloc(Tcl_DStringLength(&dsPtr) + 2);

    strcpy(res, Tcl_DStringValue(&dsPtr));

    Tcl_DStringFree(&dsPtr);
    return res;
}

char *
Tk_NsetsPrintProc(ClientData clientData, Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr)
{
    GraphItem	*graphPtr = (GraphItem *) widgRec;
    char	*res;
    char	doublestr[TCL_DOUBLE_SPACE];
    int		count;

    sprintf(doublestr, "%d", graphPtr->nchildren);

    *freeProcPtr = TCL_DYNAMIC;
    res = (char *)ckalloc(strlen(doublestr) + 2);

    strcpy(res, doublestr);

    return res;
}

int
Tk_TickorientParseProc(ClientData clientData, Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec, int offset)
{
    GraphItem	*graphPtr = (GraphItem *) widgRec;
    int newmask=0;
    char *v = value;

    while (*v) {
	switch (*v) {
	    case 'i':
		if (graphPtr->tickprops & (SOUTH_TICK))
		    newmask |= SOUTH_TICK_IN;
		if (graphPtr->tickprops & (NORTH_TICK))
		    newmask |= NORTH_TICK_IN;
		if (graphPtr->tickprops & (WEST_TICK))
		    newmask |= WEST_TICK_IN;
		if (graphPtr->tickprops & (EAST_TICK))
		    newmask |= EAST_TICK_IN;
		break;
	    case 'o':
		if (graphPtr->tickprops & (SOUTH_TICK))
		    newmask |= SOUTH_TICK_OUT;
		if (graphPtr->tickprops & (NORTH_TICK))
		    newmask |= NORTH_TICK_OUT;
		if (graphPtr->tickprops & (WEST_TICK))
		    newmask |= WEST_TICK_OUT;
		if (graphPtr->tickprops & (EAST_TICK))
		    newmask |= EAST_TICK_OUT;
		break;
	    default:  Tcl_AppendResult(interp, "Bad tick orientation ", value,
			      ".  Must be one of 'i', 'o', or 'io'.",
			      (char *)NULL);
		      return TCL_ERROR;
	}
	v++;
    }

    graphPtr->tickprops = newmask;
    return TCL_OK;
}

char *
Tk_TickpositionPrintProc(ClientData clientData, Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr)
{
    GraphItem	*graphPtr = (GraphItem *) widgRec;
    char *res;
    int index=0;

    *freeProcPtr = TCL_DYNAMIC;

    res = (char *)ckalloc(sizeof(char) * 5);

    if (graphPtr->tickprops & (NORTH_TICK_IN|NORTH_TICK_OUT))
	    res[index++] = 'n';

    if (graphPtr->tickprops & (SOUTH_TICK_IN|SOUTH_TICK_OUT))
	    res[index++] = 's';

    if (graphPtr->tickprops & (WEST_TICK_IN|WEST_TICK_OUT))
	    res[index++] = 'w';

    if (graphPtr->tickprops & (EAST_TICK_IN|EAST_TICK_OUT))
	    res[index++] = 'e';

    res[index] = (char)NULL;

    return res;
}

int
Tk_TickpositionParseProc(ClientData clientData, Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec, int offset)
{
    GraphItem	*graphPtr = (GraphItem *) widgRec;
    char	*v=value;
    int		newmask=0;

    while (*v) {
	switch (*v) {
	    case 'n':
		if (graphPtr->tickprops & (TICK_IN))
		    newmask |= NORTH_TICK_IN;
		if (graphPtr->tickprops & (TICK_OUT))
		    newmask |= NORTH_TICK_OUT;
		break;
	    case 's':
		if (graphPtr->tickprops & (TICK_IN))
		    newmask |= SOUTH_TICK_IN;
		if (graphPtr->tickprops & (TICK_OUT))
		    newmask |= SOUTH_TICK_OUT;
		break;
	    case 'w':
		if (graphPtr->tickprops & (TICK_IN))
		    newmask |= WEST_TICK_IN;
		if (graphPtr->tickprops & (TICK_OUT))
		    newmask |= WEST_TICK_OUT;
		break;
	    case 'e':
		if (graphPtr->tickprops & (TICK_IN))
		    newmask |= EAST_TICK_IN;
		if (graphPtr->tickprops & (TICK_OUT))
		    newmask |= EAST_TICK_OUT;
		break;
	    default:  Tcl_AppendResult(interp, "Bad tick orientation ", value,
			      ".  Must be a combination of 'n', 's', 'w', and 'e'",
			      (char *)NULL);
		      return TCL_ERROR;
	}

	v++;
    }

    graphPtr->tickprops = newmask;

    return TCL_OK;
}

char *
Tk_TickorientPrintProc(ClientData clientData, Tk_Window tkwin, char *widgRec, int offset, Tcl_FreeProc **freeProcPtr)
{
    GraphItem	*graphPtr = (GraphItem *) widgRec;
    char *res;
    int index=0;

    *freeProcPtr = TCL_DYNAMIC;

    res = (char *)ckalloc(sizeof(char) * 3);

    if (graphPtr->tickprops &
	    (NORTH_TICK_IN|SOUTH_TICK_IN|EAST_TICK_IN|WEST_TICK_IN))
	    res[index++] = 'i';

    if (graphPtr->tickprops &
	    (NORTH_TICK_OUT|SOUTH_TICK_OUT|EAST_TICK_OUT|WEST_TICK_OUT))
	    res[index++] = 'o';

    res[index] = (char)NULL;

    return res;
}
