/*
 * tcltidy.c --
 *
 *      This file implements a Tcl interface to the tidy command
 *
 * Copyright (c) 2000 Ajuba Solutions.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include <tcl.h>
#include "platform.h"
#include "html.h"

static int Tidy _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
                int argc, char *argv[]));

typedef struct {
    int size;
    int current;
    char *ptr;
} TclTidy_Buffer;

#ifndef max
#define max(A,B) ((A)>(B)?(A):(B))
#endif

void tcltidy_strncat(TclTidy_Buffer *buffer, char *data, int count) {

    char *newptr;

    if (buffer->size < (buffer->current + count +1)) {
	buffer->size = buffer->size + max(count +1, (buffer->size *2));
	newptr = ckalloc(buffer->size);

	if (buffer->ptr != NULL) {
	    strcpy(newptr, buffer->ptr);
	    ckfree(buffer->ptr);
	}
	buffer->ptr = newptr;
    }
    strcpy(buffer->ptr + buffer->current, data);
    buffer->current += count;
}

int tcltidy_vfprintf(tidy::ErrOut *out, const char *msg, va_list arg) {
    int result;
    TclTidy_Buffer *buffer = (TclTidy_Buffer *)out->clientData;
    char temp[4096];
    int count;
    
    count = vsprintf(temp, msg, arg);
    tcltidy_strncat(buffer, temp, count);

    return count;
}

/* wrapper for putc */
void tcltidy_putc(uint c, tidy::Out *out)
{    
    char temp[2];
    char *newptr;
    TclTidy_Buffer *buffer = (TclTidy_Buffer *)out->clientData;

    temp[0] = c;
    temp[1] = '\0';

    tcltidy_strncat(buffer, temp, 1);
}


/* wrapper for feof */
int tcltidy_eof(tidy::StreamIn *in)
{
    if (((char *)in->clientData)[0] == '\0') return 1;
    else return 0;
}

/* wrapper for getc */
uint tcltidy_getc(tidy::StreamIn *in)
{
    uint result = ((char *)in->clientData)[0];

    if (result != '\0') {
	in->clientData = (void *)(((char *)in->clientData)+1);
    }
    return result;
}


/*
 *----------------------------------------------------------------------
 *
 * Tidy --
 *
 *       Implements the new Tcl "tidy" command.
 *
 * Results:
 *      A standard Tcl result
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
Tidy (
    ClientData clientData,      /* Not used. */
    Tcl_Interp *interp,         /* Current interpreter */
    int argc,                   /* Number of arguments */
    char *argv[]                /* Argument strings */
)
{
    int count;
    char *file, *prog;
    tidy::Node *document;
    tidy::Lexer *lexer;
    char *s, c, *arg, *current_errorfile = "stderr";
    tidy::StreamIn input;
    tidy::Out out;
    tidy::ErrOut errout, warnout;
    int verboseErrors = 0;

    tidy::InitConfig();

    TclTidy_Buffer warningBuffer = {0, 0, NULL};
    TclTidy_Buffer errorBuffer = {0, 0, NULL};
    TclTidy_Buffer outBuffer = {0, 0, NULL};
    
    /* setup input, errout, warnout, out */
    warnout.clientData = (void *)&warningBuffer;
    warnout.vprintfproc = tcltidy_vfprintf;
    errout.clientData = (void *)&errorBuffer;
    errout.vprintfproc = tcltidy_vfprintf;
    out.encoding = UTF8;
    out.state = FSM_ASCII;
    out.clientData = (void *)&outBuffer;
    out.putcproc   = tcltidy_putc;
    input.clientData = (void *)(argv[argc-1]);
    input.eofproc = tcltidy_eof;
    input.getcproc = tcltidy_getc;
    input.pushed = no;
    input.c = '\0';
    input.tabs = 0;
    input.curline = 1;
    input.curcol = 1;
    input.encoding = tidy::CharEncoding;
    input.state = FSM_ASCII;
    
    outBuffer.size = strlen(argv[argc-1]+1);
    outBuffer.ptr = ckalloc(outBuffer.size);

    prog = argv[0];

    tidy::ParseConfig("show-warnings", "no", &errout);
    tidy::ParseConfig("tidy-mark", "no", &errout);
    tidy::ParseConfig("char-encoding", "utf8", &errout);

    /* the argc should be even */
    if (argc % 2) {
        tidy::tidy_out(&errout, "wrong number of arguments to %s", prog);
        if (errorBuffer.ptr != NULL) interp->result = errorBuffer.ptr;
        if (outBuffer.ptr != NULL) ckfree(outBuffer.ptr);
        if (warningBuffer.ptr != NULL) ckfree(warningBuffer.ptr);
        return TCL_ERROR;
    }

    /* last argument is the input data */
    for (count = 1; count < argc; count++) {

        if (argv[count][0] == '-') {
	    char *option = &argv[count][1];
	    char *value = argv[count+1];

	    /* 
	     * prefilter some of the options which are not supported 
	     * by the Tcl extension, run the rest through tidy::ParseConfig 
	     */

	    if (tidy::wstrcasecmp(option, "char-encoding") == 0 ||
		tidy::wstrcasecmp(option, "error-file") == 0 ||
		tidy::wstrcasecmp(option, "slide-style") == 0) {
	  
	        /* report a warning for the option */
		tidy::tidy_out(&errout, "Unsupported option %s", option);
	    } else if (tidy::wstrcasecmp(option, "verbose") == 0) {
	        if (tidy::wstrcasecmp(value, "1") == 0 ||
		    tidy::wstrcasecmp(value, "yes") == 0 ||
		    tidy::wstrcasecmp(value, "y") == 0 ||
		    tidy::wstrcasecmp(value, "true") == 0)
		{
		    verboseErrors = 1;
		}
	    } else if (tidy::ParseConfig(option, value, &errout)) {
		count++;
	    } 

	    if (errorBuffer.ptr != NULL) {
	        if (errorBuffer.ptr != NULL) interp->result = errorBuffer.ptr;
		if (outBuffer.ptr != NULL) ckfree(outBuffer.ptr);
		if (warningBuffer.ptr != NULL) ckfree(warningBuffer.ptr);
		return TCL_ERROR;
	    }
        }
    }

    
    /* ensure config is self-consistent */
    tidy::AdjustConfig();

    if (tidy::ReadInput(&input, &lexer, &document, &errout) == 1) {
        if (errorBuffer.ptr != NULL) interp->result = errorBuffer.ptr;
	if (outBuffer.ptr != NULL) ckfree(outBuffer.ptr);
	if (warningBuffer.ptr != NULL) ckfree(warningBuffer.ptr);
	return TCL_ERROR;
    }
    
    if (lexer->errors > 0) {
        if (errorBuffer.ptr != NULL) interp->result = errorBuffer.ptr;
	if (outBuffer.ptr != NULL) ckfree(outBuffer.ptr);
	if (warningBuffer.ptr != NULL) ckfree(warningBuffer.ptr);
	return TCL_ERROR;
    } else {
	tidy::WriteOutput(lexer, document, &out);
    }
    
    if (verboseErrors) tidy::ErrorSummary(lexer);
    tidy::FreeNode(document);
    tidy::FreeLexer(lexer);
    
    if (errorBuffer.ptr != NULL) {
        if (errorBuffer.ptr != NULL) interp->result = errorBuffer.ptr;
	if (outBuffer.ptr != NULL) ckfree(outBuffer.ptr);
	if (warningBuffer.ptr != NULL) ckfree(warningBuffer.ptr);
	return TCL_ERROR;
    }
    
    /* called to free hash tables etc. */
    /* DeInitTidy(); */
    if (outBuffer.ptr != NULL) interp->result = outBuffer.ptr;
    if (warningBuffer.ptr != NULL) ckfree(warningBuffer.ptr);
    ckfree(errorBuffer.ptr);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tidy_Init --
 *
 *      Initialize the new package.  The string "Examplea" in the
 *      function name must match the PACKAGE declaration at the top of
 *      configure.in.
 *
 * Results:
 *      A standard Tcl result
 *
 * Side effects:
 *      The Tclsha1 package is created.
 *      One new command "sha1" is added to the Tcl interpreter.
 *
 *----------------------------------------------------------------------
 */

extern "C" DLLEXPORT int
Tidy_Init(Tcl_Interp *interp)
{
    if (Tcl_InitStubs(interp, "8.1", 0) == NULL) {
        return TCL_ERROR;
    }
    if (Tcl_PkgRequire(interp, "Tcl", TCL_VERSION, 0) == NULL) {
        if (TCL_VERSION[0] == '7') {
            if (Tcl_PkgRequire(interp, "Tcl", "8.0", 0) == NULL) {
                return TCL_ERROR;
            }
        }
    }
    if (Tcl_PkgProvide(interp, "Tidy", VERSION) != TCL_OK) {
        return TCL_ERROR;
    }
    Tcl_CreateCommand(interp, "tidy", Tidy,
            (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);

    tidy::InitTidy();

    return TCL_OK;
}
