# sedit
#
# A simple editor for composing mail messages.
# See also the Text and Entry bindings in seditBind.tcl
#
# Copyright (c) 1993 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.

proc SeditHelp {} {
    catch {destroy .sedithelp}
    set t [Widget_Toplevel .sedithelp "Sedit help" Help]
    Widget_Frame $t but Menubar {top fill}
    Widget_AddBut $t.but quit "Dismiss" [list destroy $t]
    Widget_Message $t msg -text \
"
Sedit is a simple editor built into exmh.
The editor works on draft messages that are kept in your
MH draft folder.  When you do Compose, Reply, or Forward,
a new draft is created for you.

Here is what the buttons do:
Help - displays this window.
Bind - displays a dialog box that lets you edit keybindings.
    See the TK bind man page for keystroke syntax, if you need to.
Whom - displays the list of recipients for the message.
Abort - remove the draft message from the drafts folder.
    The editor window is removed.
Sign - insert your ~/.signature file at the end of the message.
Sign... - A menu of ~/.signature* files if you have more than one.
Send - post the message.

More... - Bring up a menu with the following options:
Keep on Send - if this button is highlighted, then Send and Save will not
    remove the editor window.  Otherwise, the window disappears afterwards.
Format mail - this breaks long lines (> 80 characters) at word boundaries.
    Note that the auto-wrapping done by the TK text widget does not insert
    newline characters.  This will be done by the editor when you save
    the message and the Format mail option is selected.
Insert File... - Insert a file into the message.
Insert @ - Insert a copy of the current message (replies only)
Save - write out the editor buffer to disk.  This is useful if you want
    to re-edit the draft message later.
Dismiss - remove the editor window.  You will be prompted if the message
    has not been saved or sent.  If you abort the message from the dialog,
    it gets deleted from the drafts folder.
"
}

proc Sedit_Start { draft } {
    global sedit
    if ![info exists sedit(init)] {
	Sedit_Init
    }
    set id [file tail $draft]
    if [catch {Widget_Toplevel .sedit$id $draft Sedit} err] {
	# Reuse existing window
	set t $sedit($id,text)
	SeditMsg $t $draft
	$t delete 1.0 end
	catch {
	    if [file readable "@"] {
		Widget_AddBut .sedit$id.but repl "@" \
		    [list SeditInsertFile $draft $t "@"]
	    } else {
		destroy .sedit$id.but.repl
	    }
	}
    } else {
	set b [Widget_Frame .sedit$id but Menubar {top fill}]
	set f [Widget_Frame .sedit$id f Frame {top expand fill}]
	set t [Widget_Text $f $sedit(height) -cursor xterm -setgrid true]
	SeditTextBindings $draft $t

	Widget_AddBut $b send "Send" [list SeditSend $draft $t]
	if [catch {glob ~/.signature*} sigFiles] {
	    set sigFiles [glob ~]/.signature
	}
	if {[llength $sigFiles] <= 1} {
	    Widget_AddBut $b sign "Sign" [list SeditSign $draft $t]	
	} else {
	    set menub [Widget_AddMenuB $b sign "Sign..." {right padx 1}]
	    foreach file $sigFiles {
		Widget_AddMenuItem $menub [file tail $file] [list SeditSign $draft $t $file]
	    }
	}
	Widget_AddBut $b whom "Whom" [list SeditWhom $draft $f]
	Widget_AddBut $b pref "Bind" [list Sedit_Pref]
	Widget_AddBut $b help "Help" [list SeditHelp]

	set menub [Widget_AddMenuB $b text "Text..." {right padx 1}]
	Widget_AddMenuItem $menub "Plain" \
	    [list SeditMimeEnriched $draft $t x-plain]
	Widget_AddMenuItem $menub "Bold" \
	    [list SeditMimeEnriched $draft $t bold]
	Widget_AddMenuItem $menub "Italic" \
	    [list SeditMimeEnriched $draft $t italic]
	Widget_AddMenuItem $menub "Underline" \
	    [list SeditMimeEnriched $draft $t underline]
	Widget_AddMenuItem $menub "Smaller" \
	    [list SeditMimeEnriched $draft $t smaller]
	Widget_AddMenuItem $menub "Bigger" \
	    [list SeditMimeEnriched $draft $t bigger]

	set menub [Widget_AddMenuB $b file "More..." {right padx 1}]
	set sedit($t,keep) 0
	set sedit($t,format) 1
	Widget_CheckMenuItem $menub "Keep on send" sedit($t,keep)
	Widget_CheckMenuItem $menub "Format mail" sedit($t,format)
	if [file readable "@"] {
	    Widget_AddMenuItem $menub "Insert @" [list SeditInsertFile $draft $t "@"]
	}    
	Widget_AddMenuItem $menub "Insert File..." [list SeditInsertFileDialog $draft $t]
	Widget_AddMenuItem $menub "Abort" [list SeditAbort $draft $t]
	Widget_AddMenuItem $menub "Save" [list SeditSave $draft $t]
	Widget_AddMenuItem $menub "Dismiss" [list SeditQuit $draft $t]

	set sedit($t,label) [Widget_Label $b status {left fill} -text $draft]
	set sedit($t,toplevel) .sedit$id
	set sedit($id,text) $t
	lappend sedit(allids) .sedit$id
    }
    set sedit($t,sent) 0
    set sedit($t,dirty) 0
    SeditMimeReset $t

    if [catch {open $draft r} in] {
	$t insert 1.0 "Cannot open $draft"
    } else {
	$t insert 1.0 [read $in]
	close $in
	SeditPositionCursor $t
    }
    focus $t
}
proc SeditPositionCursor { t } {
    set l 1
    set insert 0
    set header 0
    for {set l 1} {1} {incr l} {
	if {[$t compare $l.0 > end]} {
	    if {! $insert} {
		$t mark set insert end
	    }
	    if {! $header} {
		incr l -1
		$t mark set header $l.0
	    }
	    return
	}
	set line [$t get $l.0 $l.end]
	if [regexp {: *$} $line] {
	    if {! $insert} {
		$t mark set insert $l.end
	    }
	}
	if {$line == {}} {
	    $t mark set header $l.0
	    incr l
	    $t mark set insert $l.0
	    return
	}
    }
}

proc SeditQuit { draft t } {
    global sedit
    if [SeditIsDirty $t] {
	Widget_Toplevel .seditDirty "Dirty file"
	Widget_Message .seditDirty msg  -aspect 1000 -text "
$draft
has not been saved or sent.
Do you want to abort (remove) it,
send it now,
save it for later editting,
or do nothing?"
	Widget_Frame .seditDirty f Dialog
	.seditDirty.f configure -bd 10
	Widget_AddBut .seditDirty.f ok "Abort" [list SeditNuke $draft $t]
	Widget_AddBut .seditDirty.f send "Send" [list SeditSend $draft $t]
	Widget_AddBut .seditDirty.f save "Save" \
		[list SeditSave $draft $t SeditNuke]
	Widget_AddBut .seditDirty.f no "Do nothing" {destroy .seditDirty}
    } else {
	SeditNuke $draft $t
    }
}
proc SeditNuke { draft t } {
    global sedit
    set ix [lsearch $sedit(allids) $sedit($t,toplevel)]
    if {$ix >= 0} {
	set sedit(allids) [lreplace $sedit(allids) $ix $ix]
    }
    catch {destroy .seditUnsent}
    catch {destroy .seditDirty}
    Exmh_Focus
    destroy $sedit($t,toplevel)
}
proc SeditMsg { t text } {
    global sedit
    $sedit($t,label) configure -text $text
    update
}

proc SeditSend { draft t } {
    if [SeditSave $draft $t] {
	set id [file tail $draft]
	SeditMsg $t "Message sent"
	SeditMarkSent $t
	Edit_Done send $id
	global sedit
	if {! $sedit($t,keep)} {
	    SeditNuke $draft $t
	}
    }
}
proc SeditAbort { draft t } {
    set id [file tail $draft]
    Edit_Done abort $id
    SeditNuke $draft $t
}
proc SeditWhom { draft f } {
    global tk_version
    set parent [file root $f]
    if {[catch {destroy $parent.whom}] == 0} {
	return
    }
    set id [file tail $draft]
    catch {Mh_Whom $id} result
    set lines [llength [split $result \n]]
    set f2 [Widget_Frame $parent whom {top fill}]
    set height [expr {$lines > 8 ? 8 : $lines}]
    set t2 [Widget_Text $f2 $height]
    $t2 configure -height $height	;# Widget_Text broken
    $t2 insert 1.0 $result
    $t2 config -state disabled
    if {$tk_version >= 3.3} {
	pack $f2 -before $f -side top
    } else {
	pack before $f $f2 top
    }
}
proc SeditSign { draft t {f ~/.signature} } {
    if [catch {glob $f} sig] {
	return
    }
    if [file readable $sig] {
	set in [open $sig]
	set signature [read $in]
	$t insert end \n
	$t insert end $signature
	close $in
	WidgetTextYview $t end
    }
}
proc SeditInsertFile { draft t file } {
    global sedit
    if [file readable $file] {
	set in [open $file]
	if {$file == "@"} {
	    set inheaders 1
	    while {[gets $in line] > -1} {
		if {! $inheaders} {
		    $t insert insert $sedit(pref,replPrefix)$line\n
		} else {
		    if {[string length $line] == 0} {
			set inheaders 0
		    }
		}
	    }
	} else {
	    set body [read $in]
	    $t insert insert $body
	}
	close $in
    } else {
	SeditMsg $t "Cannot read $file"
    }
}

proc SeditInsertFileDialog { draft t } {
    if [catch {Widget_Toplevel .seditfile "Insert File..."} d] {
	catch {raise .seditfile}
	return
    }
    Widget_Message $d msg -aspect 1500 -text "Enter the pathname of the file to insert"
    set f [Widget_Frame $d name Entry]
    $f configure -bd 10
    set f [Widget_Frame $f rim Entry]
    $f configure -bd 2 -relief raised
    Widget_Label $f label {left} -text "Pathname:"
    set e [Widget_Entry $f name {right fill} -bg white -relief raised]
    bind $e <Return> [list SeditInsertFileCommit .seditfile $e $draft $t]
    focus $e

    set f [Widget_Frame $d but Entry]
    $f configure -bd 10
    Widget_AddBut $f cancel "Cancel" \
	[list SeditInsertFileDone .seditfile $t] left
    Widget_AddBut $f ok "OK" \
	[list SeditInsertFileCommit .seditfile $e $draft $t] right
}
proc SeditInsertFileCommit { top entry draft t} {
    set name [$entry get]
    SeditInsertFile $draft $t $name
    SeditInsertFileDone $top $t
    return
}
proc SeditInsertFileDone { top t} {
    focus $t
    destroy $top
}

proc SeditSave { draft t {hook {}} } {
    global sedit
    if [catch {
	set out [open $draft w]
	if {$sedit($t,format)} {
	    SeditFormatMail $t $out
	} else {
	    puts $out [$t get 1.0 end]
	}
	close $out
	SeditMsg $t "Message saved"
	if {$hook != {}} {
	    after 1 [list $hook $draft $t]
	}
    } err] {
	Exmh_Error "SeditSave $draft: $err"
	return 0
    }
    SeditMarkClean $t
    return 1
}
proc SeditFormatMail { t out } {
    global sedit exmh
    if {$sedit($t,enriched)} {
	SeditEnrichedExpand $t
	set quote 1
    } else {
	set quote 0
    }
    set inheaders 1
    scan [$t index end] "%d"  last
    for {set L 1} {$L <= $last} {incr L} {
	set line [$t get $L.0 $L.end]
	if {$inheaders} {
	    if {[regexp {^[ 	]*$} $line] || [regexp {^-+} $line]} {
		# Blank or empty line terminates headers
		# Leading --- terminates headers
		if ![info exists exmh(mhaltmsg)] {
		    puts $out "X-Mailer: exmh $exmh(version)"
		}
		SeditMimeHeaders $t $out
		set inheaders 0
	    }
	}
	set limit 80
	set cutoff 50
	set continuation 0
	while {[string length $line] > $limit} {
	    for {set c [expr $limit-1]} {$c >= $cutoff} {incr c -1} {
		set char [string index $line $c]
		if {$char == " " || $char == "\t"} {
		    break
		}
		if {$char == ">"} {	;# Hack for enriched formatting
		    break
		}
	    }
	    if {$c < $cutoff} {
		set c [expr $limit-1]
	    }
	    set newline [string range $line 0 $c]
	    if {! $continuation} {
		puts $out $newline
	    } else {
		puts $out \ $newline
	    }
	   incr c
	    set line [string range $line $c end]
	    if {$inheaders} {
		set continuation 1
		set limit 78
	    }
	}
	if {$continuation} {
	    puts $out \ $line
	} else {
	    if {$quote && !$inheaders} {
		# enriched requires two newlines for each one.
		puts $out $line\n
	    } else {
		puts $out $line
	    }
	}
    }
}

proc SeditMarkSent { t } {
    global sedit
    set sedit($t,sent) 1
}
proc SeditNotSent { t } {
    global sedit
    return [expr {! $sedit($t,sent)}]
}

proc SeditTextBindings { draft t } {
    global sedit
    bind $t $sedit(key,sendMsg) [list SeditSend $draft $t]
    bind Entry $sedit(key,sendMsg) { }
}
