#############################################################################
#
# alias.tcl: This file contains procedures related to the processing of the 
#            MH aliases
#
# AliasWindow {}
#       Produce a window with a list of the aliases, and a set of buttons
#       to edit, add and delete aliases. Also a button to compose a message
#       to the selected alias.
#
# DeleteAlias {}
#       Delete an alias from the list
#
# EditAlias {}
#       Change the address for the alias in Addresses
#
# InsertAlias {}
#       Insert an alias before the selected place. If no selection, append 
#       it to the end of the list
#
# ShowAlias {} 
#       Show the selected alias and the Address in a window
#
# SaveAlias {}
#       Save the aliases in the alias file, backing up the old one
#
# SearchAlias {Alias} {resolve 0}
#       Search variable Aliases for the alias, and return the Address. 
#       resolve is a boolean telling to resolve the address further over
#       other aliases, to the real address. all is a boolean telling
#       the procedure if it should output the path of the search.
#       TODO: add a PREF so we can control substitution of aliases
#
# ShowAliases {}
#       Display Aliases var in the listbox
#
##############################################################################

proc AliasWindow {} {
    global Changed BM AliasList
    set Changed 0

    catch {destroy .alias}
    toplevel .alias -class Config
    wm title .alias "TkMH Aliases"
    wm iconbitmap .alias @$BM(alias)
    
    scrollbar .alias.s -command {.alias.l yview}
    listbox .alias.l -yscroll {.alias.s set} -geometry 15x15

    frame .alias.b
    button .alias.b.done -text "Done" -command \
      {catch {destroy .aa}
       if $Changed {
         if [Confirm "Save Changes?"] {SaveAlias; destroy .alias} \
         else {destroy .alias}
       } else {destroy .alias}
       AdaptAliMenus
      }
    button .alias.b.edit -text "Edit" -command \
       {if {[set Alias [GetListboxSel .alias.l 1]] != ""} {
           FlashSel .alias.l 2; InsertAlias $Alias $Alias}}
    button .alias.b.delete -text "Delete" -command DeleteAlias
    button .alias.b.add -text "Insert New" -command InsertAlias
    button .alias.b.save -text "Save" -command {SaveAlias; AdaptAliMenus}
    button .alias.b.help -text "Help" -command {Help Configuration/Aliases}
    button .alias.b.reset -text "Reset" -command \
       {BuildAliasList; DisplayAliases; AdaptAliMenus; set Changed 0}

    pack append .alias.b \
      .alias.b.edit {top expand fill} \
      .alias.b.add {top expand fill} \
      .alias.b.delete {top expand fill} \
      .alias.b.help {top expand fill} \
      .alias.b.reset {top expand fill} \
      .alias.b.save {top expand fill} \
      .alias.b.done {top expand fill}
    pack append .alias \
      .alias.l {left expand fill} \
      .alias.s {left filly} \
      .alias.b {left filly}

    bind .alias.l <Double-1> {.alias.b.edit invoke}

    DisplayAliases
}

proc DeleteAlias {} {
    global Aliases Addresses Changed

    if {[set Sels [.alias.l curselection]] == {}} {return}
    FlashSel .alias.l 2
    set first [lindex $Sels 0]
    set last [lindex $Sels [expr [llength $Sels] - 1]]
    set Aliases [lreplace $Aliases $first $last]
    set Addresses [lreplace $Addresses $first $last]
    set Changed 1
    DisplayAliases
}

proc DisplayAliases {} {
    global Aliases

    set curYview [lindex [.alias.s get] 2]
    .alias.l delete 0 end
    foreach Alias $Aliases {.alias.l insert end $Alias}
    .alias.l yview $curYview
}

proc InsertAlias {{Title Alias} {AliasIn {}}} {
    global FONT
    global Aliases Addresses Changed Alias Address

    catch {destroy .aa}
    update
    toplevel .aa
    frame .aa.b
    button .aa.b.cancel -text "Cancel" -command {destroy .aa}
    button .aa.b.ok -text "OK" -command \
      {set Alias [.aa.f2.e1 get]; set Address [.aa.f2.e2 get]; destroy .aa}
    frame .aa.f1
    label .aa.f1.l1 -text "Alias:" -anchor e -font $FONT(ProB)
    label .aa.f1.l2 -text "Address:" -anchor e -font $FONT(ProB)
    frame .aa.f2
    entry .aa.f2.e1 -width 40 -font $FONT(ProR)
    entry .aa.f2.e2 -width 40 -font $FONT(ProR)
    pack append .aa.b \
      .aa.b.ok {top expand fill} \
      .aa.b.cancel {top expand fill}
    pack append .aa.f1 \
      .aa.f1.l1 {top fill} \
      .aa.f1.l2 {top fill}
    pack append .aa.f2 \
      .aa.f2.e1 {top fill} \
      .aa.f2.e2 {top fill}
    pack append .aa \
      .aa.f1 {left padx 20 pady 30} \
      .aa.f2 {left padx 20 pady 30} \
      .aa.b {left filly}

     
    wm title .aa $Title
    bind .aa.f2.e1 <Space> {}
    focus .aa.f2.e1
    grab .aa
    set Alias $AliasIn
    if {$Alias != {}} {
        set Index [lsearch $Aliases $Alias]
        set Address [lindex $Addresses $Index]
        .aa.f2.e1 delete 0 end
        .aa.f2.e2 delete 0 end
        .aa.f2.e1 insert 0 $Alias
        .aa.f2.e2 insert 0 $Address
        set Aliases [lreplace $Aliases $Index $Index]
        set Addresses [lreplace $Addresses $Index $Index]
    } else {
        if {[set Index [lindex [.alias.l curselection] 0]] == ""} {
            set Index [expr [llength $Aliases]]
        }
    }
    tkwait window .aa
    if {$Alias == ""} return
    if {$Address == ""} return
    set Aliases [linsert $Aliases $Index $Alias]
    set Addresses [linsert $Addresses $Index $Address]
    set Changed 1
    if [winfo exists .alias] {DisplayAliases}
    update
}

proc SaveAlias {} {
    global cmd MH
    global Aliases Addresses Changed ALIASREST

    if {[file exists [RF $MH(Path)/$MH(AliasFile)]] && $MH(AliasFile) != {}} {
        eval exec $cmd(mv) [RF $MH(Path)/$MH(AliasFile)] \
                           [RF $MH(Path)/$MH(AliasFile).old]
    }
    if {$MH(AliasFile) == {}} {
        set MH(AliasFile) [Ask "Name of AliasFile:"]
        if {$MH(AliasFile) == {}} {set MH(AliasFile) tkmh.aliases; ConfMH:Write}
    }
    set Fhandle [open [RF $MH(Path)/$MH(AliasFile)] w]
    set numlist [llength $Aliases]
    for {set i 0} {$i < $numlist} {incr i} {
        puts -nonewline $Fhandle [lindex $Aliases $i]
        puts -nonewline $Fhandle ":\t" 
        puts $Fhandle [lindex $Addresses $i]
    }
    foreach Line $ALIASREST {
        puts $Fhandle $Line
    }
    close $Fhandle
    set Changed 0
}

proc SearchAlias {Alias {resolve 0}} {
    global Aliases Addresses cmd MH

    set i [lsearch $Aliases $Alias]
    set Address [lindex $Addresses $i]
    if $resolve {
        switch [string index $Address 0] {
        < -
        + -
        = -
        * {set Address [eval exec $MH(bindir)/ali -nouser -nolist $Alias]}
        default 
           { if {[set i [lsearch $Aliases $Address]] != -1} {
             set Address [SearchAlias $Address 1]}
           }
        }
    }
    return $Address
}
